/*
 * ewfverify
 * Verifies the integrity of the media data within the EWF file
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "libewf.h"

#include "ewf_md5hash.h"

/* Prints the executable usage information
 */
void usage( void )
{
	fprintf( stderr, "Usage: ewfverify [ -hvV ] ewf_files\n" );

	fprintf( stderr, "\t-h: shows this help\n" );
	fprintf( stderr, "\t-v: verbose output to stderr\n" );
	fprintf( stderr, "\t-V: print version\n" );
}

/* Prints the executable version information
 */
void version( void )
{
	fprintf( stderr, "Copyright 2006 Joachim Metz, Hoffmann Investigations <%s> and contributors.\n", PACKAGE_BUGREPORT );
	fprintf( stderr, "This is free software; see the source for copying conditions. There is NO\n" );
	fprintf( stderr, "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" );
}

/* The main program
 */
int main( int argc, const char **argv )
{
	LIBEWF_HANDLE *handle                    = NULL;
	LIBEWF_STRING *stored_md5hash_string     = NULL;
	LIBEWF_STRING *calculated_md5hash_string = NULL;
	uint64_t sector                          = 0;
	uint32_t sector_count                    = 0;
	uint32_t iterator                        = 0;
	int match                                = 0;
	int option                               = 0;

	fprintf( stderr, "ewfverify version: %s\n\n", VERSION );

	while( ( option = getopt( argc, (char **) argv, "hvV" ) ) > 0 )
	{
		switch( option )
		{
			case '?':
			default:
				fprintf( stderr, "Invalid argument: %s\n", argv[ optind ] );

				usage();

				return( EXIT_FAILURE );

			case 'h':
				usage();

				return( EXIT_SUCCESS );

			case 'v':
				libewf_verbose = 1;

				break;

			case 'V':
				version();

				return( EXIT_SUCCESS );
		}
	}
	if( optind == argc )
	{
		fprintf( stderr, "Missing EWF image file(s).\n" );

		usage();

		return( EXIT_FAILURE );
	}
	handle = libewf_open( &argv[ optind ], ( argc - optind ), LIBEWF_OPEN_READ );

	if( handle == NULL )
	{
		fprintf( stderr, "Unable to open EWF image file(s).\n" );

		return( EXIT_FAILURE );
	}
	stored_md5hash_string     = libewf_data_md5hash( handle );
	calculated_md5hash_string = libewf_calculate_md5hash( handle );

	if( handle->crc_error_count > 0 )
	{
		fprintf( stderr, "Sector validation errors:\n" );
		fprintf( stderr, "\ttotal amount: %" PRIu32 "\n", handle->crc_error_count );

		for( iterator = 0; iterator < handle->crc_error_count; iterator++ )
		{
			sector       = handle->crc_error_sectors[ iterator ].sector;
			sector_count = handle->crc_error_sectors[ iterator ].sector_count;

			fprintf( stderr, "\tin sector(s): %" PRIu64 " - %" PRIu64 " amount: %" PRIu32 "\n", sector, (sector + sector_count), sector_count );
		}
		fprintf( stderr, "\n" );
	}
	libewf_close( handle );

	if( stored_md5hash_string == NULL )
	{
		fprintf( stderr, "MD5 hash stored in file:\tN/A\n" );
	}
	else
	{
		fprintf( stderr, "MD5 hash stored in file:\t%s\n", stored_md5hash_string );
	}
	if( calculated_md5hash_string == NULL )
	{
		fprintf( stderr, "MD5 hash calculated over data:\tN/A\n" );
	}
	else
	{
		fprintf( stderr, "MD5 hash calculated over data:\t%s\n", calculated_md5hash_string );
	}
	match = ( libewf_common_strncmp( stored_md5hash_string, calculated_md5hash_string, 32 ) == 0 );

	if( stored_md5hash_string != NULL )
	{
		libewf_string_free( stored_md5hash_string );
	}
	if( calculated_md5hash_string != NULL )
	{
		libewf_string_free( calculated_md5hash_string );
	}
	if( match )
	{
		fprintf( stderr, "\newfverify: SUCCESS\n" );

		return( EXIT_SUCCESS );
	}
	else
	{
		fprintf( stderr, "\newfverify: FAILURE\n" );

		return( EXIT_FAILURE );
	}
}

