/*
 * Common code for libewf - wraps external function calls
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "libewf_definitions.h"
#include "libewf_notify.h"

/* Function to wrap malloc
 */
void *libewf_common_alloc( uint32_t size )
{
	if( size > INT32_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_alloc: invalid size only values below 2^32 are supported.\n" );

		return( NULL );
	}
	return( malloc( (size_t) size ) );
}

/* Function to wrap realloc
 */
void *libewf_common_realloc( void *buffer, uint32_t size )
{
	void *reallocated_buffer = NULL;

	if( buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc: invalid buffer.\n" );

		return( NULL );
	}
	if( size > INT32_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc: invalid size only values below 2^32 are supported.\n" );

		return( NULL );
	}
	reallocated_buffer = realloc( buffer, (size_t) size );

	if( reallocated_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc: unable to reallocate buffer.\n" );

		return( NULL );
	}
	return( reallocated_buffer );
}

/* Function to wrap free
 */
void libewf_common_free( void *buffer )
{
	if( buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_free: invalid buffer.\n" );
	}
	free( buffer );
}

/* Function to wrap memset
 */
void *libewf_common_memset( void *buffer, int value, uint32_t count )
{
	if( buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_memset: invalid buffer.\n" );

		return( NULL );
	}
	if( count > INT32_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_memset: invalid count only values below 2^32 are supported.\n" );

		return( NULL );
	}
	return( memset( buffer, value, (size_t) count ) );
}

/* Function to wrap memcpy
 */
void *libewf_common_memcpy( void *destination, const void *source, uint32_t size )
{
	if( source == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_memcpy: invalid source.\n" );

		return( NULL );
	}
	if( destination == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_memcpy: invalid destination.\n" );

		return( NULL );
	}
	return( memcpy( destination, source, (size_t) size ) );
}

/* Function to wrap memcmp
 */
int32_t libewf_common_memcmp( const void *buffer1, const void *buffer2, uint32_t size )
{
	if( ( buffer1 == NULL ) || ( buffer2 == NULL ) )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_memcmp: invalid buffer.\n" );

		return( -1 );
	}
	return( (int32_t) memcmp( buffer1, buffer2, (size_t) size ) );
}

/* Function to wrap strdup
 */
char *libewf_common_strdup( const char *string )
{
	if( string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_strdup: invalid string.\n" );

		return( NULL );
	}
	return( strdup( string ) );
}

/* Function to wrap strlen
 */
uint32_t libewf_common_strlen( const char *string )
{
	if( string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_strlen: invalid string.\n" );

		return( 0 );
	}
	return( (int32_t) strlen( string ) );
}

/* Function to wrap strchr
 */
char *libewf_common_strchr( const char *string, uint8_t character )
{
	if( string == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_strchr: invalid string.\n" );

		return( NULL );
	}
	return( strchr( string, (int) character ) );
}

/* Function to wrap strncmp
 */
uint8_t libewf_common_strncmp( const char *string1, const char *string2, uint32_t length )
{
	if( ( string1 == NULL ) || ( string2 == NULL ) )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_strncmp: invalid string.\n" );

		return( -1 );
	}
	return( (uint8_t) strncmp( string1, string2, (size_t) length ) );
}

/* Function to wrap open
 */
int libewf_common_open( const char *filename, int flags, mode_t mode )
{
	int file_descriptor = 0;

	if( filename == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_open: invalid filename.\n" );

		return( -1 );
	}
	file_descriptor = open( filename, flags, mode );

#if HAVE_STRERROR
	if( file_descriptor <= -1 )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_open: error opening file %s with failure: %s.\n", filename, strerror( errno ) );
	}
#endif
	return( file_descriptor );
}

/* Function to wrap read
 */
int64_t libewf_common_read( int file_descriptor, void *buffer, uint64_t count )
{
	int64_t read_count = 0;

	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_read: invalid file descriptor.\n" );

		return( -1 );
	}
	if( buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_read: invalid buffer.\n" );

		return( -1 );
	}
	if( count > INT64_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_read: invalid count only values below 2^64 are supported.\n" );

		return( -1 );
	}
	read_count = (int64_t) read( file_descriptor, buffer, (size_t) count );

#if HAVE_STRERROR
	if( read_count <= -1 )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_read: error reading from file with failure: %s.\n", strerror( errno ) );
	}
#endif
	return( read_count );
}

/* Function to wrap lseek
 */
int64_t libewf_common_lseek( int file_descriptor, uint64_t offset, int whence )
{
	int64_t seek_offset = 0;

	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_lseek: invalid file descriptor.\n" );

		return( -1 );
	}
	if( offset > INT64_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_lseek: invalid offset only values below 2^64 are supported.\n" );

		return( -1 );
	}
	seek_offset = (int64_t) lseek( file_descriptor, (off_t) offset, whence );

#if HAVE_STRERROR
	if( seek_offset <= -1 )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_lseek: error seeking in file with failure: %s.\n", strerror( errno ) );
	}
#endif
	return( seek_offset );
}

/* Function to wrap write
 */
int64_t libewf_common_write( int file_descriptor, const void *buffer, uint64_t count )
{
	int64_t write_count = 0;

	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_write: invalid file descriptor: %d.\n", file_descriptor );

		return( -1 );
	}
	if( buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_write: invalid buffer.\n" );

		return( -1 );
	}
	if( count > INT64_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_write: invalid count only values below 2^64 are supported.\n" );

		return( -1 );
	}
	write_count = (int64_t) write( file_descriptor, buffer, (size_t) count );

#if HAVE_STRERROR
	if( write_count <= -1 )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_common_write: error writing to file with failure: %s.\n", strerror( errno ) );
	}
#endif
	return( write_count );
}

/* Function to wrap close
 */
void libewf_common_close( int file_descriptor )
{
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_close: invalid file descriptor.\n" );

		return;
	}
	close( file_descriptor );
}

/* Function to allocated wiped memory
 */
void *libewf_common_alloc_cleared( uint32_t size, int clear_value )
{
	void *allocated_buffer = NULL;
	void *cleared_buffer   = NULL;

	allocated_buffer = libewf_common_alloc( size );

	if( allocated_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_alloc_cleared: unable to allocate buffer.\n" );

		return( NULL );
	}
	cleared_buffer = libewf_common_memset( allocated_buffer, clear_value, size );

	if( cleared_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_alloc_cleared: unable to clear buffer.\n" );

		libewf_common_free( allocated_buffer );

		return( NULL );
	}
	return( allocated_buffer );
}

/* Function to reallocated fully wiped memory
 */
void *libewf_common_realloc_full_cleared( void *buffer, uint32_t previous_size, uint32_t new_size, int clear_value )
{
	void *reallocated_buffer = NULL;
	void *cleared_buffer     = NULL;

	if( previous_size > INT32_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_full_cleared: invalid size only values below 2^32 are supported.\n" );

		return( NULL );
	}
	if( new_size <= previous_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_full_cleared: new size must be greater than previous size.\n" );

		return( NULL );
	}
	reallocated_buffer = libewf_common_realloc( buffer, (size_t) new_size );

	if( reallocated_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_full_cleared: unable to reallocate buffer.\n" );

		return( NULL );
	}
	cleared_buffer = libewf_common_memset( reallocated_buffer, clear_value, new_size );

	if( cleared_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_full_cleared: unable to clear buffer.\n" );

		libewf_common_free( reallocated_buffer );

		return( NULL );
	}
	return( reallocated_buffer );
}

/* Function to reallocated wiped newly allocated memory
 */
void *libewf_common_realloc_new_cleared( void *buffer, uint32_t previous_size, uint32_t new_size, int clear_value )
{
	void *reallocated_buffer = NULL;
	void *cleared_buffer     = NULL;

	if( previous_size > INT32_MAX )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_new_cleared: invalid size only values below 2^32 are supported.\n" );

		return( NULL );
	}
	if( new_size <= previous_size )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_new_cleared: new size must be greater than previous size.\n" );

		return( NULL );
	}
	reallocated_buffer = libewf_common_realloc( buffer, (size_t) new_size );

	if( reallocated_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_new_cleared: unable to reallocate buffer.\n" );

		return( NULL );
	}
	cleared_buffer = libewf_common_memset( ( reallocated_buffer + previous_size ), clear_value, ( new_size - previous_size ) );

	if( cleared_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_realloc_new_cleared: unable to clear buffer.\n" );

		libewf_common_free( reallocated_buffer );

		return( NULL );
	}
	return( reallocated_buffer );
}

/* Check for empty block, a block that contains the same value for every byte
 * Returns 1 if block is empty, or 0 otherwise
 */
uint8_t libewf_common_test_empty_block( uint8_t *block_buffer, uint32_t block_size )
{
	int32_t iterator = 0;

	if( block_buffer == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_common_test_empty_block: invalid block buffer.\n" );

		return( 0 );
	}
	for( iterator = 1; iterator < block_size; iterator++ )
	{
		if( block_buffer[ 0 ] != block_buffer[ iterator ] )
		{
			return( 0 );
		}
	}
	return( 1 );
}

