/*
 * libewf file reading
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <stdlib.h>

#include "ewf_compress.h"
#include "ewf_crc.h"
#include "ewf_data.h"
#include "ewf_error2.h"
#include "ewf_md5hash.h"
#include "ewf_file_header.h"
#include "ewf_hash.h"
#include "ewf_header.h"
#include "ewf_header2.h"
#include "ewf_ltree.h"
#include "ewf_section.h"
#include "ewf_sectors.h"
#include "ewf_volume.h"
#include "ewf_volume_smart.h"
#include "ewf_table.h"

#include "libewf_definitions.h"
#include "libewf_debug.h"
#include "libewf_endian.h"
#include "libewf_header_values.h"
#include "libewf_notify.h"
#include "libewf_md5.h"
#include "libewf_segment_table.h"

/* Reads and processes a section start
 * Returns the section start, or NULL on error
 */
EWF_SECTION *libewf_section_start_read( LIBEWF_HANDLE *handle, int file_descriptor )
{
	EWF_SECTION *section    = NULL;
	EWF_CRC *calculated_crc = NULL;
	EWF_CRC stored_crc      = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_read: invalid handle.\n" );

		return( NULL );
	}
	if( file_descriptor <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_read: invalid file descriptor.\n" );

		return( NULL );
	}
	section = ewf_section_read( file_descriptor );

	if( section == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_read: unable to read section start.\n" );

		return( NULL );
	}
	LIBEWF_VERBOSE_EXEC( ewf_section_fprint( stderr, section ); );

#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( section->padding, 40 ); );
#endif

	calculated_crc = ewf_crc_calculate( (void *) section, ( EWF_SECTION_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_read: unable to calculate CRC.\n" );

		ewf_section_free( section );

		return( NULL );
	}
	stored_crc = libewf_endian_convert_32bit( section->crc );

	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_section_free( section );
			ewf_crc_free( calculated_crc );

			return( NULL );
		}
	}
	ewf_crc_free( calculated_crc );

	return( section );
}

/* Write a section start to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_start_write( LIBEWF_HANDLE *handle, int file_descriptor, char *section_type, uint32_t section_data_size, uint32_t start_offset )
{
	EWF_SECTION *section       = NULL;
	void *data_set             = NULL;
	uint64_t section_size      = 0;
	uint64_t section_offset    = 0;
	uint32_t section_type_size = 0;
	int32_t write_count        = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: invalid handle.\n" );

		return( -1 );
	}
	section_type_size = libewf_common_strlen( section_type );

	if( section_type_size == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: section type is empty.\n" );

		return( -1 );
	}
	if( section_type_size >= 16 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: section type is too long.\n" );

		return( -1 );
	}
	section = ewf_section_alloc();

	if( section == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: unable to create section.\n" );

		return( -1 );
	}
	data_set = libewf_common_memcpy( (void *) section->type, (void *) section_type, section_type_size );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: unable to set section type.\n" );

		ewf_section_free( section );

		return( -1 );
	}
	section_size   = EWF_SECTION_SIZE + section_data_size;
	section_offset = start_offset + section_size;

	libewf_endian_revert_64bit( section_size, section->size );
	libewf_endian_revert_64bit( section_offset, section->next );

	write_count = ewf_section_write( section, file_descriptor );

	ewf_section_free( section );

	if( write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_start_write: unable to write section to file.\n" );

		return( -1 );
	}
	return( write_count );
}

/* Reads a header section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_header_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_HEADER *header = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_read: invalid handle.\n" );

		return( -1 );
	}
	header = ewf_header_read( file_descriptor, &size );

	if( header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_read: unable to read header.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_section_header_read: Header:\n" );
	LIBEWF_VERBOSE_EXEC( ewf_header_fprint( stderr, header ); );

	if( libewf_handle_is_set_header( handle ) == 0 )
	{
		libewf_handle_set_header( handle, header );
	}
	else
	{
		ewf_header_free( header );
	}
	return( (int32_t) size );
}

/* Write a header section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_header_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset, EWF_HEADER *header, uint32_t size, int8_t compression_level )
{
	EWF_HEADER *compressed_header = NULL;
	int32_t section_write_count   = 0;
	int32_t header_write_count    = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_write: invalid handle.\n" );

		return( -1 );
	}
	if( header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_write: invalid header.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_section_header_write: Header:\n" );
	LIBEWF_VERBOSE_EXEC( ewf_header_fprint( stderr, header ); );

	compressed_header = ewf_header_compress( header, &size, compression_level );

	if( compressed_header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_write: unable to compress header.\n" );

		return( -1 );
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "header", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_write: unable to write section to file.\n" );

		ewf_header_free( compressed_header );

		return( -1 );
	}
	header_write_count = ewf_header_write( compressed_header, file_descriptor, size );

	ewf_header_free( compressed_header );

	if( header_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header_write: unable to write header to file.\n" );

		return( -1 );
	}
	return( section_write_count + header_write_count );
}

/* Reads a header2 section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_header2_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_HEADER *header2 = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_read: invalid handle.\n" );

		return( -1 );
	}
	header2 = ewf_header2_read( file_descriptor, size );

	if( header2 == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_read: unable to read header2.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_section_header2_read: Header2:\n" );
	LIBEWF_VERBOSE_EXEC( ewf_header_fprint( stderr, header2 ); );

	if( libewf_handle_is_set_header2( handle ) == 0 )
	{
		libewf_handle_set_header2( handle, header2 );
	}
	else
	{
		ewf_header_free( header2 );
	}
	return( (int32_t) size );
}

/* Write a header2 section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_header2_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset, EWF_HEADER *header, uint32_t size, int8_t compression_level )
{
	EWF_HEADER *utf16_header      = NULL;
	EWF_HEADER *compressed_header = NULL;
	uint32_t size_utf16           = 0;
	int32_t section_write_count   = 0;
	int32_t header_write_count    = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: invalid handle.\n" );

		return( -1 );
	}
	if( header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: invalid header.\n" );

		return( -1 );
	}
	LIBEWF_VERBOSE_PRINT( "libewf_section_header2_write: Header:\n" );
	LIBEWF_VERBOSE_EXEC( ewf_header_fprint( stderr, header ); );

	size_utf16   = ( size * 2 ) + 4;
	utf16_header = ewf_header2_convert_ascii_to_utf16( header, size, EWF_HEADER2_LITTLE_ENDIAN );

	if( utf16_header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: unable to convert header to UTF16.\n" );

		return( -1 );
	}
	compressed_header = ewf_header_compress( utf16_header, &size_utf16, compression_level );

	if( compressed_header == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: unable to compress header.\n" );

		ewf_header_free( utf16_header );

		return( -1 );
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "header2", size_utf16, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: unable to write section to file.\n" );

		ewf_header_free( utf16_header );
		ewf_header_free( compressed_header );

		return( -1 );
	}
	header_write_count = ewf_header_write( compressed_header, file_descriptor, size_utf16 );

	ewf_header_free( utf16_header );
	ewf_header_free( compressed_header );

	if( header_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_header2_write: unable to write header to file.\n" );

		return( -1 );
	}
	return( section_write_count + header_write_count );
}

/* Reads an EWF-S01 (SMART) volume section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_volume_s01_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_VOLUME_SMART *volume_smart = NULL;
	EWF_CRC *calculated_crc        = NULL;
	EWF_CRC stored_crc             = 0;
	int32_t bytes_per_chunk        = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: invalid handle.\n" );

		return( -1 );
	}
	if( size != EWF_VOLUME_SMART_SIZE )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: mismatch in section volume size.\n" );

		return( -1 );
	}
	volume_smart = ewf_volume_smart_read( file_descriptor );

	if( volume_smart == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: unable to read volume.\n" );

		return( -1 );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume_smart->type, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume_smart->unknown2, 20 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume_smart->unknown3, 45 ); );
#endif

	calculated_crc = ewf_crc_calculate( (void *) volume_smart, ( EWF_VOLUME_SMART_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: unable to calculate CRC.\n" );

		ewf_volume_smart_free( volume_smart );

		return( -1 );
	}
	stored_crc      = libewf_endian_convert_32bit( volume_smart->crc );
	bytes_per_chunk = ewf_volume_smart_calculate_chunk_size( volume_smart );

	if( bytes_per_chunk <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: unable to calculate chunk size - using default.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
		bytes_per_chunk = EWF_MINIMUM_CHUNK_SIZE;
	}
	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_s01_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	handle->chunk_count       = libewf_endian_convert_32bit( volume_smart->chunk_count );
	handle->sectors_per_chunk = libewf_endian_convert_32bit( volume_smart->sectors_per_chunk );
	handle->bytes_per_sector  = libewf_endian_convert_32bit( volume_smart->bytes_per_sector );
	handle->sector_count      = libewf_endian_convert_32bit( volume_smart->sector_count );
	handle->chunk_size        = bytes_per_chunk;

	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_s01_read: This volume has %" PRIu32 " chunks of %" PRIi32 " bytes each, CRC %" PRIu32 " (%" PRIu32 ").\n", handle->chunk_count, bytes_per_chunk, stored_crc, *calculated_crc );
	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_s01_read: This volume has %" PRIu32 " sectors of %" PRIi32 " bytes each.\n", handle->sector_count, handle->bytes_per_sector );

	ewf_volume_smart_free( volume_smart );
	ewf_crc_free( calculated_crc );

	return( (int32_t) size );
}

/* Write an EWF-S01 (SMART) volume section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_volume_s01_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset )
{
	EWF_VOLUME_SMART *volume    = NULL;
	int32_t section_write_count = 0;
	int32_t volume_write_count  = 0;
	uint32_t size               = EWF_VOLUME_SMART_SIZE;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_smart_write: invalid handle.\n" );

		return( -1 );
	}
	volume = ewf_volume_smart_alloc();

	if( volume == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_smart_write: unable to create volume.\n" );

		return( -1 );
	}
	libewf_endian_revert_32bit( handle->chunk_count, volume->chunk_count );
	libewf_endian_revert_32bit( handle->sectors_per_chunk, volume->sectors_per_chunk );
	libewf_endian_revert_32bit( handle->bytes_per_sector, volume->bytes_per_sector );
	libewf_endian_revert_32bit( handle->sector_count, volume->sector_count );

	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_smart_write: chunk_count: %" PRIu32 ", sectors_per_chunk: %" PRIu32 ", bytes_per_sector: %" PRIu32 ", sector_count: %" PRIu32 ".\n", volume->chunk_count, volume->sectors_per_chunk, volume->bytes_per_sector, volume->sector_count );

	if( handle->format == LIBEWF_FORMAT_SMART )
	{
		volume->signature[ 0 ] = 'S';
		volume->signature[ 1 ] = 'M';
		volume->signature[ 2 ] = 'A';
		volume->signature[ 3 ] = 'R';
		volume->signature[ 4 ] = 'T';
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "volume", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_smart_write: unable to write section to file.\n" );

		ewf_volume_smart_free( volume );

		return( -1 );
	}
	volume_write_count = ewf_volume_smart_write( volume, file_descriptor );

	ewf_volume_smart_free( volume );

	if( volume_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_smart_write: unable to write volume to file.\n" );

		return( -1 );
	}
	return( section_write_count + volume_write_count );
}

/* Reads an EWF-E01 (EnCase) volume section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_volume_e01_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_VOLUME *volume      = NULL;
	EWF_CRC *calculated_crc = NULL;
	void *data_set          = NULL;
	EWF_CRC stored_crc      = 0;
	int32_t bytes_per_chunk = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: invalid handle.\n" );

		return( -1 );
	}
	if( size != EWF_VOLUME_SIZE )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: mismatch in section volume size.\n" );

		return( -1 );
	}
	volume = ewf_volume_read( file_descriptor );

	if( volume == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: unable to read volume.\n" );

		return( -1 );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->type, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown2, 16 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown3, 3 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown4, 12 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown5, 3 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown6, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->unknown7, 963 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( volume->signature, 5 ); );
#endif

	calculated_crc = ewf_crc_calculate( (void *) volume, ( EWF_VOLUME_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: unable to calculate CRC.\n" );

		ewf_volume_free( volume );

		return( -1 );
	}
	stored_crc      = libewf_endian_convert_32bit( volume->crc );
	bytes_per_chunk = ewf_volume_calculate_chunk_size( volume );

	if( bytes_per_chunk <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: unable to calculate chunk size - using default.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
		bytes_per_chunk = EWF_MINIMUM_CHUNK_SIZE;
	}
	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	handle->chunk_count       = libewf_endian_convert_32bit( volume->chunk_count );
	handle->sectors_per_chunk = libewf_endian_convert_32bit( volume->sectors_per_chunk );
	handle->bytes_per_sector  = libewf_endian_convert_32bit( volume->bytes_per_sector );
	handle->sector_count      = libewf_endian_convert_32bit( volume->sector_count );
	handle->media_type        = volume->media_type;
	handle->compression_level = volume->compression_level;
	handle->chunk_size        = bytes_per_chunk;

	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_e01_read: this volume has %" PRIu32 " chunks of %" PRIi32 " bytes each, CRC %" PRIu32 " (%" PRIu32 ").\n", handle->chunk_count, bytes_per_chunk, stored_crc, *calculated_crc );
	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_e01_read: This volume has %" PRIu32 " sectors of %" PRIi32 " bytes each.\n", handle->sector_count, handle->bytes_per_sector );

	data_set = libewf_common_memcpy( (void *) handle->guid, (void *) volume->guid, 16 );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_e01_read: unable to set GUID.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	ewf_volume_free( volume );
	ewf_crc_free( calculated_crc );

	return( (int32_t) size );
}

/* Reads a volume section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_volume_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	int32_t count = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_read: invalid handle.\n" );

		return( -1 );
	}
	if( size == EWF_VOLUME_SMART_SIZE )
	{
		handle->ewf_format = EWF_FORMAT_S01;
		count              = libewf_section_volume_s01_read( handle, file_descriptor, size );
	}
	else if( size == EWF_VOLUME_SIZE )
	{
		handle->ewf_format = EWF_FORMAT_E01;
		count              = libewf_section_volume_e01_read( handle, file_descriptor, size );
	}
	else
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_read: mismatch in section data size.\n" );

		return( -1 );
	}
	if( ( count <= -1 ) || ( count != (int32_t) size ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_read: unable to read volume section.\n" );

		return( -1 );
	}
	if( handle->chunk_count == 0 )
	{
		handle->ewf_format = EWF_FORMAT_L01;
	}
	return( count );
}

/* Write an EWF-E01 (EnCase) volume section to file
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_volume_e01_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset )
{
	EWF_VOLUME *volume          = NULL;
	int32_t section_write_count = 0;
	int32_t volume_write_count  = 0;
	uint32_t size               = EWF_VOLUME_SIZE;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_write: invalid handle.\n" );

		return( -1 );
	}
	volume = ewf_volume_alloc();

	if( volume == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_write: unable to create volume.\n" );

		return( -1 );
	}
	libewf_endian_revert_32bit( handle->chunk_count, volume->chunk_count );
	libewf_endian_revert_32bit( handle->sectors_per_chunk, volume->sectors_per_chunk );
	libewf_endian_revert_32bit( handle->bytes_per_sector, volume->bytes_per_sector );
	libewf_endian_revert_32bit( handle->sector_count, volume->sector_count );

	LIBEWF_VERBOSE_PRINT( "libewf_section_volume_write: chunk_count: %" PRIu32 ", sectors_per_chunk: %" PRIu32 ", bytes_per_sector: %" PRIu32 ", sector_count: %" PRIu32 ".\n", volume->chunk_count, volume->sectors_per_chunk, volume->bytes_per_sector, volume->sector_count );

	if( handle->format == LIBEWF_FORMAT_FTK )
	{
		volume->type[ 0 ] = 0x01;
	}
	if( handle->format == LIBEWF_FORMAT_ENCASE5 )
	{
		volume->compression_level = handle->compression_level;
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "volume", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_write: unable to write section to file.\n" );

		ewf_volume_free( volume );

		return( -1 );
	}
	volume_write_count = ewf_volume_write( volume, file_descriptor );

	ewf_volume_free( volume );

	if( volume_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_volume_write: unable to write volume to file.\n" );

		return( -1 );
	}
	return( section_write_count + volume_write_count );
}

/* Fills the offset table
 * Returns a pointer to the instance, or NULL on error
 */
LIBEWF_OFFSET_TABLE *libewf_fill_offset_table( LIBEWF_OFFSET_TABLE *offset_table, EWF_TABLE_OFFSET *offsets, uint32_t chunk_amount, int file_descriptor, uint8_t error_tollerance )
{
	LIBEWF_OFFSET_TABLE *data_set = NULL;
	uint64_t chunk_size           = 0;
	uint64_t iterator             = 0;
	uint64_t current_offset       = 0;
	uint64_t next_offset          = 0;
	uint32_t size_of_chunks       = 0;
	uint32_t raw_offset           = 0;
	uint8_t compressed            = 0;

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_fill_offset_table: invalid offset table.\n" );

		return( NULL );
	}
	/* Correct the last offset, to fill the table it should point to the first empty entry
	 * the the last filled entry
	 */
	if( offset_table->last > 0 )
	{
		offset_table->last++;
	}
	size_of_chunks = chunk_amount + offset_table->last;

	/* Allocate additional entries in the offset table if needed
	 * - a single reallocation saves processing time
	 */
	if( offset_table->amount < size_of_chunks )
	{
		offset_table = libewf_offset_table_realloc( offset_table, size_of_chunks );

		if( offset_table == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_fill_offset_table: unable to reallocate offset table.\n" );

			return( NULL );
		}
	}
	/* Read the offsets from file
	 */
	raw_offset = libewf_endian_convert_32bit( offsets[ iterator ].offset );

	/* The size of the last chunk must be determined differently
	 */
	while( iterator < ( chunk_amount - 1 ) )
	{
		compressed     = raw_offset >> 31;
		current_offset = raw_offset & EWF_OFFSET_COMPRESSED_READ_MASK;
		raw_offset     = libewf_endian_convert_32bit( offsets[ iterator + 1 ].offset );
		next_offset    = raw_offset & EWF_OFFSET_COMPRESSED_READ_MASK;
		chunk_size     = next_offset - current_offset;

		data_set       = libewf_offset_table_set_values( offset_table, offset_table->last, file_descriptor, compressed, current_offset, chunk_size );

		if( data_set == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_fill_offset_table: unable to set value in offset table.\n" );

			if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
			{
				return( NULL );
			}
		}
		offset_table->last++;

		if( compressed == 0 )
		{
			LIBEWF_VERBOSE_PRINT( "libewf_fill_offset_table: uncompressed chunk %" PRIu64 " read with offset %" PRIu64 " and size %" PRIu64 ".\n", offset_table->last, current_offset, chunk_size );
		}
		else
		{
			LIBEWF_VERBOSE_PRINT( "libewf_fill_offset_table: compressed chunk %" PRIu64 " read with offset %" PRIu64 " and size %" PRIu64 ".\n", offset_table->last, current_offset, chunk_size );

		}
		current_offset = next_offset;

		iterator++;
	}
	raw_offset     = libewf_endian_convert_32bit( offsets[ iterator ].offset );
	compressed     = raw_offset >> 31;
	current_offset = raw_offset & EWF_OFFSET_COMPRESSED_READ_MASK;
	offset_table   = libewf_offset_table_set_values( offset_table, offset_table->last, file_descriptor, compressed, current_offset, 0 );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_fill_offset_table: unable to set value in offset table.\n" );

		if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( NULL );
		}
	}
	return( offset_table );
}

/* Calculate the last offset
 * Returns a pointer to the instance, or NULL on error
 */
LIBEWF_OFFSET_TABLE *libewf_calculate_last_offset( LIBEWF_OFFSET_TABLE *offset_table, LIBEWF_SECTION_LIST *section_list, int file_descriptor, uint8_t error_tollerance )
{
	LIBEWF_OFFSET_TABLE *data_set                 = NULL;
	LIBEWF_SECTION_LIST_ENTRY *section_list_entry = NULL;
	uint64_t last_offset                          = 0;
	uint64_t chunk_size                           = 0;

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_calculate_last_offset: invalid offset table.\n" );

		return( NULL );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_calculate_last_offset: invalid section list.\n" );

		return( NULL );
	}
	if( section_list->file_descriptor != file_descriptor )
	{
		LIBEWF_WARNING_PRINT( "libewf_calculate_last_offset: file descriptor does not match the one in the section list.\n" );

		return( NULL );
	}
	/*
	 * There is no indication how large the last chunk is. The only thing known is where it starts.
	 * However it can be determined where the next section starts within the file.
	 * The size of the last chunk is determined by subtracting the last offset from the offset of the section that follows.
	 */
	section_list_entry = section_list->first;
	last_offset        = offset_table->offset[ offset_table->last ];

	while( section_list_entry != NULL )
	{
#ifdef HAVE_DEBUG_OUTPUT
		LIBEWF_VERBOSE_PRINT( "libewf_calculate_last_offset: start offset: %" PRIu64 " last offset: %" PRIu64 ".\n", section_list_entry->start_offset, last_offset );
#endif

		if( ( section_list_entry->start_offset < last_offset ) && ( last_offset < section_list_entry->end_offset ) )
		{
			chunk_size = section_list_entry->end_offset - last_offset;
			data_set   = libewf_offset_table_set_values( offset_table, offset_table->last, file_descriptor, offset_table->compressed[ offset_table->last ], last_offset, chunk_size );

			if( data_set == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_calculate_last_offset: unable to set value in offset table.\n" );

				if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
				{
					return( NULL );
				}
			}
			LIBEWF_VERBOSE_PRINT( "libewf_calculate_last_offset: last chunk %" PRIu64 " calculated with offset %" PRIu64 " and size %" PRIu64 ".\n", ( offset_table->last + 1 ), last_offset, chunk_size );

			break;
		}
		section_list_entry = section_list_entry->next;
	}
	return( offset_table );
}

/* Reads an offset table
 * Returns a pointer to the instance, or NULL on error
 */
LIBEWF_OFFSET_TABLE *libewf_offset_table_read( LIBEWF_OFFSET_TABLE *offset_table, LIBEWF_SECTION_LIST *section_list, int file_descriptor, uint32_t size, uint8_t ewf_format, uint8_t error_tollerance )
{
	EWF_TABLE *table          = NULL;
	EWF_TABLE_OFFSET *offsets = NULL;
	EWF_CRC *calculated_crc   = NULL;
	EWF_CRC *stored_crc_read  = NULL;
	EWF_CRC stored_crc        = 0;
	uint32_t chunk_count      = 0;

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: invalid offset table.\n" );

		return( NULL );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: invalid section list.\n" );

		return( NULL );
	}
	table = ewf_table_read( file_descriptor );

	if( table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: unable to read table.\n" );

		return( NULL );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( table->padding, 16 ); );
#endif

	/* The table size contains the size of the CRC (4 bytes)
	 */
	calculated_crc = ewf_crc_calculate( (void *) table, ( EWF_TABLE_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: unable to calculate CRC.\n" );

		ewf_table_free( table );

		return( NULL );
	}
	stored_crc = libewf_endian_convert_32bit( table->crc );

	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );
		if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_crc_free( calculated_crc );
			ewf_table_free( table );

			return( NULL );
		}
	}
	chunk_count = libewf_endian_convert_32bit( table->chunk_count );

	LIBEWF_VERBOSE_PRINT( "libewf_offset_table_read: table is of size %" PRIu32 " chunks CRC %" PRIu32 " (%" PRIu32 ").\n", chunk_count, stored_crc, *calculated_crc );

	ewf_crc_free( calculated_crc );
	ewf_table_free( table );

	if( chunk_count <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_offset_table_read: table contains no offsets!.\n" );

		if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( NULL );
		}
	}
	else
	{
		offsets = ewf_table_offsets_read( file_descriptor, chunk_count );

		if( offsets == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_offset_table_read: unable to read table offsets.\n" );

			return( NULL );
		}
		if( ewf_format == EWF_FORMAT_E01 )
		{
			calculated_crc = ewf_crc_calculate( offsets, ( EWF_TABLE_OFFSET_SIZE * chunk_count ), 1 );

			if( calculated_crc == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_offset_table_read: unable to calculate CRC.\n" );

				ewf_table_offsets_free( offsets );

				return( NULL );
			}
			stored_crc_read = ewf_crc_read( file_descriptor );

			if( stored_crc_read == NULL )
			{
				LIBEWF_WARNING_PRINT( "libewf_offset_table_read: unable to read CRC from file descriptor.\n" );

				ewf_table_offsets_free( offsets );
				ewf_crc_free( calculated_crc );

				return( NULL );
			}
			if( *stored_crc_read != *calculated_crc )
			{
				LIBEWF_WARNING_PRINT( "libewf_offset_table_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", *stored_crc_read, *calculated_crc );

				if( error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
				{
					ewf_table_offsets_free( offsets );
					ewf_crc_free( calculated_crc );
					ewf_crc_free( stored_crc_read );

					return( NULL );
				}
			}
			ewf_crc_free( calculated_crc );
			ewf_crc_free( stored_crc_read );
		}
		offset_table = libewf_fill_offset_table( offset_table, offsets, chunk_count, file_descriptor, error_tollerance );
		offset_table = libewf_calculate_last_offset( offset_table, section_list, file_descriptor, error_tollerance );

		ewf_table_offsets_free( offsets );
	}
	return( offset_table );
}

/* Compare the offsets in tabel and table2 sections
 * Returns a 0 if table differ
 */
uint8_t libewf_compare_offset_tables( LIBEWF_OFFSET_TABLE *offset_table1, LIBEWF_OFFSET_TABLE *offset_table2 )
{
	uint64_t iterator = 0;

	if( ( offset_table1 == NULL ) || ( offset_table2 == NULL ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_compare_offset_tables: invalid offset table.\n" );

		return( 0 );
	}
	/* Check if table and table2 are the same
	 */
	if( offset_table1->amount != offset_table2->amount )
	{
		LIBEWF_VERBOSE_PRINT( "libewf_compare_offset_tables: offset tables differ in size.\n" );

		return( 0 );
	}
	else
	{
		for( iterator = 0; iterator < offset_table1->amount; iterator++ )
		{
			if( offset_table1->offset[ iterator ] != offset_table2->offset[ iterator ] )
			{
				LIBEWF_VERBOSE_PRINT( "libewf_compare_offset_tables: offset tables differ in offset for chunk: %" PRIu64 " (table1: %" PRIu64 ", table2: %" PRIu64 ").\n", iterator, offset_table1->offset[ iterator ], offset_table2->offset[ iterator ] );

				return( 0 );
			}
		}
	}
	return( 1 );
}

/* Reads a table section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_table_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size, LIBEWF_SECTION_LIST *section_list )
{
	LIBEWF_OFFSET_TABLE *offset_table = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_read: invalid handle.\n" );

		return( -1 );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_read: invalid section list.\n" );

		return( -1 );
	}
	if( handle->offset_table == NULL )
	{
		handle->offset_table = libewf_offset_table_alloc( handle->chunk_count );

		if( handle->offset_table == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_table_read: unable to create offset table.\n" );

			return( -1 );
		}
	}
	offset_table = libewf_offset_table_read( handle->offset_table, section_list, file_descriptor, size, handle->ewf_format, handle->error_tollerance );

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_read: unable to read offset table.\n" );

		return( -1 );
	}
	handle->offset_table = offset_table;

	return( (int32_t) size );
}

/* Reads an EWF-S01 table2 section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_table2_s01_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size, LIBEWF_SECTION_LIST *section_list )
{
	LIBEWF_OFFSET_TABLE *offset_table = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_s01_read: invalid handle.\n" );

		return( -1 );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_s01_read: invalid section list.\n" );

		return( -1 );
	}
	offset_table = libewf_offset_table_read( handle->offset_table, section_list, file_descriptor, size, handle->ewf_format, handle->error_tollerance );

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_s01_read: unable to read offset table.\n" );

		return( -1 );
	}
	handle->offset_table = offset_table;

	return( (int32_t) size );
}

/* Reads an EWF-E01 table2 section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_table2_e01_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size, LIBEWF_SECTION_LIST *section_list )
{
	LIBEWF_OFFSET_TABLE *offset_table = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_e01_read: invalid handle.\n" );

		return( -1 );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_e01_read: invalid section list.\n" );

		return( -1 );
	}
	if( handle->secondary_offset_table == NULL )
	{
		handle->secondary_offset_table = libewf_offset_table_alloc( handle->chunk_count );

		if( handle->secondary_offset_table == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_table2_e01_read: unable to create secondairy offset table.\n" );

			return( -1 );
		}
	}
	offset_table = libewf_offset_table_read( handle->secondary_offset_table, section_list, file_descriptor, size, handle->ewf_format, handle->error_tollerance );

	if( offset_table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_e01_read: unable to read offset table.\n" );

		return( -1 );
	}
	handle->secondary_offset_table = offset_table;

	if( libewf_compare_offset_tables( handle->offset_table, handle->secondary_offset_table ) == 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_e01_read: table1 and table2 differ.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
		/* TODO Try to correct the table
		 */
	}
	return( (int32_t) size );
}

/* Write a table or table2 section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_table_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset, EWF_TABLE_OFFSET *offsets, uint32_t offsets_amount, char *section_header, uint32_t additional_size )
{
	EWF_TABLE *table                  = NULL;
	int32_t section_write_count       = 0;
	int32_t table_write_count         = 0;
	int32_t table_offsets_write_count = 0;
	uint32_t size                     = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_write: invalid handle.\n" );

		return( -1 );
	}
	size  = EWF_TABLE_SIZE + ( EWF_TABLE_OFFSET_SIZE * offsets_amount ) + EWF_CRC_SIZE + additional_size;
	table = ewf_table_alloc();

	if( table == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_write: unable to create table.\n" );

		return( -1 );
	}
	libewf_endian_revert_32bit( offsets_amount, table->chunk_count );

	section_write_count = libewf_section_start_write( handle, file_descriptor, section_header, size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_write: unable to write section to file.\n" );

		ewf_table_free( table );

		return( -1 );
	}
	table_write_count         = ewf_table_write( table, file_descriptor );
	table_offsets_write_count = ewf_table_offsets_write( offsets, file_descriptor, offsets_amount );

	ewf_table_free( table );

	if( table_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_write: unable to write table to file.\n" );

		return( -1 );
	}
	if( table_offsets_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table_write: unable to write table offsets to file.\n" );

		return( -1 );
	}
	return( section_write_count + table_write_count + table_offsets_write_count );
}

/* Reads a table2 section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_table2_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size, LIBEWF_SECTION_LIST *section_list )
{
	int32_t count = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_read: invalid handle.\n" );

		return( -1 );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_read: invalid section list.\n" );

		return( -1 );
	}
	/* In the EWF-S01 format the table2 section is an addition on the table section
	 */
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		count = libewf_section_table2_s01_read( handle, file_descriptor, size, section_list );
	}
	/* In the EWF-E01 format the table2 section is a replicate of the table section
	 */
	else if( handle->ewf_format == EWF_FORMAT_E01 )
	{
		count = libewf_section_table2_e01_read( handle, file_descriptor, size, section_list );
	}
	if( ( count <= -1 ) || ( count != (int32_t) size ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_table2_read: unable to read table2 section.\n" );

		return( -1 );
	}
	return( count );
}

/* Reads a sectors section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_sectors_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_sectors_read: invalid handle.\n" );

		return( -1 );
	}
	/* In the EWF-E01 format the sectors section holds the actual data chunks
	 */
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_sectors_read: EWF-S01 format should not contain sectors section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	return( (int32_t) size );
}

/* Reads a ltree section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_ltree_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	LIBEWF_MD5_CTX md5;

	EWF_LTREE *ltree       = NULL;
	EWF_HEADER *tree_data  = NULL;
/*
	EWF_CRC calculated_crc = 0;
	EWF_CRC stored_crc     = 0;
*/
        EWF_MD5HASH *calculated_md5hash          = NULL;
        LIBEWF_STRING *calculated_md5hash_string = NULL;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_ltree_read: invalid handle.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_sectors_read: EWF-S01 format should not contain ltree section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	handle->ewf_format = EWF_FORMAT_L01;
/*
	if( size != EWF_LTREE_SIZE )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_ltree_read: mismatch in section data size.\n" );

		return( -1 );
	}
*/
	ltree = ewf_ltree_read( file_descriptor );

	LIBEWF_VERBOSE_EXEC( libewf_dump_data( ltree->unknown1, 16 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( ltree->tree_size, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( ltree->unknown2, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( ltree->unknown3, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( ltree->unknown4, 20 ); );

/*
	fprintf( stderr, "remaining lree size: %" PRIu32 " (0x%" PRIx32 ").\n", ( size - EWF_LTREE_SIZE ), ( size - EWF_LTREE_SIZE ) );
*/

/*
	LIBEWF_VERBOSE_EXEC( libewf_debug_read_section( handle, file_descriptor, ( size - EWF_LTREE_SIZE ) ); );
*/
	tree_data = ewf_tree_data_read( file_descriptor, ( size - EWF_LTREE_SIZE ) );

	LIBEWF_MD5_INIT( &md5 );

	LIBEWF_MD5_UPDATE( &md5, tree_data, ( size - EWF_LTREE_SIZE ) );

/*
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( tree_data, ( size - EWF_LTREE_SIZE ) ); );
*/
	LIBEWF_VERBOSE_EXEC( ewf_header_fprint( stderr, tree_data ); );

	ewf_header_free( tree_data );

        calculated_md5hash = ewf_md5hash_alloc();

        if( calculated_md5hash == NULL )
        {
                LIBEWF_WARNING_PRINT( "libewf_section_ltree_read: unable to create MD5 hash.\n" );

                return( -1 );
        }
        LIBEWF_MD5_FINAL( calculated_md5hash, &md5 );

        calculated_md5hash_string = ewf_md5hash_to_string( calculated_md5hash );

        ewf_md5hash_free( calculated_md5hash );

        if( calculated_md5hash_string == NULL )
        {
                LIBEWF_WARNING_PRINT( "libewf_section_ltree_read: unable to create MD5 hash string.\n" );

                return( -1 );
        }
	LIBEWF_VERBOSE_PRINT( "libewf_section_ltree_read: %s.\n", calculated_md5hash_string );

	libewf_string_free( calculated_md5hash_string );

	return( (int32_t) size );
}

/* Reads a data section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_data_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_DATA *data          = NULL;
	EWF_CRC *calculated_crc = NULL;
	EWF_CRC stored_crc      = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: invalid handle.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_sectors_read: EWF-S01 format should not contain data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	if( size != EWF_DATA_SIZE )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: mismatch in section data size.\n" );

		return( -1 );
	}
	data = ewf_data_read( file_descriptor );

	if( data == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: unable to read data.\n" );

		return( -1 );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->type, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown2, 16 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown3, 3 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown4, 12 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown5, 3 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown6, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->unknown7, 963 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( data->signature, 5 ); );
#endif
	calculated_crc = ewf_crc_calculate( (void *) data, ( EWF_DATA_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: unable to calculate CRC.\n" );

		ewf_data_free( data );

		return( -1 );
	}
	stored_crc = libewf_endian_convert_32bit( data->crc );

	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: CRC does not match (in file: %" PRIu32 " calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );
			ewf_crc_free( calculated_crc );

			return( -1 );
		}
	}
	ewf_crc_free( calculated_crc );

	/* TODO add more checks
	 */
	if( handle->chunk_count != libewf_endian_convert_32bit( data->chunk_count ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: chunk count does not match in data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );

			return( -1 );
		}
	}
	if( handle->sectors_per_chunk != libewf_endian_convert_32bit( data->sectors_per_chunk ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: sectors per chunk does not match in data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );

			return( -1 );
		}
	}
	if( handle->bytes_per_sector != libewf_endian_convert_32bit( data->bytes_per_sector ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: bytes per sector does not match in data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );

			return( -1 );
		}
	}
	if( handle->sector_count != libewf_endian_convert_32bit( data->sector_count ) )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: sector count does not match in data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );

			return( -1 );
		}
	}
	if( handle->media_type != data->media_type )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_read: media type does not match in data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_data_free( data );

			return( -1 );
		}
	}
	ewf_data_free( data );

	return( (int32_t) size );
}

/* Write a data section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_data_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset )
{
	EWF_DATA *data              = NULL;
	int32_t section_write_count = 0;
	int32_t data_write_count    = 0;
	uint32_t size               = EWF_DATA_SIZE;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_write: invalid handle.\n" );

		return( -1 );
	}
	data = ewf_data_alloc();

	if( data == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_write: unable to create data.\n" );

		return( -1 );
	}
	libewf_endian_revert_32bit( handle->chunk_count, data->chunk_count );
	libewf_endian_revert_32bit( handle->sectors_per_chunk, data->sectors_per_chunk );
	libewf_endian_revert_32bit( handle->bytes_per_sector, data->bytes_per_sector );
	libewf_endian_revert_32bit( handle->sector_count, data->sector_count );

	if( handle->format == LIBEWF_FORMAT_FTK )
	{
		data->type[ 0 ] = 0x01;
	}
	if( handle->format == LIBEWF_FORMAT_ENCASE5 )
	{
		data->compression_level = handle->compression_level;
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "data", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_write: unable to write section to file.\n" );

		ewf_data_free( data );

		return( -1 );
	}
	data_write_count = ewf_data_write( data, file_descriptor );

	ewf_data_free( data );

	if( data_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_data_write: unable to write data to file.\n" );

		return( -1 );
	}
	return( section_write_count + data_write_count );
}

/* Reads a error2 section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_error2_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_ERROR2 *error2         = NULL;
	EWF_ERROR2_SECTOR *sectors = NULL;
	EWF_CRC *calculated_crc    = NULL;
	EWF_CRC *stored_crc_read   = NULL;
	EWF_CRC stored_crc         = 0;
	uint32_t error_count       = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_read: invalid handle.\n" );

		return( -1 );
	}
	if( handle->ewf_format == EWF_FORMAT_S01 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_sectors_read: EWF-S01 format should not contain data section.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	error2 = ewf_error2_read( file_descriptor );

	if( error2 == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_read: unable to read error2.\n" );

		return( -1 );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( error2->unknown, 200 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( (uint8_t *) sectors, ( EWF_ERROR2_SECTOR_SIZE * error_count ) ); );
#endif

	calculated_crc = ewf_crc_calculate( (void *) error2, ( EWF_ERROR2_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_read: unable to calculate CRC.\n" );

		ewf_error2_free( error2 );

		return( -1 );
	}
	stored_crc  = libewf_endian_convert_32bit( error2->crc );
	error_count = libewf_endian_convert_32bit( error2->error_count );

	ewf_error2_free( error2 );

	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_crc_free( calculated_crc );

			return( -1 );
		}
	}
	ewf_crc_free( calculated_crc );

	if( error_count <= 0 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_read: error2 contains no sectors!.\n" );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			return( -1 );
		}
	}
	else
	{
		sectors = ewf_error2_sectors_read( file_descriptor, error_count );

		if( sectors == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_error2_read: unable to read error2 sectors.\n" );

			return( -1 );
		}
		calculated_crc = ewf_crc_calculate( (void *) sectors, ( EWF_ERROR2_SECTOR_SIZE * error_count ), 1 );

		if( calculated_crc == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_error2_read: unable to calculate CRC.\n" );

			ewf_error2_sectors_free( sectors );

			return( -1 );
		}
		stored_crc_read = ewf_crc_read( file_descriptor );

		if( stored_crc_read == NULL )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_error2_read: unable to read CRC from file descriptor.\n" );

			ewf_error2_sectors_free( sectors );
			ewf_crc_free( calculated_crc );

			return( -1 );
		}
		if( *stored_crc_read != *calculated_crc )
		{
			LIBEWF_WARNING_PRINT( "libewf_section_error2_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", *stored_crc_read, *calculated_crc );

			if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
			{
				ewf_error2_sectors_free( sectors );
				ewf_crc_free( calculated_crc );
				ewf_crc_free( stored_crc_read );

				return( -1 );
			}
		}
		handle->error2_error_count = error_count;
		handle->error2_sectors     = sectors;

		ewf_crc_free( calculated_crc );
		ewf_crc_free( stored_crc_read );
	}
	return( (int32_t) size );
}

/* Write a error2 section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_error2_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset, EWF_ERROR2_SECTOR *sectors, uint32_t sectors_amount )
{
	EWF_ERROR2 *error2                 = NULL;
	int32_t section_write_count        = 0;
	int32_t error2_write_count         = 0;
	int32_t error2_sectors_write_count = 0;
	uint32_t size                      = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_write: invalid handle.\n" );

		return( -1 );
	}
	size   = EWF_ERROR2_SIZE + ( EWF_ERROR2_SECTOR_SIZE * sectors_amount ) + EWF_CRC_SIZE;
	error2 = ewf_error2_alloc();

	if( error2 == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_write: unable to create error2.\n" );

		return( -1 );
	}
	libewf_endian_revert_32bit( sectors_amount, error2->error_count );

	section_write_count = libewf_section_start_write( handle, file_descriptor, "error2", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_write: unable to write section to file.\n" );

		ewf_error2_free( error2 );

		return( -1 );
	}
	error2_write_count         = ewf_error2_write( error2, file_descriptor );
	error2_sectors_write_count = ewf_error2_sectors_write( sectors, file_descriptor, sectors_amount );

	ewf_error2_free( error2 );

	if( error2_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_write: unable to write error2 to file.\n" );

		return( -1 );
	}
	if( error2_sectors_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_error2_write: unable to write error2 sectors to file.\n" );

		return( -1 );
	}
	return( section_write_count + error2_write_count + error2_sectors_write_count );
}

/* Reads a hash section
 * Returns the amount of bytes read, or -1 on error
 */
int32_t libewf_section_hash_read( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t size )
{
	EWF_HASH *hash          = NULL;
	EWF_CRC *calculated_crc = NULL;
	EWF_CRC stored_crc      = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_read: invalid handle.\n" );

		return( -1 );
	}
	if( size != EWF_HASH_SIZE )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_read: mismatch in section data size.\n" );

		return( -1 );
	}
	hash = ewf_hash_read( file_descriptor );

	if( hash == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_read: unable to read hash.\n" );

		return( -1 );
	}
#ifdef HAVE_DEBUG_OUTPUT
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( hash->unknown1, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( hash->unknown2, 4 ); );
	LIBEWF_VERBOSE_EXEC( libewf_dump_data( hash->signature, 8 ); );
#endif

	calculated_crc = ewf_crc_calculate( (void *) hash, ( EWF_HASH_SIZE - EWF_CRC_SIZE ), 1 );

	if( calculated_crc == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_read: unable to calculate CRC.\n" );

		ewf_hash_free( hash );

		return( -1 );
	}
	stored_crc = libewf_endian_convert_32bit( hash->crc );

	if( stored_crc != *calculated_crc )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_read: CRC does not match (in file: %" PRIu32 ", calculated: %" PRIu32 ").\n", stored_crc, *calculated_crc );

		if( handle->error_tollerance < LIBEWF_ERROR_TOLLERANCE_COMPENSATE )
		{
			ewf_hash_free( hash );
			ewf_crc_free( calculated_crc );

			return( -1 );
		}
	}
	libewf_handle_set_md5hash( handle, hash->md5hash );

	ewf_hash_free( hash );
	ewf_crc_free( calculated_crc );

	return( (int32_t) size );
}

/* Write a hash section to file
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_hash_write( LIBEWF_HANDLE *handle, int file_descriptor, uint32_t start_offset, EWF_MD5HASH *md5hash )
{
	EWF_HASH *hash              = NULL;
	void *data_set              = NULL;
	int32_t section_write_count = 0;
	int32_t hash_write_count    = 0;
	uint32_t size               = EWF_HASH_SIZE;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_write: invalid handle.\n" );

		return( -1 );
	}
	hash = ewf_hash_alloc();

	if( hash == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_write: unable to create hash.\n" );

		return( -1 );
	}
	data_set = libewf_common_memcpy( (void *) hash->md5hash, (void *) md5hash, EWF_MD5HASH_SIZE );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_write: unable to set hash.\n" );

		ewf_hash_free( hash );

		return( -1 );
	}
	section_write_count = libewf_section_start_write( handle, file_descriptor, "hash", size, start_offset );

	if( section_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_write: unable to write section to file.\n" );

		ewf_hash_free( hash );

		return( -1 );
	}
	hash_write_count = ewf_hash_write( hash, file_descriptor );

	ewf_hash_free( hash );

	if( hash_write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_hash_write: unable to write hash to file.\n" );

		return( -1 );
	}
	return( section_write_count + hash_write_count );
}

/* Write the last section start to file
 * This is used for the next and done sections, these sections point back towards themselves
 * Returns the amount of bytes written, or -1 on error
 */
int32_t libewf_section_last_write( LIBEWF_HANDLE *handle, int file_descriptor, char *section_type, uint32_t start_offset )
{
	EWF_SECTION *section       = NULL;
	void *data_set             = NULL;
	uint64_t section_size      = 0;
	uint64_t section_offset    = 0;
	uint32_t section_type_size = 0;
	int32_t write_count        = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_last_section_write: invalid handle.\n" );

		return( -1 );
	}
	section = ewf_section_alloc();

	if( section == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_last_section_write: unable to create section.\n" );

		return( -1 );
	}
	section_type_size = libewf_common_strlen( section_type );
	section_size      = EWF_SECTION_SIZE;
	section_offset    = start_offset;

	if( section_type_size >= 16 )
	{
		LIBEWF_WARNING_PRINT( "libewf_last_section_write: section type is too long.\n" );

		ewf_section_free( section );

		return( -1 );
	}
	data_set = libewf_common_memcpy( (void *) section->type, (void *) section_type, section_type_size );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_last_section_write: unable to set section type.\n" );

		ewf_section_free( section );

		return( -1 );
	}
	libewf_endian_revert_64bit( section_size, section->size );
	libewf_endian_revert_64bit( section_offset, section->next );

	write_count = ewf_section_write( section, file_descriptor );

	ewf_section_free( section );

	if( write_count == -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_last_section_write: unable to write section to file.\n" );

		return( -1 );
	}
	return( write_count );
}

/* Reads and processes a section
 * Returns the section start, or NULL on error
 */
EWF_SECTION *libewf_section_read( LIBEWF_HANDLE *handle, int file_descriptor, LIBEWF_SECTION_LIST *section_list, uint64_t section_start_offset )
{
	LIBEWF_SECTION_LIST *data_set = NULL;
	EWF_SECTION *section          = NULL;
	uint64_t section_end_offset   = 0;
	uint32_t size                 = 0;
	int32_t count                 = 0;

	if( handle == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: invalid handle.\n" );

		return( NULL );
	}
	if( section_list == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: invalid section list.\n" );

		return( NULL );
	}
	section = libewf_section_start_read( handle, file_descriptor );

	if( section == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: unable to read section start.\n" );

		return( NULL );
	}
	size               = libewf_endian_convert_64bit( section->size );
	section_end_offset = section_start_offset + size;
	data_set           = libewf_section_list_append( section_list, file_descriptor, section_start_offset, section_end_offset );

	if( data_set == NULL )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: unable to append value.\n" );

		return( NULL );
	}
	size -= EWF_SECTION_SIZE;

	/* Skip the next and done section
	 */
	if( ewf_section_is_type_next( section ) || ewf_section_is_type_done( section ) )
	{
		/* Just skip these */
	}
	/* Read the header2 section
	 */
	else if( ewf_section_is_type_header2( section ) )
	{
		count = libewf_section_header2_read( handle, file_descriptor, size );
	}
	/* Read the header section
	 */
	else if( ewf_section_is_type_header( section ) )
	{
		count = libewf_section_header_read( handle, file_descriptor, size );
	}
	/* Read the volume or disk section
	 */
	else if( ewf_section_is_type_volume( section ) || ewf_section_is_type_disk( section ) )
	{
		count = libewf_section_volume_read( handle, file_descriptor, size );
	}
	/* Read the table2 section
	 */
	else if( ewf_section_is_type_table2( section ) )
	{
		count = libewf_section_table2_read( handle, file_descriptor, size, section_list );
	}
	/* Read the table section
	 */
	else if( ewf_section_is_type_table( section ) )
	{
		count = libewf_section_table_read( handle, file_descriptor, size, section_list );
	}
	/* Read the sectors section
	 */
	else if( ewf_section_is_type_sectors( section ) )
	{
		count = libewf_section_sectors_read( handle, file_descriptor, size );
	}
	/* Read the ltree section
	 */
	else if( ewf_section_is_type_ltree( section ) )
	{
		count = libewf_section_ltree_read( handle, file_descriptor, size );
	}
	/* Read the data section
	 */
	else if( ewf_section_is_type_data( section ) )
	{
		count = libewf_section_data_read( handle, file_descriptor, size );
	}
	/* Read the hash section
	 */
	else if( ewf_section_is_type_hash( section ) )
	{
		count = libewf_section_hash_read( handle, file_descriptor, size );
	}
	/* Read the error2 section
	 */
	else if( ewf_section_is_type_error2( section ) )
	{
		count = libewf_section_error2_read( handle, file_descriptor, size );
	}
	else
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: unsupported section type: %s.\n", section->type );

#ifdef HAVE_DEBUG_OUTPUT
		LIBEWF_VERBOSE_EXEC( libewf_debug_read_section( handle, file_descriptor, size ); );
#endif
	}
	if( count <= -1 )
	{
		LIBEWF_WARNING_PRINT( "libewf_section_read: unable to read section: %s.\n", section->type );
	}
	return( section );
}

