% SLIRP: The (Sl)ang (I)nte(r)face (P)ackage (a code generator for S-Lang) {{{
%
%  Copyright (c) 2003-2005 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
%
%  This software was partially developed by the MIT Center for Space
%  Research under contract SV1-61010 from the Smithsonian Institution.
%  
%  Permission to use, copy, modify, distribute, and sell this software
%  and its documentation for any purpose is hereby granted without fee,
%  provided that the above copyright notice appear in all copies and
%  that both that copyright notice and this permission notice appear in
%  the supporting documentation, and that the name of the Massachusetts
%  Institute of Technology not be used in advertising or publicity
%  pertaining to distribution of the software without specific, written
%  prior permission.  The Massachusetts Institute of Technology makes
%  no representations about the suitability of this software for any
%  purpose.  It is provided "as is" without express or implied warranty.
%  
%  THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
%  WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%  MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
%  INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
%  CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
%  OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
%  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
%  WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
% }}}

_debug_info = 1;
require("slirpmaps");
require("slirpc++");

% Support functions, iterators, etc {{{

static define warn_ignore(func,reason)
{
   % Trick: call as (NULL, func, reason) to simulate NULL return value
   dprintf("\n*** Ignored Function: %s\n*** Reason: %s\n", func,reason);
}

static define emit_constants(consts, kind)
{
   !if (length(consts)) return;

   variable name, value;
   emit("static SLang_%sConstant_Type %s_%sConsts [] =\n{\n",
						kind, SC.modname, kind);
   foreach(consts) using ("keys","values") {
	(name, value) = ();
	emit("   MAKE_%sCONSTANT((char*)\"%s\",%s),\n", kind, name, value);
   }

   emit("   SLANG_END_%sCONST_TABLE\n};\n\n", kind);
}

static define emit_intrinsic_vars(interface)
{
   variable name, value, i = 0, sarray_name;
   if (length(interface.string_consts)) {

	sarray_name = SC.modname + "_strings";

	emit("static const char *%s[] = {\n", sarray_name);

	foreach(interface.string_consts) using ("values") {
	   value = ();
	   emit("   %s,\n", value);
	}

	emit("NULL\n};\n\n");
   }
   else !if (length(interface.intrin_vars))
	return;

   emit("static SLang_Intrin_Var_Type %s_IVars [] =\n{\n", SC.modname);

   foreach(interface.string_consts) using ("keys") {
	name = ();
	emit("   MAKE_VARIABLE(\"%s\",&%s[%d], SLANG_STRING_TYPE,1),\n",
	      					name, sarray_name, i);
	i++;
   }

   variable arrays = Struct_Type[0];

   foreach(interface.intrin_vars) using ("keys", "values") {
	(name, value) = ();

	variable is_array = string_match(name, "\\[[^]]*\\]", 1);
	if (is_array) {
	   value = @value;			% preserve original typemap
	   value.name = name[ [0:is_array-2] ];
	   arrays = [ arrays, value ];
	}
	else
	   emit("   MAKE_VARIABLE(\"%s\",&%s, %s, 1),\n", name, value.name,
		 						value.typeid);
   }

   emit("   SLANG_END_INTRIN_VAR_TABLE\n};\n\n");

   !if (length(arrays)) return;

   emit("static int make_intrinsic_arrays(SLang_NameSpace_Type *ns)\n{");

   foreach(arrays) {
	 variable arr = ();
	emit("\n   if (-1 == SLang_add_intrinsic_array (\"%s\", %s, 1,\n"+
	      "\t\t%s, 1, sizeof(%s) / sizeof(%s)))\n"+
		"\treturn -1;",
		arr.name, arr.typeid, arr.name, arr.name, arr.type);
   }
	    
   emit("\n}\n");
   interface.intrin_arrays = arrays;		% hint to generate_module_init
}

static define emit_usage_strings()
{
   !if ( length(SC.usage_strings)) return;
   emit("static const char* usage_strings[] = {\n");
   array_map(Void_Type, &emit, "   \"%s\",\n", SC.usage_strings);
   emit("NULL\n};\n\n");
   inject_file("usage.c");
}

static define iterate_over_functions()
{
   variable args = __pop_args(_NARGS - 2), action = (), container = ();
   foreach(container.interface.functions) using ("values")
	(@action)((), __push_args(args));
}

static define iterate_over_classes()
{
   variable args = __pop_args(_NARGS - 1), action = ();
   foreach(SC.classes) using ("values") {
	variable class = ();
	(@action)(class, __push_args(args));
   }
}

% }}}

% Usage statements {{{
static variable Commas = [ EMPTY, ","];
static define generate_usage_statement(fmap)
{
   variable arg, outputs = TypeMapping[0];

   foreach (fmap.argmaps[AM_Out]) {
	variable amap = @();
	if (amap.which < 0) continue;		% an omitted return value
	arg = @amap.args[amap.which];
	if (amap.usage != NULL)
	   arg.mnemonic = amap.usage;
	else if (andelse {arg.typeclass == TYPECL_POINTER} {not(arg.marshal)}) {
	   % C pointers changed from an argument list parameter to an output
	   % value effectively have their indirection reduced by one, which
	   % we reflect by setting usage to that of the type being pointed to
	   arg.mnemonic = arg.aux.mnemonic;
	}
	outputs = [ arg, outputs];
   }

   variable num = length(outputs);
   outputs = strjoin( struct_map( String_Type, outputs, "mnemonic"), ",");
   if (outputs != EMPTY)
	if (num > 1)
	   outputs = sprintf("(%s) = ",outputs);
	else
	   outputs = strcat(outputs, " = ");

   variable args = fmap.args, narg = "SLang_Num_Function_Args";
   variable inputs = where( struct_map(Integer_Type, args, "marshal"));
   variable defaults = struct_map(String_Type, args[inputs], "defval");
   variable optional = where ( defaults != EMPTY);
   variable num_optional = length(optional);

   if (num_optional) {
	num = length(inputs) - num_optional;
	narg = sprintf("%s < %d || %s > %d", narg, num, narg, num+num_optional);
	optional = sprintf(" [%s%s]", Commas[ (num > 0)],
		strjoin ( array_map(String_Type, &sprintf, "%s=%s",
		struct_map(String_Type, args[inputs][optional], "mnemonic"),
		defaults[optional]), ","));
	optional = str_quote_string(optional, "\"",'\\');
   }
   else {
	num = length(inputs);
	narg = sprintf("%s != %d", narg, num);
	optional = EMPTY;
   }

   if (num > 0) {
	args = args[inputs];
	args = strjoin (struct_map(String_Type, args[[0:num-1]],
		 					"mnemonic"), ",");
   }
   else
	args = EMPTY;

   variable usg = sprintf("%s%s(%s)", outputs, fmap.slname, args+optional);
   SC.usage_strings = [SC.usage_strings, usg ];
   return ( narg, usg);

} % }}}

static define emit_declaration_block(fmap)	% {{{
{
   variable r = fmap.result;
   if (r.ltype != VOID)
	emit("   %s%s %s;\n", r.const, r.type, r.lname);

   foreach(fmap.local_vars) {
	variable var = ();
	emit("   %s %s%s;\n", var.type, var.lname, var.arrayspec);
   }

   !if (fmap.pop_args) return;
  
   foreach(fmap.args) {  variable arg = (); (@arg.declarer)(arg); }
} % }}}

% Argmap emission {{{ 
static define emit_argmap_codefrag(argmap)
{
   % Use %s to ensure formats w/in fragment are not interpreted by S-Lang
   emit("%s", do_parameter_substitutions(argmap));
}

static define emit_argmap_code_fragments(fmap, kind)
{
   foreach(fmap.argmaps[kind])
	emit_argmap_codefrag(());
}
% }}}

static define generate_marshaling_code(fmap)	% {{{
{
   variable mstmts = EMPTY;
   if (fmap.pop_args) {

	variable arg, argno = 0;

   	foreach(fmap.args) {

	   arg = ();
	   !if (arg.marshal) continue;
	   if (arg.proxy != NULL) arg = arg.proxy;

	   argno++;			% arg #s w/in interface files start at 1

	   mstmts = sprintf("%s == -1 ||\n\t%s",
		 	(@arg.marshaler)(arg, argno), mstmts);
	}

	if (mstmts != EMPTY) {
	   (mstmts,) = strreplace(mstmts,"||\n\t","",-1);
	   mstmts = sprintf(" ||\n\t%s",mstmts);
	}
   }

   return mstmts;

} % }}}

static define emit_call_block(fmap)	% {{{
{
   emit("   ");

   if (fmap.result.ltype != VOID) {
	if (SC.cfront)
	   if (fmap.result.typeclass == TYPECL_POINTER) 	% de-constify,
		emit("return (%s) ", fmap.result.type);		% if necessary
	   else
		emit("return ");
	else
	   emit("result = ");
   }

   if (fmap.call_emitter == NULL) {

	% Fabricate call to underlying function or class method
	variable argno = 0, fcall = (@fmap.referer) (fmap, &argno);
	variable arg_separator = "," + fmap.arg_separator;
	foreach(fmap.args[[argno:]]) {
	   variable arg = ();
	   arg = (@arg.referer)(arg, 0, fmap.pop_args);
	   fcall = strcat(fcall, arg, arg_separator);
	}
	fcall = strtrim_end(fcall, arg_separator);
	emit(strcat(fcall,");\n"));
   }
   else
	(@fmap.call_emitter)(fmap);

} % }}}

static define emit_default_output_mapping(outmap) % {{{
{
   variable output = outmap.args[0];
   variable cast = "", isconst = (output.const != EMPTY);
   if (isconst)
	cast = strcat("(",output.type,")");		% cast, to de-const-ify

   if (output.typeclass == TYPECL_OPAQUE)

	emit("   (void) SLang_push_opaque(%s, %s result, 0);\n",
							output.aux, cast);

   else if (orelse {output.ltype != SLang_String_Type}
		   {isconst}
		   {SC.assume_const_strings} )

	emit("   (void) SLang_push_%s (%s result);\n", output.mnemonic, cast);

   else
	% This will free the string after it's pushed onto stack
	emit("   (void) SLang_push_malloced_string(result);\n");
} % }}}

static define emit_return_block(fmap)	% {{{
{
   if (fmap.pop_args) {

	variable argno = 0;
	foreach(fmap.args) {

	   variable arg = (); 

	   if (arg.proxy != NULL) arg = arg.proxy;
	   !if (arg.marshal) continue;

	   % It is assumed that underlying funcs called by cleaners (e.g.
	   % SLang_free_array) gracefully handle attempts to free NULL

	   (@arg.cleaner) (arg, argno);
	   argno++;
	}
   }

   emit_argmap_code_fragments(fmap, AM_Final);

   foreach ( reverse(fmap.argmaps[AM_Out]) ) {

	variable outmap = ();
	if (outmap.code == EMPTY)
	   emit_default_output_mapping(outmap);
	else
	   emit_argmap_codefrag(outmap);
   }
} % }}}

static define emit_function_table_entry(fmap) % {{{
{
   if (orelse {fmap.pop_args} {fmap.nargs == 0}) {
	emit("   MAKE_INTRINSIC_0((char*)\"%s\",%s,V),\n",
	      					fmap.slname, fmap.wname);
	return;
   }

   % We will override the signature of the underlying C function, by
   % assigning a void arg list to the ftable entry, if the C func
   % signature includes one or more of the following:
   %
   %	struct : in order to to use struct layout functions 
   %	ref    : in order to treat refs and arrays uniformly
   %
   % Instead of relying upon the built-in S-Lang-to-C arg transfer
   % mechanism, we generate additional code to explicitly pop args
   % off the stack, and gain the ability to perform the appropriate
   % finalization (e.g. free an mmt) prior to returning from the
   % glue layer.

   variable args = String_Type[0];
   foreach (fmap.args) {

	variable arg = ();

	switch (arg.ltype)
	{ case SLang_Array_Type: arg = "SLANG_ARRAY_TYPE"; }
	{ arg = arg.typeid; }

	args = [ args, arg];
   }
   emit("   MAKE_INTRINSIC_%d((char*)\"%s\", %s, V, %s),\n", fmap.nargs,
				fmap.slname, fmap.wname, strjoin(args,","));
}

% }}}

% Wrapper function emission {{{

static define emit_header_includes()
{
   variable h = SC.header_names;
   emit( strjoin(array_map(String_Type, &sprintf, "#include \"%s\"\n", h),""));
}

static define emit_wrapper(fmap, usage_index, num_funcs_with_same_name)
{
   variable proto = generate_prototype(fmap);

   if (SC.interface.prototypes[proto] != NULL)
      return;					% never emit duplicate wrappers

   emit(proto);
   emit("\n{\n");

   SC.interface.prototypes [ proto ] = EMPTY;

   !if (SC.cfront) {

	emit_declaration_block(fmap);

	variable nargs_test, usg, marshal_code = generate_marshaling_code(fmap);

	% Usage statements are generated even for zero-arg functions; its
	% preferable to leaving crud on the stack when they're called with
	% too many args, as well as didactic for the case when an annotation
	% causes none of the C function arguments to be visible from S-Lang
	(nargs_test, usg) = generate_usage_statement(fmap);
	emit("\n   if (%s%s)\n\t{Slirp_usage_err(%d, %d); return;}\n\n",
				nargs_test, marshal_code, usage_index,
				usage_index + num_funcs_with_same_name);

	emit_argmap_code_fragments(fmap, AM_In);
	emit_argmap_code_fragments(fmap, AM_Init);
   }

   emit_call_block(fmap);

   !if (SC.cfront) emit_return_block(fmap);

   emit("}\n\n");
}

static define funcmap_2_wrapper(fmap)
{
   variable overloads = length(fmap.overloads), dispatch_entry = @List;
   variable usage_index = length(SC.usage_strings);

   foreach(fmap.overloads) {

	variable o = (), types;

	if (length(o.args)) {
	   types = map_args_to_type_abbrevs(o.args);
	   types = strjoin(types, EMPTY);
	   o.wname = o.slname + "_" + types;
	}
	else {
	   types = EMPTY;
	   o.wname = o.slname + "_v";
	}

	_list_append(dispatch_entry, list_node_new( SC.wrapper_prefix + o.wname,
		 							types));
	emit_wrapper(o, usage_index, overloads);
   }

   if (overloads) {
	if (SC.cfront) return;		% dispatcher not needed
	variable i = length(SC.dispatch_table);
 	make_dispatch_table_entry(dispatch_entry);
	emit( generate_prototype(fmap) );
	emit(" { dispatch(%d, %d, %d); }\n\n", i, i + overloads, usage_index);
   }
   else
	emit_wrapper(fmap, usage_index, 0);
}

static define emit_wrappers()
{
   if (andelse {SC.cplusplus} {not(SC.cfront)} )
	emit("static void dispatch (int first, int stop, int usage_index);\n");

   iterate_over_functions(SC, &funcmap_2_wrapper);
   iterate_over_classes(&iterate_over_functions, &funcmap_2_wrapper);

   if (SC.cfront) return;

   emit("static SLang_Intrin_Fun_Type %s_Funcs [] =\n{\n", SC.modname);

   iterate_over_functions(SC, &emit_function_table_entry);
   iterate_over_classes(&iterate_over_functions, &emit_function_table_entry);

   emit("   SLANG_END_INTRIN_FUN_TABLE\n};\n\n");
}

static define emit_cfront_style_wrappers()
{
   emit_header_includes;
   emit("#include \"%s_glue.h\"\n", SC.modname);
   emit("#define OBJECT(o)		o\n\n");
   emit_wrappers();
   iterate_over_classes(&emit_destructor);

   close_file(SC.gluefp);

   SC.gluefp = create_output_file("_glue",".h");

   emit("#ifdef __cplusplus\nextern \"C\" {\n#endif\n\n");
   foreach(SC.interface.prototypes) using ("keys") {
	variable proto = ();
	emit( strcat("extern ", proto, ";\n"));
   }
   emit("\n#ifdef __cplusplus\n}\n#endif\n");

   close_file(SC.gluefp);

   return;
}
% }}}

static define generate_code() % {{{
{
   if (SC.srcfp == NULL) return;		% no files processed

   if (andelse {SC.cfront} {not(SC.cplusplus)})
	abort("C wrappers not generated, since no C++ code was processed.");

   if (SC.genstubs) {
	SC.gluefp = create_output_file("_stubs");
	emit_header_includes;
	generate_stubs(SC.interface,"");
	foreach(SC.classes) using ("values") {
	   variable class = ();
	   generate_stubs(class.interface, class.name+"::");
	}
	close_file(SC.gluefp);
	return;
   }

   SC.gluefp = create_output_file("_glue");

   inject_file("intro.h");
   inject_file("compat.h");

   if (SC.cfront) {
	emit("#define INVOKE_METHOD(type,obj,method) ((type*)(obj))->method\n");
	return emit_cfront_style_wrappers();
   }

   if (SC.geninit)
	emit_header_includes;

   if (SC.cplusplus) {
	emit("#define OBJECT(o)			SAFE_DEREF_OPAQUE(o)\n");
	emit("#define BEGIN_DECLS			extern \"C\" {\n"+
	     "#define END_DECLS			}\n\n");
	iterate_over_classes(&emit_destructor);
   }
   else
	emit("#define BEGIN_DECLS\n"+
	     "#define END_DECLS\n\n");

   inject_file("misc.c");

   variable num_new_opaques = generate_opaque_code();

   % Conditional file inclusions
   if (SC.have_refs) {
	emit("static unsigned char map_scalars_to_refs = %d;\n",
						SC.map_scalars_to_refs);
	inject_file("refs.c");
   }

   if (SC.have_null_term_str_arrays)
	inject_file("ntstrarr.c");

   if (SC.have_fortran)
	inject_file("fortran.c");

   if (SC.tmap_outfile != NULL)
	emit_typemaps(SC.tmap_outfile);

   emit("\n");
   emit_inlines(SC.interface.inlines);
   emit("\n");
   emit_wrappers();
   emit_usage_strings();
   emit_constants(SC.interface.int_consts,"I");
   emit_constants(SC.interface.double_consts,"D");
   emit_intrinsic_vars(SC.interface);
   emit_dispatch_table();

   if (SC.geninit)
	generate_module_init(num_new_opaques);

   close_file(SC.gluefp);
   
} % }}}

static define parse_files() % {{{
{
   variable multiple_files = (__argc - SC.argc > 1);

   while (SC.argc < __argc) {

	SC.srcfp = init_file_processor(__argv[SC.argc]);
	if (SC.srcfp == NULL) { SC.argc++; continue; }

	while( not(feof(SC.srcfp)) )
	   !if (prepr_macro_or_type_declaration(SC.srcfp, NULL))
		!if (cplusplus_token(SC.srcfp))
		    !if (variable_declaration(SC.srcfp))
			function_declaration(SC.srcfp, NULL, not(SC.cplusplus));

	() = fclose(SC.srcfp);

	if (multiple_files) {
	   () = printf(".");		% Output progress indicator
	   () = fflush(stdout);
	}

	SC.argc++;
   }

   % Late bind macros which reference symbols undefined when they were seen
   determine_unresolved_macro_disposition(SC.interface);

   if (multiple_files) () = printf("\n");
} % }}}

define slsh_main()  % {{{
{
   slirp_initialize();
   parse_files();

   if (SC.print_interface)
	print_public_interface();
   else
	generate_code();

   if (SC.genmakef) generate_makefile();

   exit(SC.status);
} % }}}
