/* -*- mode: C; mode: fold; -*- */

/* This file is part of
 *
 * 	SLgtk:  S-Lang bindings for GTK+ widget set
 *
 * Copyright (C) 2003-2005 Massachusetts Institute of Technology 
 * Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
 *
 */

#include <string.h>
#include <gdk/gdkpixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-io.h>

/* Wrappers and support functions {{{ */

static void
gerr_return (void *thing, SLtype type, SLang_Ref_Type *gerr_ref, GError *gerr)
{
   if (thing == NULL)
	SLang_push_null();
   else
	SLang_push_opaque(type, thing, 0);

   if (gerr_ref != NULL) {
	if (gerr) {
	   (void)SLang_assign_cstruct_to_ref(gerr_ref,gerr,GError_Layout);
	   g_error_free(gerr);
	}
	else
	   (void) SLang_assign_to_ref (gerr_ref, SLANG_NULL_TYPE, NULL);

	SLang_free_ref(gerr_ref);
   }
}

static void sl_gdk_pixbuf_new_from_file(void)
{
   char *f = NULL;
   GError *gerr = NULL;
   SLang_Ref_Type *gerr_ref = NULL;

   if (SLang_Num_Function_Args < 1 || SLang_Num_Function_Args > 2 ||
	 (SLang_Num_Function_Args == 2 && -1 == SLang_pop_ref(&gerr_ref)) ||
	 (-1 == SLang_pop_slstring(&f)))

	SLang_verror(SLEU,
		"pixbuf = gdk_pixbuf_new_from_file(fname [,error_ref])");

   else {
	GdkPixbuf* pb = gdk_pixbuf_new_from_file(f, &gerr);
	gerr_return( pb, GdkPixbuf_Type, gerr_ref, gerr);
   }

   SLang_free_slstring(f);
}

static void sl_gdk_pixbuf_animation_new(void)
{
   char *f = NULL;
   GError *gerr = NULL;
   SLang_Ref_Type *gerr_ref = NULL;

   if (SLang_Num_Function_Args < 1 || SLang_Num_Function_Args > 2 ||
	 (SLang_Num_Function_Args == 2 && -1 == SLang_pop_ref(&gerr_ref)) ||
	 (-1 == SLang_pop_slstring(&f)))

	SLang_verror(SLEU,
	"animation = gdk_pixbuf_animation_new_from_file(fname [,error_ref])");

   else {
	GdkPixbufAnimation *a = gdk_pixbuf_animation_new_from_file(f, &gerr);
	gerr_return(a, GdkPixbufAnimation_Type, gerr_ref, gerr);
   }

   SLang_free_slstring(f);
}


/* Wrapper for gdk_pixbuf_new_from_data() and gdk_pixbuf_save(), as
 * well as support routines in module.c, contributed by John E. Davis
 * (davis@space.mit.edu) */

static int pop_image_array (SLang_Array_Type **ap,
			    unsigned int *width, unsigned int *height,
			    unsigned int *stride,
			    int *is_alpha, int *is_grayscale)
{
   SLang_Array_Type *at;
   
   *ap = NULL;
   if (-1 == SLang_pop_array_of_type (&at, SLANG_UCHAR_TYPE))
     return -1;

   if (at->num_elements == 0)
     {
	SLang_verror (SL_INVALID_PARM, "Empty image-array not supported");
	SLang_free_array (at);
	return -1;
     }

   *ap = at;
   *is_grayscale = 1;
   *is_alpha = 0;
   switch (at->num_dims)
     {
      case 1:
	*height = at->dims[0];
	*stride = *width = 1;
	return 0;
	
      case 2:
	*height = at->dims[0];
	*stride = *width = at->dims[1];
	return 0;
	
      case 3:
	switch (at->dims[2])
	  {
	   case 1:
	     *height = at->dims[0];
	     *stride = *width = at->dims[1];
	     return 0;

	   case 3:
	     *is_grayscale = 0;
	     *height = at->dims[0];
	     *width = at->dims[1];
	     *stride = 3 * (*width);
	     return 0;
	     
	   case 4:
	     *is_grayscale = 0;
	     *is_alpha = 1;
	     *height = at->dims[0];
	     *width = at->dims[1];
	     *stride = 4 * (*width);
	     return 0;
	  }
	break;
     }
   
   *ap = NULL;
   SLang_free_array (at);
   SLang_verror (SL_INVALID_PARM, "Expected an image array");
   return -1;
}

static int pop_color_image (SLang_Array_Type **ap,
			    unsigned int *width, unsigned int *height,
			    unsigned int *stride,
			    int *is_alpha)
{
   int is_grayscale;
   SLang_Array_Type *at;

   if (-1 == pop_image_array (&at, width, height, stride, 
	    				is_alpha, &is_grayscale))
     return -1;
   
   if (is_grayscale)
     {
	SLang_verror (SL_INVALID_PARM, 
	      		"Expecting a color image, found a grayscale one");
	SLang_free_array (at);
	return -1;
     }
   *ap = at;
   return 0;
}

#if 0
static int pop_grayscale_image (SLang_Array_Type **ap,
				unsigned int *width, unsigned int *height,
				unsigned int *stride)
{
   int is_grayscale, is_alpha;
   SLang_Array_Type *at;

   if (-1 == pop_image_array (&at, width, height, stride, 
	    					&is_alpha, &is_grayscale))
     return -1;
   
   if (is_grayscale == 0)
     {
	SLang_verror (SL_INVALID_PARM, 
	      		"Expecting a grayscale image, found a color one");
	SLang_free_array (at);
	return -1;
     }
   *ap = at;
   return 0;
}
#endif

static void destroy_pixbuf_data_array (guchar *pixels, gpointer data_array)
{
   SLang_Array_Type *at = (SLang_Array_Type *)data_array;
   SLang_free_array (at);
   (void) pixels;
}

static void sl_gdk_pixbuf_new_from_data (void)
{
   GdkPixbuf* rtn;
   SLang_Array_Type *at;
   int is_alpha;
   unsigned int width, height, stride;
   
   if (SLang_Num_Function_Args == 0) 
     {
	SLang_verror (SL_USAGE_ERROR,
		"Usage: GdkPixbuf = gdk_pixbuf_new_from_data(image-array)");
	return;
     }

   if (-1 == pop_color_image (&at, &width, &height, &stride, &is_alpha))
     return;

   /* Note: the returned pixbuf has a reference to the array */
   rtn = gdk_pixbuf_new_from_data ((const guchar *) at->data,
				   GDK_COLORSPACE_RGB,
				   is_alpha,
				   8,
				   width,
				   height,
				   stride,
				   destroy_pixbuf_data_array, (gpointer)at);

   if (rtn == NULL)
     {
	SLang_free_array (at);
	return;
     }

   if (-1 == SLang_push_opaque(GdkPixbuf_Type, rtn, 0))
     gdk_pixbuf_unref (rtn);
}

static void sl_gdk_pixbuf_save (void)
{
   unsigned int nopts;
   char **keys, **vals;
   char *type, *file;
   GdkPixbuf *pixbuf;
   Slirp_Opaque *pixbuf_o = NULL;
   GError *error = NULL;

   if (SLang_Num_Function_Args < 3)
     {
	SLang_verror (SL_USAGE_ERROR,
	"Usage: gdk_pixbuf_save (pixbuf, file, type [,\"key=val\",...])");
	return;
     }
   
   nopts = (unsigned int) (SLang_Num_Function_Args - 3);

   if (-1 == pop_key_val_pairs (nopts, &keys, &vals))
     return;

   type = NULL;
   file = NULL;

   if ((-1 == SLang_pop_slstring (&type))
       || (-1 == SLang_pop_slstring (&file))
       || (-1 == SLang_pop_opaque (GdkPixbuf_Type, (void**)&pixbuf, &pixbuf_o)))
     goto free_and_return;

   if (FALSE == gdk_pixbuf_savev (pixbuf,
				  (const char *)file,
				  (const char *)type,
				  keys, vals,
				  &error))

     SLang_verror (SL_INTRINSIC_ERROR, "gdk_pixbuf_save: %s",
			(error != NULL) ? error->message : "Unknown Error");

   /* drop */
   free_and_return:
   free_malloced_string_array (keys, nopts);
   free_malloced_string_array (vals, nopts);
   SLang_free_slstring (type);
   SLang_free_slstring (file);
   SLang_free_opaque(pixbuf_o);
}
/* }}} */

static SLang_Intrin_Fun_Type Intrin_Funcs[] =
{

  MAKE_INTRINSIC_0("gdk_pixbuf_new_from_file",sl_gdk_pixbuf_new_from_file,V),
  MAKE_INTRINSIC_0("gdk_pixbuf_new_from_data",sl_gdk_pixbuf_new_from_data,V),
  MAKE_INTRINSIC_0("gdk_pixbuf_save",sl_gdk_pixbuf_save,V),
  MAKE_INTRINSIC_0("gdk_pixbuf_animation_new_from_file",sl_gdk_pixbuf_animation_new,V),

  SLANG_END_INTRIN_FUN_TABLE
};

static int init_sl_gdkpixbuf(SLang_NameSpace_Type *ns)
{

#ifdef SLGTK_STATIC_MODULE
   extern gboolean load_fits_pixbuf_module();
   static unsigned int loaded = 0;
   if ( !loaded && !load_fits_pixbuf_module())
	fprintf(stderr,"\nWarning: could not load FITS pixbuf module\n");
   loaded = 1;
#endif

   patch_ftable(Intrin_Funcs, O, GtkOpaque_Type);
   return SLns_add_intrin_fun_table (ns, Intrin_Funcs, "__GDKPIXBUF__");
}
