
% SLgtk RGB image displayer, with support for Log/Sqrt/Zero/Pow scaling,
% and Zoom in/out.  Can also be extended by adding other widgets to the
% button box fields.
%
% For sample usage, see evt2img guilet, distributed with histgram module.
%
% Contributed by John E. Davis (davis@space.mit.edu)

#ifnexists rgb_display_widget_new

require ("gtk");

% Version information % {{{
private variable version = "300";
private variable version_string = "0.3.0";
define _rgbwidget_version_string() { return version_string; }
define _rgbwidget_version() { return version; }
% }}}

private define zoomed_xy_to_unzoomed_xy (rgb_widget, x, y) % {{{
{
   x = (x - rgb_widget.ax)/(1.0*rgb_widget.zx);
   y = (y - rgb_widget.ay)/(1.0*rgb_widget.zy);
   return x,y;
   return int (x+0.5), int (y+0.5);
} % }}}

private define unzoomed_xy_to_zoomed_xy (rgb_widget, x, y) % {{{
{
   x = rgb_widget.zx*x + rgb_widget.ax;
   y = rgb_widget.zy*y + rgb_widget.ay;
   return int (x+0.5), int (y+0.5);
} % }}}

private define pan_to_center_unzoomed_pixel (rgb_widget, x0, y0) % {{{
{
   rgb_widget.ax = 0.5*rgb_widget.da_width - rgb_widget.zx * x0;
   rgb_widget.ay = 0.5*rgb_widget.da_height - rgb_widget.zy * y0;
   rgb_widget.dirty = 1;
} % }}}

private define zoom_by_factor (rgb_widget, fx, fy) % {{{
{
   variable zx = rgb_widget.zx;
   variable zy = rgb_widget.zy;
   variable w = rgb_widget.width;
   variable h = rgb_widget.height;
   variable x0, y0;

   (x0, y0) = zoomed_xy_to_unzoomed_xy (rgb_widget, 
					rgb_widget.da_width*0.5, 
					rgb_widget.da_height*0.5);

   zx *= fx;
   zy *= fy;
   if (zx <= 0) zx = 1.0;
   if (zx > 100*w) zx = w;

   if (zy <= 0) zy = 1.0;
   if (zy > 100*w) zy = h;

   rgb_widget.zx = zx;
   rgb_widget.zy = zy;

   pan_to_center_unzoomed_pixel (rgb_widget, x0, y0);
} % }}}

private define rgb_widget_expose (win, event, rgb_widget) % {{{
{
   variable drawable = gtk_widget_get_window (win);
   variable pixbuf = rgb_widget.pixbuf;
   variable width, height;
   variable unzoomed_width, unzoomed_height;

   unzoomed_width = rgb_widget.width;
   unzoomed_height = rgb_widget.height;
   variable da_width = rgb_widget.da_width;
   variable da_height = rgb_widget.da_height;

   gdk_drawable_get_size (drawable, &width, &height);
   variable dest = rgb_widget.zoomed_pixbuf;
   if ((da_width != width) or (da_height != height))
     {
	variable xc, yc;
	
	(xc, yc) = zoomed_xy_to_unzoomed_xy (rgb_widget, 0.5*da_width, 0.5*da_height);

	rgb_widget.da_width = width;
	rgb_widget.da_height = height;
	pan_to_center_unzoomed_pixel (rgb_widget, xc, yc);
	rgb_widget.dirty = 1;
     }

   if ((dest == NULL) or (rgb_widget.dirty))
     {
	rgb_widget.da_width = width;
	rgb_widget.da_height = height;

	dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, 0, 8, width, height);
	gdk_pixbuf_fill(dest, 0);
   
	% gdk_pixbuf_scale does not appear to be very robust and can lead to 
	% SEGVs if given scaling parameters that lead it off the input pixbuf.
	variable dest_x, dest_y;
	(dest_x, dest_y) = unzoomed_xy_to_zoomed_xy (rgb_widget, 0, 0);
	if (dest_x < 0) dest_x = 0;
	if (dest_y < 0) dest_y = 0;
	variable max_dest_x, max_dest_y;
	(max_dest_x, max_dest_y) = unzoomed_xy_to_zoomed_xy (rgb_widget, unzoomed_width, unzoomed_height);
	if (max_dest_x > width) max_dest_x = width;
	if (max_dest_y > height) max_dest_y = height;
   
	gdk_pixbuf_scale (pixbuf, dest, 
			  dest_x, dest_y, 
			  max_dest_x - dest_x, max_dest_y - dest_y,
			  rgb_widget.ax, rgb_widget.ay,
			  rgb_widget.zx, rgb_widget.zy,
			  rgb_widget.interp_method);
     }

   variable gc =  rgb_widget.gc;
   if (gc == NULL)
     {
	gc = gdk_gc_new (drawable);
	rgb_widget.gc = gc;
     }

   gdk_draw_pixbuf (drawable, gc, dest, 0, 0, 0, 0, -1, -1, 
		    GDK_RGB_DITHER_NORMAL, 0, 0);

   rgb_widget.zoomed_pixbuf = dest;
   rgb_widget.dirty = 0;
   return 1;
} % }}}

% Scaling parameters % {{{
private variable ZERO_SCALING = 0;
private variable LOG_SCALING = 1;
private variable LINEAR_SCALING = 2;
private variable LOGLOG_SCALING = 3;
private variable SQRT_SCALING = 4;
private variable POW_SCALING = 5;
% }}}

private define normalize (r) % {{{
{
   variable min_r = min(r);
   variable dr = 1.0 * (max(r) - min_r);

   if (dr != 0.0)
     r = 255.0 * ((r - min_r)/dr);
   else if (min_r != 0)
     {
	r = @r;
	r[*] = 255;
     }

   return typecast (r, UChar_Type);
} % }}}

private define apply_color_scaling (color, scale_method, scale_parm) % {{{
{
   switch (scale_method)
     {
      case SQRT_SCALING:
	color = sqrt (__tmp(color));
     }
     {
      case LOG_SCALING:
	color = log (__tmp(color) + 1.0);
     }
     {
      case POW_SCALING:
	color = __tmp(color)^scale_parm[0];
     }
     {
      case LOGLOG_SCALING:
	variable bad, good;
	bad = where (color<=0.0);
	good = where (color > 0);
	if (length (good))
	  {
	     color = log (__tmp(color)+2);
	     variable min_color = min(color[good]);
	     color = log (__tmp(color)-min_color+2);
	     color[bad] = 0;
	  }
     }
     {
      case ZERO_SCALING:
	color = @color;
	color[*,*] = 0;
     }
   % Now perform a normalization to the range 0-256
   return normalize (color);
} % }}}

private define scale_rgb (rgb_widget, rscale, gscale, bscale) % {{{
{
   variable r = rgb_widget.r;
   variable g = rgb_widget.g;
   variable b = rgb_widget.b;

   variable w = rgb_widget.width, h = rgb_widget.height;
   variable rgbbuf = rgb_widget.rgbbuf;
   
   variable init = 0;

   if (rgbbuf == NULL)
     {
	init = 1;
	rgbbuf = UChar_Type [h, w, 3];
     }

   variable i = [h-1:0:-1];

   if ((rgb_widget.rscale != rscale) or init)
     rgbbuf[i,*,0] = apply_color_scaling (rgb_widget.r, rscale, rgb_widget.rscale_parm);

   if ((rgb_widget.gscale != gscale) or init)
     rgbbuf[i,*,1] = apply_color_scaling (rgb_widget.g, gscale, rgb_widget.gscale_parm);

   if ((rgb_widget.bscale != bscale) or init)
     rgbbuf[i,*,2] = apply_color_scaling (rgb_widget.b, bscale, rgb_widget.bscale_parm);

   rgb_widget.rscale = rscale;
   rgb_widget.gscale = gscale;
   rgb_widget.bscale = bscale;

   rgb_widget.rgbbuf = rgbbuf;
   rgb_widget.pixbuf = gdk_pixbuf_new_from_data (rgbbuf);

   rgb_widget.dirty = 1;
   if (rgb_widget.widget != NULL)
     gtk_widget_queue_draw (rgb_widget.widget);
} % }}}

public define rgb_display_widget_set_rgb (rgb_widget, r, g, b) % {{{
{
   rgb_widget.r = r;
   rgb_widget.g = g;
   rgb_widget.b = b;

   variable dims; (dims,,) = array_info (r);
   variable w = dims[1];
   variable h = dims[0];
   
   if ((h != rgb_widget.height) or (w != rgb_widget.width) 
       or (rgb_widget.pixbuf == NULL))
     {
	rgb_widget.height = h;
	rgb_widget.width = w;
	rgb_widget.zx = 1;
	rgb_widget.zy = 1;
	rgb_widget.ax = 0;
	rgb_widget.ay = 0;
	rgb_widget.rgbbuf = NULL;
     }
   
   variable rscale = rgb_widget.rscale;
   variable gscale = rgb_widget.gscale;
   variable bscale = rgb_widget.bscale;
   rgb_widget.rscale = LINEAR_SCALING;
   rgb_widget.gscale = LINEAR_SCALING;
   rgb_widget.bscale = LINEAR_SCALING;

   scale_rgb (rgb_widget, rscale, gscale, bscale);
} % }}}

private define zoom_by_factor_callback (rgb_widget, fx, fy) % {{{
{
   zoom_by_factor (rgb_widget, fx, fy);
   gtk_widget_queue_draw (rgb_widget.widget);
} % }}}

private define motion_notify_callback (win, event, rgb_widget) % {{{
{
   if (rgb_widget.freeze_wcs_display)
     return 1;

   variable x, y;
   gtk_widget_get_pointer (win, &x, &y);
   (x,y) = zoomed_xy_to_unzoomed_xy (rgb_widget, x, y);
   % The y values are flipped.  So, 
   y = rgb_widget.height - y;

   variable s;
   if (rgb_widget.wcs_callback != NULL)
     s = (@rgb_widget.wcs_callback)(x, y, __push_args (rgb_widget.wcs_callback_args));
   else
     s = sprintf ("(%g,%g)", x, y);

   if (s != NULL)
     gtk_label_set_text (rgb_widget.xy_label_area, s);

   return 1;
} % }}}

private define rgb_widget_button_press_callback (win, event, rgb_widget) % {{{
{
   variable f = rgb_widget.freeze_wcs_display;
   if (event.button == 3)
     {
	rgb_widget.freeze_wcs_display = 0;
	motion_notify_callback (win, event, rgb_widget);
	rgb_widget.freeze_wcs_display = not (f);
	return 1;
     }
   if (f)
     {
	rgb_widget.freeze_wcs_display = 0;
	motion_notify_callback (win, event, rgb_widget);
	rgb_widget.freeze_wcs_display = 1;
	return 1;
     }
   variable x = event.x;
   variable y = event.y;
   
   (x,y) = zoomed_xy_to_unzoomed_xy (rgb_widget, x, y);
   pan_to_center_unzoomed_pixel (rgb_widget, x, y);

   gtk_widget_queue_draw (rgb_widget.widget);
   return 1;
} % }}}

private define zoom_out_callback (rgb_widget) % {{{
{
   zoom_by_factor_callback (rgb_widget, 0.5, 0.5);
} % }}}

private define zoom_in_callback (rgb_widget) % {{{
{
   zoom_by_factor_callback (rgb_widget, 2.0, 2.0);
} % }}}

private define pow_expon_changed_callback (spinner_adj,  % {{{
		spinner, parm_array, button_callback, button, callback_args)
{
   variable val = gtk_spin_button_get_value (spinner);
   variable inc = 1;
   
   if (val <= 0.019)
     inc = 0.001;
   else if (val <= 0.19)
     inc = 0.01;
   else if (val <= 1.0)
     inc = 0.1;
   gtk_spin_button_set_increments (spinner, inc, 0);
   parm_array[0] = val;

   (@button_callback) (button, __push_args(callback_args));
} % }}}

private define make_spin_box (parm_array, button_callback, button, callback_args) % {{{
{
   parm_array[0] = 0.25;
   variable spinner_adj = gtk_adjustment_new (parm_array[0], 0.001, 10.0, 0.1, 0.1, 0);
   variable spinner = gtk_spin_button_new (spinner_adj, 0.001, 3);
   gtk_spin_button_set_numeric (spinner, 1);
   gtk_spin_button_set_update_policy (spinner, GTK_UPDATE_IF_VALID);
   () = g_signal_connect (spinner_adj, "value_changed", 
		     &pow_expon_changed_callback, spinner, parm_array,
		     button_callback, button, callback_args);
   
   return spinner;
} % }}}

private define make_radio_button () % {{{
{
   % Usage: make_radio_button (box, group, label, callback, ...)
   variable box, group, label, parm_array, callback, callback_args;
   callback_args = __pop_args (_NARGS-5);
   (box, group, label, parm_array, callback) = ();

   variable button = gtk_radio_button_new_with_label (group, label);
   () = g_signal_connect (button, "clicked",
			  callback, __push_args(callback_args));
   if (label == "Pow")
     {
	variable spacing = 0;
	variable hbox = gtk_hbox_new (0, spacing);
	gtk_box_pack_start (box, hbox, 1, 1, 0);

	variable spinbox = make_spin_box (parm_array, callback, button, callback_args);
	gtk_box_pack_start (hbox, button, 0, 0, spacing);
	gtk_box_pack_end (hbox, spinbox, 0, 0, 0);
     }
   else gtk_box_pack_start (box, button, 1, 1, 0);
   group = gtk_radio_button_get_group (button);

   return (button, group);
} % }}}

private define color_scale_button_callback (button, rgb_widget, color, scaling) % {{{
{
   variable rscale, bscale, gscale;

   if (0 == gtk_toggle_button_get_active (button))
     return;

   rscale = rgb_widget.rscale;
   gscale = rgb_widget.gscale;
   bscale = rgb_widget.bscale;

   switch (color)
     {
      case "Red":
	rgb_widget.rscale = NULL;
	rscale = scaling;
     }
     {
      case "Green":
	rgb_widget.gscale = NULL;
	gscale = scaling;
     }
     {
      case "Blue":
	rgb_widget.bscale = NULL;
	bscale = scaling;
     }

   scale_rgb (rgb_widget, rscale, gscale, bscale);
} % }}}

private define make_color_scaling_buttons (color, rgb_widget, parm_array) % {{{
{
   variable frame = gtk_frame_new (sprintf ("%s Scaling", color));
   variable button, box, group;
   
   box = gtk_vbox_new (1, 0);
   gtk_container_add (frame, box);

   group = NULL;

   (button, group) = make_radio_button (box, group, "Linear", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, LINEAR_SCALING);

   gtk_toggle_button_set_active (button, TRUE);
   
   (button, group) = make_radio_button (box, group, "Sqrt", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, SQRT_SCALING);

   (button, group) = make_radio_button (box, group, "Log", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, LOG_SCALING);
#iffalse
   (button, group) = make_radio_button (box, group, "LogLog", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, LOGLOG_SCALING);
#endif
   (button, group) = make_radio_button (box, group, "Pow", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, POW_SCALING);

   (button, group) = make_radio_button (box, group, "Zero", parm_array,
					&color_scale_button_callback,
					rgb_widget, color, ZERO_SCALING);
   return frame;
} % }}}

private define interp_button_callback (button, rgb_widget, method) % {{{
{
   if (0 == gtk_toggle_button_get_active (button))
     return;

   rgb_widget.interp_method = method;
   rgb_widget.dirty = 1;
   gtk_widget_queue_draw (rgb_widget.widget);
} % }}}

private define make_interp_buttons (rgb_widget) % {{{
{
   variable frame = gtk_frame_new ("Interpolation");
   variable button, box, group;

   box = gtk_vbox_new (1, 0);
   gtk_container_add (frame, box);

   group = NULL;
   variable default_method = GDK_INTERP_NEAREST;

   (button, group) = make_radio_button (box, group, "Nearest Neighbor", NULL,
					&interp_button_callback, rgb_widget,
					GDK_INTERP_NEAREST);
   if (default_method == GDK_INTERP_NEAREST)
     gtk_toggle_button_set_active (button, 1);

   (button, group) = make_radio_button (box, group, "Bilinear", NULL,
					&interp_button_callback, rgb_widget,
					GDK_INTERP_BILINEAR);
   if (default_method == GDK_INTERP_BILINEAR)
     gtk_toggle_button_set_active (button, 1);
   
   (button, group) = make_radio_button (box, group, "Hyperbolic", NULL,
					&interp_button_callback, rgb_widget,
					GDK_INTERP_HYPER);
   if (default_method == GDK_INTERP_HYPER)
     gtk_toggle_button_set_active (button, 1);

   rgb_widget.interp_method = GDK_INTERP_NEAREST;

   return frame;
} % }}}

private define ok_callback (rgb_widget) % {{{
{
   if (rgb_widget.ok_callback == NULL)
     {
	gtk_widget_destroy(rgb_widget.window);
	gtk_main_quit ();
	return;
     }

   (@rgb_widget.ok_callback) (__push_args (rgb_widget.ok_callback_args));
} % }}}

public define rgb_display_widget_set_wcs_callback () % {{{
{
   variable rgb_widget, func, args;
   args = __pop_args (_NARGS-2);
   (rgb_widget, func) = ();
   rgb_widget.wcs_callback = func;
   rgb_widget.wcs_callback_args = args;
} % }}}

public define rgb_display_widget_set_ok_callback () % {{{
{
   variable rgb_widget, func, args;
   args = __pop_args (_NARGS-2);
   (rgb_widget, func) = ();
   rgb_widget.ok_callback = func;
   rgb_widget.ok_callback_args = args;
} % }}}

public define rgb_display_widget_new () % {{{
{
   variable r, g, b;
   switch(_NARGS) 
   { case 1: r = (); r = normalize(r); g = r; b = r; }
   { case 3: (r,g,b) = (); }
   { usage( sprintf("%s (red [, green , blue])",_function_name)); }

   variable rgb_widget = struct
     {
	window, widget,
	gc, pixbuf, width, height, 	       %  unzoomed
	zoomed_pixbuf, da_width, da_height,
	zx, ax, zy, ay,	       %  zoom and offsets
	dirty,
	interp_method,
	
	% hooks
	wcs_callback, wcs_callback_args,
	ok_callback, ok_callback_args,

	main_vbox,
	main_hbox,
	button_hbox,
	xy_label_area,
	r, g, b, rgbbuf,
	rscale, gscale, bscale,
	rscale_parm, gscale_parm, bscale_parm,
	freeze_wcs_display,
     };
   rgb_widget.rscale_parm = Double_Type[1];
   rgb_widget.gscale_parm = Double_Type[1];
   rgb_widget.bscale_parm = Double_Type[1];

   rgb_display_widget_set_rgb (rgb_widget, r, g, b);
   
   variable window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   rgb_widget.window = window;

   gtk_window_set_title(window, "RGB Image Display");
   gtk_window_set_resizable(window, 1);
   () = g_signal_connect(window, "destroy", &gtk_main_quit);

   variable spacing = 5;

   variable vbox = gtk_vbox_new (0, spacing);
   gtk_container_add (window, vbox);
   rgb_widget.main_vbox = vbox;

   variable hbox_top = gtk_hbox_new (0, spacing);
   gtk_box_pack_start (vbox, hbox_top, 1, 1, 0);
   rgb_widget.main_hbox = hbox_top;

   variable widget = gtk_layout_new (NULL,NULL);
   rgb_widget.widget = widget;
   gtk_box_pack_start (hbox_top, widget, 1, 1, 1);
   rgb_widget.da_width = min([512, rgb_widget.width]);
   rgb_widget.da_height = min([512, rgb_widget.height]);

   pan_to_center_unzoomed_pixel (rgb_widget, 0.5*rgb_widget.width,
	 						0.5*rgb_widget.height);

   gtk_widget_set_size_request (widget, rgb_widget.da_width, rgb_widget.da_height);
   rgb_widget.dirty = 1;
   () = g_signal_connect (widget, "expose-event", &rgb_widget_expose, rgb_widget);
   () = g_signal_connect (widget, "button_press_event", &rgb_widget_button_press_callback, rgb_widget);
   () = g_signal_connect (widget, "motion_notify_event", &motion_notify_callback, rgb_widget);
   gtk_widget_set_events (widget, GDK_EXPOSURE_MASK
			  | GDK_BUTTON_PRESS_MASK
			  | GDK_POINTER_MOTION_MASK 
			  | GDK_POINTER_MOTION_HINT_MASK);

   rgb_widget.freeze_wcs_display = 0;

   variable vbox_right = gtk_vbox_new (0, spacing);
   gtk_box_pack_start (hbox_top, vbox_right, 0, 0, spacing);

   % Red, Green, Blue scaling buttons
   variable scale_box = make_color_scaling_buttons ("Red", rgb_widget, rgb_widget.rscale_parm);
   gtk_box_pack_start (vbox_right, scale_box, 0, 0, spacing);
   
   scale_box = make_color_scaling_buttons ("Green", rgb_widget, rgb_widget.gscale_parm);
   gtk_box_pack_start (vbox_right, scale_box, 0, 0, spacing);

   scale_box = make_color_scaling_buttons ("Blue", rgb_widget, rgb_widget.bscale_parm);
   gtk_box_pack_start (vbox_right, scale_box, 0, 0, spacing);

   % Interpolation Methods/Buttons
   scale_box = make_interp_buttons (rgb_widget);
   gtk_box_pack_start (vbox_right, scale_box, 0, 0, spacing);

   % Button area below main drawing area
   variable hbox = gtk_hbox_new (0, 0);
   gtk_box_pack_end (vbox, hbox, 0, 0, spacing);
   rgb_widget.button_hbox = hbox;

   variable button;

   button = gtk_button_new_with_label ("Zoom In");
   () = g_signal_connect_swapped (button, "clicked", &zoom_in_callback, rgb_widget);
   gtk_box_pack_start (hbox, button, 0, 0, spacing);
   
   button = gtk_button_new_with_label ("Zoom Out");
   () = g_signal_connect_swapped (button, "clicked", &zoom_out_callback, rgb_widget);
   gtk_box_pack_start (hbox, button, 0, 0, spacing);

   variable xy_label_area = gtk_label_new ("(0000,0000)");
   gtk_box_pack_start (hbox, xy_label_area, 0, 0, spacing);
   rgb_widget.xy_label_area = xy_label_area;

   button = gtk_button_new_with_label ("Ok");
   () = g_signal_connect_swapped (button, "clicked", &ok_callback, rgb_widget);
   gtk_box_pack_end (hbox, button, 0, 0, spacing);

   return rgb_widget;
} % }}}

private define img_motion_callback (x, y, img) % {{{
{
   variable i = int(y);
   variable j = int(x);
   variable dims; (dims,,) = array_info (img);
   if ((i < 0) or (i >= dims[0])
       or ((j < 0) or (j >= dims[1])))
     return sprintf ("(%g,%g,x)",x,y);

   sprintf ("(%g,%g,%g)", x, y, img[i,j]);
} % }}}

public define rgb_display_widget_view (img) % {{{
{
   % img is a 2d image.  Map it to grayscale 
   
   variable nimg = normalize (img);
   variable rgb_widget = rgb_display_widget_new (nimg, nimg, nimg);
   rgb_display_widget_set_wcs_callback (rgb_widget, &img_motion_callback, img);
   gtk_widget_show_all (rgb_widget.window);
   gtk_main ();
} % }}}

provide("rgbwidget");

#endif
