%  vwdraw.sl: this file is part of the VWhere SLgtk guilet {{{
%
%  Copyright (c) 2003-2009 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
%
% }}}

private variable FilterDescriptor = struct {
	plotd,			% GtkPlotDescriptor
	zoom,			% fractional zoom
	hshift, vshift,		% fractional shift left/right/up/down
	regions,		% list of regions applied to plot
	prefs,			% Plot-specific preferences
	x,y,			% x,y vectors
	key_being_pressed
};

% Polygon drawing functions {{{

private define poly_refresh(canvas, event, poly, dc)
{
   % Erase previous segment & draw new one (using drawing context dc)
   gdk_draw_line(dc.drawable, dc.gc, poly.x[-1], poly.y[-1], dc.x, dc.y);

   % Draw new segment [avoid get_pointer() trip to X server, use event x,y]
   gdk_draw_line(dc.drawable, dc.gc, poly.x[-1], poly.y[-1], event.x, event.y);
   dc.x = event.x;
   dc.y = event.y;

   return TRUE;					% mark event as handled
}

private define poly_next(canvas, event, poly, filtd)
{
   variable plotd = filtd.plotd;		% plot descriptor

   switch (event.button)
   { case 1:

	poly.x = [poly.x, event.x];
	poly.y = [poly.y, event.y];
	return TRUE;				% mark event as handled
   }
   { case 2:

	% Polygon now closed ...

 	if (length(poly.x) > 2) {		% exclude vacuous polygons

	   % ... so convert vertices from pixel- to canvas-space coords ...

	   variable i, x, y;

	   poly.x = typecast(poly.x, Double_Type);
	   poly.y = typecast(poly.y, Double_Type);

	   for (i=0; i < length(poly.x); i++) {
	      gtk_plot_canvas_get_position(plotd.canvas,
				int(poly.x[i]), int(poly.y[i]), &x, &y);
	      poly.x[i] = x;
	      poly.y[i] = y;
	   }

	   % ... and formally record the region
	   filtd.regions.append(SHAPE_POLYGON, poly.x, poly.y);
	}
   }

   % Both Mouse2 and Mouse3 terminate polygon drawing
   plotd.stop_interactive_draw();
   gtk_plot_canvas_refresh(plotd.canvas);

   return FALSE;				% mark event as unhandled
}

private define poly_expose(canvas, event, poly, filtd)
{
   variable pd = filtd.plotd;		% plot descriptor
   variable dc = pd.drawc;		% drawing context

   pd.disconnect(dc.expose_id);

   % We may now safely begin interactive drawing of polygon segment(s)
   dc.bpress_id = pd.connect("button_press_event", &poly_next, poly, filtd);
   dc.motion_id = pd.connect ("motion_notify_event", &poly_refresh, poly, dc);

   return FALSE;		% give next expose handler a chance, too
}

private define poly_start(event, vwd)
{
   variable filtd = vwd.filtd;
   variable pd = filtd.plotd;
   variable dc = pd.drawc;

   % Prohibit canvas selections while drawing polygon
   pd.block(g_object_get_data(pd.canvas, "select_item_id"));

   % Intercept next expose event, for proper segment drawing on slow X servers
   variable poly = struct{x, y};
   dc.expose_id = pd.connect("expose_event", &poly_expose, poly, filtd);

   poly.x = [event.x];		% Convenient to keep poly coords in pixel
   poly.y = [event.y];		% space until poly is fully drawn
   dc.x = event.x;
   dc.y = event.y;
}
% }}}

% Canvas {{{

private define canvas_rect_drag(canvas, x1, y1, x2, y2, vwd) % {{{
{
   % Don't add single points, lines, or other uselessly tiny regions
   if (orelse {abs(x1 - x2) < 0.005} {abs(y1 - y2) < 0.005} )
	return;

   % Ensure that (x1,y1) is top left, (x2,y2) is bottom right (canvas coords)
   variable t;
   if (x1 > x2) { t = x2; x2 = x1; x1 = t; }
   if (y1 > y2) { t = y2; y2 = y1; y1 = t; }

   variable type = vwd.tbox.curr;
   variable filtd = vwd.filtd;

   switch(type)
	{ case SHAPE_RECTANGLE or case SHAPE_ELLIPSE:

		filtd.regions.append(type, [x1, x2], [y1, y2]);
	}
	{ case TB_ZOOM_FIT:

		variable new_zoom = zoom_in(filtd.zoom);
		if (new_zoom != filtd.zoom) {

		   variable plotd = filtd.plotd;
		   (x1, y1) = _gtk_plot_transform(plotd, x1, y1, 1);
		   (x2, y2) = _gtk_plot_transform(plotd, x2, y2, 1);
		   _gtk_plot_set_xrange(plotd, x1, x2);
		   _gtk_plot_set_yrange(plotd, y2, y1);

		   gtk_plot_canvas_unselect(canvas);	% remove rubberband box
		   transform_regions(vwd);
		   filtd.zoom = new_zoom;
		   display_zoom(vwd);
		}

		return;
	}
	{ return; }

   gtk_plot_canvas_unselect(canvas);	% remove rubberband box
   gtk_plot_canvas_refresh(canvas);

} % }}}

private define canvas_changed(canvas, vwd) % {{{
{
   if (vwd.active_child == NULL)
      return;

   variable regions = vwd.filtd.regions;
   variable region = regions.widget2struct(vwd.active_child);

   if (region != NULL)
      regions.set_bbox(region);

   vwd.active_child = NULL;
} % }}}

private define canvas_select(canvas, event, child, vwd) % {{{
{
   switch (gtk_plot_canvas_get_child_type(child))
	{ case GTK_PLOT_CANVAS_PLOT:
	   
		if (vwd.tbox.curr == SHAPE_POLYGON)
		   poly_start(event, vwd);
	}
	{ case GTK_PLOT_CANVAS_NONE:

		if (vwd.tbox.curr != SHAPE_POLYGON) {
		   display_coords(vwd);
		   return TRUE;			% permit the selection
		}

		poly_start(event, vwd);
	}
	{ case GTK_PLOT_CANVAS_ELLIPSE	 or
	  case GTK_PLOT_CANVAS_RECTANGLE or
	  case GTK_PLOT_CANVAS_POLYGON   or
	  case GTK_PLOT_CANVAS_DATA:

		display_coords(vwd);
		return TRUE;			% permit the selection
	}

   return FALSE;				% veto the selection
} % }}}

private define canvas_delete(canvas,child,vwd) { return TRUE; }

private define canvas_motion(canvas,event,vwd) % {{{
{
   display_coords(vwd);
   return FALSE;			% propagate event, mark as unhandled
} % }}}

private define canvas_move_resize(canvas, child, dummy1, dummy2, vwd) % {{{
{
   switch (gtk_plot_canvas_get_child_type(child))
	{ case GTK_PLOT_CANVAS_ELLIPSE   or
	  case GTK_PLOT_CANVAS_RECTANGLE or
	  case GTK_PLOT_CANVAS_POLYGON:

		vwd.active_child = child;
	 	return TRUE;
	}
   return FALSE;
} % }}}

private define canvas_keypress(canvas, key, vwd) % {{{
{
   variable regions = vwd.filtd.regions;

   switch(key.keyval)
   { case GDK_Delete or case GDK_KP_Delete or case GDK_BackSpace:

	variable action = regions.delete;
   }
   { case GDK_e:

	action = regions.toggle_exclude;
	vwd.filtd.key_being_pressed = key.keyval;
   }
   { return FALSE; }

   variable selection = gtk_plot_canvas_get_active_item(canvas);
   variable type =  gtk_plot_canvas_get_child_type(selection);

    !if (type == GTK_PLOT_CANVAS_ELLIPSE ||
	 type == GTK_PLOT_CANVAS_RECTANGLE ||
	 type == GTK_PLOT_CANVAS_POLYGON)
	return FALSE;

   variable region_widget = gtk_plot_canvas_get_child_data(selection);
   variable region = regions.widget2struct(region_widget);
   @action(regions, region);
   _gtk_plot_redraw(vwd.filtd.plotd);

   return TRUE;
} % }}}

private define canvas_keyrelease(canvas, key, vwd) % {{{
{
   vwd.filtd.key_being_pressed = 0;
   return FALSE;
} % }}}

private define canvas_bpress(canvas, event, vwd) % {{{
{
   if (event.button == 3) { prefs_select_full(vwd); return TRUE; }
   return FALSE;
} % }}}

% }}}

% Point drawing {{{
private define maximize_axes_ranges(filtd)
{
   variable plotd = filtd.plotd;
   variable xmin = min(filtd.x), xmax = max(filtd.x);
   variable ymin = min(filtd.y), ymax = max(filtd.y);
   _gtk_plot_set_xrange(plotd, xmin, xmax);
   _gtk_plot_set_yrange(plotd, ymin, ymax);
   
   variable axis = plotd.dset.axes[0];
   axis.min = xmin; axis.max = xmax; axis.range = (xmax - xmin);
   axis = plotd.dset.axes[1];
   axis.min = ymin; axis.max = ymax; axis.range = (ymax - ymin);
}

define draw_points(vwd, x, y)
{
   variable fd, pd;				% filter and plot descriptors
   variable pp, fp = vwd.fprefs;		% plot and filter preferences

   if (x == NULL and y == NULL) {
	fd = vwd.filtd;				% drawing on existing plot pane
	pd = fd.plotd;
	pp = fd.prefs;
   }
   else
   {
	if (not (length(x) or length(y))) {
	   _error_dialog("empty plot","X or Y axis data is zero-length");
	   return 0;
	}

	fd = @FilterDescriptor;			% drawing on new plot pane
	fd.x = x;
	fd.y = y;
	pd   = NULL;

	if (vwd.filtd == NULL) {
	   fd.zoom    = 1.0;
	   fd.hshift  = 0.0;
	   fd.vshift  = 0.0;
	}
	else {
	   fd.zoom    = vwd.filtd.zoom;		% default new plots to
	   fd.hshift  = vwd.filtd.hshift;	% current zoom/shift
	   fd.vshift  = vwd.filtd.vshift;
	}

	fd.regions = region_list_new(fd);
	pp = plot_prefs_new();
	fd.prefs   = pp;
   }

   % Incrementally filter points before drawing, if requested
   variable incl_x, incl_y, excl_x = NULL, excl_y = NULL, filter = NULL;
   variable selected, num_selected, filtered;
   if (fp.incrfilt.value)
	filter = apply_region_filters(vwd);	% may also return NULL

   if (filter == NULL) {
	incl_x = fd.x;
	incl_y = fd.y;
	num_selected = vwd.veclen;
   }
   else {
	selected = where(filter);
	incl_x = fd.x[ selected ];
	incl_y = fd.y[ selected ];
	num_selected = length(selected);
   }
   if (num_selected == 0 and not fp.drawbg.value) {
	_error_dialog("empty plot","All points have been filtered, and "+
	      "you've\nswitched off background point drawing");
	return 0;
   }

   % If requested, plot the background (filtered) points (if any)
   if (andelse {fp.drawbg.value} {num_selected != vwd.veclen} ) {

	if (pp.align.value) {
	   excl_x = fd.x;
	   excl_y = fd.y;
	}
	else {
	   filtered = where(not(filter));
	   excl_x = fd.x[filtered];
	   excl_y = fd.y[filtered];
	}

	% In aligned mode draw foreground and background with same style
	variable linestyle;
	if (pp.align.value)
	   linestyle = pp.fgstyle.value;
	else
	   linestyle = pp.bgstyle.value;

	% Connect background (filtered) points with line color == symbol color
	if (pd == NULL)
	   pd = _gtk_plot(excl_x,excl_y, pp.bgcolor.value, linestyle);
	else
	   _gtk_oplot(pd, excl_x,excl_y, pp.bgcolor.value, linestyle);

	_gtk_plot_set_symbol(pd.dset, pp.sym.value, pp.bgcolor.value,
					pp.symsize.value, pp.symstyle.value);
   }

   % Now overplot the points which pass filters.  Using this ordering
   % ensures that filtered pts (when drawn) never eclipse selected pts
   if (num_selected) {

	if (pd == NULL) 
	   pd = _gtk_plot(incl_x, incl_y, gdk_black, pp.fgstyle.value);
	else
	   _gtk_oplot(pd, incl_x, incl_y, gdk_black, pp.fgstyle.value);

	_gtk_plot_set_symbol(pd.dset, pp.sym.value, pp.fgcolor.value,
					pp.symsize.value, pp.symstyle.value);

	% Ensure secondary, tertiary, etc plot panes match existing dimensions
	if (vwd.filtd != NULL) {

	   variable w, h, win = gtk_widget_get_window(vwd.filtd.plotd.canvas);
	   if (win != NULL) {
		gdk_drawable_get_size(win,&w,&h);
		gtk_plot_canvas_set_size(pd.canvas,w,h);
	   }
	}
   }

   vwd.filtd = fd;		% Reset current filter descriptor to this
   fd.plotd = pd;		% Rset current plot descriptor to this

   % It is visually convenient to draw selected (foreground) points relative
   % to the full range of the original dataset.  This happens automatically
   % when filtered (background) points are also drawn (because, by definition,
   % unfiltered + filtered == entire dataset).  When background is not drawn
   % the axes ranges will shrink to fit the foreground (or not), per whether
   % the fullrange filtering preference is off (or on).

   if (andelse {num_selected != vwd.veclen} {fp.fullrange.value})
	maximize_axes_ranges(fd);

   set_viewpoint(vwd);

   % Install custom signal handlers on the canvas
   () = pd.connect("changed", &canvas_changed, vwd);
   () = pd.connect("select_region", &canvas_rect_drag, vwd);
   () = pd.connect("select_item", &canvas_select, vwd ; save_id);
   () = pd.connect("delete_item", &canvas_delete, vwd);
   () = pd.connect("move_item", &canvas_move_resize, vwd);
   () = pd.connect("resize_item", &canvas_move_resize, vwd);
   () = pd.connect("key_press_event", &canvas_keypress, vwd);
   () = pd.connect("key_release_event", &canvas_keyrelease, vwd);
   () = pd.connect("motion_notify_event", &canvas_motion, vwd);
   () = pd.connect("button_press_event", &canvas_bpress, vwd);

   % Finally, display the plot(s) in the current pane
   gtk_widget_show_all(pd.canvas);

   return 1;
}
% }}}
