#ifndef __vtkTagTable_h
#define __vtkTagTable_h

#include "vtkObject.h"
#include "vtkTable.h"
#include "vtkMRML.h"

#include <vtksys/SystemTools.hxx>
#include <string>
#include <map>

class VTK_MRML_EXPORT vtkTagTable : public vtkObject
{
  public:
  // The Usual vtk class functions
  static vtkTagTable *New();
  vtkTypeRevisionMacro(vtkTagTable, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Get the vtkTable that contains user-defined attribute/value pairs.
  vtkGetStringMacro ( Name );
  vtkSetStringMacro ( Name );

  // Description:
  // When a new server (or the same server) has been selected by a user,
  // the available tags for query are refreshed. If the same server is selected
  // we would like to restore the 'selected' status of tags and the values
  // a user had chosen for each attribute. Teh RestoreSelectionState
  // flag is set to 1 if the same server has been re-selected by a user,
  // which lets the logic catch this condition and restore the state if possible.
  vtkGetMacro ( RestoreSelectionState, int);
  vtkSetMacro ( RestoreSelectionState, int);
  
  // Description:
  // Method that sets up default and required tags for a service.
  // Each derived class should fill out this method.
  virtual void Initialize() { };

  // Description:
  // Add a tag column to the TagTable
  virtual void AddOrUpdateTag ( const char *attribute, const char *value );
  // Description:
  // Add a tag column to the TagTable
  virtual void AddOrUpdateTag ( const char *attribute, const char *value, int selected );
  // Description:
  // Add a tag column to the TagTable
  virtual void AddUniqueTag ( const char *attribute, const char *value );
  // Description:
  // Replace a tag column in the TagTable
  virtual int UpdateTag ( const char *attribute, const char *value );
  virtual int UpdateTag ( const char *attribute, const char *value, int selected );

  // Description:
  // Get the attribute at a particular index in the TagTable.
  virtual const char *GetTagAttribute ( int index);

  // Description:
  // Get the value of a tag in the TagTable
  virtual const char *GetTagValue ( int index );

  // Description:
  // Get the value of a tag in the TagTable
  virtual const char *GetTagValue ( const char* attribute);
  
  // Description:
  // Checks TagTable to see if a attribute already exists.
  // Returns index of attribute if found, or -1 if it's not found.
  virtual int CheckTableForTag ( const char *attribute, const char *value );

  // Description:
  // Returns index of table for which tag exists, or returns -1
  virtual int CheckTableForTag ( const char *attribute );

  virtual void SelectTag ( const char *attribute );
  virtual void DeselectTag ( const char *attribute );
  virtual int IsTagSelected ( const char *attribute );
  
  
  // Description:
  // Deletes a column with the name matching "attribute" in the TagTable
  virtual void DeleteTag ( const char *attribute );

  // Description:
  // Clears all tags out of table.
  virtual void ClearTagTable ( );
  
  // Description:
  // Convenience method that returns the number of tags in TagTable.
  virtual int GetNumberOfTags ( );

  // Description:
  virtual void Copy( vtkTagTable *t);
  
  //BTX
  // describes attribute value pairs
  std::map<std::string, std::string> TagTable;
  // describes whether each attribute is selected or not.
  std::map<std::string, int > TagSelectionTable;
  //ETX

 private:
  char *Name;
  int RestoreSelectionState;

 protected:
  vtkTagTable();
  virtual ~vtkTagTable();
  vtkTagTable(const vtkTagTable&);
  void operator=(const vtkTagTable&);

};

#endif






