package SliMP3::Pages;

# $Id: Pages.pm,v 1.93 2003/06/30 23:42:07 blackketter Exp $
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License, 
# version 2.

use strict;
use File::Spec::Functions qw(:ALL);
use POSIX;
use SliMP3::Misc;

use SliMP3::Strings qw(string);

my($NEWLINE) = "\012";

sub home {
	my($client, $paramsref) = @_;
	my $output = "";
	my %listform;

	if (defined($$paramsref{'forget'})) {
		SliMP3::Client::forgetClient($$paramsref{'forget'});
	}
	if ($::nosetup) {
		$$paramsref{'nosetup'} = 1;
	}
	if ($::newVersion) {
		$$paramsref{'newVersion'} = $::newVersion;
	}
	
	# fill out the client setup choices
	foreach my $player (SliMP3::Client::clients()) {
		# every player gets a page.
		# next if (!SliMP3::Client::isSliMP3($player));
		$listform{'playername'} = SliMP3::Client::name($player);
		$listform{'playerid'} = SliMP3::Client::id($player);
		$listform{'player'} = $$paramsref{'player'};
		$listform{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'player_list'} .= &SliMP3::HTTP::filltemplatefile("homeplayer_list.html", \%listform);;
	}

	if (SliMP3::iTunes::useiTunesLibrary()) {
		$$paramsref{'nofolder'} = 1;
	}
	
	addStats($paramsref, [],[],[],[]);
 
	my $template = 'index.html';
	if ( $$paramsref{"path"}  =~ /home\.(htm|xml)/) { $template = 'home.html'; }
	
	$output .= &SliMP3::HTTP::filltemplatefile($template, $paramsref);
	return $output;
}

sub addStats {
	my $paramsref = shift;
	my $genreref = shift;
	my $artistref = shift;
	my $albumref = shift;
	my $songref = shift;
	
	if (!SliMP3::Misc::stillScanning()) {
		my $count = SliMP3::Info::songCount($genreref,$artistref,$albumref,$songref);
		$$paramsref{'song_count'} = $count . " " . lc(($count == 1 ? string('SONG') : string('SONGS')));
		$count = SliMP3::Info::artistCount($genreref,$artistref,$albumref,$songref);
		$$paramsref{'artist_count'} = $count . " " . lc(($count == 1 ? string('ARTIST') : string('ARTISTS')));
		$count	= SliMP3::Info::albumCount($genreref,$artistref,$albumref,$songref);
		$$paramsref{'album_count'} = $count . " " . lc(($count == 1 ? string('ALBUM') : string('ALBUMS')));
	}
}

sub browser {
	my($client, $httpclientsock, $output, $paramsref) = @_;

	my $dir = defined($$paramsref{'dir'}) ? $$paramsref{'dir'} : "";

	my $item;
	my $itempath;
	my @items;
	my %list_form;
	my $current_player = "";
	my $playlist;
	my $fulldir = SliMP3::Misc::virtualToAbsolute($dir);

	$::d_http && msg("browse virtual path: " . $dir . "\n");
	$::d_http && msg("with absolute path: " . $fulldir . "\n");

	if (defined($client)) {
		$current_player = SliMP3::Client::id($client);
	}

	if ($dir =~ /^__playlists/) {
		$playlist = 1;
		$$paramsref{'playlist'} = 1;

		if (!defined(SliMP3::Prefs::get("playlistdir") && !SliMP3::iTunes::useiTunesLibrary())) {
			$::d_http && msg("no valid playlists directory!!\n");
			return &SliMP3::HTTP::filltemplatefile("badpath.html", $paramsref);
		}

		if ($dir =~ /__current.m3u$/) {
			# write out the current playlist to a file with the special name __current
			if (defined($client)) {
				my $count = SliMP3::Playlist::count($client);
				$::d_http && msg("Saving playlist of $count items to $fulldir\n");
				if ($count) {
					SliMP3::Command::execute($client, ['playlist', 'save', '__current']);
				}
			} else {
				$::d_http && msg("no client, so we can't save a file!!\n");
				return &SliMP3::HTTP::filltemplatefile("badpath.html", $paramsref);
			}
		}

	} else {
		if (!defined(SliMP3::Prefs::get("mp3dir"))) {
			$::d_http && msg("no mp3dir, so we can't save a file!!\n");
			return &SliMP3::HTTP::filltemplatefile("badpath.html", $paramsref);;
		}
	}

	if (!$fulldir || !SliMP3::Info::isList($fulldir)) {
		# check if we're just showing itunes playlists
		if (SliMP3::iTunes::useiTunesLibrary()) {
			browser_addtolist_done($current_player, $httpclientsock, $paramsref, [], $output);
			return;
		} else {
			$::d_http && msg("the selected playlist $fulldir isn't good!!.\n");
			return &SliMP3::HTTP::filltemplatefile("badpath.html", $paramsref);
		}
	}

	# if they are renaming the playlist, let 'em.
	if ($playlist && $$paramsref{'newname'}) {
		my $newname = $$paramsref{'newname'};
		# don't allow periods, colons, control characters, slashes, backslashes, just to be safe.
		$newname =~ tr/.:\x00-\x1f\/\\/ /s;
		if (SliMP3::Info::isM3U($fulldir)) { $newname .= ".m3u"; };
		if (SliMP3::Info::isPLS($fulldir)) { $newname .= ".pls"; };
		
		if ($newname ne "") {
			my @newpath  = splitdir($fulldir);
			pop @newpath;
			my $container = catdir(@newpath);
			push @newpath, $newname;
			my $newfullname = catdir(@newpath);

			$::d_http && msg("renaming $fulldir to $newfullname\n");
			if ($newfullname ne $fulldir && !-e $newfullname && rename $fulldir, $newfullname) {
				SliMP3::Info::clearCache($container);
				SliMP3::Info::clearCache($fulldir);
				$fulldir = $newfullname;

				$dir = SliMP3::Misc::descendVirtual(SliMP3::Misc::ascendVirtual($dir), $newname);

				$$paramsref{'dir'} = $dir;
				$::d_http && msg("new dir value: $dir\n");
			} else {
				$::d_http && msg("Rename failed!\n");
				$$paramsref{'RENAME_WARNING'} = 1;
			}
		}
	} elsif ($playlist && $$paramsref{'delete'}) {
		if (-f $fulldir && unlink $fulldir) {
			my @newpath  = splitdir($fulldir);
			pop @newpath;
			my $container = catdir(@newpath);
			SliMP3::Info::clearCache($container);
			SliMP3::Info::clearCache($fulldir);
			$dir = SliMP3::Misc::ascendVirtual($dir);
			$$paramsref{'dir'} = $dir;
			$fulldir = SliMP3::Misc::virtualToAbsolute($dir);
		}
	}

	#
	# Make separate links for each component of the pwd
	#
	%list_form=();
#	$list_form{'player'} = SliMP3::HTTP::escape($current_player);
	$list_form{'player'} = $current_player;
	$list_form{'myClientState'} = $client;
	$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
	my $aggregate = "";
	if ($playlist) { $aggregate = "__playlists" };
	$$paramsref{'pwd_list'} = " ";
	my $lastpath;
	foreach my $c (splitdir($dir)) {
	    if ($c ne "" && $c ne "__playlists" && $c ne "__current.m3u") {
		    $aggregate = (defined($aggregate) && $aggregate ne '') ? catdir($aggregate, $c) : $c;        # incrementally build the path for each link
		    $list_form{'dir'}=SliMP3::HTTP::escape($aggregate);
            if ($c =~ /(.*)\.m3u$/) { $c = $1; }
		    
		    if (SliMP3::Info::isURL($c)) {
		    	$list_form{'shortdir'}= SliMP3::Info::standardTitle(undef, $c);
		    } else {
		    	$list_form{'shortdir'}=$c; #possibly make this into the TITLE of the playlist if this is a number
		    }
		    $$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browse_pwdlist.html", \%list_form);
		}
		if ($c) { $lastpath = $c; }
	}

	if (SliMP3::Info::isM3U($fulldir)) {
		$::d_http && msg("lastpath equals $lastpath\n");
		if ($lastpath eq "__current.m3u") {
			$$paramsref{'playlistname'} = string("UNTITLED");
			$$paramsref{'savebuttonname'} = string("SAVE");
		} else {
			$lastpath =~ s/(.*)\.m3u$/$1/;
			$$paramsref{'playlistname'} = $lastpath;
			$$paramsref{'savebuttonname'} = string("RENAME");
			$$paramsref{'titled'} = 1;
		}
	} elsif (SliMP3::Info::isPLS($fulldir)) {
			$lastpath =~ s/(.*)\.pls$/$1/;
			$$paramsref{'playlistname'} = $lastpath;
			$$paramsref{'savebuttonname'} = string("RENAME");
			$$paramsref{'titled'} = 1;
	}

	my $items = [];

	SliMP3::Scan::addToList($items, $fulldir, 0, undef,  
			\&browser_addtolist_done, $current_player, $httpclientsock, $paramsref, $items, $output);  
	
	# when this finishes, it calls the next function, browser_addtolist_done:
	return ''; # return '' means we'll take care of sending the output to the client (special case because we're going into the background)
}

sub browser_addtolist_done {
	my (	$current_player,
		$httpclientsock,
	 	$paramsref,
		$itemsref,
		$output,
		$numitems
	) = @_;
	
	if (defined $paramsref->{'dir'} && $paramsref->{'dir'} eq '__playlists' && SliMP3::iTunes::useiTunesLibrary()) {
		$::d_http && msg("just showing itunes playlists\n");
		push @$itemsref, @{SliMP3::iTunes::playlists()};
		if (SliMP3::iTunes::stillScanning()) {
			$paramsref->{'warn'} = 1;
		}
	} 
	
	$numitems = scalar @{ $itemsref };
	
	$::d_http && msg("browser_addtolist_done with $numitems items (". scalar @{ $itemsref } . ")\n");

	my $shortitem;
	${$paramsref}{'browse_list'} = " ";
	if ($numitems) {
		my %list_form;
		my $item;
		my $otherparams = 	'player=' . SliMP3::HTTP::escape($current_player) .
							'&dir=' . SliMP3::HTTP::escape($paramsref->{'dir'}) . '&';
							
		my @namearray;
		foreach $item ( @{$itemsref} ) {
			$::d_http && msg("browser_addtolist_done getting name for $item\n");
			if ($item) {
				push @namearray, SliMP3::Info::standardTitle(undef,$item);
			}
		}
		my ($start,$end);
		
		if (defined $paramsref->{'nopagebar'}){
			($start, $end) = simpleheader($numitems,
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											0);
		}
		else{
			($start,$end) = pagebar($numitems,
								$$paramsref{'path'},
								0,
								$otherparams,
								\$$paramsref{'start'},
								\$$paramsref{'browselist_header'},
								\$$paramsref{'browselist_pagebar'},
								$$paramsref{'skinOverride'}
								,$$paramsref{'itemsPerPage'});
		}
		my $itemnumber = $start;
		my $offset = $start % 2 ? 0 : 1;
		my $lastAnchor = '';
		my $filesort = SliMP3::Prefs::get('filesort');
		my ($cover,$thumb, $body, $type);
		
		# don't look up cover art info if we're browsing a playlist.
		if ($paramsref->{'dir'} && $paramsref->{'dir'} =~ /^__playlists/) { $thumb = 1; $cover = 1;}
		
		foreach $item ( @{$itemsref}[$start..$end] ) {
			
			# make sure the players get some time...
			SliMP3::Protocol::idle();
			
			# don't display and old unsaved playlist
			if ($item =~ /__current.m3u$/) { next; }
			%list_form=();
			#
			# There are different templates for directories and playlist items:
			#
			$shortitem = SliMP3::Misc::descendVirtual($paramsref->{'dir'},$item,$itemnumber);
			if (SliMP3::Info::isList($item)) {
				$list_form{'descend'}     = $shortitem;
			} elsif (SliMP3::Info::isSong($item)){
				$list_form{'descend'}     = 0;
				if (!defined $cover) {
					($body, $type) =  SliMP3::Info::coverArt($item,'cover');
					if (defined($body)) { $$paramsref{'coverart'} = 1; $$paramsref{'coverartpath'} = $shortitem;}
					$cover = $item;
				}
				if (!defined $thumb) {
					($body, $type) =  SliMP3::Info::coverArt($item,'thumb');
					if (defined($body)) { $$paramsref{'coverthumb'} = 1; $$paramsref{'thumbpath'} = $shortitem;}
					$thumb = $item;
				}
			}

			if ($filesort) {
				$list_form{'title'}		= SliMP3::Info::fileName($item);
			} else {
				$list_form{'title'}		= SliMP3::Info::standardTitle(undef,$item);
				$list_form{'artist'}    = SliMP3::Info::artist($item);
				$list_form{'album'}     = SliMP3::Info::album($item);
			}
			
			$list_form{'itempath'}  = SliMP3::Misc::virtualToAbsolute($item);
			$list_form{'odd'}	  	= ($itemnumber + $offset) % 2;
			$list_form{'player'}	= $current_player;
			addStats($paramsref, [],[],[],[]);
 						
			my $anchor = anchor($list_form{'title'},1);
			if ($lastAnchor ne $anchor) {
				$list_form{'anchor'}  = $anchor;
				$lastAnchor = $anchor;
			}
			$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
			$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile(($$paramsref{'playlist'} ? "browse_playlist_list.html" : "browse_list.html"), \%list_form);
			$itemnumber++;
		}
	} else {
		$$paramsref{'browse_list'} = &SliMP3::HTTP::filltemplatefile("browse_list_empty.html", {'skinOverride' => $$paramsref{'skinOverride'}});
	}

	$output .=  &SliMP3::HTTP::filltemplatefile(($$paramsref{'playlist'} ? "browse_playlist.html" : "browse.html"), $paramsref);

	&SliMP3::HTTP::addresponse($httpclientsock, $output);
}

#
# Send the status page (what we're currently playing, contents of the playlist)
#
sub status {
	my($client, $main_form_ref, $add_playlist) = @_;
	my $current_player;
	my $songcount = 0;
	 
	if (defined($client)) {
		$songcount = SliMP3::Playlist::count($client);

		if (SliMP3::Client::defaultName($client) ne SliMP3::Client::name($client)) {
			$$main_form_ref{'player_name'} = SliMP3::Client::name($client);
		}
		
		if (SliMP3::Playlist::shuffle($client) == 1) {
			$$main_form_ref{'shuffleon'} = "on";
		} elsif (SliMP3::Playlist::shuffle($client) == 2) {
			$$main_form_ref{'shufflealbum'} = "album";
		} else {
			$$main_form_ref{'shuffleoff'} = "off";
		}
	
		$$main_form_ref{'songtime'} = int(SliMP3::Playlist::songTime($client));
		if (SliMP3::Playlist::song($client)) { 
			my $dur = SliMP3::Info::durationSeconds(SliMP3::Playlist::song($client));
			if ($dur) { $dur = int($dur); }
			$$main_form_ref{'songduration'} = $dur; 
		}
		
		if (!SliMP3::Playlist::repeat($client)) {
			$$main_form_ref{'repeatoff'} = "off";
		} elsif (SliMP3::Playlist::repeat($client) == 1) {
			$$main_form_ref{'repeatone'} = "one";
		} else {
			$$main_form_ref{'repeatall'} = "all";
		}
	
		if("play" eq SliMP3::Playlist::playmode($client)) {
			$$main_form_ref{'modeplay'} = "Play";
		} elsif ("pause" eq SliMP3::Playlist::playmode($client)) {
			$$main_form_ref{'modepause'} = "Pause";
		} else {
			$$main_form_ref{'modestop'} = "Stop";
		}
		if (SliMP3::Playlist::rate($client) > 1) {
			$$main_form_ref{'rate'} = 'ffwd';
		} elsif (SliMP3::Playlist::rate($client) < 0) {
			$$main_form_ref{'rate'} = 'rew';
		} else {
			$$main_form_ref{'rate'} = 'norm';
		}
		
		$$main_form_ref{'sync'} = SliMP3::Playlist::syncwith($client);
		
		$$main_form_ref{'mode'} = SliMP3::Buttons::Common::mode($client);
		if (SliMP3::Client::isSliMP3($client)) {
			$$main_form_ref{'slimp3'} = 1;
			$$main_form_ref{'volume'} = int(SliMP3::Prefs::clientGet($client, "volume") + 0.5);
			$$main_form_ref{'bass'} = int(SliMP3::Prefs::clientGet($client, "bass") + 0.5);
			$$main_form_ref{'treble'} = int(SliMP3::Prefs::clientGet($client, "treble") + 0.5);
		}
		
		$$main_form_ref{'player'} = SliMP3::Client::id($client);
	}
	
	if ($songcount > 0) {
		my $song = SliMP3::Playlist::song($client);
		$$main_form_ref{'currentsong'}    = SliMP3::Playlist::currentSongIndex($client) + 1;
		$$main_form_ref{'thissongnum'}    = SliMP3::Playlist::currentSongIndex($client);
		$$main_form_ref{'songcount'}      = $songcount;
		$$main_form_ref{'songtitle'}      = SliMP3::Info::standardTitle(undef,$song);
		$$main_form_ref{'artist'} 	  = SliMP3::Info::artist($song);
		$$main_form_ref{'album'} 	  = SliMP3::Info::album($song);
		addsonginfo($client, $song, $main_form_ref);

		if (SliMP3::Prefs::get("playlistdir"))  { $$main_form_ref{'cansave'} = 1; };
	}
	
	my $output = "";

	$$main_form_ref{'playercount'} = SliMP3::Client::clientCount();
	
	my @players = SliMP3::Client::clients();
	if (scalar(@players) > 1) {
		my %clientlist = ();
		foreach my $eachclient (@players) {
			$clientlist{SliMP3::Client::id($eachclient)} =  SliMP3::Client::name($eachclient);
			if (SliMP3::Playlist::isSynced($eachclient)) {
				$clientlist{SliMP3::Client::id($eachclient)} .= " (".string('SYNCHRONIZED_WITH')." ".SliMP3::Playlist::syncwith($eachclient).")";
			}	
		}
		$$main_form_ref{'player_chooser_list'} = options(SliMP3::Client::id($client),\%clientlist,$$main_form_ref{'skinOverride'});
	}
	
	if ($add_playlist) {
		$$main_form_ref{'playlist'} = playlist($client, $main_form_ref);
	}
			  
	$output .= &SliMP3::HTTP::filltemplatefile($add_playlist ? "status.html" : "status_header.html", $main_form_ref);
	return $output;
}

sub playlist {
	my $client = shift; 
	my $main_form_ref = shift;
	
	my $songcount = 0;
	if (defined($client)) {$songcount = SliMP3::Playlist::count($client);}
	my $playlist_items = '';
	if ($songcount > 0) {
		my $item;
		my %list_form;
		if (SliMP3::Prefs::get("playlistdir"))  { $$main_form_ref{'cansave'} = 1; };
		
		my ($start,$end);
		
		if (defined $main_form_ref->{'nopagebar'}){
			($start, $end) = simpleheader($songcount,
											\$$main_form_ref{'start'},
											\$$main_form_ref{'playlist_header'},
											$$main_form_ref{'skinOverride'},
											$$main_form_ref{'itemsPerPage'},
											0);
		}
		else{
			($start,$end) = pagebar($songcount,
								$$main_form_ref{'path'},
								SliMP3::Playlist::currentSongIndex($client),
								"player=" . SliMP3::HTTP::escape(SliMP3::Client::id($client)) . "&", 
								\$$main_form_ref{'start'}, 
								\$$main_form_ref{'playlist_header'},
								\$$main_form_ref{'playlist_pagebar'},
								$$main_form_ref{'skinOverride'}
								,$$main_form_ref{'itemsPerPage'});
		}
		my $offset = $start % 2 ? 0 : 1; 
		my $webFormat = SliMP3::Prefs::getInd("titleFormat",SliMP3::Prefs::get("titleFormatWeb"));
		my $includeArtist =  ($webFormat !~ /ARTIST/);
		my $includeAlbum = ($webFormat !~ /ALBUM/) ;
		my $currsongind = SliMP3::Playlist::currentSongIndex($client);
		for( $item = $start; $item < $end + 1; $item++) {
			%list_form = ();
			$list_form{'myClientState'} = $client;
			$list_form{'num'}=$item;
			$list_form{'odd'} = ($item + $offset) % 2;
			if ($item == $currsongind) {
				$list_form{'currentsong'} = "current";
			} else {
				$list_form{'currentsong'} = undef;
			}
			$list_form{'nextsongind'} = $currsongind + (($item > $currsongind) ? 1 : 0);
			my $song = SliMP3::Playlist::song($client, $item);
			$list_form{'player'}	= $$main_form_ref{'player'};
			$list_form{'title'} 	= SliMP3::Info::standardTitle(undef,$song);
			if ($includeArtist) { $list_form{'artist'} 	= SliMP3::Info::artist($song);}
			if ($includeAlbum) { $list_form{'album'} 	= SliMP3::Info::album($song);} 
			$list_form{'start'}		= $$main_form_ref{'start'};
			$list_form{'skinOverride'}		= $$main_form_ref{'skinOverride'};
			$playlist_items .= &SliMP3::HTTP::filltemplatefile("status_list.html", \%list_form);
			SliMP3::Protocol::idle();
		}
	}
	$$main_form_ref{'playlist_items'} = $playlist_items;
	return  SliMP3::HTTP::filltemplatefile("playlist.html", $main_form_ref);
}

sub search {
	my($client, $paramsref) = @_;
	my $output = "";
	$$paramsref{'browse_list'} = " ";

	my $player = $$paramsref{'player'};
	my $query = $$paramsref{'query'};
	$$paramsref{'numresults'} = -1;

	my $descend;
	my $lastAnchor = '';
	if ($query) {
		my $otherparams = 'player=' . SliMP3::HTTP::escape($player) . 
		                  '&type=' . ($$paramsref{'type'} ? $$paramsref{'type'} : ''). 
		                  '&query=' . SliMP3::HTTP::escape($$paramsref{'query'}) . '&';
		if ($$paramsref{'type'} eq 'artist') {
			my @searchresults = SliMP3::Info::artists([], searchStringSplit($query), [], []);
			$$paramsref{'numresults'} = scalar @searchresults;
			$descend = 'true';
			my $itemnumber = 0;
			if ($$paramsref{'numresults'}) {
			
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar @searchresults,
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											0);
				}
				else{
					($start,$end) = alphapagebar(\@searchresults
								,$$paramsref{'path'}
								,$otherparams
								,\$$paramsref{'start'}
								,\$$paramsref{'searchlist_pagebar'}
								,1
								,$$paramsref{'skinOverride'}
								,$$paramsref{'itemsPerPage'});
				}
				foreach my $item ( @searchresults[$start..$end] ) {
					my %list_form=();
					$list_form{'genre'}	  = '*';
					$list_form{'artist'}  = $item;
					$list_form{'album'}	  = '';
					$list_form{'song'}	  = '';
					$list_form{'title'}   = $item;
					$list_form{'descend'} = $descend;
					$list_form{'player'} = $player;
					$list_form{'odd'}	  = ($itemnumber + 1) % 2;
					$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
					my $anchor = anchor($item, 1);
					if ($lastAnchor ne $anchor) {
						$list_form{'anchor'}  = $anchor;
						$lastAnchor = $anchor;
					}
					$itemnumber++;
					$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				}
			}
		} elsif ($$paramsref{'type'} eq 'album') {
			my @searchresults = SliMP3::Info::albums([], [], searchStringSplit($query), []);
			$$paramsref{'numresults'} = scalar @searchresults;
			$descend = 'true';
			my $itemnumber = 0;
			if ($$paramsref{'numresults'}) {
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar @searchresults,
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											0);
				}
				else{
					($start,$end) = alphapagebar(\@searchresults
								,$$paramsref{'path'}
								,$otherparams
								,\$$paramsref{'start'}
								,\$$paramsref{'searchlist_pagebar'}
								,1
								,$$paramsref{'skinOverride'}
								,$$paramsref{'itemsPerPage'});
				}
				foreach my $item ( @searchresults[$start..$end] ) {
					my %list_form=();
					$list_form{'genre'}	  = '*';
					$list_form{'artist'}  = '*';
					$list_form{'album'}	  = $item;
					$list_form{'song'}	  = '';
					$list_form{'title'}   = $item;
					$list_form{'descend'} = $descend;
					$list_form{'player'} = $player;
					$list_form{'odd'}	  = ($itemnumber + 1) % 2;
					$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
					my $anchor = anchor($item,1);
					if ($lastAnchor ne $anchor) {
						$list_form{'anchor'}  = $anchor;
						$lastAnchor = $anchor;
					}
					$itemnumber++;
					$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				}
			}
		} elsif ($$paramsref{'type'} eq 'song') {
			my @searchresults = SliMP3::Info::songs([], [], [], searchStringSplit($query), 1);
			$$paramsref{'numresults'} = scalar @searchresults;
			my $itemnumber = 0;
			if ($$paramsref{'numresults'}) {
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar @searchresults,
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											0);
				}
				else{
					($start,$end) = pagebar(scalar(@searchresults)
								,$$paramsref{'path'}
								,0
								,$otherparams
								,\$$paramsref{'start'}
								,\$$paramsref{'searchlist_header'}
								,\$$paramsref{'searchlist_pagebar'}
								,$$paramsref{'skinOverride'}
								,$$paramsref{'itemsPerPage'});
				}
				foreach my $item ( @searchresults[$start..$end] ) {
					my %list_form=();
					$list_form{'genre'}	  = SliMP3::Info::genre($item);
					$list_form{'artist'}  = SliMP3::Info::artist($item);
					$list_form{'album'}	  = SliMP3::Info::album($item);
					$list_form{'itempath'} = $item;
					$list_form{'title'}   = SliMP3::Info::standardTitle(undef, $item);
					$list_form{'descend'} = undef;
					$list_form{'player'} = $player;
					$list_form{'odd'}	  = ($itemnumber + 1) % 2;
					$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
					$itemnumber++;
					$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				}
			}
		}
	}

	$output .= &SliMP3::HTTP::filltemplatefile("search.html", $paramsref);
}

sub addsonginfo {
	my($client, $song, $paramsref) = @_;

	if (!$song) {
		my $song = SliMP3::Info::songPath($$paramsref{'genre'}, $$paramsref{'artist'}, $$paramsref{'album'}, $$paramsref{'track'});
	}

	if ($song) {
		$$paramsref{'genre'} = SliMP3::Info::genre($song);
		$$paramsref{'artist'} = SliMP3::Info::artist($song);
		$$paramsref{'composer'} = SliMP3::Info::composer($song);
		$$paramsref{'album'} = SliMP3::Info::album($song);
		$$paramsref{'title'} = SliMP3::Info::title($song);
		if (SliMP3::Info::fileLength($song)) { $$paramsref{'filelength'} = SliMP3::Misc::delimitThousands(SliMP3::Info::fileLength($song)); }
		$$paramsref{'duration'} = SliMP3::Info::duration($song);
		$$paramsref{'track'} = SliMP3::Info::trackNumber($song);
		$$paramsref{'year'} = SliMP3::Info::year($song);
		$$paramsref{'type'} = string(uc(SliMP3::Info::contentType($song)));
		$$paramsref{'tagversion'} = SliMP3::Info::tagVersion($song);
		
		my ($body, $type) =  SliMP3::Info::coverArt($song,'cover');
		if (defined($body)) { $$paramsref{'coverart'} = 1; }
		($body, $type) =  SliMP3::Info::coverArt($song,'thumb');
		if (defined($body)) { $$paramsref{'coverthumb'} = 1; }
		
		$$paramsref{'modtime'} = SliMP3::Misc::longDateF(SliMP3::Info::age($song)) . ", " . SliMP3::Misc::timeF(SliMP3::Info::age($song));

		# make urls in comments into links
		$$paramsref{'comment'} = SliMP3::Info::comment($song);
		my $comment = SliMP3::Info::comment($song);
		if ($comment) {
			if (!($comment =~ s!\b(http://[\-~A-Za-z0-9_/\.]+)!<a href=\"$1\" target=\"_blank\">$1</a>!igo)) {
				# handle emusic-type urls which don't have http://
				$comment =~ s!\b(www\.[\-~A-Za-z0-9_/\.]+)!<a href=\"http://$1\" target=\"_blank\">$1</a>!igo;
			}
		}
		$$paramsref{'comment'} = $comment;

		$$paramsref{'bitrate'} = SliMP3::Info::bitrate($song);

		my $url;
		if (SliMP3::Info::isURL($song)) {
			$url = $song;
		} else {
			my $loc = $song;
			my @path;
			$loc = SliMP3::Misc::fixPath($loc);
			my $curdir = SliMP3::Prefs::get('mp3dir');
			if ($loc =~ /^\Q$curdir\E(.*)/) {
				$url = '/music';
				@path = splitdir($1);
			} else {
				$url = 'file://';
				@path  = splitdir($song);
			}
			foreach my $item (@path) {
				$url .= '/' . SliMP3::HTTP::escape($item);
			}
			$url =~ s/\/\//\//;
		}
		$$paramsref{'url'} = $url;
		
		$$paramsref{'itempath'} = $song;
	}
}

sub songinfo {
	my($client, $paramsref) = @_;
	my $output = "";

	my $song = $$paramsref{'songurl'};
	addsonginfo($client, $song, $paramsref);
	$output .= &SliMP3::HTTP::filltemplatefile("songinfo.html", $paramsref);
}

sub generate_pwd_list {
        my ($genre, $artist, $album, $player) = @_;
        my %list_form;
        my $pwd_list = "";
        
	if (defined($genre) && $genre eq '*' && 
	    defined($artist) && $artist eq '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = '';
		$list_form{'artist'} = '*';
		$list_form{'genre'} = '*';
		$list_form{'player'} = $player;
                $list_form{'pwditem'} = string('BROWSE_BY_ALBUM');
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	} elsif (defined($genre) && $genre eq '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = '*';
		$list_form{'player'} = $player;
                $list_form{'pwditem'} = string('BROWSE_BY_ARTIST');
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	} else {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = '';
		$list_form{'player'} = $player;
                $list_form{'pwditem'} = string('BROWSE_BY_GENRE');
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	};

	if ($genre && $genre ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = $genre;
		$list_form{'player'} = $player;
		$list_form{'pwditem'} = $genre;
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}

	if ($artist && $artist ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = '';
		$list_form{'artist'} = $artist;
		$list_form{'genre'} = $genre;
		$list_form{'pwditem'} = $artist;
		$list_form{'player'} = $player;
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}

	if ($album && $album ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = $album;
		$list_form{'artist'} = $artist;
		$list_form{'genre'} = $genre;
		$list_form{'pwditem'} = $album;
		$list_form{'player'} = $player;
		$pwd_list .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}
	
	return $pwd_list;
}

sub browseid3 {
	my($client, $paramsref) = @_;
	my $output = "";
	my @items = ();

	my $song = $$paramsref{'song'};
	my $artist = $$paramsref{'artist'};
	my $album = $$paramsref{'album'};
	my $genre = $$paramsref{'genre'};
	my $player = $$paramsref{'player'};
	my $descend;
	my %list_form;
	my $itemnumber = 0;
	my $lastAnchor = '';

	# warn the user if the scanning isn't complete.
	if (SliMP3::Misc::stillScanning()) {
		$$paramsref{'warn'} = 1;
	}

	if (SliMP3::iTunes::useiTunesLibrary()) {
		$$paramsref{'itunes'} = 1;
	}

	my $genreref = [];  	if (defined($genre) && $genre ne '') { $genreref = [$genre]; }
	my $artistref = [];  	if (defined($artist) && $artist ne '') { $artistref = [$artist]; }
	my $albumref = [];  	if (defined($album) && $album ne '') { $albumref = [$album]; }
	my $songref = [];  		if (defined($song) && $song ne '') { $songref = [$song]; }
	
	addStats($paramsref, $genreref, $artistref, $albumref, $songref);

	if (defined($genre) && $genre eq '*' && 
	    defined($artist) && $artist eq '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = '';
		$list_form{'artist'} = '*';
		$list_form{'genre'} = '*';
		$list_form{'player'} = $player;
		$list_form{'pwditem'} = string('BROWSE_BY_ALBUM');
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
		$$paramsref{'browseby'} = 'BROWSE_BY_ALBUM';
	} elsif (defined($genre) && $genre eq '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = '*';
		$list_form{'player'} = $player;
		$list_form{'pwditem'} = string('BROWSE_BY_ARTIST');
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
		$$paramsref{'browseby'} = 'BROWSE_BY_ARTIST';
	} else {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = '';
		$list_form{'player'} = $player;
		$list_form{'pwditem'} = string('BROWSE_BY_GENRE');
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
		$$paramsref{'browseby'} = 'BROWSE_BY_GENRE';
	};

	if ($genre && $genre ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'artist'} = '';
		$list_form{'album'} = '';
		$list_form{'genre'} = $genre;
		$list_form{'player'} = $player;
		$list_form{'pwditem'} = $genre;
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}

	if ($artist && $artist ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = '';
		$list_form{'artist'} = $artist;
		$list_form{'genre'} = $genre;
		$list_form{'pwditem'} = $artist;
		$list_form{'player'} = $player;
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}

	if ($album && $album ne '*') {
		%list_form=();
		$list_form{'song'} = '';
		$list_form{'album'} = $album;
		$list_form{'artist'} = $artist;
		$list_form{'genre'} = $genre;
		$list_form{'pwditem'} = $album;
		$list_form{'player'} = $player;
		$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
		$$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_pwdlist.html", \%list_form);
	}
	my $otherparams = 'player=' . SliMP3::HTTP::escape($player) . 
					  '&genre=' . SliMP3::HTTP::escape($genre) . 
					  '&artist=' . SliMP3::HTTP::escape($artist) . 
					  '&album=' . SliMP3::HTTP::escape($album) . 
					  '&song=' . SliMP3::HTTP::escape($song) . '&';
	if (!$genre) {
		# Browse by Genre
		@items = SliMP3::Info::genres([], [$artist], [$album], [$song]);
		if (scalar(@items)) {
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar(@items),
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											0);
				}
				else{
					($start,$end) = alphapagebar(\@items
							,$$paramsref{'path'}
							,$otherparams
							,\$$paramsref{'start'}
							,\$$paramsref{'browselist_pagebar'}
							,0
							,$$paramsref{'skinOverride'}
							,$$paramsref{'itemsPerPage'});
				}
			$descend = 'true';
			foreach my $item ( @items[$start..$end] ) {
				%list_form=();
				$list_form{'genre'}	  = $item;
				if ($$paramsref{'includeItemStats'} && !SliMP3::Misc::stillScanning()) {
					$list_form{'artist_count'}	= SliMP3::Info::artistCount([$item],[],[],[]);
					$list_form{'album_count'}	= SliMP3::Info::albumCount([$item],[],[],[]);
					$list_form{'song_count'}	= SliMP3::Info::songCount([$item],[],[],[]);
				}
				$list_form{'artist'}  = $artist;
				$list_form{'album'}	  = $album;
				$list_form{'song'}	  = $song;
				$list_form{'title'}   = $item;
				$list_form{'descend'} = $descend;
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'mixable_descend'} = SliMP3::Info::isGenreMixable($item) && ($descend eq "true");
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				my $anchor = anchor($item);
				if ($lastAnchor ne $anchor) {
					$list_form{'anchor'}  = $anchor;
					$lastAnchor = $anchor;
				}
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				
				SliMP3::Protocol::idle();
			}
		}
	} elsif (!$artist) {
		# Browse by Artist
		@items = SliMP3::Info::artists([$genre], [], [$album], [$song]);
		if (scalar(@items)) {
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar(@items),
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											(scalar(@items) > 1));
				}
				else{
					($start,$end) = alphapagebar(\@items
							,$$paramsref{'path'}
							,$otherparams
							,\$$paramsref{'start'}
							,\$$paramsref{'browselist_pagebar'}
							,1
							,$$paramsref{'skinOverride'}
							,$$paramsref{'itemsPerPage'});
				}
			$descend = 'true';

			if (scalar(@items) > 1) {
				%list_form=();
				if ($$paramsref{'includeItemStats'} && !SliMP3::Misc::stillScanning()) {
					$list_form{'album_count'}	= SliMP3::Info::albumCount([$genre],['*'],[],[]);
					$list_form{'song_count'}	= SliMP3::Info::songCount([$genre],['*'],[],[]);
				}
				$list_form{'genre'}	  = $genre;
				$list_form{'artist'}  = '*';
				$list_form{'album'}	  = $album;
				$list_form{'song'}	  = $song;
				$list_form{'title'}   = string('ALL_ALBUMS');
				$list_form{'descend'} = $descend;
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
			}
			
			foreach my $item ( @items[$start..$end] ) {
				%list_form=();
				if ($$paramsref{'includeItemStats'} && !SliMP3::Misc::stillScanning()) {
					$list_form{'album_count'}	= SliMP3::Info::albumCount([$genre],[$item],[],[]);
					$list_form{'song_count'}	= SliMP3::Info::songCount([$genre],[$item],[],[]);
				}
				$list_form{'genre'}	  = $genre;
				$list_form{'artist'}  = $item;
				$list_form{'album'}	  = $album;
				$list_form{'song'}	  = $song;
				$list_form{'title'}   = $item;
				$list_form{'descend'} = $descend;
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'mixable_descend'} = SliMP3::Info::isArtistMixable($item) && ($descend eq "true");
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				my $anchor = anchor($item, 1);
				if ($lastAnchor ne $anchor) {
					$list_form{'anchor'}  = $anchor;
					$lastAnchor = $anchor;
				}
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				SliMP3::Protocol::idle();
			}
		}
	} elsif (!$album) {
		# Browse by Album
		@items = SliMP3::Info::albums([$genre], [$artist], [], [$song]);
		if (scalar(@items)) {
				my ($start,$end);
				if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar(@items),
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											(scalar(@items) > 1));
				}
				else{
					($start,$end) = alphapagebar(\@items
							,$$paramsref{'path'}
							,$otherparams
							,\$$paramsref{'start'}
							,\$$paramsref{'browselist_pagebar'}
							,1
							,$$paramsref{'skinOverride'}
							,$$paramsref{'itemsPerPage'});
				}
			$descend = 'true';
			
			if (scalar(@items) > 1) {
				%list_form=();
				if ($$paramsref{'includeItemStats'} && !SliMP3::Misc::stillScanning()) {
					$list_form{'song_count'}	= SliMP3::Info::songCount([$genre],[$artist],['*'],[]);
				}
				$list_form{'genre'}	  = $genre;
				$list_form{'artist'}  = $artist;
				$list_form{'album'}	  = '*';
				$list_form{'song'}	  = $song;
				$list_form{'title'}   = string('ALL_SONGS');
				$list_form{'descend'} = 1;
				$list_form{'player'} =  $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
			}
			
			foreach my $item ( @items[$start..$end] ) {
				%list_form=();
				if ($$paramsref{'includeItemStats'} && !SliMP3::Misc::stillScanning()) {
					$list_form{'song_count'}	= SliMP3::Info::songCount([$genre],[$artist],[$item],[]);
				}
				$list_form{'genre'}	  = $genre;
				$list_form{'artist'}  = $artist;
				$list_form{'album'}	  = $item;
				$list_form{'song'}	  = $song;
				$list_form{'title'}   = $item;
				$list_form{'descend'} = $descend;
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				my $anchor = anchor($item,1);
				if ($lastAnchor ne $anchor) {
					$list_form{'anchor'}  = $anchor;
					$lastAnchor = $anchor;
				}
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				SliMP3::Protocol::idle();
			}
		}
	} else {
		# Browse by ?
		@items = SliMP3::Info::songs([$genre], [$artist], [$album], [], ($album eq '*'));
		if (scalar(@items)) {
			my ($start,$end);
			if (defined $paramsref->{'nopagebar'}){
					($start, $end) = simpleheader(scalar(@items),
											\$$paramsref{'start'},
											\$$paramsref{'browselist_header'},
											$$paramsref{'skinOverride'},
											$$paramsref{'itemsPerPage'},
											(scalar(@items) > 1));
			}
			else{
				($start,$end) = pagebar(scalar(@items)
							,$$paramsref{'path'}
							,0
							,$otherparams
							,\$$paramsref{'start'}
							,\$$paramsref{'browselist_header'}
							,\$$paramsref{'browselist_pagebar'}
							,$$paramsref{'skinOverride'}
							,$$paramsref{'itemsPerPage'});
			}
			$descend = undef;
			
			if (scalar(@items) > 1) {
				%list_form=();
				$list_form{'genre'}	  = $genre;
				$list_form{'artist'}  = $artist;
				$list_form{'album'}	  = $album;
				$list_form{'song'}	  = '*';
				$list_form{'descend'} = 'true';
				$list_form{'title'}   = string('ALL_SONGS');
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
			}

			foreach my $item ( @items[$start..$end] ) {
				%list_form=();
				my $title = SliMP3::Info::standardTitle(undef, $item);
				$list_form{'genre'}	  = SliMP3::Info::genre($item);
				$list_form{'artist'}  = SliMP3::Info::artist($item);
				$list_form{'album'}	  = $album; # SliMP3::Info::album($item);
				$list_form{'itempath'} = $item; 
				$list_form{'item'} = $item;
				$list_form{'title'}   = $title;
				$list_form{'descend'} = $descend;
				$list_form{'player'} = $player;
				$list_form{'odd'}	  = ($itemnumber + 1) % 2;
				$list_form{'mixable_not_descend'} = SliMP3::Info::isSongMixable($item);
				$list_form{'skinOverride'} = $$paramsref{'skinOverride'};
				$itemnumber++;
				$$paramsref{'browse_list'} .= &SliMP3::HTTP::filltemplatefile("browseid3_list.html", \%list_form);
				SliMP3::Protocol::idle();
			}
			my ($body, $type) =  SliMP3::Info::coverArt($items[$start]);
			if (defined($body)) { $$paramsref{'coverart'} = 1; $$paramsref{'coverartpath'} = $items[$start];}
		}
	}
	$$paramsref{'descend'} = $descend;

	$output .= &SliMP3::HTTP::filltemplatefile("browseid3.html", $paramsref);
}

sub mood_wheel {
	my($client, $paramsref) = @_;
	my $output = "";
	my @items = ();

	my $song = $$paramsref{'song'};
	my $artist = $$paramsref{'artist'};
	my $album = $$paramsref{'album'};
	my $genre = $$paramsref{'genre'};
	my $player = $$paramsref{'player'};
	my $itemnumber = 0;
	
        if (defined $artist && $artist ne "") {
                @items = SliMP3::MoodLogic::getMoodWheel(SliMP3::Info::moodLogicArtistId($artist), 'artist');
        } elsif (defined $genre && $genre ne "" && $genre ne "*") {
                @items = SliMP3::MoodLogic::getMoodWheel(SliMP3::Info::moodLogicGenreId($genre), 'genre');
	} else {
                $::d_moodlogic && msg('no/unknown type specified for mood wheel');
                return undef;
	}

        $$paramsref{'pwd_list'} = &generate_pwd_list($genre, $artist, $album, $player);
        $$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("mood_wheel_pwdlist.html", $paramsref);
        
	foreach my $item ( @items ) {
                my %list_form=();
                $list_form{'mood'} = $item;
                $list_form{'genre'} = $genre;
		$list_form{'artist'}  = $artist;
		$list_form{'album'} = $album;
		$list_form{'player'} = $player;
		$list_form{'odd'} = ($itemnumber + 1) % 2;
		$itemnumber++;
                $$paramsref{'mood_list'} .= &SliMP3::HTTP::filltemplatefile("mood_wheel_list.html", \%list_form);
	}
        
	$output .= &SliMP3::HTTP::filltemplatefile("mood_wheel.html", $paramsref);
	return $output;
}

sub instant_mix {
	my($client, $paramsref) = @_;
	my $output = "";
	my @items = ();

	my $song = $$paramsref{'song'};
	my $artist = $$paramsref{'artist'};
	my $album = $$paramsref{'album'};
	my $genre = $$paramsref{'genre'};
	my $player = $$paramsref{'player'};
	my $mood = $$paramsref{'mood'};
	my $p0 = $$paramsref{'p0'};
	my $itemnumber = 0;

        $$paramsref{'pwd_list'} = &generate_pwd_list($genre, $artist, $album, $player);
        if (defined $mood && $mood ne "") {
            $$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("mood_wheel_pwdlist.html", $paramsref);
        }
        
        if (defined $song && $song ne "") {
            $$paramsref{'src_mix'} = SliMP3::Info::standardTitle(undef, $song);
        } elsif (defined $mood && $mood ne "") {
            $$paramsref{'src_mix'} = $mood;
        }
        
        $$paramsref{'pwd_list'} .= &SliMP3::HTTP::filltemplatefile("instant_mix_pwdlist.html", $paramsref);
        
        if (defined $song && $song ne "") {
                @items = SliMP3::MoodLogic::getMix(SliMP3::Info::moodLogicSongId($song), undef, 'song');
        } elsif (defined $artist && $artist ne "" && $artist ne "*" && $mood ne "") {
                @items = SliMP3::MoodLogic::getMix(SliMP3::Info::moodLogicArtistId($artist), $mood, 'artist');
        } elsif (defined $genre && $genre ne "" && $genre ne "*" && $mood ne "") {
                @items = SliMP3::MoodLogic::getMix(SliMP3::Info::moodLogicGenreId($genre), $mood, 'genre');
        } else {
            $::d_moodlogic && msg('no/unknown type specified for instant mix');
            return undef;
        }
        
	foreach my $item ( @items ) {
                my %list_form=();
                $list_form{'title'} = SliMP3::Info::infoFormat($item, 'TITLE (ARTIST)', 'TITLE');
		$list_form{'odd'} = ($itemnumber + 1) % 2;
		$itemnumber++;
                $$paramsref{'instant_mix_list'} .= &SliMP3::HTTP::filltemplatefile("instant_mix_list.html", \%list_form);
	}
        
        if (defined $p0) {
            SliMP3::Command::execute($client, ["playlist", $p0 eq "append" ? "append" : "play", $items[0]]);
		
	    for (my $i=1; $i<=$#items; $i++) {
                        SliMP3::Command::execute($client, ["playlist", "append", $items[$i]]);
	    }
        }
        
	$output .= &SliMP3::HTTP::filltemplatefile("instant_mix.html", $paramsref);
	return $output;
}

sub searchStringSplit {
	my $search_string = shift;
	my @strings = ();
	foreach my $ss (split(' ',$search_string)) {
		push @strings, "*" . $ss . "*";
	}
	return \@strings;
}

sub anchor {
	my $item = shift;
	my $suppressArticles = shift;
	
	if ($suppressArticles) {
		my $articles =  SliMP3::Prefs::get("ignoredarticles");
		$articles =~ s/\s+/|/g;
		$item =~ s/^($articles)\s+//i;
	}

	return SliMP3::Info::matchCase(substr($item, 0, 1));
}

sub options {
	#pass in the selected value and a hash of value => text pairs to get the option list filled
	#with the correct option selected.
	my ($selected,$optionref,$skinOverride) = @_;
	my $optionlist = '';
	foreach my $curroption (sort keys %{$optionref}) {
		$optionlist .= SliMP3::HTTP::filltemplatefile("select_option.html",{'selected' => ($curroption eq $selected)
											, 'key' => $curroption
											, 'value' => $optionref->{$curroption}
											, 'skinOverride' => $skinOverride});
	}
	return $optionlist;
}

#
# Build a simple header 
#

sub simpleheader {
	my $itemcount = shift;
	my $startref = shift; #will be modified
	my $headerref = shift; #will be modified
	my $skinOverride = shift;
	my $count = shift || SliMP3::Prefs::get('itemsPerPage');
	my $offset = shift;
	
	my $start = (defined($$startref) && $$startref ne '') ? $$startref : 0;
	if ($start >= $itemcount) { $start = $itemcount - $count; }
	$$startref = $start;
	my $end = $start+$count-1-$offset;
	if ($end >= $itemcount) { $end = $itemcount - 1;}
	$$headerref = &SliMP3::HTTP::filltemplatefile("pagebarheader.html", { "start" => ($start), "end" => ($end), "itemcount" => ($itemcount-1), 'skinOverride' => $skinOverride});
	return ($start,$end);
}


#
# Build a bar of links to multiple pages of items
#
sub pagebar {
	my $itemcount = shift;
	my $path = shift;
	my $currentitem = shift;
	my $otherparams = shift;
	my $startref = shift; #will be modified
	my $headerref = shift; #will be modified
	my $pagebarref = shift; #will be modified
	my $skinOverride = shift;
	my $count = shift || SliMP3::Prefs::get('itemsPerPage');

	my $start = (defined($$startref) && $$startref ne '') ? $$startref : (int($currentitem/$count)*$count);
	if ($start >= $itemcount) { $start = $itemcount - $count; }
	$$startref = $start;
	my $end = $start+$count-1;
	if ($end >= $itemcount) { $end = $itemcount - 1;}
	if ($itemcount > $count) {
		$$headerref = &SliMP3::HTTP::filltemplatefile("pagebarheader.html", { "start" => ($start+1), "end" => ($end+1), "itemcount" => $itemcount, 'skinOverride' => $skinOverride});

		my %pagebar = ();

		my $numpages = ceil($itemcount/$count);
		my $curpage = int($start/$count);
		my $pagesperbar = 10; #make this a preference
		my $pagebarstart = (($curpage - int($pagesperbar/2)) < 0 || $numpages <= $pagesperbar) ? 0 : ($curpage - int($pagesperbar/2));
		my $pagebarend = ($pagebarstart + $pagesperbar) > $numpages ? $numpages : ($pagebarstart + $pagesperbar);

		$pagebar{'pagesstart'} = ($pagebarstart > 0);

		if ($pagebar{'pagesstart'}) {
			$pagebar{'pagesprev'} = ($curpage - $pagesperbar) * $count;
			if ($pagebar{'pagesprev'} < 0) { $pagebar{'pagesprev'} = 0; };
		}

		if ($pagebarend < $numpages) {
			$pagebar{'pagesend'} = ($numpages -1) * $count;
			$pagebar{'pagesnext'} = ($curpage + $pagesperbar) * $count;
			if ($pagebar{'pagesnext'} > $pagebar{'pagesend'}) { $pagebar{'pagesnext'} = $pagebar{'pagesend'}; }
		}

		$pagebar{'pageprev'} = $curpage > 0 ? (($curpage - 1) * $count) : undef;
		$pagebar{'pagenext'} = ($curpage < ($numpages - 1)) ? (($curpage + 1) * $count) : undef;
		$pagebar{'otherparams'} = defined($otherparams) ? $otherparams : '';
		$pagebar{'skinOverride'} = $skinOverride;
		$pagebar{'path'} = $path;

		for (my $j = $pagebarstart;$j < $pagebarend;$j++) {
			$pagebar{'pageslist'} .= SliMP3::HTTP::filltemplatefile('pagebarlist.html'
							,{'currpage' => ($j == $curpage)
							,'itemnum0' => ($j * $count)
							,'itemnum1' => (($j * $count) + 1)
							,'pagenum' => ($j + 1)
							,'otherparams' => $otherparams
							,'skinOverride' => $skinOverride
							,'path' => $path});
		}
		$$pagebarref = SliMP3::HTTP::filltemplatefile("pagebar.html", \%pagebar);
	}
	return ($start,$end);
}

sub alphapagebar {
	my $itemsref = shift;
	my $path = shift;
	my $otherparams = shift;
	my $startref = shift; #will be modified
	my $pagebarref = shift; #will be modified
	my $ignorearticles = shift;
	my $skinOverride = shift;
	my $maxcount = shift || SliMP3::Prefs::get('itemsPerPage');
	
	my $itemcount = scalar(@$itemsref);

	my $start = $$startref;

	if (!$start) { 
		$start = 0;
	}
	
	if ($start >= $itemcount) { 
		$start = $itemcount - $maxcount; 
	}
	
	$$startref = $start;
	
	my $end = $itemcount-1;

	if ($itemcount > $maxcount/2) {

		my %pagebar_params = ();
		if (!defined($otherparams)) {
			$otherparams = '';
		}
		$pagebar_params{'otherparams'} =  $otherparams;
		
		my $lastLetter = '';
		my $lastLetterIndex = 0;
		my $pageslist = '';
		$end = -1;
		for (my $j = 0; $j < $itemcount; $j++) {

			my $curLetter = anchor($$itemsref[$j], $ignorearticles);

			if ($lastLetter ne $curLetter) {

				if (($j - $lastLetterIndex) > $maxcount) {
					if ($end == -1 && $j > $start) {
						$end = $j - 1;
					}
					$lastLetterIndex = $j;
				}
				$pageslist .= SliMP3::HTTP::filltemplatefile('alphapagebarlist.html'
								,{'currpage' => ($lastLetterIndex == $start)
								,'itemnum0' => $lastLetterIndex
								,'itemnum1' => ($lastLetterIndex + 1)
								,'pagenum' => $curLetter
								,'fragment' => ("#" . $curLetter)
								,'otherparams' => $otherparams
								,'skinOverride' => $skinOverride
								,'path' => $path});
				
				$lastLetter = $curLetter;
			}
		}
		
		if ($end == -1) {
			$end = $itemcount - 1;
		}

		$pagebar_params{'pageslist'} = $pageslist;
		$pagebar_params{'skinOverride'} = $skinOverride;
		$$pagebarref = SliMP3::HTTP::filltemplatefile("pagebar.html", \%pagebar_params);
	}
	
	return ($start,$end);
}
