/*
** Copyright (C) 2000 Rene Puls <rpuls@gmx.net>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>

#include <ctapi.h>
#include <ctbcs.h>

#include "smartcard.h"

extern int debug;		/* option: debug mode (from main.c) */

/* These bit masks are used to analyze the ATR ("answer to reset") of
   the card. */

#define ATR_H1_PROTO_MASK      240       /* 11110000 */
#define ATR_H1_PROTO_NONISO    128       /* 1000.... */
#define ATR_H1_PROTO_I2C       128       /* 1000.... */
#define ATR_H1_PROTO_3W        144       /* 1001.... */
#define ATR_H1_PROTO_2W        160       /* 1010.... */

#define ATR_H1_STRUCT_MASK     7         /* 00000111 */
#define ATR_H1_STRUCT_ISO_MASK 3         /* ......11 */
#define ATR_H1_STRUCT_COMMON   2         /* .....010 */
#define ATR_H1_STRUCT_PROPR    6         /* .....110 */
#define ATR_H1_STRUCT_SPECIAL  1         /* .......1 */

#define ATR_H2_READERR_FLAG    128       /* 1....... */
#define ATR_H2_UNIT_COUNT_MASK 120       /* 01111000 */
#define ATR_H2_UNIT_SIZE_MASK  7         /* 00000111 */

#define ATR_H3_SYNC_CARD       16        /* 00010000 */

#define ATR_H4_DIR_IS_VALID    128       /* 1....... */

/**************************************************************************
 * The smartcard_iso_xxx functions follow. These are direct wrappers for  *
 * sending the corresponding ISO 7816-4 commands.                         *
 **************************************************************************/

/* This executes the "select file" command, which selects the data
   area on that card that is to be accessed. (I really wouldn't call
   *that* a file though.) FID_AID is either 0x00 for file ID, or 0x04
   for application ID. XID is the AID/FID to be selected. CNR is the
   handle that was given to init_terminal(). */

static int 
smartcard_iso_select_file(unsigned short int const cnr, 
			  unsigned char const fid_aid,
			  unsigned short int const xid) 
{
  unsigned char cmd[7];          /* APDU */
  unsigned char response[512];   /* response APDU */
  unsigned char sad, dad;        /* source/destination address */
  unsigned short int lenc, lenr; /* command/reply length */
  int ret;                       /* return code of CT_data() */
  
  cmd[0] = 0x00;                /* CLA (class): ISO-7816-4 */
  cmd[1] = 0xA4;                /* INS (instruction): select file */
  cmd[2] = fid_aid;		/* P1 (parameter 1): 0x00=FID; 0x04=AID */
  cmd[3] = 0x00;                /* P2 (parameter 2): always 0x00 */
  cmd[4] = 0x02;                /* Lc (length of command data) */

  /* FIXME: this does not take the byte order into account! */
  cmd[5] = ((char *)(&xid))[1];
  cmd[6] = ((char *)(&xid))[0];

  sad = 2;			/* source address: host */
  dad = 0;			/* destination address: chip card 1 */
  lenc = 7;			/* length of command data */
  lenr = sizeof(response);	/* max. length of response */

  /* send the APDU */
  ret = CT_data(cnr, &dad, &sad, lenc, cmd, &lenr, response);

  if (debug)
    fprintf(stderr, "debug: iso_select_file(%d,0x%X,0x%X) = %d, lenr=%d\n",
	    cnr, fid_aid, xid, ret, lenr);
  if (debug)
    fprintf(stderr, "debug: response trail = %X %X\n", 
	    response[lenr-2], response[lenr-1]);

  if ((response[lenr-2] != 0x90) || (response[lenr-1] != 00))
    return -111;
  
  return ret;
}

/* This executes the "read_binary" command, which reads SIZE bytes
   from the card into BUF, beginning at (card) offset OFFSET. CNR is
   the handle that was given to init_terminal(). If SIZE is 0, this
   reads *all* data. The return value is the number of bytes that were
   actually read (or a CT-API error code). */

static int 
smartcard_iso_read_binary(unsigned short int const cnr, 
			  unsigned short int const offset,
			  char * const buf, 
			  unsigned char const size)
{
  unsigned char cmd[5];          /* APDU */
  unsigned char *response;       /* response APDU */
  unsigned char sad, dad;        /* source/destination address */
  unsigned short int lenc, lenr; /* command/reply length */
  int ret;                       /* return code of CT_data() */

  cmd[0] = 0x00;                /* CLA (class): ISO-7816-4 */
  cmd[1] = 0xB0;                /* INS (instruction): read binary */

  /* FIXME: will probably not work on machines with different byte order! */
  cmd[2] = ((char *)(&offset))[1];
  cmd[3] = ((char *)(&offset))[0];

  cmd[4] = size;                /* Le (length expected): bytes to read */

  lenc = sizeof(cmd);		/* length of command data */

  lenr = size+2;		/* max. response length: +2 bytes for trail */
  sad = 2;			/* source address: host */
  dad = 0;			/* destination address: chip card 1 */

  response = malloc(lenr);	/* allocate some memory for the response  */
  
  /* send the APDU */
  ret = CT_data(cnr, &dad, &sad, lenc, cmd, &lenr, response);
  
  if (debug)
    fprintf(stderr, "debug: iso_read_binary(%d,%d,...,%d) = %d, lenr=%d\n",
	    cnr, offset, size, ret, lenr);

  if (ret != OK)
    return ret;
  
  if (debug)
    fprintf(stderr, "debug: copying %d bytes to user buffer\n",
	    lenr-2);
  if (debug)
    fprintf(stderr, "debug: response trail = %X %X\n", 
	    response[lenr-2], response[lenr-1]);

  if ((response[lenr-2] != 0x90) || (response[lenr-1] != 00)) {
    free(response);
    return -111;
  }

  if (lenr > 2)
    memcpy(buf, response, lenr-2); /* copy to user buffer */

  free(response);

  return lenr-2;
}

/* This executes the "update_binary" command, which writes SIZE bytes
   from BUF to the card, beginning at (card) offset OFFSET. CNR is the
   handle that was given to init_terminal(). */

static int 
smartcard_iso_update_binary(unsigned short int const cnr, 
			    unsigned short int const offset,
			    char const * const buf, 
			    unsigned char const size)
{
  unsigned char cmd[5];          /* APDU */
  unsigned char response[512];   /* response APDU */
  unsigned char *apdu;		 /* constructed APDU */
  unsigned char sad, dad;        /* source/destination address */
  unsigned short int lenc, lenr; /* command/reply length */
  int ret;                       /* return code of CT_data() */

  cmd[0] = 0x00;                /* CLA (class): ISO-7816-4 */
  cmd[1] = 0xD6;                /* INS (instruction): update binary */
  cmd[2] = ((char *)(&offset))[1]; /* P1 (parameter 1): high part offset */
  cmd[3] = ((char *)(&offset))[0]; /* P2 (parameter 2): low part offset */
  cmd[4] = size;                /* Lc (length of command data) */

  lenc = sizeof(cmd) + size;	/* length of command data */
  lenr = sizeof(response);	/* max. length of response */
  sad = 2;			/* source address: host */
  dad = 0;			/* destination address: chip card 1 */

  apdu = malloc(lenc);		/* construct the APDU */
  memcpy(apdu, cmd, 5);
  memcpy(apdu+5, buf, size);

  /* send the APDU */
  ret = CT_data(cnr, &dad, &sad, lenc, apdu, &lenr, response);

  if (debug)
    fprintf(stderr, "debug: iso_update_binary(%d,%d,...,%d) = %d, lenr=%d\n",
	    cnr, offset, size, ret, lenr);
  if (debug)
    fprintf(stderr, "debug: response trail = %X %X\n", 
	    response[lenr-2], response[lenr-1]);

  free(apdu);

  if ((response[lenr-2] != 0x90) || (response[lenr-1] != 00))
    return -111;

  return ret;			/* that's it! */
}

/* This sends the "REQUEST ICC" command to the terminal, which
   activates the card and optionally returns some information about
   it. CNR is the handle that was given to init_terminal(). WHAT can
   be either 0x00 to return nothing, 0x01 to return the ATR (answer to
   reset), or 0x02 to return the historical bytes. The returned data
   is placed in BUF, and will always be at most six bytes long -- uh,
   I *think*.

   And don't ask me what the "historical bytes" are. The only thing is
   know is that these are exactly the same as the ATR on all my memory
   cards. Cool. */

int 
smartcard_ctbcs_request_icc(unsigned short int const cnr,
			    unsigned char const what,
			    char * const buf)
{
  unsigned char cmd[5];          /* APDU */
  unsigned char response[512];   /* response APDU */
  unsigned char sad, dad;        /* source/destination address */
  unsigned short int lenc, lenr; /* command/reply length */
  int ret;                       /* return code of CT_data() */
  
  cmd[0] = 0x20;                /* CLA (class): CTBCS command */
  cmd[1] = 0x12;                /* INS (instruction): Request ICC */
  cmd[2] = 0x01;		/* P1 (parameter 1): interface 1 */
  cmd[3] = what;                /* P2 (parameter 2): no/atr/hist */
  cmd[4] = 0;			/* Lc (length of command data) */

  sad = 2;			/* source address: host */
  dad = 1;			/* destination address: card terminal */
  lenc = 5;			/* length of command data */
  lenr = sizeof(response);	/* max. length of response */

  /* send the APDU */
  ret = CT_data(cnr, &dad, &sad, lenc, cmd, &lenr, response);

  if (debug)
    fprintf(stderr, "debug: ctbcs_request_icc(%d,%d,...) = %d, lenr=%d\n",
	    cnr, what, ret, lenr);
  if (debug)
    fprintf(stderr, "debug: response trail = %X %X\n", 
	    response[lenr-2], response[lenr-1]);

  if (ret != OK)
    return ret;
  else {
    if ((response[lenr-2] != 0x90) || ((response[lenr-1] != 00)
				       && (response[lenr-1] != CTBCS_SW2_REQUEST_ASYNC_OK)))
      return -111;

    if (lenr <= 6)		/* prevent a buffer overrun */
      memcpy(buf, response, lenr);
    return 0;
  }
}

/* This sends the command "GET STATUS" to the terminal, requesting
   information about the current card (WHAT=0x80) or the terminal
   manufacturer (0x46), which is then put into BUF, with SIZE bytes
   maximal length. CNR is the handle that was given to
   init_terminal(). */

int smartcard_ctbcs_get_status(unsigned short int const cnr,
			       unsigned char const what,
			       char * const buf,
			       unsigned short int const size)
{
  unsigned char cmd[5];          /* APDU */
  unsigned char response[512];   /* response APDU */
  unsigned char sad, dad;        /* source/destination address */
  unsigned short int lenc, lenr; /* command/reply length */
  int ret;                       /* return code of CT_data() */
  
  cmd[0] = 0x20;                /* CLA (class): CTBCS command */
  cmd[1] = 0x13;                /* INS (instruction): Get Status */
  cmd[2] = 0x00;		/* P1 (parameter 1): kernel */
  cmd[3] = what;		/* P2 (parameter 2): manufacturer/icc */
  cmd[4] = 0;			/* Lc (length of command data) */

  sad = 2;			/* source address: host */
  dad = 1;			/* destination address: card terminal */
  lenc = 5;			/* length of command data */
  lenr = sizeof(response);	/* max. length of response */

  /* send the APDU */
  ret = CT_data(cnr, &dad, &sad, lenc, cmd, &lenr, response);

  if (debug)
    fprintf(stderr, "debug: ctbcs_get_status(%d,%d,...,%d) = %d, lenr=%d\n",
	    cnr, what, size, ret, lenr);
  if (debug)
    fprintf(stderr, "debug: response trail = %X %X\n", 
	    response[lenr-2], response[lenr-1]);

  if (ret != OK)
    return ret;
  else {

    if ((response[lenr-2] != 0x90) || (response[lenr-1] != 00))
      return -111;
    
    memcpy(buf, response, lenr > size ? size : lenr);
    return 0;
  }
}

/**************************************************************************
 * All other functions which are accessible from outside follow:          *
 **************************************************************************/

/* Convert the CT-API error code ERRNUM to a human-readable error
   message. */

char const * smartcard_ctapi_error_msg(int const errnum)
{
  switch (errnum) {
  case -111:	/* this is not really CT-API, but who cares? */
    return "Unexpected APDU trail received";
  case ERR_INVALID:
    return "invalid argument to function call (ERR_INVALID)";
  case ERR_CT:
    return "CT-API error (ERR_CT)";
  case ERR_TRANS:
    return "transport error (ERR_TRANS)";
  case ERR_MEMORY:
    return "memory allocation error (ERR_MEMORY)";
  case ERR_HTSI:
    return "host transport service interface error (ERR_HTSI)";
  default:
    return "unknown error code";
  }
}

/* Initialize a smart card terminal. CNR is a handle that can be
   chosen by the caller and must be used for all subsequent calls for
   this terminal. PORT identifies the port to use (use PORT=0 for
   COM1, PORT=1 for COM2, and so on). */

int 
smartcard_init_terminal(unsigned short int const cnr, 
			unsigned short int const port)
{
  int ret;
  
  ret = CT_init(cnr, port);
  if (ret != OK) {
    if (debug)
      fprintf(stderr, "debug: CT_init(%d,%d) failed, ret=%d\n", 
	      cnr, port, ret);
    return ret;
  }
  
  return 0;
}

/* Close a smart card terminal. CNR is the handle that was given to
   init_terminal(). */

int 
smartcard_close_terminal(unsigned short int const cnr)
{
  return CT_close(cnr);		/* simple and stupid */
}

/* Activates the card in the terminal. Returns 0 if activated, or -1
   if the activation failed. CNR is the handle that was given to
   init_terminal(). */

int
smartcard_activate_card(unsigned short int const cnr)
{
  unsigned char buf[6];
  
  if (smartcard_ctbcs_request_icc(cnr, 0x00, buf) != 0)
    return -1;
  
  if ((buf[0] == 0x90) && (buf[1] == 0x00))
    return 0;
  else if ((buf[0] == CTBCS_SW1_REQUEST_ASYNC_OK)
	   && (buf[1] == CTBCS_SW2_REQUEST_ASYNC_OK))
    return 1;
  else
    return -1;
}

/* This tries to read SIZE bytes of data from the card at position
   OFFSET, storing the result in BUF. CNR is the handle that was given
   to init_terminal(). */

int smartcard_read_data(unsigned short int const cnr, 
			char * const buf,
			unsigned short int const size,
			unsigned short int const offset)
{
  int ret;			/* stores return codes */
  char *tmp = buf;		/* pointer for sequential reading */
  unsigned short int count = 0;	/* number of bytes read */
  int i;			/* temporary */

  ret = smartcard_iso_select_file(cnr, SMARTCARD_SELECT_FID, 
				  SMARTCARD_FID_ALL);
  if (ret != OK)
    return -1;

  if ((size > 0) && (size <= 255)) {
    if (debug)
      fprintf(stderr, "debug: read_data(%d,...,%d,%d) reading one chunk\n",
	      cnr, size, offset);
    return smartcard_iso_read_binary(cnr, offset, buf, size);
  } else {
    if (debug)
      fprintf(stderr, "debug: read_data(%d,...,%d,%d) reading multi-chunk\n",
	      cnr, size, offset);

    count = offset;
    do {			/* read all data, sequentially */
      if (size - count < 255)
	i = smartcard_iso_read_binary(cnr, count, tmp, size - count);
      else
	i = smartcard_iso_read_binary(cnr, count, tmp, 255);
      if (i > 0) {
	tmp += i;		/* move the pointer forward          */
	count += i;		/* ... and increase the byte counter */
      }
      if (debug)
	fprintf(stderr, "debug: reading chunk of %d bytes (%d total)\n",
		i, count);
    } while ((i > 0) && (count < (size ? size : 8192)));
    return count;
  }
}

/* This tries to write SIZE bytes of BUF to the card, starting at
   offset POS. CNR is the handle that was given to init_terminal(). */

int smartcard_write_data(unsigned short int cnr, 
			 char const * const buf, 
			 unsigned short int const size,
			 unsigned short int const offset)
{
  int ret;			/* stores return codes */
  char const * tmp = buf;	/* pointer for sequential reading */
  unsigned short int count = 0;	/* number of bytes written (total) */
  int c;			/* number of bytes to write per chunk */

  ret = smartcard_iso_select_file(cnr, SMARTCARD_SELECT_FID, 
				  SMARTCARD_FID_ALL);
  if (ret != OK)
    return -1;

  if (size <= 255) {
    if (debug)
      fprintf(stderr, "debug: write_data(%d,...,%d,%d) writing one chunk\n",
	      cnr, size, offset);
    return smartcard_iso_update_binary(cnr, offset, buf, size);
  } else {
    if (debug)
      fprintf(stderr, "debug: write_data(%d,...,%d,%d) writing multi-chunk\n",
	      cnr, size, offset);

    count = offset;
    do {			/* write all data, sequentially */

      if (size - count < 255)
	c = size - count;
      else
	c = 255;

      ret = smartcard_iso_update_binary(cnr, count, tmp, c);

      if (debug)
	fprintf(stderr, "debug: writing chunk of %d bytes at %d, ret=%d\n", 
		c, count, ret);

      if (ret != 0)
	return -1;

      count += c;
      tmp += c;

    } while (count < size);
    return 0;
  }
}

/* This function calls lots of other functions defined above to get
   some information about the card reader and the currently inserted
   card (if available). It also decodes the ATR if it seems valid. If
   FORCE_READ_ATR is set, the ATR is always decoded, even if it does
   not look sane (usually resulting in completely useless data). CNR
   is the handle that was given to init_terminal(). */

int smartcard_display_info(unsigned short int const cnr, 
			   int const force_read_atr) 
{
  int ret;
  char temp_buf[256];
  char manufacturer[6], model[4], revision[11];
  unsigned char atr[10];
  int i;

  /* First, get the manufacturer, model, and revision. */
  ret = smartcard_ctbcs_get_status(cnr, 0x46, temp_buf, sizeof(temp_buf));

  strncpy(manufacturer, temp_buf, 5);
  manufacturer[5] = '\0';

  strncpy(model, &temp_buf[5], 3);
  model[3] = '\0';

  strncpy(revision, &temp_buf[10], 5);
  revision[5] = '\0';

  printf("Terminal Manufacturer: %s\n", manufacturer);
  printf("Terminal Model: %s\n", model);
  printf("Terminal Revision: %s\n", revision);

  /* Now check the current status. */

  printf("Card Status: ");
  ret = smartcard_ctbcs_get_status(cnr, 0x80, temp_buf, sizeof(temp_buf));

  switch(temp_buf[0]) {
  case CTBCS_DATA_STATUS_CARD:
    printf("Card Inserted");
    break;
  case CTBCS_DATA_STATUS_CARD_CONNECT:
    printf("Card Inserted (connected)");
    break;
  case CTBCS_DATA_STATUS_NOCARD:
    printf("No Card Inserted");
    break;
  default:
    printf("Unknown (%X)", ret);
    break;
  }
  printf("\n");

  /* Exit here if there is no card in the terminal. */
  if (temp_buf[0] == CTBCS_DATA_STATUS_NOCARD)
    return 0;
  
  /* Get and decode the ATR. */
  ret = smartcard_ctbcs_request_icc(cnr, 0x01, atr);

  if (((atr[4] == CTBCS_SW1_REQUEST_SYNC_OK)
       && (atr[5] == CTBCS_SW2_REQUEST_SYNC_OK)
       && (atr[2] == ATR_H3_SYNC_CARD))
      || (force_read_atr)) {

    if (ret == 0) {

      /* Thanks to c't Magazine for the description of the ATR fields.
	 However, I still have no idea what some of these fields mean,
	 so don't expect too much from the displayed descriptions. */

      /* H1 byte */
      printf("ATR Protocol: ");
      if ((atr[0] & ATR_H1_PROTO_NONISO) == 0)
        printf("ISO protocol");
      else if ((atr[0] & ATR_H1_PROTO_MASK) == ATR_H1_PROTO_I2C)
        printf("I2C");
      else if ((atr[0] & ATR_H1_PROTO_MASK) == ATR_H1_PROTO_3W)
        printf("3W");
      else if ((atr[0] & ATR_H1_PROTO_MASK) == ATR_H1_PROTO_2W)
        printf("2W");
      else 
        printf("unknown");
      printf("\n");
      
      printf("ATR Structure Ident: ");
      if ((atr[0] & ATR_H1_STRUCT_ISO_MASK) == 0)
        printf("ISO");
      else if ((atr[0] & ATR_H1_STRUCT_MASK) == ATR_H1_STRUCT_COMMON)
        printf("common use");
      else if ((atr[0] & ATR_H1_STRUCT_MASK) == ATR_H1_STRUCT_PROPR)
        printf("proprietary use");
      else
        printf("special use");
      printf("\n");
      
      /* H2 byte */
      printf("ATR Read Error Flag: ");
      if ((atr[1] & ATR_H2_READERR_FLAG))
        printf("finish");
      else
        printf("defined length");
      printf("\n");
      
      /* This will look a bit scary... */
      printf("ATR Memory Size: %d blocks, %d bits wide = %d bytes\n",
             1 << (((atr[1] & ATR_H2_UNIT_COUNT_MASK) >> 3)+6),
             1 << (atr[1] & ATR_H2_UNIT_SIZE_MASK),
             
             1 << (((int)(atr[1] & ATR_H2_UNIT_COUNT_MASK) >> 3)+6) *
             1 << (int)(atr[1] & ATR_H2_UNIT_SIZE_MASK) / 8);

      printf("ATR DIR: %s\n", (atr[3]&ATR_H4_DIR_IS_VALID)?"valid":"invalid");
      
    } 
    
    /* Print the raw ATR. */
    printf("Answer To Reset (ATR): ");
    for (i=0; i<4; ++i)
      printf("%X ", atr[i]);    
    if (atr[2] != ATR_H3_SYNC_CARD)
      printf("(probably fake)\n");
    printf("\n");

    /* That's all we have for now. If you know how to decode more information
       from the card, please let me know (or implement it yourself). */
  }
  
  return 0;
}
