/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 * 
 * Copyright(C) 2001-2005 Roderick Colenbrander
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "nvclock.h"
#include "backend.h"

/* This function is actually a basic version of set_card.
/  It mainly copies the entries of the card list and maps 
/  the video registers. We need this function as we need access
/  to the videocard from the config file creation code. Further it
/  is being used from set_card.
*/
int set_card_info(int number)
{
    nv_card.card_name = nvclock.card[number].card_name;
    nv_card.number = number;
    nv_card.device_id = nvclock.card[number].device_id;
    nv_card.arch = nvclock.card[number].arch;
    nv_card.devbusfn = nvclock.card[number].devbusfn;
    nv_card.irq = nvclock.card[number].irq;
    nv_card.reg_address = nvclock.card[number].reg_address;
    nv_card.gpu = nvclock.card[number].gpu;
    nv_card.bios = nvclock.card[number].bios;
    nv_card.supported = 0; /* Reset the capabilities of the card */

    if(!map_mem(nvclock.card[number].dev_name))
    {
	/* The card isn't set, so set it to no card */
	nv_card.number = -1;
	return 0; /* map_mem already took care of the error */
    }
    
    return 1;
}

/* Set the card object to the requested card */
int set_card(int number)
{
    /* When the internal card number is the same as the number passed, don't do anything. */
    if((nv_card.number == number) && (nv_card.device_id != 0)) 
	return 1;

    if(!set_card_info(number))
	return 0;

    /* For hardware monitoring support we need access to the i2c bus. Each card
    /  has 2 or 3 of these busses. Currently we will create these busses here in set_card
    /  and store them in the card list. It should also be possible to create just 3 busses and store
    /  those in the nv_card struct and have each card use them and limit the number of used busses by
    /  a variable but the issue is that this is not very nice and perhaps even incorrect. For example
    /  different cards might need different i2c bus timings and further we mess up the xfree86 i2c structures
    /  if we mix dev/bus structures from different cards.
    /
    /  Once the busses have been created they will be probed for sensor chips. If a sensor is found
    /  a i2c device pointer will be stored. Using this pointer we can access the chip using standard
    /  xf86 i2c functions.
    */
    if(nvclock.card[number].busses[0] == NULL)
    {
	nvclock.card[number].num_busses = 2;
        nvclock.card[number].busses[0] = NV_I2CCreateBusPtr("BUS0", 0x3e); /* available on riva128 and higher */
        nvclock.card[number].busses[1] = NV_I2CCreateBusPtr("BUS1", 0x36); /* available on rivatnt hardware and higher */

	/* There's an extra bus available on geforce4mx, geforcefx and geforce6 cards (not on geforce4ti!).
	/  The check below looks for geforce4mx/geforcefx/geforce6 architecture.
	*/
	if(nv_card.arch & (NV17 | NV3X | NV4X))
	{
	    nvclock.card[number].num_busses = 3;
    	    nvclock.card[number].busses[2] = NV_I2CCreateBusPtr("BUS2", 0x50); 
	}
        nvclock.card[number].sensor = I2cProbeDevices(nvclock.card[number].busses, nvclock.card[number].num_busses);
    }

    /* When a sensor is available, enable the correct function pointers */
    if(nvclock.card[number].sensor)
    {
	nv_card.sensor = nvclock.card[number].sensor;
	switch(nv_card.sensor->chip_id)
	{
	    case LM99:
	    case MAX6559:
		nv_card.supported |= I2C_TEMP_MONITORING_SUPPORTED;
		nv_card.get_board_temp = lm99_get_board_temp;
        	nv_card.get_gpu_temp = lm99_get_gpu_temp;
		break;
	    case F75375:
		nv_card.supported |= I2C_TEMP_MONITORING_SUPPORTED | I2C_FANSPEED_MONITORING_SUPPORTED;
		nv_card.get_board_temp = f75375_get_board_temp;
        	nv_card.get_gpu_temp = f75375_get_gpu_temp;
		nv_card.get_i2c_fanspeed_rpm = f75375_get_fanspeed_rpm;
		nv_card.get_i2c_fanspeed_pwm = f75375_get_fanspeed_pwm;
		nv_card.set_i2c_fanspeed_pwm = f75375_set_fanspeed_pwm;
		break;
	    case W83L785R:
		nv_card.supported |= I2C_TEMP_MONITORING_SUPPORTED | I2C_FANSPEED_MONITORING_SUPPORTED;
		nv_card.get_board_temp = w83l785r_get_board_temp;
        	nv_card.get_gpu_temp = w83l785r_get_gpu_temp;
		nv_card.get_i2c_fanspeed_rpm = w83l785r_get_fanspeed_rpm;
		nv_card.get_i2c_fanspeed_pwm = w83l785r_get_fanspeed_pwm;
		nv_card.set_i2c_fanspeed_pwm = w83l785r_set_fanspeed_pwm;
		break;
	}
    }
    else
	nv_card.sensor = NULL;
	
    /* Set the default pll's */
    nv_card.mpll = nvclock.card[number].mpll;
    nv_card.mpll2 = nvclock.card[number].mpll2;
    nv_card.nvpll = nvclock.card[number].nvpll;
    nv_card.nvpll2 = nvclock.card[number].nvpll2;

    /* Find out what memory is being used */
    nv_card.mem_type = (nv_card.PFB[0x200/4] & 0x01) ? DDR : SDR;
    nv_card.get_bus_type = get_bus_type;

    if(strcmp(nv_card.get_bus_type(), "AGP") == 0)
    {
	nv_card.get_agp_rate = get_agp_rate;
	nv_card.get_agp_status = get_agp_status;
	nv_card.get_fw_status = get_fw_status;
	nv_card.get_sba_status = get_sba_status;
	nv_card.get_supported_agp_rates = get_supported_agp_rates;
    }
    else
    {
	nv_card.get_agp_rate = get_agp_rate;
	nv_card.get_agp_status = dummy_str;
	nv_card.get_fw_status = dummy_str;
	nv_card.get_sba_status = dummy_str;
	nv_card.get_supported_agp_rates = dummy_str;
    }
    
    nv_card.get_memory_size = get_memory_size;
    nv_card.get_memory_type = get_memory_type;
    nv_card.get_memory_width = get_memory_width;

    /* GeforceFX 5600/5700 */
    if(nv_card.arch == NV31)
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
        nv_card.get_gpu_speed = nv31_get_gpu_speed;
        nv_card.get_memory_speed = nv31_get_memory_speed;
        nv_card.set_memory_speed = nv31_set_memory_speed;
        nv_card.set_gpu_speed = nv31_set_gpu_speed;
        nv_card.reset_gpu_speed = nv31_reset_gpu_speed;
        nv_card.reset_memory_speed = nv31_reset_memory_speed;
    }
    /* GeforceFX 5800/5900 */
    else if(nv_card.arch & (NV30|NV35))
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
        nv_card.get_gpu_speed = nv30_get_gpu_speed;
        nv_card.get_memory_speed = nv30_get_memory_speed;
        nv_card.set_memory_speed = nv30_set_memory_speed;
        nv_card.reset_gpu_speed = nv30_reset_gpu_speed;
        nv_card.reset_memory_speed = nv30_reset_memory_speed;

	/* HW monitoring; bit 31 is an indication if fanspeed monitoring is available
	/  Note this bit isn't very reliable as it is set on cards with advanced sensors too.
	/
	/  Only support this on NV35/NV38 hardware for now as it works differently on NV30 boards
	*/
	if((nv_card.PMC[0x10f0/4] & 0x80000000) && (nv_card.arch == NV35) && !(nv_card.supported & I2C_FANSPEED_MONITORING_SUPPORTED))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv30_get_fanspeed;
	    nv_card.set_fanspeed = nv30_set_fanspeed;
	}
    }
    /* Geforce 6200/6600/6800 */
    else if(nv_card.arch & NV4X)
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
        nv_card.get_gpu_speed = nv40_get_gpu_speed;
        nv_card.get_memory_speed = nv40_get_memory_speed;
        nv_card.set_memory_speed = nv40_set_memory_speed;
        nv_card.set_gpu_speed = nv40_set_gpu_speed;
        nv_card.reset_gpu_speed = nv40_reset_gpu_speed;
        nv_card.reset_memory_speed = nv40_reset_memory_speed;

	/* Pipeline code; note for this we need checks as not all boards support modding */
	nv_card.supported |= PIPELINE_MODDING_SUPPORTED;
	nv_card.get_pixel_pipelines = nv40_get_pixel_pipelines;
	nv_card.set_pixel_pipelines = nv40_set_pixel_pipelines;
	nv_card.get_vertex_pipelines = nv40_get_vertex_pipelines;
	nv_card.set_vertex_pipelines = nv40_set_vertex_pipelines;
	
	/* HW monitoring; bit 31 is an indication if fanspeed monitoring is available
	/  Note this bit isn't very reliable as it is set on cards with advanced sensors too.
	*/
	if((nv_card.arch == NV40) && (nv_card.PMC[0x10f0/4] & 0x80000000))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv40_get_fanspeed;
	    nv_card.set_fanspeed = nv40_set_fanspeed;
	}
	else if((nv_card.arch == NV43) && (nv_card.PMC[0x15f4/4] & 0x80000000))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv43_get_fanspeed;
	    nv_card.set_fanspeed = nv43_set_fanspeed;
	}
    }
    else if(nv_card.gpu == NFORCE)
    {
	/* Only support gpu overclocking as the memory is (shared) system memory */
	nv_card.supported |= GPU_OVERCLOCKING_SUPPORTED;
    	nv_card.get_gpu_speed = get_gpu_speed;
	nv_card.set_gpu_speed = set_gpu_speed;
	nv_card.get_memory_speed = nforce_get_memory_speed;
	nv_card.set_memory_speed = dummy;
        nv_card.reset_gpu_speed = reset_gpu_speed;
        nv_card.reset_memory_speed = dummy;
    }
    /* All other cards including the GeforceFX 5200 and GeforceFX 5500 */
    else
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	nv_card.get_gpu_speed = get_gpu_speed;
    	nv_card.set_gpu_speed = set_gpu_speed;
	nv_card.get_memory_speed = get_memory_speed;
	nv_card.set_memory_speed = set_memory_speed;
        nv_card.reset_gpu_speed = reset_gpu_speed;
        nv_card.reset_memory_speed = reset_memory_speed;
    }

    /* Mobile GPU check; we don't want to overclock those unless the user wants it */
    if(nv_card.gpu == MOBILE)
    {
	/* Don't support overclocking as laptops aren't made for it; the user should be able to override this for atleast underclocking */
	nv_card.supported = ~(~nv_card.supported | GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	nv_card.set_gpu_speed = dummy;
	nv_card.set_memory_speed = dummy;
        nv_card.reset_gpu_speed = dummy;
        nv_card.reset_memory_speed = dummy;
    }

    /* Set the speed range */
    /*
    set_speed_range();
    */
    
    return 1;
}

void unset_card()
{
    nv_card.number=-1;
    unmap_mem();
}
