-- See the Copyright notice at the end of this file.
--
class WINDOWS_DIRECTORY_NOTATION
	-- The Windows like file path notation looks like:
	--   C:\SmartEiffel\sys\system.se

inherit
	DIRECTORY_NOTATION

feature {ANY}
	is_current_directory (path: STRING): BOOLEAN is
		do
			if path.compare(once ".") = 0 or else path.compare(once ".\") = 0 then
				Result := True
			else
				Result := path.occurrences('.') + path.occurrences('\') = path.count and not path.has_substring("..")
			end
		end

	is_parent_directory (path: STRING): BOOLEAN is
		local
			pos: INTEGER
		do
			if path.compare(once "..") = 0 or else path.compare(once "..\") = 0 then
				Result := True
			else
				Result := path.occurrences('.') + path.occurrences('\') = path.count and path.has_substring("..")
				if Result then
					pos := path.first_substring_index("..")
					Result := Result and path.substring_index("..", pos + 2) = 0
				end
			end
		end

	to_parent_directory (some_path: STRING) is
		do
			--|*** "." and ".." are not handled correctly <FM-24/03/2005>
			from
				some_path.remove_last
			until
				some_path.is_empty or else some_path.last = '\'
			loop
				some_path.remove_last
			end
			if not some_path.is_empty then
				check
					some_path.last = '\'
				end
				some_path.remove_last
			end
		end

	to_subdirectory_with (parent_path, entry_name: STRING) is
		do
			if (once ".").is_equal(entry_name) then
				-- Because you would get the same directory as `parent_path' and
				-- not a new subdirectory as explained before.
				parent_path.clear_count
			elseif (once "..").is_equal(entry_name) then
				-- Because you would not get a subdirectory of `parent_path'.
				parent_path.clear_count
			else
				if entry_name.first /= '\' then
					parent_path.extend_unless('\')
				end
				parent_path.append(entry_name)
				if parent_path.last = '\' then
					parent_path.remove_last
				end
			end
		end

	to_file_path_with (parent_path, file_name: STRING) is
		do
			if file_name.first /= '\' then
				parent_path.extend_unless('\')
			end
			parent_path.append(file_name)
		end

	to_subpath_with (parent_path, subpath: STRING) is
		do
			if parent_path.last /= ':' then
				parent_path.extend_unless('\')
			end
			parent_path.append(subpath)
		end

	to_directory_path (path: STRING) is
		do
			from
			until path.is_empty or else path.last /= '\'
			loop
				path.remove_last
			end
		end

feature {ANY}
	can_map_drive (source_notation: DIRECTORY_NOTATION; drive: STRING): BOOLEAN is
		do
			Result := ({WINDOWS_DIRECTORY_NOTATION} ?:= source_notation) and then drive.count = 1
		end

	to_root (source_notation: DIRECTORY_NOTATION; drive: STRING) is
		do
			drive.extend(':')
			drive.extend('\')
		end

	to_default_root (directory: STRING) is
		local
			bd: BASIC_DIRECTORY
		do
			if {WINDOWS_DIRECTORY_NOTATION} ?:= bd.system_notation then
				directory.copy(bd.current_working_directory)
				directory.keep_head(3)
			else
				directory.clear_count
				directory.extend('C')
				directory.extend(':')
				directory.extend('\')
			end
		end

	to_current_directory (directory: STRING) is
		do
			directory.clear_count
			directory.extend('.')
		end

feature {ANY}
	is_case_sensitive: BOOLEAN is False

	is_valid_path, is_valid_directory_path (path: STRING): BOOLEAN is
		do
			--|*** Not nearly strict enough <FM-24/03/2003>
			Result := not path.is_empty
		end

	is_valid_file_name (name: STRING): BOOLEAN is
		do
			--|*** Not nearly strict enough <FM-24/03/2003>
			Result := not name.is_empty
		end

	is_absolute_path (path: STRING): BOOLEAN is
		do
			Result := path.count >= 3 and then path.item(2) = ':' and then path.item(3) = '\'
		end

	to_notation (path: STRING; destination_notation: DIRECTORY_NOTATION): STRING is
		local
			i, j: INTEGER; file_name: STRING
		do
			if {WINDOWS_DIRECTORY_NOTATION} ?:= destination_notation then
				Result := path
			else
				Result := once ""
				file_name := once ""
				if is_absolute_path(path) then
					file_name.clear_count
					file_name.extend(path.item(2))
					if destination_notation.can_map_drive(Current, file_name) then
						destination_notation.to_root(Current, file_name)
					else
						destination_notation.to_default_root(Result)
					end
					i := 4
				elseif path.first = '\' then
					destination_notation.to_default_root(Result)
					i := 2
				elseif path.count >= 2 and then path.item(2) = ':' then
					-- This is a relative path with a drive specifier, such
					-- as "D:lib\foo\bar"
					not_yet_implemented
				else
					destination_notation.to_current_directory(Result)
					i := 1
				end
				from
					j := path.index_of('\', i)
				until
					j = 0
				loop
					file_name.copy(path)
					file_name.shrink(i, j - 1)
					if file_name.is_equal(once ".") then
					elseif file_name.is_equal(once "..") then
						destination_notation.to_parent_directory(Result)
					elseif destination_notation.is_valid_file_name(file_name) then
						destination_notation.to_subdirectory_with(Result, file_name)
						i := j + 1
						j := path.index_of('\', i)
					elseif destination_notation.can_sanitize(file_name) then
						destination_notation.to_valid_file_name(file_name)
						destination_notation.to_subdirectory_with(Result, file_name)
						i := j + 1
						j := path.index_of('\', i)
					else
						Result.clear_count
						i := path.count + 1
						j := 0
					end
				end
				if i <= path.count then
					file_name.copy(path)
					file_name.shrink(i, path.count)
					if file_name.is_equal(once ".") then
					elseif file_name.is_equal(once "..") then
						destination_notation.to_parent_directory(Result)
					elseif destination_notation.is_valid_file_name(file_name) then
						destination_notation.to_file_path_with(Result, file_name)
					elseif destination_notation.can_sanitize(file_name) then
						destination_notation.to_valid_file_name(file_name)
						destination_notation.to_file_path_with(Result, file_name)
					else
						Result.clear_count
					end
				end
			end
		end

end -- class WINDOWS_DIRECTORY_NOTATION
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
