-- See the Copyright notice at the end of this file.
--
class HTTP_CONNECTION
	--
	-- Handle one connection to the HTTP_SERVER. The real HTTP protocol is defined here.
	--

inherit
	CONNECTION
		redefine set_io
		end

insert
	DISPOSABLE
	RECYCLABLE

creation {ANY}
	make

feature {LOOP_ITEM}
	continue is
		local
			log: STRING
		do
			ios.read_line
			state.call([ios.last_string])
			if method_handler.is_ready then
				method_handler.answer
				log := once ""
				log.copy(request_line)
				log.extend(' ')
				method_handler.code.append_in(log)
				server.log(log)
				ios.disconnect
			end
		end

feature {HTTP_SERVER}
	set_server (a_server: like server) is
		do
			server := a_server
			state := states.item(0)
		end

feature {SERVER}
	set_io (a_io: like ios) is
		do
			Precursor(a_io)
			a_io.when_disconnect(agent handle_disconnect)
		end

feature {}
	method_handler_factory: FUNCTION[TUPLE[STRING, STRING, STRING, OUTPUT_STREAM], HTTP_METHOD_HANDLER]
			-- The first argument is the method, the second is the URI, the third is the version; the fourth is
			-- the output stream to the client socket

	make (a_method_handler_factory: like method_handler_factory) is
		do
			method_handler_factory := a_method_handler_factory
			create request_line.make_empty
		end

feature {}
	server: HTTP_SERVER

	handle_disconnect (a_io: like ios) is
		require
			a_io = ios
			done
		do
			server.connection_done(Current)
		end

feature {} -- The HTTP protocol (see RFC 2616)
	state: PROCEDURE[TUPLE[STRING]]

	states: FAST_ARRAY[PROCEDURE[TUPLE[STRING]]] is
		once
			Result := {FAST_ARRAY[PROCEDURE[TUPLE[STRING]]] << agent a_request_line, agent a_header, agent a_body >> }
		end

	method_handler: HTTP_METHOD_HANDLER
	request_line: STRING

	a_request_line (line: STRING) is
		local
			i, j: INTEGER
			method, uri, version: STRING
		do
			i := line.first_index_of(' ')
			j := line.index_of(' ', i + 1)
			method := once ""
			method.copy(line)
			method.shrink(1, i - 1)
			method.to_upper
			uri := once ""
			uri.copy(line)
			if j = 0 then
				uri.shrink(i + 1, line.count)
				version := once "HTTP/1.0"
			else
				uri.shrink(i + 1, j - 1)
				version := once ""
				version.copy(line)
				version.shrink(j + 1, line.count)
			end
			request_line.copy(method)
			request_line.extend(' ')
			request_line.append(uri)
			state := states.item(1)
			method_handler := method_handler_(method, uri, version)
		end

	a_header (line: STRING) is
		do
			if line.is_empty then
				state := states.item(2)
			else
				method_handler.add_header(line.twin)
			end
		end

	a_body (line: STRING) is
		do
			method_handler.add_body(line.twin)
		end

feature {} -- method handlers reuse
	method_handlers: DICTIONARY[FAST_ARRAY[HTTP_METHOD_HANDLER], STRING] is
		once
			create {HASHED_DICTIONARY[FAST_ARRAY[HTTP_METHOD_HANDLER], STRING]} Result.make
		end
	no_method_handlers: FAST_ARRAY[HTTP_NO_METHOD_HANDLER] is
		once
			create Result.make(0)
		end

	method_handler_ (method, uri, version: STRING): HTTP_METHOD_HANDLER is
		require
			method.as_upper.is_equal(method)
		local
			mh: FAST_ARRAY[HTTP_METHOD_HANDLER]
		do
			mh := method_handlers.reference_at(method)
			if mh /= Void and then not mh.is_empty then
				Result := mh.last
				mh.remove_last
				Result.make(uri, version, ios)
			else
				if method_handler_factory /= Void then
					Result := method_handler_factory.item([method, uri, version, ios])
				end
				if Result = Void then
					inspect method
					when "GET" then
						create {HTTP_GET_HANDLER} Result.make(uri, version, ios)
					else
						if no_method_handlers.is_empty then
							create {HTTP_NO_METHOD_HANDLER} Result.make(uri, version, ios)
						else
							Result := no_method_handlers.last
							no_method_handlers.remove_last
							Result.make(uri, version, ios)
						end
					end
				end
			end
		ensure
			Result.method.is_equal(method) or else ({HTTP_NO_METHOD_HANDLER} ?:= Result)
		end

	free_method_handler (a_method_handler: HTTP_METHOD_HANDLER) is
		local
			mh: FAST_ARRAY[HTTP_METHOD_HANDLER]
			nmh: HTTP_NO_METHOD_HANDLER
		do
			if nmh ?:= a_method_handler then
				nmh ::= a_method_handler
				no_method_handlers.add_last(nmh)
			else
				mh := method_handlers.reference_at(a_method_handler.method)
				if mh = Void then
					create mh.make(0)
					method_handlers.add(mh, a_method_handler.method)
				end
				mh.add_last(a_method_handler)
			end
		end

feature {WEAK_ARRAY}
	recycle is
		do
			handle_disconnect(ios)
		end

feature {}
	dispose is
		do
			handle_disconnect(ios)
		end

end -- class HTTP_CONNECTION
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
