-- See the Copyright notice at the end of this file.
--
class URL
	--
	-- Provides meaning to URL (Universal Resource Locator) strings, by the use of accessors to different parts
	-- of it: `protocol', `username', `password', `host', `port', `path', `arguments' and `anchor' are
	-- different parts of the URL:
	--
	--   protocol://username:password@host:port/path?arguments#anchor
	--
	-- See also [http://www.faqs.org/rfcs/rfc1738.html RFC 1738]
	--
	-- Some ideas and first draft of this class kindly provided by Serge Romanchenko
	-- [mailto:se@sir.nensi.net]
	--

creation {ANY}
	make, set_url

feature {ANY} -- URL connection:
	is_connected: BOOLEAN is
			-- True if the URL is connected to the resource it points to.
		do
			Result := protocol.is_connected
		end

	connect is
			-- Connect to the resource pointed by this URL.
		require
			not is_connected
		do
			protocol.connect_to(Current)
			if protocol.error /= Void and then error_handler /= Void then
				error_handler.call([protocol.error])
			end
		end

	disconnect is
			-- Disconnect from the resource pointed by this URL.
		require
			is_connected
		do
			protocol.disconnect
		ensure
			not is_connected
		end

	input: INPUT_STREAM is
			-- Data coming from the resource this URL points to.
		require
			is_connected
		do
			Result := protocol.input
		end

	output: OUTPUT_STREAM is
			-- Data going to the resource this URL points to.
		require
			is_connected
		do
			Result := protocol.output
		end

feature {ANY} -- URL data:
	make (a_protocol: like protocol; a_host: like host; a_port: like port; a_path: like path) is
			-- Make a URL with the minimum information. If `port' is 0, then the standard port of the protocol is
			-- used (e.g. 80 for http).
		require
			a_protocol /= Void
			not a_host.is_empty
			not a_path.is_empty
			a_port >= 0
		do
			set_protocol(a_protocol)
			set_host(a_host)
			set_port(a_port)
			set_path(a_path)
		end

	set_error_handler(a_error_handler: like error_handler) is
		do
			error_handler := a_error_handler
		end

	set_protocol (a_protocol: like protocol) is
			-- Sets the protocol.
		require
			a_protocol /= Void
		do
			protocol := a_protocol
		ensure
			protocol = a_protocol
		end

	set_host (a_host: like host) is
			-- Sets the host name.
		require
			not a_host.is_empty
		do
			host := a_host
		ensure
			host = a_host
		end

	set_port (a_port: like port) is
			-- Sets the port number; 0 means use the standard port of the protocol.
		require
			a_port >= 0
		do
			port := a_port
		ensure
			port = a_port
		end

	set_username (a_username: like username) is
			-- Sets an optional user name; Void clears it.
		do
			username := a_username
		ensure
			username = a_username
		end

	set_password (a_password: like password) is
			-- Sets an optional password; Void clears it.
		require
			a_password /= Void implies username /= Void
		do
			password := a_password
		ensure
			password = a_password
		end

	set_path (a_path: like path) is
			-- Sets the path.
		require
			not a_path.is_empty
		do
			path := a_path
		ensure
			path = a_path
		end

	set_arguments (a_argument_name, a_argument_value: STRING) is
			-- Sets an optional argument. May be called more than once to add more arguments. A Void name clears
			-- all the arguments.
		require
			a_argument_value /= Void implies a_argument_name /= Void
		do
			if a_argument_name = Void then
				arguments := Void
			else
				if arguments = Void then
					arguments := a_argument_name.twin
				else
					arguments.extend('&')
					arguments.append(a_argument_name)
				end
				if a_argument_value /= Void then
					arguments.extend('=')
					arguments.append(encode(a_argument_value))
				end
			end
		end

	set_anchor (a_anchor: like anchor) is
			-- Sets an optional anchor.
		do
			anchor := a_anchor
		ensure
			anchor = a_anchor
		end

	set_url (a_url: STRING) is
			-- Sets all parts of the URL from the given full URL string.
			--|*** lots of memory leaks!
		require
			is_a_url(a_url)
		local
			i, j, k: INTEGER; s: STRING; protocols: PROTOCOLS
		do
			-- look for scheme
			i := a_url.first_index_of(':')
			s := once ""
			s.copy(a_url)
			s.shrink(1, i-1)
			set_protocol(protocols.protocol(s))
			j := a_url.index_of(':', i+1)
			k := a_url.first_index_of('@')
			if a_url.valid_index(j) then
				if j < k then
					-- look for user
					set_username(a_url.substring(i+1, j-1))
					-- look for password
					set_password(a_url.substring(j+1, k-1))
					j := a_url.index_of(':', j+1)
				else
					-- look for user
					set_username(a_url.substring(i+1, k-1))
				end
				i := k
			else
				i := i + 2
			end
			k := a_url.index_of('/', i+1)
			if a_url.valid_index(k) then
				if a_url.valid_index(j) then
					-- look for host
					set_host(a_url.substring(i+1, j-1))
					-- look for port
					s.copy(a_url)
					s.shrink(j+1, k-1)
					set_port(s.to_integer)
				else
					-- look for host
					set_host(a_url.substring(i+1, k-1))
				end
				i := a_url.index_of('?', k+1)
				j := a_url.index_of('#', k+1)
				if a_url.valid_index(i) then
					-- look for path
					set_path(a_url.substring(k, i-1))
					if a_url.valid_index(j) then
						-- look for arguments
						arguments := a_url.substring(i+1, j-1)
						-- look for anchor
						set_anchor(a_url.substring(j+1, a_url.upper))
					else
						-- look for arguments
						arguments := a_url.substring(i+1, a_url.upper)
					end
				elseif a_url.valid_index(j) then
					-- look for path
					set_path(a_url.substring(k, j-1))
					-- look for anchor
					set_anchor(a_url.substring(j+1, a_url.upper))
				else
					-- look for path
					set_path(a_url.substring(k, a_url.upper))
				end
			else
				-- no path
				set_path("")
			end
		ensure
			url.is_equal(a_url)
		end

	is_a_url (a_url: STRING): BOOLEAN is
			-- True if `a_url' contains a correctly formatted URL.
		require
			not a_url.is_empty
		local
			i, j, k: INTEGER; s: STRING; protocols: PROTOCOLS
		do
			-- look for scheme
			i := a_url.first_index_of(':')
			s := once ""
			s.copy(a_url)
			s.shrink(1, i-1)
			if protocols.protocol(s) /= Void then
				j := a_url.index_of(':', i+1)
				k := a_url.first_index_of('@')
				if a_url.valid_index(j) then
					if j < k then
						Result := k > j and then j > i
						j := a_url.index_of(':', j+1)
					else
						-- look for user
						Result := k > i
					end
					i := k
				else
					Result := True
					i := i + 2
				end
				if Result then
					k := a_url.index_of('/', i+1)
					if a_url.valid_index(k) then
						if a_url.valid_index(j) then
							-- look for host
							Result := k > j and then j > i
							if Result then
								-- look for port
								s.copy(a_url)
								s.shrink(j+1, k-1)
								Result := s.is_integer
							end
						else
							-- look for host
							Result := k > i
						end
						if Result then
							i := a_url.index_of('?', k+1)
							j := a_url.index_of('#', k+1)
							if a_url.valid_index(i) then
								-- look for path
								Result := i >= k
								if a_url.valid_index(j) then
									-- look for arguments
									Result := Result and then j > i
									-- look for anchor
									Result := Result and then a_url.upper > j
								else
									-- look for arguments
									Result := Result and then a_url.upper > i
								end
							elseif a_url.valid_index(j) then
								-- look for path
								Result := Result and then j >= k
								-- look for anchor
								Result := Result and then a_url.upper > j
							else
								-- look for path
								Result := Result and then a_url.upper >= k
							end
						end
					else
						-- fine: no path
					end
				end
			end
		end

	url: STRING is
			-- Returns the full URL string. Always returns the same string.
			--   http://user:pass@example.com:8000/path?query=1&show=true#ANCHOR
		do
			Result := once ""
			Result.clear_count
			Result.append(protocol.name)
			Result.append(once "://")
			if username /= Void then
				Result.append(username)
				if password /= Void then
					Result.extend(':')
					Result.append(password)
				end
			end
			if username /= Void then
				Result.extend('@')
			end
			Result.append(host)
			if port /= 0 then
				Result.extend(':')
				port.append_in(Result)
			end
			if path /= Void then
				Result.append(path)
			end
			if arguments /= Void then
				Result.extend('?')
				Result.append(arguments)
			end
			if anchor /= Void then
				Result.extend('#')
				Result.append(anchor)
			end
			Result := decode(Result)
		end

	protocol: PROTOCOL
			-- The protocol.

	username: STRING
			-- The optional user name.

	password: STRING
			-- The optional password.

	host: STRING
			-- The host name.

	port: INTEGER
			-- The port; 0 means use the standard port of the protocol.

	path: STRING
			-- The path.

	arguments: STRING
			-- The optional arguments.

	anchor: STRING
			-- The optional anchor.

feature {}
	encode (string: STRING): STRING is
		require
			string /= Void
		local
			i: INTEGER; c: CHARACTER
		do
			Result := once ""
			Result.clear_count
			from
				i := string.lower
			until
				i > string.upper
			loop
				c := string.item(i)
				inspect
					c
				when '%U' .. ' ', '%%', '<', '>', '%"', '#', '{', '}', '|', '\', '^', '~', '[', ']', '`', '%/127/' .. '%/255/',
					';', '/', '?', ':', '@', '=', '&' then
					-- list of "unsafe" and "reserved" characters from RFC1738
					Result.extend('%%')
					c.to_hexadecimal_in(Result)
				else
					Result.extend(c)
				end
				i := i + 1
			end
		ensure
			Result /= Void
		end

	decode (string: STRING): STRING is
		require
			string /= Void
		local
			i, n: INTEGER; c: CHARACTER
		do
			Result := once ""
			Result.clear_count
			from
				i := string.lower
			until
				i > string.upper
			loop
				c := string.item(i)
				if c /= '%%' then
					Result.extend(c)
					i := i + 1
				else
					c := string.item(i + 1)
					inspect
						c
					when '0'..'9' then
						n := c.code - '0'.code
					when 'A'..'Z' then
						n := c.code - 'A'.code + 10
					when 'a' .. 'z' then
						n := c.code - 'a'.code + 10
					end
					c := string.item(i + 2)
					inspect
						c
					when '0'..'9' then
						n := n*16 + c.code - '0'.code
					when 'A'..'Z' then
						n := n*16 + c.code - 'A'.code + 10
					when 'a'..'z' then
						n := n*16 + c.code - 'a'.code + 10
					end
					Result.extend(n.to_character)
					i := i + 3
				end
			end
		end

	error_handler: PROCEDURE[TUPLE[STRING]]
			-- An optional error handler

invariant
	has_protocol: protocol /= Void
	has_host: not host.is_empty
	meaningful_port: port >= 0
	has_path: not path.is_empty

end -- class URL
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
