-- See the Copyright notice at the end of this file.
--
class HASHED_DICTIONARY[V_, K_ -> HASHABLE]
	--
	-- Associative memory. Values of type `V_' are stored using Keys of type `K_'.
	--
	-- Efficient implementation of DICTIONARY using `hash_code' on keys.
	--

inherit
	SIMPLE_DICTIONARY[V_, K_]
		redefine key_map_in, item_map_in, copy
		end

creation {ANY}
	make, with_capacity, manifest_creation

creation {HASHED_DICTIONARY}
	special_common_dictionary

feature {HASHED_DICTIONARY}
	buckets: NATIVE_ARRAY[like cache_node]
			-- The `buckets' storage area is the primary hash table of
			-- `capacity' elements. To search some key, the first access is
			-- done in `buckets' using the remainder of the division of the
			-- key `hash_code' by `capacity'. In order to try to avoid clashes,
			-- `capacity' is always a prime number (selected using
			-- HASH_TABLE_SIZE).

feature {ANY}
	Default_size: INTEGER is 193
			-- Default size for the storage area in number of items.

	capacity: INTEGER
			-- Of the `buckets' storage area. Note: this not the accurate capacity value, but it
			-- does not hurt.

	count: INTEGER

	has (k: K_): BOOLEAN is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else key_safe_equal(node.key, k)
			loop
				node := node.next
			end
			Result := node /= Void
		end

	at (k: K_): V_ is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				key_safe_equal(node.key, k)
			loop
				node := node.next
			end
			Result := node.item
		end

	reference_at (k: K_): V_ is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else key_safe_equal(node.key, k)
			loop
				node := node.next
			end
			if node /= Void then
				Result := node.item
			end
		end

	fast_has (k: K_): BOOLEAN is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else node.key = k
			loop
				node := node.next
			end
			Result := node /= Void
		end

	fast_at (k: K_): V_ is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				node.key = k
			loop
				node := node.next
			end
			Result := node.item
		end

	fast_reference_at (k: K_): V_ is
		local
			idx: INTEGER; node: like cache_node
		do
			from
				idx := k.hash_code #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else node.key = k
			loop
				node := node.next
			end
			if node /= Void then
				Result := node.item
			end
		end

feature {ANY}
	put (v: V_; k: K_) is
		local
			h, idx: INTEGER; node: like cache_node
		do
			from
				h := k.hash_code
				idx := h #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else key_safe_equal(node.key, k)
			loop
				node := node.next
			end
			if node = Void then
				if capacity = count then
					increase_capacity
					idx := h #\\ capacity
				end
				node := new_node(v, k, buckets.item(idx))
				buckets.put(node, idx)
				count := count + 1
				cache_user := -1
			else
				node.set_item(v)
			end
		end

	fast_put (v: V_; k: K_) is
		local
			h, idx: INTEGER; node: like cache_node
		do
			from
				h := k.hash_code
				idx := h #\\ capacity
				node := buckets.item(idx)
			until
				node = Void or else node.key = k
			loop
				node := node.next
			end
			if node = Void then
				if capacity = count then
					increase_capacity
					idx := h #\\ capacity
				end
				node := new_node(v, k, buckets.item(idx))
				buckets.put(node, idx)
				count := count + 1
				cache_user := -1
			else
				node.set_item(v)
			end
		end

	add (v: V_; k: K_) is
		local
			idx: INTEGER; node: like cache_node
		do
			cache_user := -1
			if capacity = count then
				increase_capacity
			end
			idx := k.hash_code #\\ capacity
			node := new_node(v, k, buckets.item(idx))
			buckets.put(node, idx)
			count := count + 1
		end

	remove (k: K_) is
		local
			h, idx: INTEGER; node, previous_node: like cache_node
		do
			cache_user := -1
			h := k.hash_code
			idx := h #\\ capacity
			node := buckets.item(idx)
			if node /= Void then
				if key_safe_equal(node.key, k) then
					count := count - 1
					node := dispose_node(node)
					buckets.put(node, idx)
				else
					from
						previous_node := node
						node := node.next
					until
						node = Void or else key_safe_equal(node.key, k)
					loop
						previous_node := node
						node := node.next
					end
					if node /= Void then
						count := count - 1
						previous_node.set_next(dispose_node(node))
					end
				end
			end
		end

	fast_remove (k: K_) is
		local
			h, idx: INTEGER; node, previous_node: like cache_node
		do
			cache_user := -1
			h := k.hash_code
			idx := h #\\ capacity
			node := buckets.item(idx)
			if node /= Void then
				if node.key = k then
					count := count - 1
					node := dispose_node(node)
					buckets.put(node, idx)
				else
					from
						previous_node := node
						node := node.next
					until
						node = Void or else node.key = k
					loop
						previous_node := node
						node := node.next
					end
					if node /= Void then
						count := count - 1
						previous_node.set_next(dispose_node(node))
					end
				end
			end
		end

	clear_count, clear_count_and_capacity is
		local
			i: INTEGER; node: like cache_node
		do
			cache_user := -1
			count := 0
			from
				i := capacity - 1
			until
				i < 0
			loop
				node := buckets.item(i)
				buckets.put(Void, i)
				from
				until
					node = Void
				loop
					node := dispose_node(node)
				end
				i := i - 1
			end
		ensure then
			capacity = old capacity
		end

	item (index: INTEGER): V_ is
		do
			set_cache_user(index)
			Result := cache_node.item
		end

	key (index: INTEGER): K_ is
		do
			set_cache_user(index)
			Result := cache_node.key
		end

	get_new_iterator_on_keys: ITERATOR[K_] is
		do
			create {ITERATOR_ON_DICTIONARY_KEYS[V_, K_]} Result.make(Current)
		end

	key_map_in (buffer: COLLECTION[K_]) is
		local
			node: like cache_node; i, idx: INTEGER
		do
			from
				i := count
				node := buckets.item(idx)
			until
				i <= 0
			loop
				from
				until
					node /= Void
				loop
					idx := idx + 1
					check
						idx < capacity
					end
					node := buckets.item(idx)
				end
				buffer.add_last(node.key)
				node := node.next
				i := i - 1
			end
		end

	item_map_in (buffer: COLLECTION[V_]) is
		local
			node: like cache_node; i, idx: INTEGER
		do
			from
				i := count
				node := buckets.item(idx)
			until
				i <= 0
			loop
				from
				until
					node /= Void
				loop
					idx := idx + 1
					check
						idx < capacity
					end
					node := buckets.item(idx)
				end
				buffer.add_last(node.item)
				node := node.next
				i := i - 1
			end
		end

	copy (other: like Current) is
		local
			i: INTEGER
		do
			clear_count
			from
				if capacity < other.count then
					with_capacity(other.count + 1)
				elseif capacity = 0 then
					make
				end
				i := 1
			until
				i > other.count
			loop
				put(other.item(i), other.key(i))
				i := i + 1
			end
		end

	internal_key (k: K_): K_ is
		local
			node: like cache_node
		do
			from
				node := buckets.item(k.hash_code #\\ capacity)
				Result := node.key
			until
				key_safe_equal(Result, k)
			loop
				node := node.next
				Result := node.key
			end
		end

feature {}
	increase_capacity is
			-- There is no more free slots: the dictionary must grow.
		require
			capacity = count
		local
			i, idx, new_capacity: INTEGER; old_buckets: like buckets; node1, node2: like cache_node; hts: HASH_TABLE_SIZE
		do
			from
				new_capacity := hts.prime_number_ceiling(capacity + 1)
				old_buckets := buckets
				buckets := buckets.calloc(new_capacity)
				i := capacity - 1
				capacity := new_capacity
			until
				i < 0
			loop
				from
					node1 := old_buckets.item(i)
				until
					node1 = Void
				loop
					node2 := node1.next
					idx := node1.key.hash_code #\\ capacity
					node1.set_next(buckets.item(idx))
					buckets.put(node1, idx)
					node1 := node2
				end
				i := i - 1
			end
			cache_user := -1
		ensure
			count = old count
			capacity > old capacity
		end

	set_cache_user (index: INTEGER) is
			-- Set the internal memory cache (`cache_user', `cache_node' and
			-- `cache_buckets') to the appropriate valid value.
		require
			valid_index(index)
		do
			if index = cache_user + 1 then
				from
					cache_user := index
					cache_node := cache_node.next
				until
					cache_node /= Void
				loop
					cache_buckets := cache_buckets + 1
					cache_node := buckets.item(cache_buckets)
				end
			elseif index = cache_user then
			elseif index = 1 then
				from
					cache_user := 1
					cache_buckets := 0
					cache_node := buckets.item(cache_buckets)
				until
					cache_node /= Void
				loop
					cache_buckets := cache_buckets + 1
					cache_node := buckets.item(cache_buckets)
				end
			else
				from
					set_cache_user(1)
				until
					cache_user = index
				loop
					set_cache_user(cache_user + 1)
				end
			end
		ensure
			cache_user = index
			cache_buckets.in_range(0, capacity - 1)
			cache_node /= Void
		end

	cache_user: INTEGER
			-- The last user's external index in range [1 .. `count'] (see `item'
			-- and `valid_index' for example) may be saved in `cache_user' otherwise
			-- -1 to indicate that the cache is not active. When the cache is
			-- active, the corresponding index in `buckets' is save in
			-- `cache_buckets' and the corresponding node in `cache_node'.

	cache_node: HASHED_DICTIONARY_NODE[V_, K_]
			-- Meaningful only when `cache_user' is not -1.

	cache_buckets: INTEGER
			-- Meaningful only when `cache_user' is not -1.

	free_nodes: WEAK_REFERENCE[HASHED_DICTIONARY_NODE[V_, K_]]
			-- If any, they are ready to be recycled.

	common_free_nodes: DICTIONARY[WEAK_REFERENCE[ANY_HASHED_DICTIONARY_NODE], STRING] is
		local
			fn: WEAK_REFERENCE[HASHED_DICTIONARY_NODE[WEAK_REFERENCE[ANY_HASHED_DICTIONARY_NODE], STRING]]
		once
			create fn.set_item(Void)
			create {HASHED_DICTIONARY[WEAK_REFERENCE[ANY_HASHED_DICTIONARY_NODE], STRING]} Result.special_common_dictionary(fn)
			Result.add(fn, Result.generating_type)
		end

	dispose_node (node: HASHED_DICTIONARY_NODE[V_, K_]): HASHED_DICTIONARY_NODE[V_, K_] is
			-- Add `node' in the `free_nodes' list.
		require
			node /= Void
		do
			Result := node.next
			node.set_next(free_nodes.item)
			free_nodes.set_item(node)
		ensure
			Result = old node.next
		end

	new_node (v: V_; k: K_; next: HASHED_DICTIONARY_NODE[V_, K_]): HASHED_DICTIONARY_NODE[V_, K_] is
			-- Recycle from `free_nodes' or create a new one.
		do
			Result := free_nodes.item
			if Result = Void then
				create Result.make(v, k, next)
			else
				free_nodes.set_item(Result.next)
				Result.make(v, k, next)
			end
		end

	make is
			-- Create an empty dictionary. Internal storage `capacity' of the
			-- dictionary is initialized using the `Default_size' value. Then,
			-- tuning of needed storage `capacity' is performed automatically
			-- according to usage. If you are really sure that your dictionary
			-- is always really bigger than `Default_size', you may consider to use
			-- `with_capacity' to save some execution time.
		do
			with_capacity(Default_size)
		ensure then
			capacity = Default_size
		end

	with_capacity (medium_size: INTEGER) is
			-- May be used to save some execution time if one is sure that
			-- storage size will rapidly become really bigger than `Default_size'.
			-- When first `remove' occurs, storage size may naturally become
			-- smaller than `medium_size'. Afterall, tuning of storage size is
			-- done automatically according to usage.
		require
			medium_size > 0
		local
			new_capacity: INTEGER; hts: HASH_TABLE_SIZE
		do
			free_nodes ::= common_free_nodes.fast_reference_at(generating_type)
			if free_nodes = Void then
				create free_nodes.set_item(Void)
				common_free_nodes.add(free_nodes, generating_type)
			end
			new_capacity := hts.prime_number_ceiling(medium_size)
			buckets := buckets.calloc(new_capacity)
			capacity := new_capacity
			cache_user := -1
			count := 0
		ensure
			is_empty
			capacity >= medium_size
		end

	special_common_dictionary (fn: like free_nodes) is
			-- Used to avoid having a recursive once function while initializing `common_free_nodes'.
		require
			fn /= Void
			-- common_free_nodes = Void
			-- {V_} ?:= free_nodes
			-- {K_} ?:= generating_type
		local
			new_capacity: INTEGER; hts: HASH_TABLE_SIZE
		do
			new_capacity := hts.prime_number_ceiling(Default_size)
			buckets := buckets.calloc(new_capacity)
			capacity := new_capacity
			cache_user := -1
			count := 0
			free_nodes := fn
		ensure
			count = 0
		end

invariant
	capacity > 0
	capacity >= count
	cache_user.in_range(-1, count)
	cache_user > 0 implies cache_node /= Void
	cache_user > 0 implies cache_buckets.in_range(0, capacity - 1)
	free_nodes /= Void

end -- class HASHED_DICTIONARY
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
