-- See the Copyright notice at the end of this file.
--
deferred class XML_REPOSITORY_IMPL[O_ -> STORABLE]
	--
	-- Implementation of the XML store and retrieval
	--
	-- Schema:
	--
	--   <!DOCTYPE repository [
	--
	--   <!ELEMENT repository (layout | reference | embedded | basic)* >
	--   <!ATTLIST repository version CDATA #REQUIRED >
	--
	--   <!ELEMENT layout (reference | embedded | basic | array)* >
	--   <!ATTLIST layout type CDATA #REQUIRED >
	--   <!ATTLIST layout ref ID #REQUIRED >
	--
	--   <!ELEMENT reference (EMPTY) >
	--   <!ATTLIST reference name CDARA #REQUIRED >
	--   <!ATTLIST reference ref IDREF #REQUIRED >
	--
	--   <!ELEMENT embedded (reference | embedded | basic | array)* >
	--   <!ATTLIST embedded name CDATA #REQUIRED >
	--   <!ATTLIST embedded type CDATA #REQUIRED >
	--
	--   <!ELEMENT basic (EMPTY) >
	--   <!ATTLIST basic name #REQUIRED >
	--   <!ATTLIST basic type #REQUIRED >
	--   <!ATTLIST basic value #REQUIRED >
	--
	--   <!ELEMENT array (reference | embedded | basic)* >
	--   <!ATTLIST array name CDATA #REQUIRED >
	--   <!ATTLIST array type CDATA #REQUIRED >
	--
	--   ] >
	--
	-- Vocabulary:
	--
	-- * layout: the description of the contents of an object (its layout). This object is referenced
	-- by a reference (see below).
	--
	-- * reference: a reference to an object. Note that the layout of the object MUST have been defined before
	-- any reference links to it. A special 'Void' ref indicates a Void object.
	--
	-- * embedded: a user-defined expanded object (i.e. not one of the basic objects)
	--
	-- * basic: a special expanded object, with a simple value, specially treated by the compiler and by this
	-- class. Basic types are INTEGER and co., READ and co., CHARACTER and BOOLEAN.
	--
	-- * array: a NATIVE_ARRAY of objects. The type is, in that particular case, the type of the items.
	--
inherit
	REPOSITORY[O_]
	XML_CALLBACKS

feature {ANY} -- Error handling on repository update
	register_update_error_handler (a_error_handler: PROCEDURE[TUPLE[INTEGER, INTEGER, STRING]]) is
		do
			update_error_handlers.add_last(a_error_handler)
		end

feature {}
	update_error_handlers: FAST_ARRAY[PROCEDURE[TUPLE[INTEGER, INTEGER, STRING]]]

	fire_update_error (line, column: INTEGER; message: STRING) is
		local
			i: INTEGER
		do
			sedb_breakpoint
			from
				i := update_error_handlers.lower
			until
				i > update_error_handlers.upper
			loop
				update_error_handlers.item(i).call([line, column, message])
				i := i + 1
			end
		end

feature {} -- Implementation of update
	update_name: STRING is ""

	update_ref: STRING is ""

	update_type: STRING is ""

	update_value: STRING is ""

	update_capacity: STRING is ""

	update_version: STRING is ""

	solve_again: BOOLEAN

	update_layouts: STACK[XML_REPOSITORY_LAYOUT] is
		once
			create Result.make
		end

	open_nodes: FAST_ARRAY[STRING] is
		once
			create Result.with_capacity(4)
		end

	updated_internals: HASHED_DICTIONARY[INTERNALS, STRING] is
		once
			create Result.make
		end

	layouts: FAST_ARRAY[XML_REPOSITORY_LAYOUT] is
		once
			create Result.make(0)
		end

	objects: HASHED_DICTIONARY[STRING, STRING] is
		once
			create Result.make
		end

	solve (ref: STRING): INTERNALS is
		do
			if not ref.is_equal(once "Void") then
				Result := updated_internals.reference_at(ref)
				if Result = Void then
					solve_again := True
				end
			end
		end

	solver: FUNCTION[TUPLE[STRING], INTERNALS] is
		once
			Result := agent solve
		end

	xml: XML_PARSER is
		once
			create Result.make
		end

	read_from_stream (in_stream: INPUT_STREAM) is
		local
			string: STRING
		do
			repository.clear_count
			--|*** (CAD) I think it is necessary
			check
				open_nodes.is_empty
			end
			from
			until
				updated_internals.is_empty
			loop
				string := updated_internals.key(updated_internals.lower)
				updated_internals.remove(string)
				release_string(string)
			end
			update_from_stream(in_stream)
		end

	update_from_stream (in_stream: INPUT_STREAM) is
		do
			xml.connect_to(in_stream)
			xml.parse(Current)
			if not open_nodes.is_empty then
				fire_update_error(xml.line, xml.column, once "open_nodes should be empty")
			end
		end

	record_object (ref, name: STRING; line, column: INTEGER) is
			-- Register the object as a high-level one, i.e. put it in the repository.
		local
			typed: TYPED_INTERNALS[O_]; error: STRING
		do
			if not updated_internals.has(ref) then
				error := once ""
				error.copy(once "Unknown reference: ")
				error.append(ref)
				fire_update_error(line, column, error)
			else
				typed ::= solve(ref)
				put(typed.object, name)
			end
		end

	prepare_layout (a_layout: XML_REPOSITORY_LAYOUT) is
		do
			if not update_name.is_empty then
				a_layout.set_name(update_name)
			end
			if not update_ref.is_empty then
				a_layout.set_ref(update_ref)
			end
			if not update_type.is_empty then
				a_layout.set_type(update_type)
			end
			if not update_value.is_empty then
				a_layout.set_value(update_value)
			end
			if not update_capacity.is_empty then
				a_layout.set_capacity(update_capacity.to_integer)
			end
		end

feature {XML_PARSER}
	with_attribute (attribute_name: STRING; attribute_value: STRING; line, column: INTEGER) is
		local
			error: STRING
		do
			inspect
				attribute_name
			when "name" then
				update_name.copy(attribute_value)
			when "type" then
				update_type.copy(attribute_value)
			when "ref" then
				update_ref.copy(attribute_value)
			when "value" then
				update_value.copy(attribute_value)
			when "capacity" then
				update_capacity.copy(attribute_value)
			when "version" then
				update_version.copy(attribute_value)
			else
				error := once ""
				error.copy(once "Unknown attribute: ")
				error.append(attribute_name)
				fire_update_error(line, column, error)
			end
		end

	open_node (node_name: STRING; line, column: INTEGER) is
		local
			s, error: STRING; layout: XML_REPOSITORY_LAYOUT
		do
			s := new_string
			s.copy(node_name)
			open_nodes.add_last(s)
			layout := new_layout
			layout.set(node_name)
			update_layouts.push(layout)
			inspect
				node_name
			when "repository" then
				if not update_version.is_equal(version) then
					error := once ""
					error.copy(once "Incompatible versions: expected ")
					error.append(version)
					error.append(once " but got ")
					error.append(update_version)
					fire_update_error(line, column, error)
				end
				objects.clear_count
				layouts.clear_count
			when "layout" then
				prepare_layout(layout)
			when "reference" then
				prepare_layout(layout)
			when "embedded" then
				prepare_layout(layout)
			when "basic" then
				prepare_layout(layout)
			when "array" then
				prepare_layout(layout)
			else
				error := once ""
				error.copy(once "Unknown node: ")
				error.append(node_name)
				fire_update_error(line, column, error)
			end
			clear_attributes
		end

	close_node (node_name: STRING; line, column: INTEGER) is
		local
			layout: XML_REPOSITORY_LAYOUT; internals: INTERNALS; ref, error: STRING
			i: INTEGER
		do
			release_string(open_nodes.last)
			open_nodes.remove_last
			layout := update_layouts.top
			update_layouts.pop
			inspect
				node_name
			when "repository" then
				from
					solve_again := True
				until
					not solve_again
				loop
					solve_again := False
					from
						i := layouts.lower
					until
						i > layouts.upper
					loop
						layout := layouts.item(i)
						internals := layout.solve(solver)
						if internals /= Void then
							if updated_internals.has(layout.ref) then
								ref := updated_internals.key(index_of_key(layout.ref))
								updated_internals.remove(ref)
							else
								ref := new_string
							end
							ref.copy(layout.ref)
							updated_internals.add(internals, ref)
						end
						i := i + 1
					end
				end
				from
					i := objects.lower
				until
					i > objects.upper
				loop
					record_object(objects.item(i), objects.key(i), line, column)
					i := i + 1
				end
			when "layout" then
				layouts.add_last(layout)
			when "reference" then
				if update_layouts.count = 1 then
					-- means only the "repository" node is open
					objects.put(layout.ref, layout.name)
				else
					update_layouts.top.add_layout(layout)
				end
			when "embedded" then
				update_layouts.top.add_layout(layout)
			when "basic" then
				update_layouts.top.add_layout(layout)
			when "array" then
				update_layouts.top.add_layout(layout)
			else
				error := once ""
				error.copy(once "Unknown node: ")
				error.append(node_name)
				fire_update_error(line, column, error)
			end
		end

	open_close_node (node_name: STRING; line, column: INTEGER) is
		do
			open_node(node_name, line, column)
			close_node(node_name, line, column)
		end

	current_node: STRING is
		do
			Result := open_nodes.last
		end

	xml_header (line, column: INTEGER) is
		do
			clear_attributes
		end

	processing_instruction (a_target, a_data: STRING) is
		do
			check
				not_called: False
			end
		end

	entity (a_entity: STRING; line, column: INTEGER): STRING is
		do
			check
				not_called: False
			end
		end

	data (a_data: STRING; line, column: INTEGER) is
		local
			i: INTEGER; break: BOOLEAN
		do
			from
				i := a_data.lower
			until
				i > a_data.upper or else break
			loop
				if not a_data.item(i).is_separator then
					fire_update_error(line, column, once "Separator expected")
				end
				i := i + 1
			end
		end

	parse_error (line, column: INTEGER; message: STRING) is
		do
			at_error := True
			fire_update_error(line, column, message)
		end

	at_error: BOOLEAN

feature {} -- Implementation of commit
	commit_map: SET[POINTER] is
			-- Used when committing object not to commit them twice
		once
			create {HASHED_SET[POINTER]} Result.make
		end

	write_to_stream (out_stream: OUTPUT_STREAM) is
		require
			out_stream.is_connected
		local
			i: INTEGER
		do
			commit_map.clear_count
			out_stream.put_string(once "<?xml version='1.1'?>%N<repository version='")
			out_stream.put_string(version)
			out_stream.put_string(once "'>%N")
			from
				i := lower
			until
				i > upper
			loop
				write_object(key(i), item(i), out_stream)
				i := i + 1
			end
			out_stream.put_string(once "</repository>%N")
		end

	write_object (name: like key; object: like item; out_stream: OUTPUT_STREAM) is
		local
			int: INTERNALS
		do
			if object = Void then
				write_reference(Void, name, out_stream)
			else
				int := object.to_internals
				if int.type_is_expanded then
					write_expanded(int, name, out_stream)
				else
					if not commit_map.has(int.object_as_pointer) then
						write_layout(int, out_stream)
					end
					write_reference(int, name, out_stream)
				end
			end
		end

	write_reference (layout: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		require
			layout /= Void implies not layout.type_is_expanded
		do
			out_stream.put_string(once "<reference name='")
			out_stream.put_string(name)
			if layout = Void then
				out_stream.put_string(once "' ref='Void")
			else
				out_stream.put_string(once "' ref='")
				out_stream.put_integer(layout.object_as_pointer.hash_code)
			end
			out_stream.put_string(once "'/>%N")
		end

	write_layout (layout: INTERNALS; out_stream: OUTPUT_STREAM) is
		require
			not commit_map.has(layout.object_as_pointer)
			not layout.type_is_expanded
		local
			i: INTEGER; int: INTERNALS
		do
			-- Add the pointer to the map of "known objects" (those already written). It must be done first
			-- because of possible recursion
			commit_map.add(layout.object_as_pointer)
			-- Write the nested objects not already defined
			from
				i := 1
			until
				i > layout.type_attribute_count
			loop
				int := layout.object_attribute(i)
				if int /= Void then
					if int.type_is_expanded then
						if int.type_is_native_array then
							write_array_fields_layouts(int, out_stream)
						end
					elseif not commit_map.has(int.object_as_pointer) then
						write_layout(int, out_stream)
					end
				end
				i := i + 1
			end
			out_stream.put_string(once "<layout ref='")
			out_stream.put_integer(layout.object_as_pointer.hash_code)
			out_stream.put_string(once "' type='")
			out_stream.put_string(layout.type_generating_type)
			out_stream.put_string(once "'>%N")
			write_contents(layout, out_stream)
			out_stream.put_string(once "</layout>%N")
		ensure
			commit_map.has(layout.object_as_pointer)
		end

	write_contents (layout: INTERNALS; out_stream: OUTPUT_STREAM) is
		local
			i: INTEGER; int: INTERNALS
		do
			from
				i := 1
			until
				i > layout.type_attribute_count
			loop
				int := layout.object_attribute(i)
				if int = Void then
					write_reference(Void, layout.type_attribute_name(i), out_stream)
				else
					if int.type_is_expanded then
						write_expanded(int, layout.type_attribute_name(i), out_stream)
					else
						write_reference(int, layout.type_attribute_name(i), out_stream)
					end
				end
				i := i + 1
			end
		end

	write_array_fields_layouts (array: INTERNALS; out_stream: OUTPUT_STREAM) is
		require
			array.type_is_expanded and then array.type_is_native_array
		local
			f: INTEGER; int: INTERNALS
		do
			from
				f := 1
			until
				f > array.type_attribute_count
			loop
				int := array.object_attribute(f)
				if int /= Void and then not int.type_is_expanded and then not commit_map.has(int.object_as_pointer) then
					write_layout(int, out_stream)
				end
				f := f + 1
			end
		end

	write_expanded (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		require
			internals.type_is_expanded
		local
			type: STRING
		do
			type := internals.type_generating_type
			inspect
				type
			when "CHARACTER" then
				write_character_layout_object(internals, name, out_stream)
			when "BOOLEAN" then
				write_boolean_layout_object(internals, name, out_stream)
			when "INTEGER_8" then
				write_integer_8_layout_object(internals, name, out_stream)
			when "INTEGER_16" then
				write_integer_16_layout_object(internals, name, out_stream)
			when "INTEGER" then
				write_integer_layout_object(internals, name, out_stream)
			when "INTEGER_32" then
				write_integer_32_layout_object(internals, name, out_stream)
			when "INTEGER_64" then
				write_integer_64_layout_object(internals, name, out_stream)
			when "REAL" then
				write_real_layout_object(internals, name, out_stream)
			when "REAL_32" then
				write_real_32_layout_object(internals, name, out_stream)
			when "REAL_64" then
				write_real_64_layout_object(internals, name, out_stream)
			when "REAL_80" then
				write_real_80_layout_object(internals, name, out_stream)
			when "REAL_128" then
				write_real_128_layout_object(internals, name, out_stream)
			when "REAL_EXTENDED" then
				write_real_expanded_layout_object(internals, name, out_stream)
			else
				if internals.type_is_native_array then
					write_array_layout_object(internals, name, out_stream)
				else
					write_embedded_layout_object(internals, name, out_stream)
				end
			end
		end

	write_character_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[CHARACTER]; c: CHARACTER
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='CHARACTER' value='")
			out_stream.put_integer(c.code)
			out_stream.put_string(once "'/>%N")
		end

	write_boolean_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[BOOLEAN]; c: BOOLEAN
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='BOOLEAN' value='")
			out_stream.put_boolean(c)
			out_stream.put_string(once "'/>%N")
		end

	write_integer_8_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[INTEGER_8]; c: INTEGER_8
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='INTEGER_8' value='")
			out_stream.put_integer(c)
			out_stream.put_string(once "'/>%N")
		end

	write_integer_16_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[INTEGER_16]; c: INTEGER_16
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='INTEGER_16' value='")
			out_stream.put_integer(c)
			out_stream.put_string(once "'/>%N")
		end

	write_integer_32_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[INTEGER_32]; c: INTEGER_32
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='INTEGER_32' value='")
			out_stream.put_integer(c)
			out_stream.put_string(once "'/>%N")
		end

	write_integer_64_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[INTEGER_64]; c: INTEGER_64
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='INTEGER_64' value='")
			out_stream.put_integer(c)
			out_stream.put_string(once "'/>%N")
		end

	write_integer_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[INTEGER]; c: INTEGER
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='INTEGER' value='")
			out_stream.put_integer(c)
			out_stream.put_string(once "'/>%N")
		end

	write_real_32_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL_32]; c: REAL_32
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL_32' value='")
			out_stream.put_real(c)
			out_stream.put_string(once "'/>%N")
		end

	write_real_64_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL_64]; c: REAL_64
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL_64' value='")
			out_stream.put_real(c)
			out_stream.put_string(once "'/>%N")
		end

	write_real_80_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL_80]; c: REAL_80
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL_80' value='")
			out_stream.put_real(c.force_to_real_64)
			out_stream.put_string(once "'/>%N")
		end

	write_real_128_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL_128]; c: REAL_128
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL_128' value='")
			out_stream.put_real(c.force_to_real_64)
			out_stream.put_string(once "'/>%N")
		end

	write_real_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL]; c: REAL
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL' value='")
			out_stream.put_real(c)
			out_stream.put_string(once "'/>%N")
		end

	write_real_expanded_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			t: TYPED_INTERNALS[REAL_EXTENDED]; c: REAL_EXTENDED
		do
			t ::= internals
			c := t.object
			out_stream.put_string(once "<basic name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='REAL_EXTENDED' value='")
			out_stream.put_real(c.force_to_real_64)
			out_stream.put_string(once "'/>%N")
		end

	write_array_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		local
			type: STRING; f: INTEGER; int: INTERNALS
		do
			if internals.type_attribute_count > 0 then
				type := once ""
				type.copy(internals.type_generating_type)
				type.remove_prefix(once "NATIVE_ARRAY[")
				type.remove_suffix(once "]")
				out_stream.put_string(once "<array name='")
				out_stream.put_string(name)
				out_stream.put_string(once "' type='")
				out_stream.put_string(type)
				out_stream.put_string(once "' capacity='")
				out_stream.put_integer(internals.type_attribute_count)
				out_stream.put_string(once "'>%N")
				from
					f := 1
				until
					f > internals.type_attribute_count
				loop
					int := internals.object_attribute(f)
					if int = Void then
						write_reference(Void, internals.type_attribute_name(f), out_stream)
					else
						if int.type_is_expanded then
							write_expanded(int, internals.type_attribute_name(f), out_stream)
						else
							write_reference(int, internals.type_attribute_name(f), out_stream)
						end
					end
					f := f + 1
				end
				out_stream.put_string(once "</array>%N")
			end
		end

	write_embedded_layout_object (internals: INTERNALS; name: STRING; out_stream: OUTPUT_STREAM) is
		do
			out_stream.put_string(once "<embedded name='")
			out_stream.put_string(name)
			out_stream.put_string(once "' type='")
			out_stream.put_string(internals.type_generating_type)
			out_stream.put_string(once "'>%N")
			write_contents(internals, out_stream)
			out_stream.put_string(once "</embedded>%N")
		end

feature {} -- Internals
	frozen version: STRING is "1"

	free_layouts: STACK[XML_REPOSITORY_LAYOUT] is
		once
			create Result.make
		end

	new_layout: XML_REPOSITORY_LAYOUT is
		do
			if free_layouts.is_empty then
				create Result.make
			else
				Result := free_layouts.top
				free_layouts.pop
			end
		ensure
			Result.is_clear
		end

	release_layout (a_layout: XML_REPOSITORY_LAYOUT) is
		do
			a_layout.clear
			free_layouts.push(a_layout)
		end

	free_strings: STACK[STRING] is
		once
			create Result.make
		end

	new_string: STRING is
		do
			if free_strings.is_empty then
				Result := ""
			else
				Result := free_strings.top
				free_strings.pop
			end
		ensure
			Result.is_empty
		end

	release_string (a_string: STRING) is
		do
			a_string.clear_count
			free_strings.push(a_string)
		end

	index_of_key (a_string: STRING): INTEGER is
		local
			i: INTEGER; done: BOOLEAN
		do
			from
				i := updated_internals.lower
			until
				done
			loop
				if updated_internals.key(i).is_equal(a_string) then
					Result := i
					done := True
				elseif i = updated_internals.upper then
					done := True
				else
					i := i + 1
				end
			end
		end

	clear_attributes is
		do
			update_name.clear_count
			update_ref.clear_count
			update_type.clear_count
			update_value.clear_count
			update_capacity.clear_count
			update_version.clear_count
		end

feature {} -- Creation
	make is
			-- Create a not-connected empty repository.
		do
			if repository = Void then
				create {HASHED_DICTIONARY[O_, STRING]} repository.make
				create update_error_handlers.with_capacity(2)
			else
				repository.clear_count
				update_error_handlers.clear_count
			end
		end

end -- class XML_REPOSITORY_IMPL
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
