class WEAK_ARRAY [R_ -> RECYCLABLE]
--
-- Useful to implement recycling support in libraries.
-- Has no sense for expanded types (it will not even compile).
--
-- This class is designed to work with and without the garbage collector. It takes advantage of the GC if 
-- it is used, but it also minimises the memory footprint when the GC is ''not'' used.
--
-- See also STRING_WEAK_ARRAY, especially tuned for STRING usage.
--

insert
	MEMORY

creation {ANY}
	make

feature {ANY}
	recycle (a_item: like item) is
		require
			a_item /= Void
		local
			i: INTEGER; done: BOOLEAN; wr: WEAK_REFERENCE[R_]
		do
			a_item.recycle
			if cursor /= no_item then
				from
					i := cursor
				until
					done or else i > ring.upper
				loop
					wr := ring.item(i)
					if wr.item = Void then
						wr.set_item(a_item)
						done := True
						cursor := i
					end
					i := i + 1
				end
				if not done then
					from
						i := ring.lower
					until
						done or else i = cursor
					loop
						wr := ring.item(i)
						if wr.item = Void then
							wr.set_item(a_item)
							done := True
							cursor := i
						end
						i := i + 1
					end
				end
			end
			if not done then
				create wr.set_item(a_item)
				ring.add_last(wr)
				cursor := ring.upper
			end
		ensure
			cursor /= no_item
			ring.valid_index(cursor)
		end

	item: R_ is
		local
			i: INTEGER; gc: BOOLEAN
		do
			if cursor /= no_item then
				gc := collecting
				collection_off
				i := next(cursor)
				if i /= no_item then
					Result := item_at(i)
					check
						Result /= Void
					end
				end
				if gc then
					collection_on
				end
			end
		ensure
			Result = Void implies cursor = no_item
		end

feature {} -- internals -- also used by STRING_WEAK_ARRAY
	no_item: INTEGER is -1

	next (start: INTEGER): INTEGER is
		require
			not collecting
			start /= no_item
		local
			i: INTEGER
		do
			Result := no_item
			from
				i := start
			until
				Result /= no_item or else i > ring.upper
			loop
				if ring.item(i).item /= Void then
					Result := i
				end
				i := i + 1
			end
			if Result = no_item then
				from
					i := ring.lower
				until
					Result /= no_item or else i = cursor
				loop
					if ring.item(i).item /= Void then
						Result := i
					end
					i := i + 1
				end
			end
		ensure
			Result /= no_item implies ring.item(Result).item /= Void
		end

	item_at (i: INTEGER): R_ is
		require
			not collecting
			i /= no_item
		local
			wr: WEAK_REFERENCE[R_]
		do
			wr := ring.item(i)
			Result := wr.item
			if Result /= Void then
				wr.set_item(Void)
				cursor := i + 1
			else
				cursor := no_item
			end
		end

feature {}
	cursor: INTEGER
	ring: RING_ARRAY[WEAK_REFERENCE[R_]]

	make is
		do
			create ring.with_capacity(16, 0)
			cursor := no_item
		end

invariant
	ring.lower = 0

end
