-- See the Copyright notice at the end of this file.
--
class XMLNS_PARSER
	--
	-- A namespace-aware XML parser
	--

inherit
	XML_CALLBACKS
		rename
			validator as xml_validator
			set_validator as set_xml_validator
		end

insert
	XML_NAMESPACES

creation {ANY}
	connect_to, make

feature {ANY}
	parse (a_callbacks: XMLNS_CALLBACKS) is
			-- Parse an XML documents by sending parsing events to the given `callbacks'
		do
			callbacks := a_callbacks
			parser.parse(Current)
			if validator /= Void then
				validator.the_end
			end
		end

	connect_to (a_stream: INPUT_STREAM) is
			-- Connect to the given XML document
		require
			a_stream.is_connected
		do
			make
			parser.connect_to(a_stream)
		end

	line: INTEGER is
		do
			Result := parser.line
		end

	column: INTEGER is
		do
			Result := parser.column
		end

feature {}
	make is
			-- Create a not connected parser
		do
			if parser = Void then
				create parser.make
				create namespaces.make(0)
			end
		end

	parser: XML_PARSER

	callbacks: XMLNS_CALLBACKS

feature {XML_NAMESPACES}
	set_validator (a_validator: XMLNS_VALIDATOR) is
		do
			callbacks.set_validator(a_validator)
		end

	validator: XMLNS_VALIDATOR is
		do
			Result := callbacks.validator
		end

feature {XML_PARSER}
	with_attribute (attribute_name: STRING; attribute_value: STRING; l, c: INTEGER) is
		do
			if not attributes_for_new_node then
				namespaces.add_last(new_namespaces)
				attributes_for_new_node := True
			end
			if attribute_name.has_prefix(once "xml") then
				xml_attribute(attribute_name, attribute_value, l, c)
			else
				split_namespace(attribute_name, l, c)
				if validator /= Void then
					validator.with_attribute(namespace, name, attribute_value, l, c)
				end
				callbacks.with_attribute(namespace, name, attribute_value, l, c)
			end
		end

	open_node (node_name: STRING; l, c: INTEGER) is
		do
			if attributes_for_new_node then
				attributes_for_new_node := False
			else
				namespaces.add_last(new_namespaces)
			end
			split_namespace(node_name, l, c)
			if validator = Void then
				callbacks.open_node(namespace, name, l, c)
			elseif validator.is_valid_open_node(namespace, name, l, c) then
				validator.open_node(namespace, name, l, c)
				callbacks.open_node(namespace, name, l, c)
			else
				parse_error(l, c, once "Invalid opening tag")
			end
		end

	close_node (node_name: STRING; l, c: INTEGER) is
		do
			split_namespace(node_name, l, c)
			if validator = Void then
				callbacks.close_node(namespace, name, l, c)
			elseif validator.is_valid_close_node(namespace, name, l, c) then
				validator.close_node(namespace, name, l, c)
				callbacks.close_node(namespace, name, l, c)
			else
				parse_error(l, c, once "Invalid closing tag")
			end
			old_namespaces(namespaces.last)
			namespaces.remove_last
		end

	open_close_node (node_name: STRING; l, c: INTEGER) is
		local
			local_namespaces: BOOLEAN
		do
			if attributes_for_new_node then
				attributes_for_new_node := False
				local_namespaces := True
			end
			split_namespace(node_name, l, c)
			if validator = Void then
				callbacks.open_close_node(namespace, name, l, c)
			elseif validator.is_valid_open_close_node(namespace, name, l, c) then
				validator.open_close_node(namespace, name, l, c)
				callbacks.open_close_node(namespace, name, l, c)
			else
				parse_error(l, c, once "Invalid empty tag")
			end
			if local_namespaces then
				old_namespaces(namespaces.last)
				namespaces.remove_last
			end
		end

	xml_header (l, c: INTEGER) is
		do
			callbacks.xml_header(l, c)
		end

	processing_instruction (a_target, a_data: STRING) is
		do
			callbacks.processing_instruction(a_target, a_data)
		end

	entity (a_entity: STRING; l, c: INTEGER): STRING is
		do
			if validator = Void then
				Result := callbacks.entity(a_entity, l, c)
			else
				Result := validator.entity(a_entity, l, c)
				if Result = Void then
					Result := callbacks.entity(a_entity, l, c)
				end
			end
		end

	current_node: STRING is
		local
			ns: STRING
		do
			Result := once ""
			ns := callbacks.current_namespace
			if ns = Void then
				Result.copy(callbacks.current_node)
			else
				Result.copy(ns)
				Result.extend(':')
				Result.append(callbacks.current_node)
			end
		end

	data (a_data: STRING; l, c: INTEGER) is
		do
			if validator = Void then
				callbacks.data(a_data, l, c)
			elseif validator.is_valid_data(a_data, l, c) then
				validator.data(a_data, l, c)
				callbacks.data(a_data, l, c)
			else
				parse_error(l, c, once "Invalid data")
			end
		end

	parse_error (l, c: INTEGER; message: STRING) is
		do
			callbacks.parse_error(l, c, message)
		end

	at_error: BOOLEAN is
		do
			Result := callbacks.at_error
		end

feature {}
	namespace: STRING
			-- set by `split_namespace'

	name: STRING
			-- set by `split_namespace'

	split_namespace (a_name: STRING; l, c: INTEGER) is
			-- Sets `namespace' and `name' according to the given name, splitting at the first colon (':'). If
			-- there is no colon, `namespace' is Void and `name' contains the full given name. Otherwise
			-- `namespace' contains the URI of the namespace and `name' contains the part of the given name after
			-- the first colon.
		local
			i: INTEGER; ns, error: STRING
		do
			name := once ""
			name.copy(a_name)
			i := a_name.first_index_of(':')
			if a_name.valid_index(i) then
				namespace := once ""
				namespace.copy(a_name)
				namespace.shrink(1, i - 1)
				name.shrink(i + 1, name.count)
			else
				namespace := once ""
			end
			ns := find_namespace(namespace)
			if ns = Void then
				error := once ""
				error.copy(once "Unknown namespace prefix: ")
				error.append(namespace)
				parse_error(l, c, error)
				namespace := Void
			else
				namespace.copy(ns)
			end
		end

	namespaces: FAST_ARRAY[HASHED_DICTIONARY[STRING, STRING]]
			-- The known namespaces

	xml_attribute (attribute_name, attribute_value: STRING; l, c: INTEGER) is
		local
			i: INTEGER; pfx, ns, error: STRING
			action: PROCEDURE[TUPLE[XMLNS_PARSER, STRING]]
		do
			ns := once ""
			ns.copy(attribute_name)
			i := attribute_name.first_index_of(':')
			if attribute_name.valid_index(i) then
				pfx := once ""
				pfx.copy(attribute_name)
				pfx.shrink(1, i)
				ns.shrink(i + 1, ns.count)
			else
				pfx := ns
				ns := once ""
			end
			inspect
				pfx
			when "xmlns" then
				action := namespace_actions.reference_at(attribute_value)
				if action /= Void then
					action.call([Current, ns])
				end
				namespaces.last.put(attribute_value, ns)
			else
				error := once ""
				error.copy(once "Bad namespace prefix (must not begin by %"xml%"): ")
				error.append(pfx)
				parse_error(l, c, error)
			end
		end

	find_namespace (a_namespace_ref: STRING): STRING is
		local
			i: INTEGER
		do
			from
				i := namespaces.upper
			until
				Result /= Void or else i < namespaces.lower
			loop
				Result := namespaces.item(i).reference_at(a_namespace_ref)
				i := i - 1
			end
		end

	attributes_for_new_node: BOOLEAN

feature {} -- Memory management
	unused_namespaces: FAST_ARRAY[WEAK_REFERENCE[HASHED_DICTIONARY[STRING, STRING]]] is
		once
			create Result.make(0)
		end

	new_namespaces: HASHED_DICTIONARY[STRING, STRING] is
		local
			i: INTEGER
		do
			from
				i := unused_namespaces.upper
			until
				Result /= Void or else i < unused_namespaces.lower
			loop
				Result := unused_namespaces.item(i).item
				i := i - 1
			end
			if Result = Void then
				create Result.make
			else
				Result.clear_count
			end
		ensure
			Result.is_empty
		end

	old_namespaces (a_namespaces: HASHED_DICTIONARY[STRING, STRING]) is
		local
			i: INTEGER; done: BOOLEAN; wr: WEAK_REFERENCE[HASHED_DICTIONARY[STRING, STRING]]
		do
			from
				i := unused_namespaces.lower
			until
				done or else i > unused_namespaces.upper
			loop
				wr := unused_namespaces.item(i)
				if wr.item = Void then
					wr.set_item(a_namespaces)
					done := True
				end
				i := i + 1
			end
			if not done then
				create wr.set_item(a_namespaces)
				unused_namespaces.add_last(wr)
			end
		end

end -- class XMLNS_PARSER
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
