-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class ACE
	--
	-- Assembly of Classes in Eiffel.
	--
	-- Both an ACE file parser and a facade to the universe (i.e. all the known classes).
	--
	-- This singleton is shared via the GLOBALS.`ace' once function.
	--

inherit
	PARSER

insert
	ASSERTION_LEVEL_NUMBERING
	DEBUG_KEY_SUPPORT
	FLAG_NAME_LIST

feature {ANY}
	file_path: STRING
			-- Non Void when a ACE file is in use (keep in mind that one can
			-- stil launch a compilation without any ACE file, passing all the
			-- information using command arguments only).
			-- Usually, ACE files are suffixed with ".ace" or ".ACE".

	executable_name: STRING
			-- The name of the executable to build (after the "system" keyword
			-- in the ACE file or after the -o flag in the command line).
			-- In command line mode, a Void value means that "a.out" is to be
			-- used for C mode while using gcc for example. For the Java
			-- byte-code this name is used as the name of the main output class
			-- file and as the name of the directory used to store auxilliary
			-- class files.

	root_class_name: HASHED_STRING is
			-- The name of the root class using only upper case letters. This
			-- is actually the generating type of the very first live object at
			-- runtime. This `root_class_name' is after the "root" keyword in
			-- the ACE file or is given as a command line argument.
		do
			if root_class_names.count > 0 then
				Result := root_class_names.last
			end
		end

	root_procedure_name: STRING is
			-- The name of the procedure which is actually the main program.
			-- This procedure is supposed to be member of the creation clause
			-- of `root_class_name'.
		require
			root_class_names.count = 1
		local
			bc: CLASS_TEXT; cn: CLASS_NAME
		do
			if root_procedure_name_memory = Void then
				create cn.unknown_position(root_class_name)
				bc := smart_eiffel.class_text(cn, True)
				root_procedure_name_memory := bc.default_root_procedure_name
			end
			Result := root_procedure_name_memory
		end

	root_class_names: FAST_ARRAY[HASHED_STRING] is
			-- All the class names given on the command line
		once
			create Result.with_capacity(1)
		end
		--|*** Boost level for debuging purpose. Should decode boost option
		--|if we decide to keep it.

	boost1: BOOLEAN is True

	boost2: BOOLEAN is False

	boost3: BOOLEAN is False

	boost: BOOLEAN is
		do
			Result := highest_encountered_level = level_boost
		end

	no_check: BOOLEAN is
			-- The system level value (see also {CLASS_TEXT}.no_check).
		do
			Result := highest_encountered_level >= level_no
		end

	require_check: BOOLEAN is
		do
			Result := highest_encountered_level >= level_require
		end

	ensure_check: BOOLEAN is
		do
			Result := highest_encountered_level >= level_ensure
		end

	invariant_check: BOOLEAN is
		do
			Result := highest_encountered_level >= level_invariant
		end

	loop_check: BOOLEAN is
		do
			Result := highest_encountered_level >= level_loop
		end

	all_check: BOOLEAN is
		do
			Result := highest_encountered_level >= level_all
		end

	flat_check: BOOLEAN
			-- Each assertion will be executed in no_check mode

	no_main: BOOLEAN
			-- Don't include a main() in the generated executable.

	safety_check: BOOLEAN

	manifest_string_trace: BOOLEAN

	high_memory_compiler: BOOLEAN
			--|*** Is it still used ?? *** Dom feb 7th 2005 ***

	profile: BOOLEAN
			-- Is the -profile flag used?

	relax: BOOLEAN
			-- Is the -relax flag used?

	precompile_header: BOOLEAN
			-- Is the -precompile_header flag used ?
			--|*** ensure Result implies system_tools.precompile_header_supported	 (as soon as ensure on attribute is supported)

	sedb: BOOLEAN is
			-- The -sedb flag is used or some class of the ACE file is
			-- in trace mode. (When `sedb' is set, files sys/runtime/sedb.[ch]
			-- are included.)
		do
			Result := no_check and then sedb_flag
		end

	jar: BOOLEAN

	run: BOOLEAN

	use_jar: STRING

	use_jvm: STRING

	java_compiler: STRING

	ss: STRING

	mx: STRING

	ms: STRING

	classpath: STRING

	clean_classes: BOOLEAN

	hard_clean: BOOLEAN

feature {CLASS_TEXT}
	trace_of (a_class_text: CLASS_TEXT): BOOLEAN is
			-- Is the -sedb trace mode enabled for `a_class_text'.
		local
			cluster: CLUSTER
		do
			cluster := a_class_text.cluster
			if cluster /= Void then
				Result := cluster.trace(a_class_text.name)
			else
				Result := default_trace
			end
			if Result then
				if default_assertion_level = level_boost then
					default_assertion_level := level_no
				end
			end
		end

feature {E_DEBUG}
	is_debug_checked (e_debug: E_DEBUG; type: TYPE): BOOLEAN is
			-- Is this `e_debug' statement is active or not?
			-- Note: during the execution of this routine, the
			-- `default_assertion_level' may be switched from `level_boost' to
			-- `level_no'.
		require
			e_debug /= Void
		local
			ct: CLASS_TEXT; cluster: CLUSTER
		do
			--|*** I think the test belongs in E_DEBUG (requires a few
			--|*** exportation changes in CLUSTER) (FM - june 9th 2004)
			ct := type.class_text
			cluster := ct.cluster
			if cluster /= Void then
				Result := cluster.debug_check(ct.name, e_debug)
			else
				Result := default_debug(e_debug)
			end
			if Result then
				--|*** This feels like it is at the right place in ACE (FM - june 9th 2004)
				if default_assertion_level = level_boost then
					default_assertion_level := level_no
				end
			end
		end

feature {CLASS_TEXT}
	assertion_level_of (a_class_text: CLASS_TEXT): INTEGER is
		require
			avoid_recomputation: a_class_text.assertion_level_not_yet_computed
		local
			cluster: CLUSTER
		do
			cluster := a_class_text.cluster
			if cluster /= Void then
				Result := cluster.assertion_level_of(a_class_text.name)
			else
				Result := default_assertion_level
			end
		ensure
			Result.in_range(level_boost, level_all)
		end

feature {COMMAND_LINE_TOOLS}
	analyse_ace_file (fp: like file_path) is
			-- Parse `fp' which is supposed to be some file containing an ACE description.
		require
			not fp.is_empty
			not string_aliaser.registered_one(fp)
		local
			echo_set_verbose_delayed: BOOLEAN
		do
			if file_path /= Void then
				echo.w_put_string("Multiple ACE files in the command line: %"")
				echo.w_put_string(file_path)
				echo.w_put_string("%" and %"")
				echo.w_put_string(fp)
				echo.w_put_string("%".%N")
				die_with_code(exit_failure_code)
			end
			file_path := string_aliaser.string(fp)
			parser_buffer.load_file(fp)
			if not parser_buffer.is_ready then
				error_handler.append("Cannot open %"")
				error_handler.append(file_path)
				error_handler.append("%" file.%NACE file not found.")
				error_handler.print_as_fatal_error
			end
			line := 1
			column := 1
			current_line := parser_buffer.item(line)
			set_highest_encountered_level(level_boost)
			if current_line.count = 0 then
				cc := '%N'
			else
				cc := current_line.first
			end
			drop_comments := True
			skip_comments
			if not a_keyword(once "system") then
				error_handler.add_position(current_position)
				error_handler.append("Keyword %"system%" expected. Invalid ACE file.")
				error_handler.print_as_fatal_error
			end
			executable_name := string_aliaser.string(a_string)
			if not a_keyword(once "root") then
				error_handler.add_position(current_position)
				error_handler.append("Keyword %"root%" expected. Invalid ACE file.")
				error_handler.print_as_fatal_error
			end
			set_root_class_name(string_aliaser.hashed_string(a_identifier))
			a_cluster_mark
			if skip1(':') then
				root_procedure_name_memory := a_identifier
			end
			if a_keyword(fz_default) then
				echo_set_verbose_delayed := a_system_level_defaults
			end
			if a_cluster_clause_list then
			end
			if a_keyword(fz_external) then
				a_external
			end
			if a_keyword(fz_generate) then
				a_generate
			end
			if a_keyword(fz_end) then
			end
			if cc /= end_of_text then
				error_handler.add_position(current_position)
				error_handler.append("End of text expected (invalid ACE file).")
				error_handler.print_as_fatal_error
			end
			parser_buffer.release
			get_started
			if echo_set_verbose_delayed then
				echo.set_verbose
			end
			echo.put_string("Just finished parsing of %"")
			echo.put_string(file_path)
			echo.put_string("%" ACE file.%N")
		ensure
			file_path.is_equal(fp)
			default_assertion_level /= level_not_computed
		end

feature {}
	set_default_c_mode is
		local
			c_mode: STRING
		do
			if default_debug_keys /= Void then
				c_mode := once "debug_check"
			else
				inspect highest_encountered_level
				when level_boost then
					c_mode := once "boost"
				else
					c_mode := once ""
					c_mode.copy(level_name(highest_encountered_level))
					c_mode.append(once "_check")
				end
			end
			system_tools.set_default_c_mode(c_mode)
		end

feature {COMMAND_LINE_TOOLS, SYSTEM_TOOLS}
	set_root_class_name_using (command_line_name: STRING) is
			-- Compute the `root_class_name' name using the `command_line_name' as a model.
			-- Trailing Eiffel file suffix is removed if any.
			-- Leading path is also removed if any.
			-- Finally, the feature `to_upper' is applied.
		require
			not command_line_name.is_empty
		local
			rcn: HASHED_STRING
		do
			rcn := string_aliaser.hashed_string(class_name_using(command_line_name))
			root_class_names.add_last(rcn)
		ensure
			root_class_names.count = old root_class_names.count + 1
			not root_class_names.last.to_string.has_suffix(eiffel_suffix)
		end

feature {}
	class_name_using (name: STRING): STRING is
			-- Compute some Eiffel class name using the `name' information which can be some complete file path
			-- notation with some Eiffel file suffix. The result is written using the standard Eiffel notation
			-- for class names (upper case letters only).
		require
			not name.is_empty
		local
			i: INTEGER; c: CHARACTER
		do
			Result := once ""
			Result.copy(name)
			-- Removes the ".e" or the ".E" suffix if any:
			i := Result.count
			if i > 2 then
				if Result.item(i - 1) = '.' then
					if Result.item(i).same_as('e') then
						Result.remove_tail(2)
					end
				end
			end
			from
				i := Result.count
			until
				i = 0
			loop
				c := Result.item(i)
				if c.is_letter then
					i := i - 1
				elseif c = '_' then
					i := i - 1
				elseif c.is_digit then
					i := i - 1
				else
					Result.remove_head(i)
					i := 0
				end
			end
			Result.to_upper
		end

feature {COMPILE}
	clean: BOOLEAN
			-- Should the clean command be launched after compilation?

feature {COMPILE_TO_JVM, SYSTEM_TOOLS, EXTERNAL_TOOL}
	set_root_procedure_name (rp: STRING) is
		do
			root_procedure_name_memory := rp
		ensure
			root_procedure_name = rp
		end

feature {COMMAND_LINE_TOOLS}
	set_relax is
		do
			relax := True
		end

	set_precompile_header is
		require
			system_tools.precompile_header_supported
		do
			precompile_header := True
		end

	set_default_level is
		do
			if default_assertion_level = level_not_computed then
				default_assertion_level := level_all
			end
			set_highest_encountered_level(default_assertion_level)
			set_default_c_mode
		end

	set_boost is
		do
			default_assertion_level := level_boost
		end

	set_no_check is
		do
			default_assertion_level := level_no
		end

	set_require_check is
		do
			default_assertion_level := level_require
		end

	set_ensure_check is
		do
			default_assertion_level := level_ensure
		end

	set_invariant_check is
		do
			default_assertion_level := level_invariant
		end

	set_loop_check is
		do
			default_assertion_level := level_loop
		end

	set_all_check is
		do
			default_assertion_level := level_all
		end

	set_debug_check is
		do
			error_handler.append(once "-debug_check is deprecated. Please use -debug and another -*_check assertion level. This defaults to -all_check.")
			error_handler.print_as_warning
			set_all_check
			set_command_line_debug
		end

	set_command_line_debug is
		do
			default_debug_keys := {FAST_ARRAY[STRING] <<fz_yes>>}
		end

	set_flat_check is
		do
			flat_check := True
		end

	command_line_parsed (command_name: STRING) is
			-- Should be called the end of command line argument parsing
			-- (i.e. only in command line mode) to check among other things
			-- that the root class was actually given as argument.
		require
			file_path = Void
		do
			if root_class_name = Void then
				echo.w_put_string(command_name)
				echo.w_put_string(": error: No <Root-Class> in command line.%N")
				die_with_code(exit_failure_code)
			end
			if sedb and then boost then
				echo.w_put_string(command_name)
				echo.w_put_string(": cannot use -sedb with -boost flag.%N")
				die_with_code(exit_failure_code)
			end
			if not is_launcher then
				system_tools.read_loadpath_files
			end
			get_started
		ensure
			default_assertion_level /= level_not_computed
		end

	set_default_trace is
		do
			default_trace := True
			sedb_flag := True
		ensure
			default_trace = True
		end

	set_executable_name (name: STRING) is
		do
			executable_name := name
		ensure
			executable_name = name
		end

	set_clean (flag: BOOLEAN) is
		do
			clean := flag
		end

	set_safety_check is
		do
			safety_check := True
		end

	set_manifest_string_trace is
		do
			manifest_string_trace := True
		end

	set_high_memory_compiler is
		do
			high_memory_compiler := True
		end

	set_profile is
		do
			profile := True
		end

	set_jar is
		do
			jar := True
		end

	set_run is
		do
			run := True
		end

	set_use_jar(new_jar : like use_jar) is
		require
			new_jar /= Void
		do
			use_jar := new_jar
			set_jar
		end
	
	set_use_jvm(new_jvm : like use_jvm) is
		require
			new_jvm /= Void
		do
			use_jvm := new_jvm
			set_run
		end

	set_java_compiler(jc : like java_compiler) is
		require
			jc /= Void
		do
			java_compiler := jc
		end

	set_ss(new_ss: like ss) is
		require
			new_ss /= Void
		do
			ss := new_ss
			set_run
		end
	
	set_mx(new_mx: like mx) is
		require
			new_mx /= Void
		do
			mx := new_mx
			set_run
		end

	set_ms(new_ms: like ms) is
		require
			new_ms /= Void
		do
			ms := new_ms
			set_run
		end

	set_classpath(cp: like classpath) is
		require
			cp /= Void
		do
			classpath := cp
			set_run
		end

	set_clean_classes is
		do
			clean_classes := True
			set_run
		end

	set_hard_clean is
		do
			hard_clean := True
		end

feature {}
	best_cluster_of (origin: CLASSES; class_name: CLASS_NAME; report_error: BOOLEAN; skip: CLASSES): CLUSTER is
		local
			c: like clusters_
			d: like distances_
		do
			c := clusters_
			d := distances_
			c.clear_count
			d.clear_count
			origin.clusters_of(class_name, report_error, skip, c, d, 0)
			inspect
				c.count
			when 0 then
				-- no class found
			when 1 then
				-- cool, only one class found!
				Result := c.first
			else
				if d.item(d.lower+1) > d.first then
					-- OK, the second one is strictly further; keep the closest
					Result := c.first
				elseif report_error then
					check
						d.item(d.lower+1) = d.first
					end
					error_handler.append(once "More than one class in the system is named ")
					error_handler.append(class_name.to_string)
					error_handler.append(once ". The first one is in the cluster %"")
					error_handler.append(c.first.name)
					error_handler.append(once "%" while the second is in the cluster %"")
					error_handler.append(c.item(c.lower+1).name)
					error_handler.append(once "%" (and there may be more). Search started from ")
					error_handler.append(origin.to_string)
					error_handler.append(once ".%N")
					error_handler.print_as_warning
				end
			end
		end

	clusters_: FAST_ARRAY[CLUSTER] is
		once
			create Result.make(0)
		end

	distances_: FAST_ARRAY[INTEGER] is
		once
			create Result.make(0)
		end

feature {SMART_EIFFEL} -- Class loading
	cluster_named (cluster_name: STRING): CLUSTER is
		require
			not cluster_name.is_empty
			string_aliaser.registered_one(cluster_name)
		do
			Result := universe.cluster_named(cluster_name)
		ensure
			Result /= Void implies Result.name.is_equal(cluster_name)
		end

	cluster_of (class_name: CLASS_NAME; report_error: BOOLEAN): CLUSTER is
		require
			class_name /= Void
		local
			origin, last_origin: CLUSTERS; ct: CLASS_TEXT
		do
			-- The principle is the following: let's say we try to find the class C, whose `class_name' is
			-- given. Usually that `class_name' is written in a class A, somewhere belonging to some cluster. We
			-- try to find C in a cluster which is closest to A's cluster. It means:
			-- * First, look at subclusters of A's cluster. The less deep, the better.
			-- * If not found, try to find C in the closest cluster of the parent cluster A's cluster; go up the
			-- tree if not found, recursively.
			--
			-- Note that we optimize a bit to avoid scanning again and again a growing subtree.
			-- CLASSES.clusters_of knows how to skip a subtree, and in this feature we use `last_origin' for the
			-- same effect.

			ct := class_name.start_position.class_text
			if ct = Void then
				Result := best_cluster_of(universe, class_name, report_error, Void)
			else
				-- OK, the class name is written in some other class.
				-- We try to find the closest class to that one in steps getting gradually up the tree.
				check
					ct.cluster /= Void
				end
				origin := ct.cluster.tree
				from
				until
					Result /= Void or else origin = Void
				loop
					Result := best_cluster_of(origin, class_name, report_error, last_origin)
					last_origin := origin
					origin := origin.parent
				end
			end

			if Result = Void and then report_error then
				error_handler.add_position(class_name.start_position)
				error_handler.append("Cannot find the class %"")
				error_handler.append(class_name.to_string)
				error_handler.append("%" in any known cluster.")
				error_handler.print_as_fatal_error
			end
		ensure
			report_error implies Result /= Void
		end

	all_class_texts (class_name: CLASS_NAME): FAST_ARRAY[CLASS_TEXT] is
		local
			i: INTEGER; cluster: CLUSTER
			c: like clusters_; d: like distances_
		do
			c := clusters_
			d := distances_
			c.clear_count
			d.clear_count
			universe.clusters_of(class_name, False, Void, c, d, 0)
			create Result.with_capacity(c.count)
			from
				i := c.lower
			until
				i > c.upper
			loop
				cluster := c.item(i)
				Result.add_last(cluster.class_text(class_name, False, True))
				i := i + 1
			end
		end

	class_text (class_name: CLASS_NAME; report_error, load: BOOLEAN; cluster: CLUSTER): CLASS_TEXT is
			-- The `load' flag must be True if we want to load the class if it is not already loaded. If False,
			-- we don't load the class and the result may be Void.
			-- If `cluster' is not Void it overrides the information in `class_name'.
		require
			class_name /= Void
		do
			Result := find_class_text(class_name, report_error, False, cluster)
			if Result = Void and then load then
				Result := find_class_text(class_name, report_error, True, cluster)
				if Result = Void and then report_error then
					if class_name.is_tuple_related then
						smart_eiffel.tuple_class_not_found_fatal_error(class_name)
					else
						error_handler.add_position(class_name.start_position)
						error_handler.append("Unable to load class %"")
						error_handler.append(class_name.to_string)
						error_handler.append("%".")
						error_handler.print_as_fatal_error
					end
				end
			end
		ensure
			report_error implies Result /= Void
		end

	remove (a_class_text: CLASS_TEXT) is
		do
			a_class_text.cluster.remove(a_class_text)
		end

	class_text_count: INTEGER is
		do
			Result := universe.loaded_class_count
		end

	generic_formal_arguments_check is
		do
			universe.generic_formal_arguments_check
		end

feature {}
	find_class_text (class_name: CLASS_NAME; report_error, load: BOOLEAN; a_cluster: CLUSTER): CLASS_TEXT is
		require
			class_name /= Void
		local
			ct: CLASS_TEXT; cluster: CLUSTER
			pov: DICTIONARY[CLASS_TEXT, HASHED_STRING]
		do
			-- With some result caching depending on the "pov" (Point Of View: client cluster)

			if a_cluster /= Void then
				cluster := a_cluster
			else
				ct := class_name.start_position.class_text
				if ct /= Void then
					cluster := ct.cluster
				end
			end
			if cluster /= Void then
				pov := pov_classes.reference_at(cluster.name)
				if pov /= Void then
					Result := pov.reference_at(class_name.hashed_name)
				else
					create {HASHED_DICTIONARY[CLASS_TEXT, HASHED_STRING]} pov.make
					pov_classes.put(pov, cluster.name)
				end
			else
				pov := no_pov_classes
				echo.put_string(once "*** Looking for ")
				echo.put_string(class_name.to_string)
				echo.put_string(once " without point of view!%N")
				Result := pov.reference_at(class_name.hashed_name)
			end
			if Result = Void then
				cluster := cluster_of(class_name, report_error)
				if cluster /= Void then
					Result := cluster.class_text(class_name, report_error, load)
					if Result /= Void then
						pov.put(Result, class_name.hashed_name) --|*** ???? add should work instead of put!!
					end
				end
			end
		end

	no_pov_classes: DICTIONARY[CLASS_TEXT, HASHED_STRING] is
		once
			create {HASHED_DICTIONARY[CLASS_TEXT, HASHED_STRING]} Result.make
		end

	pov_classes: DICTIONARY[DICTIONARY[CLASS_TEXT, HASHED_STRING], STRING] is
		once
			create {HASHED_DICTIONARY[DICTIONARY[CLASS_TEXT, HASHED_STRING], STRING]} Result.make
		end

feature {ANY}
	has (class_name: HASHED_STRING): BOOLEAN is
		do
			Result := universe.has_class(class_name)
		end

feature {SMART_EIFFEL}
	parse_include is
			-- Look for some class(es) to be loaded first because of some "include" option.
		do
			if file_path /= Void then
				universe.parse_include
			end
		end

feature {ACE_CHECK}
	pretty_in (txt: STRING) is
			-- Performs the `ace_check' and also prepare in `txt' a pretty version of the Ace file as 
			-- it is memorized (can be also used to pretty one's ACE file).
		require
			file_path /= Void
		do
			txt.append("system ")
			txt.append(executable_name)
			txt.append("%Nroot ")
			txt.append(root_class_name.to_string)
			if root_procedure_name_memory /= Void then
				txt.append(": %"")
				txt.append(root_procedure_name_memory)
				txt.extend('%"')
			end
			txt.append("%Ndefault%N	  assertion (")
			txt.append(level_name(default_assertion_level))
			txt.append(")%N	debug (")
--|*** txt.append(default_debug_key)
			txt.append(")%N")
			if default_trace then
				txt.append("	trace (yes)%N")
			else
				txt.append("	trace (no)%N")
			end
			if gc_handler.is_off then
				txt.append("	collect (no)%N")
			else
				txt.append("	collect (yes)%N")
			end
			if eiffel_parser.no_style_warning then
				txt.append("	no_style_warning (yes)%N")
			else
				txt.append("	no_style_warning (no)%N")
			end
			if error_handler.no_warning then
				txt.append("	no_warning (yes)%N")
			else
				txt.append("	no_warning (no)%N")
			end
			if echo.is_verbose then
				txt.append("	verbose (yes)%N")
			else
				txt.append("	verbose (no)%N")
			end
			if manifest_string_trace then
				txt.append("	manifest_string_trace (yes)%N")
			else
				txt.append("	manifest_string_trace (no)%N")
			end
			if high_memory_compiler then
				txt.append("	high_memory_compiler (yes)%N")
			else
				txt.append("	high_memory_compiler (no)%N")
			end
			if profile then
				txt.append("	profile (yes)%N")
			else
				txt.append("	profile (no)%N")
			end
			if safety_check then
				txt.append("	safety_check (yes)%N")
			else
				txt.append("	safety_check (no)%N")
			end
			txt.append("cluster%N")
			universe.pretty_in(txt)
			txt.append("external%N")
			if not system_tools.external_object_files.is_empty then
				txt.append("	external_object_files: %"")
				txt.append(system_tools.external_object_files)
				txt.append("%"%N")
			end
			if not system_tools.external_c_files.is_empty then
				txt.append("	external_c_files: %"")
				txt.append(system_tools.external_c_files)
				txt.append("%"%N")
			end
			if not system_tools.external_c_plus_plus_files.is_empty then
				txt.append("	external_c_plus_plus_files: %"")
				txt.append(system_tools.external_c_plus_plus_files)
				txt.append("%"%N")
			end
			--|*** cecil_pool.pretty_ace_in(txt)
			--|*** to continue...
			txt.append("generate%N")
			if system_tools.no_strip then
				txt.append("	no_strip (yes)%N")
			else
				txt.append("	no_strip (no)%N")
			end
			if no_split then
				txt.append("	no_split (yes)%N")
			else
				txt.append("	no_split (no)%N")
			end
			if clean then
				txt.append("	clean (yes)%N")
			else
				txt.append("	clean (no)%N")
			end
			--|*** to continue...
			txt.append("end%N")
		end

feature {SYSTEM_TOOLS, CLUSTER}
	view_in (msg: STRING) is
			-- Append in `msg' a viewable version of the `cluster_list' as well as
			-- some other informations to help the user to fix the problem.
		require
			msg /= Void
		local
			sed: STRING
		do
			check
			-- Because the `pretty' command do not accept class names as arguments.
				not smart_eiffel.pretty_flag
			end
			if file_path = Void then
				msg.append("%N%
				 %You are in command line mode (i.e. no ACE file is used).%N%
				 %The load path can be changed using a file called%N%
				 %loadpath.se in the current working directory.%N%
				 %Usually, this loadpath.se file is a simple list of directories.%N%
				 %It is also possible to use system variables or include files. See%N%
				 %the documentation for the finder command for more information.%N")
			else
				msg.append("%N%
				 %Eiffel class file searching is being done according to the ACE %
				 %file %"")
				msg.append(file_path)
				msg.append("%".%N")
			end
			msg.append("Files are being searched for in the following list of clusters (")
			cluster_count.append_in(msg)
			msg.append(" items):%N")
			universe.view_in(msg)
			system_tools.system_name_in(msg)
			msg.append("The value of the environment variable %"SmartEiffel%" is:%N%"")
			sed := echo.getenv(fz_smarteiffel, Void)
			if sed /= Void then
				msg.append(sed)
			end
			msg.append("%".%N")
			if file_path /= Void then
				msg.append("Please, also note that you can use the %"ace_check%" command%N%
				 %to view all informations stored into your ACE file.%N")
			end
		end

feature {ANY}
	cluster_count: INTEGER is
		do
			Result := universe.cluster_count
		end

	cluster_at (i: INTEGER): CLUSTER is
		do
			Result := universe.cluster_at(i)
		end

feature {}
	get_started is
			-- Should be called to set some default values at the end of
			-- command line parsing or at the end of the ACE file parsing.
		do
			set_default_level
			system_tools.get_started
			if default_debug_keys = Void then
				default_debug_keys := {FAST_ARRAY[STRING] <<fz_no>>}
			end
			universe.get_started(highest_encountered_level)
			if echo.is_verbose and then not is_launcher then
				echo.put_string(once "Cluster tree:%N")
				universe.show(0)
			end
		end

	highest_encountered_level: INTEGER
			-- The highest encountered one in the whole ACE file. (Used to compute the C mode to use.)

feature {CLUSTER}
	default_trace: BOOLEAN
			-- Code generated with the -sedb flag way is the default for all
			-- classes of the ACE file (see also `sedb').

	default_assertion_level: INTEGER
			-- The default assertion level mangled using constants and tools from class 
			-- ASSERTION_LEVEL_NUMBERING.
			-- This value memorize the command line flag such as (-boost, -no_check, 
			-- -require_check, ...). When the ACE file is used, this value memorize the 
			-- information after "assertion" tag of the default section.

	default_debug (e_debug: E_DEBUG): BOOLEAN is
		require
			e_debug /= Void
		do
			check
				default_debug_keys /= Void
			end
			Result := match_debug_keys(e_debug, default_debug_keys)
		end

feature {}
	set_root_class_name (rcn: HASHED_STRING) is
		do
			root_class_names.add_last(rcn)
		ensure
			root_class_name = rcn
		end

feature {SYSTEM_TOOLS}
	add_loadpath (name, path: STRING; distance: INTEGER) is
		require
			name /= Void
			path /= Void
			string_aliaser.registered_one(path)
			;(create {FILE_TOOLS}).file_exists(path)
			distance > 0
		local
			n: like name; p: like path
		do
			if string_aliaser.registered_one(name) then
				n := name
			else
				n := string_aliaser.string(name)
			end
			if universe.is_system_path(path) then
				p := string_aliaser.string(universe.system_path_to_classes_path(path))
			else
				p := path
			end
			universe.add_entry(distance, path, p, n)
		end

feature {C_PRETTY_PRINTER, COMPILE_TO_C, STRING_COMMAND_LINE, LIVE_TYPE, INSTALL, SYSTEM_TOOLS}
	no_split: BOOLEAN
			-- True when the "-no_split" flag or equivalent is in use.

	set_no_split (flag: BOOLEAN) is
		do
			no_split := flag
		end

	set_no_main is
		do
			no_main := True
		ensure
			no_main
		end

feature {ANY}
	for_all (action: PROCEDURE[TUPLE[CLASS_TEXT]]) is
		do
			universe.for_all(action)
		end

	for_all_clusters (action: PROCEDURE[TUPLE[CLUSTER]]) is
		do
			universe.for_all_clusters(action)
		end

	loaded_class_count: INTEGER is
		do
			Result := universe.loaded_class_count
		end

feature {CLUSTER}
	new_cluster (a_cluster: CLUSTER) is
		do
			new_clusters.add_last(a_cluster)
		end

feature {}
	new_clusters: FAST_ARRAY[CLUSTER] is
		once
			create Result.make(0)
		end

	update_last_manifest_string (p: POSITION; once_flag, unicode_flag: BOOLEAN; string, source_view: STRING) is
		do
		end

	may_expand_var is
		local
			envar, value: STRING; c, l: INTEGER
		do
			from
				l := line
				c := column
				next_char
				if cc = '{' then
					next_char
					create envar.make(12)
				else
					buffer.extend('$')
					go_back_at(l, c)
				end
			until
				envar = Void
			loop
				inspect
					cc
				when '}' then
					value := echo.getenv(envar, file_path)
					if value /= Void then
						buffer.append(value)
					end
					envar := Void
				when end_of_text then
					error_handler.add_position(pos(l, c))
					error_handler.append("Bad Environment variable.%N%
												%(Closing %"}%" not found.)")
					error_handler.print_as_fatal_error
				else
					envar.extend(cc)
					next_char
				end
			end
		end

	a_identifier: STRING is
			-- Analyse an Eiffel identifier.
		local
			stop: BOOLEAN
		do
			Result := buffer
			if cc = '%"' then
				error_handler.add_position(current_position)
				error_handler.append(once "Quoted identifiers are deprecated. Please remove quotes here.")
				error_handler.print_as_warning
				--|*** Inefficient but since this will eventually go away... <FM-21/08/2004>
				Result := a_string
			else
				from
					Result.clear_count
				until
					stop
				loop
					inspect
						cc
					when 'a' .. 'z', 'A' .. 'Z', '0' .. '9', '_' then
						Result.extend(cc)
						next_char
					else
						stop := True
					end
				end
				if Result.is_empty then
					error_handler.add_position(current_position)
					error_handler.append("Non empty unquoted name expected here.")
					error_handler.print_as_fatal_error
				end
				skip_comments
			end
			Result := string_aliaser.string(Result)
			check
				Result /= buffer
			end
		ensure
			Result /= Void
			string_aliaser.registered_one(Result)
		end

	a_string: STRING is
			-- Analyse a notation like the one for manifest strings, 
			-- returning only the content of the manifest string and 
			-- expanding system environment variables (syntax: ${name})
		do
			Result := buffer
			if not a_manifest_string(True) then
				error_handler.add_position(current_position)
				error_handler.append(once "Unquoted filenames are deprecated. Please add quotes here.")
				error_handler.print_as_warning
				--|*** Inefficient but since this will eventually go away... <FM-21/08/2004>
				Result := a_identifier
				check
					Result.is_equal(buffer)
				end
				Result := buffer
			elseif Result.is_empty then
				error_handler.add_position(current_position)
				error_handler.append("Non empty quoted string expected here.")
				error_handler.print_as_fatal_error
			end
		ensure
			Result = buffer
			not Result.is_empty
		end

	a_cluster_mark is
		do
			-- At time being, this is just syntactic sugar because
			-- clusters are not handled :-(
			if skip1('(') then
				if a_identifier /= Void then
				end
				if skip1(')') then
				end
			end
		end

	a_system_level_defaults: BOOLEAN is
			-- The `Result' is used to delay the `echo.is_verbose' value setting
		local
			stop: BOOLEAN
			dummy: like default_assertion_level
		do
			from
			until
				stop
			loop
				if cc = end_of_text then
					stop := True
				elseif a_cluster_clause_list then
					stop := True
				elseif a_keyword(fz_external) then
					a_external
					stop := True
				elseif a_keyword(fz_generate) then
					a_generate
					stop := True
				elseif skip1(';') then
				elseif a_keyword(fz_assertion) then
					if not smart_eiffel.short_or_class_check_flag then
						default_assertion_level := a_assertion_level
					else
						dummy := a_assertion_level
					end
				elseif a_keyword(fz_assertion_flat_check) then
					if a_yes_no_all then
						set_flat_check
					end
				elseif a_keyword(fz_debug) then
					add_default_debug_key(a_debug_key)
				elseif a_keyword(once "collect") then
					if not a_yes_no_all then
						gc_handler.no_gc
					end
				elseif a_keyword(fz_case_insensitive) then
					error_handler.add_position(current_position)
					error_handler.append("The %"case_insensitive%" option is no longer supported.")
					error_handler.print_as_warning
					if a_yes_no_all then
					end
				elseif a_keyword(fz_no_style_warning) then
					if a_yes_no_all then
						eiffel_parser.set_no_style_warning
					end
				elseif a_keyword(fz_no_warning) then
					if a_yes_no_all then
						error_handler.set_no_warning
					end
				elseif a_keyword(fz_trace) then
					if a_yes_no_all then
						set_default_trace
					end
				elseif a_keyword(fz_relax) then
					if a_yes_no_all then
						set_relax
					end
				elseif a_keyword(fz_safety_check) then
					if a_yes_no_all then
						set_safety_check
					end
				elseif a_keyword(fz_verbose) then
					Result := a_yes_no_all
				elseif a_keyword(fz_manifest_string_trace) then
					manifest_string_trace := a_yes_no_all
				elseif a_keyword(fz_high_memory_compiler) then
					high_memory_compiler := a_yes_no_all
				elseif a_keyword(fz_profile) then
					profile := a_yes_no_all
				elseif a_keyword(fz_relax) then
					relax := a_yes_no_all
				else
					stop := True
				end
			end
		end

	a_cluster_clause_list: BOOLEAN is
			--++ clusters --> "cluster" { cluster_clause ";" ...}
		local
			stop: BOOLEAN
		do
			if a_keyword(fz_cluster) then
				Result := True
				from
				until
					stop
				loop
					if a_cluster_clause then
						if skip1(';') then
						end
						if a_keyword(fz_end) then
						end
					else
						stop := True
					end
				end
			end
		end

	a_cluster_clause: BOOLEAN is
			--++ cluster_clause --> [cluster_tag]
			--++							directory_name
			--++							directory_name/loadpath.se
			--++							cluster_properties
		local
			cluster_name: STRING; path: STRING
		do
			if cc = end_of_text then
			elseif a_keyword(fz_external) then
				a_external
			elseif a_keyword(fz_generate) then
				a_generate
			elseif a_keyword(fz_end) then
			elseif cc = '%"' then
				Result := True
				path := a_string
				cluster_name := string_aliaser.string(path)
				system_tools.environment_variable_substitution(file_path, path)
				if universe.is_system_path(path) then
					path := universe.system_path_to_classes_path(path)
				end
				new_universe_entry(1, string_aliaser.string(path), cluster_name)
				a_cluster_properties
			elseif cc.is_letter then
				Result := True
				cluster_name := a_identifier
				if skip1(':') then
					if cc = '%"' then
						path := a_string
						system_tools.environment_variable_substitution(file_path, path)
						if universe.is_system_path(path) then
							path := universe.system_path_to_classes_path(path)
						end
						new_universe_entry(1, string_aliaser.string(path), cluster_name)
					else
						error_handler.add_position(current_position)
						error_handler.append("Cluster path expected after cluster name.")
						error_handler.print_as_fatal_error
					end
				else
					path := cluster_name
					if universe.is_system_path(path) then
						path := universe.system_path_to_classes_path(path)
					end
					new_universe_entry(1, string_aliaser.string(path), cluster_name)
				end
				a_cluster_properties
			else
				fatal_error_in(fz_cluster)
			end
		end

	new_universe_entry (distance: INTEGER; path, name: STRING) is
		require
			distance > 0
			path /= Void
			string_aliaser.registered_one(path)
			name /= Void
		do
			if universe.is_system_path(path) then
				universe.add_entry(distance, path, path, name)
			else
				universe.add_entry(distance, Void, path, name)
			end
		end

	a_cluster_properties is
			--++ cluster_properties -->
			--++	  [use]
			--++	  [include]
			--++	  [exclude]
			--++	  [name_adaptation]
			--++	  [default]
			--++	  [options]
			--++	  [visible]
		local
			stop, yna: BOOLEAN; i, l: INTEGER; n: STRING
		do
			if a_keyword(fz_use) then
				error_handler.add_position(current_position)
				error_handler.append("The %"use%" clause is not yet implemented.")
				error_handler.print_as_fatal_error
			end
			if a_keyword(fz_include) then
				from
				until
					cc /= '%"'
				loop
					n := string_aliaser.string(a_string)
					from
						i := new_clusters.upper
					until
						i < new_clusters.lower
					loop
						new_clusters.item(i).include_add_last(n)
						i := i - 1
					end
					if skip1(';') then
					end
				end
			end
			if a_keyword(fz_exclude) then
				from
				until
					cc /= '%"'
				loop
					n := string_aliaser.string(a_string)
					from
						i := new_clusters.upper
					until
						i < new_clusters.lower
					loop
						new_clusters.item(i).exclude_add_last(n)
						i := i - 1
					end
					if skip1(';') then
					end
				end
			end
			if a_keyword(fz_adapt) then
				error_handler.add_position(current_position)
				error_handler.append("The %"adapt%" clause is not yet implemented.")
				error_handler.print_as_fatal_error
			end
			if a_keyword(fz_default) then
				from
				until
					stop
				loop
					if a_keyword(fz_assertion) then
						l := a_assertion_level
						from
							i := new_clusters.upper
						until
							i < new_clusters.lower
						loop
							new_clusters.item(i).set_default_assertion_level(l)
							i := i - 1
						end
					elseif a_keyword(fz_trace) then
						yna := a_yes_no_all
						sedb_flag := sedb_flag or else yna
						from
							i := new_clusters.upper
						until
							i < new_clusters.lower
						loop
							new_clusters.item(i).set_default_trace(yna)
							i := i - 1
						end
					elseif a_keyword(fz_debug) then
						n := a_debug_key
						from
							i := new_clusters.upper
						until
							i < new_clusters.lower
						loop
							new_clusters.item(i).add_default_debug_key(n)
							i := i - 1
						end
					else
						stop := True
					end
					if skip1(';') then
					end
				end
			end
			if a_keyword(fz_option) then
				from
				until
					not a_option_in_cluster_properties
				loop
				end
			end
			new_clusters.clear_count
		end

	a_option_in_cluster_properties: BOOLEAN is
			-- Possible option after the keyword "option" in one
			-- `cluster' description.
		local
			level, i: INTEGER; class_name: CLASS_NAME; debug_key: STRING
		do
			if a_keyword(fz_assertion) then
				Result := True
				level := a_assertion_level
				if not skip1(':') then
					error_handler.add_position(current_position)
					error_handler.append(em29)
					error_handler.print_as_warning
				end
				from
				until
					not a_class_name
				loop
					class_name := token_buffer.to_class_name
					from
						i := new_clusters.upper
					until
						i < new_clusters.lower
					loop
						new_clusters.item(i).set_option_assertion_level(class_name, level)
						i := i - 1
					end
					if skip1(',') then
					end
				end
			elseif a_keyword(fz_trace) then
				sedb_flag := True
				Result := True
				if not skip1(':') then
					error_handler.add_position(current_position)
					error_handler.append(em29)
					error_handler.print_as_warning
				end
				from
				until
					not a_class_name
				loop
					class_name := token_buffer.to_class_name
					from
						i := new_clusters.upper
					until
						i < new_clusters.lower
					loop
						new_clusters.item(i).add_option_trace(class_name)
						i := i - 1
					end
					if skip1(',') then
					end
				end
			elseif a_keyword(fz_debug) then
				Result := True
				debug_key := a_debug_key
				if not skip1(':') then
					error_handler.add_position(current_position)
					error_handler.append(em29)
					error_handler.print_as_warning
				end
				from
				until
					not a_class_name
				loop
					class_name := token_buffer.to_class_name
					from
						i := new_clusters.upper
					until
						i < new_clusters.lower
					loop
						new_clusters.item(i).add_option_debug_key(class_name, debug_key)
						i := i - 1
					end
					if skip1(',') then
					end
				end
			end
		end

	a_class_name: BOOLEAN is
			-- A single class name strictly written using only uppercase letter
			-- in order to avoid any possible ambiguities. When the `Result' is
			-- True, the corresponding class name is stored as usual in the
			-- `token_buffer' buffer.
		do
			if cc.is_upper then
				from
					token_buffer.reset(line, column)
					token_buffer.extend(cc)
				until
					Result
				loop
					next_char
					inspect
						cc
					when 'A' .. 'Z', '0' .. '9', '_' then
						token_buffer.extend(cc)
					else
						Result := True
					end
				end
				skip_comments
			end
		end

	a_external is
		local
			stop: BOOLEAN
		do
			from
			until
				stop
			loop
				if cc = end_of_text then
					stop := True
				elseif a_keyword(fz_generate) then
					a_generate
					stop := True
				elseif skip1(';') then
				elseif a_keyword(once "external_c_files") then
					if skip1(':') then
					end
					system_tools.external_c_files.copy(string_aliaser.string(a_string))
				elseif a_keyword(once "external_header_path") then
					if skip1(':') then
					end
					system_tools.set_external_header_path(string_aliaser.string(a_string))
				elseif a_keyword(once "external_c_plus_plus_files") then
					if skip1(':') then
					end
					system_tools.external_c_plus_plus_files.copy(string_aliaser.string(a_string))
				elseif a_keyword(once "external_object_files") then
					if skip1(':') then
					end
					system_tools.external_object_files.copy(string_aliaser.string(a_string))
				elseif a_keyword(fz_cecil) then
					if cecil_pool /= Void then
						cecil_pool.add_cecil_file(a_cecil_file)
					elseif a_cecil_file = Void then
						-- Skip the cecil file's name
					end
				elseif a_keyword(once "external_lib_path") then
					if skip1(':') then
					end
					system_tools.set_ace_external_lib_path(string_aliaser.string(a_string))
				elseif a_keyword(once "external_lib") then
					if skip1(':') then
					end
					system_tools.set_ace_external_lib(string_aliaser.string(a_string))
				else
					stop := True
				end
			end
		end

	a_generate is
		local
			stop, value: BOOLEAN
		do
			from
			until
				stop
			loop
				if cc = end_of_text then
					stop := True
				elseif skip1(';') then
				elseif a_keyword(fz_cc) then
					if skip1(':') then
					end
					system_tools.set_c_compiler(string_aliaser.string(a_string))
				elseif a_keyword(fz_gc_info) then
					if a_yes_no_all then
						gc_handler.set_info_flag
					end
				elseif a_keyword(fz_no_strip) then
					if a_yes_no_all then
						system_tools.set_no_strip
					end
				elseif a_keyword(fz_no_main) then
					if a_yes_no_all then
						set_no_main
					end
				elseif a_keyword(fz_no_split) then
					set_no_split(a_yes_no_all)
				elseif a_keyword(fz_clean) then
					value := a_yes_no_all -- Priority to the command line -clean flag:
					if not clean then
						set_clean(value)
					end

					--|*** 
				elseif a_keyword(once "wedit") then
					error_handler.add_position(current_position)
					error_handler.append("Unused obsolete flag -wedit / option wedit.")
					error_handler.print_as_warning
					value := a_yes_no_all
					--|*** wedit part to be removed one day ***
					--|*** Warning added 1st feb 2005 *** Dom ***


				elseif a_keyword(once "c_compiler_options") then
					if skip1(':') then
					end
					system_tools.set_ace_compiler_options(string_aliaser.string(a_string))
				elseif a_keyword(once "linker_options") then
					if skip1(':') then
					end
					system_tools.set_ace_linker_options(string_aliaser.string(a_string))
				elseif a_keyword(once "c_mode") then
					if skip1(':') then
					end
					system_tools.set_alternate_c_mode(a_identifier)
				else
					stop := True
				end
			end
		end

	a_yes_no_all: BOOLEAN is
			-- Return True for a notation like "(yes)" or for a notation
			-- like "(all)". Return False for a notation like "(no").
		do
			if not skip1('(') then
				error_handler.add_position(current_position)
				error_handler.append(em27)
				error_handler.print_as_warning
			end
			if a_keyword(fz_no) then
			elseif a_keyword(fz_all) or else a_keyword(fz_yes) then
				Result := True
			else
				error_handler.add_position(current_position)
				error_handler.append("At this point in the ACE file, you are supposed to %
											%say %"yes%", %"no%", or %"all%".")
				error_handler.print_as_fatal_error
			end
			if not skip1(')') then
				error_handler.add_position(current_position)
				error_handler.append(em28)
				error_handler.print_as_warning
			end
		end

	a_assertion_level: INTEGER is
		do
			if not skip1('(') then
				error_handler.add_position(current_position)
				error_handler.append(em27)
				error_handler.print_as_warning
			end
			if a_keyword(fz_boost) then
				Result := level_boost
			elseif a_keyword(fz_no) then
				Result := level_no
			elseif a_keyword(fz_require) then
				Result := level_require
			elseif a_keyword(fz_ensure) then
				Result := level_ensure
			elseif a_keyword(fz_invariant) then
				Result := level_invariant
			elseif a_keyword(fz_loop) then
				Result := level_loop
			elseif a_keyword(fz_check) or else a_keyword(fz_all) or else a_keyword(fz_yes) then
				Result := level_all
			elseif a_keyword(fz_debug) then
				error_handler.add_position(current_position)
				error_handler.append(once "The %"debug%" assertion level is deprecated. Please use %"all%" and debug statements.")
				error_handler.print_as_warning
				Result := level_all
			else
				error_handler.add_position(current_position)
				error_handler.append("Unknown assertion level tag.")
				error_handler.print_as_error
				error_handler.append("You have to fix the problem in your ACE file. Valid %
				%assertion level tags are: %"no%", %"require%", %"ensure%",%
				% %"invariant%", %"loop%", %"check%", %"all%", and %"debug%".")
				error_handler.print_as_fatal_error
			end
			if not skip1(')') then
				error_handler.add_position(current_position)
				error_handler.append(em28)
				error_handler.print_as_warning
			end
			set_highest_encountered_level(Result)
		ensure
			Result.in_range(level_boost, level_all)
		end

	a_debug_key: STRING is
			-- Return some acceptable notation for a debug key: "yes",
			-- "no" or some user defined key.
		do
			if not skip1('(') then
				error_handler.add_position(current_position)
				error_handler.append(em27)
				error_handler.print_as_warning
			end
			if a_keyword(fz_no) then
				Result := fz_no
			elseif a_keyword(fz_yes) or else a_keyword(fz_all) then
				set_highest_encountered_level(level_all)
				Result := fz_yes
			else
				set_highest_encountered_level(level_all)
				Result := string_aliaser.string(a_string)
			end
			if not skip1(')') then
				error_handler.add_position(current_position)
				error_handler.append(em28)
				error_handler.print_as_warning
			end
		end

	a_cecil_file: STRING is
			-- Return some acceptable notation for a cecil file
		do
			if not skip1('(') then
				error_handler.add_position(current_position)
				error_handler.append(em27)
				error_handler.print_as_warning
			end
			Result := a_string.twin
			if not skip1(')') then
				error_handler.add_position(current_position)
				error_handler.append(em28)
				error_handler.print_as_warning
			end
		end

	pos (l, c: INTEGER): POSITION is
		do
			Result.set_in_ace_file(l, c)
		end

	universe: UNIVERSE is
		local
			factory: CLASSES_TREE_FACTORY
		once
			Result := factory.universe
		end

	root_procedure_name_memory: STRING

	cluster_by_directory_path (path: STRING): CLUSTER is
		require
			not path.is_empty
		do
			Result := universe.cluster_by_directory_path(path)
		ensure
			Result /= Void implies Result.directory_path.is_equal(path)
		end

	has_cluster (c: CLUSTER): BOOLEAN is
		require
			c /= Void
		do
			Result := universe.has_cluster(c)
		end

	fatal_error_in (section_name: STRING) is
		do
			error_handler.add_position(current_position)
			error_handler.append("Error in the %"")
			error_handler.append(section_name)
			error_handler.append("%" section.")
			error_handler.print_as_fatal_error
		end

	set_highest_encountered_level (level: INTEGER) is
		do
			if level > highest_encountered_level then
				highest_encountered_level := level
			end
		ensure
			highest_encountered_level >= old highest_encountered_level
		end

	sedb_flag: BOOLEAN

	singleton_memory: ACE is
		once
			Result := Current
		end

invariant
	is_real_singleton: Current = singleton_memory
	file_path /= Void implies string_aliaser.registered_one(file_path)

end -- class ACE
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.		 - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
