-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class CECIL_ENTRY

inherit
	VISITABLE
insert
	GLOBALS

creation {CECIL_FILE}
	make

feature {ANY}
	c_name: STRING
			-- The external C name used to call the Eiffel code.

	type_mark: TYPE_MARK
			-- The one which is written in the -cecil file (i.e. its `start_position' is in the -cecil file).

	feature_name: FEATURE_NAME
			-- Of the feature to be called (also the one of the -cecil file).

	type: TYPE
			-- Alias of `type_mark.type'. Static `type' of the target of the feature to be called.

	feature_stamp: FEATURE_STAMP
			-- The corresponding one of `type' and `feature_name'.

	is_creation: BOOLEAN
			-- True if the function must call a creation procedure and return the created object.

	accept (visitor: CECIL_ENTRY_VISITOR) is
		do
			visitor.visit_cecil_entry(Current)
		end

feature {CECIL_FILE}
	get_started (cecil_file: CECIL_FILE) is
			-- Called only once to initialize `type' and `feature_stamp'.
		require
			cecil_file /= Void
			type = Void
			feature_stamp = Void
			may_report_an_error: error_handler.is_empty
		local
			dummy: BOOLEAN
			fs: FEATURE_STAMP
		do
			type := type_mark.type
			if not type.valid_feature_name(feature_name) then
				error_handler.add_position(type_mark.start_position)
				error_handler.add_position(feature_name.start_position)
				error_handler.append("Unknown feature name ")
				error_handler.append(feature_name.to_string)
				error_handler.append(" in type ")
				error_handler.append(type.name.to_string)
				error_handler.append(".")
				error_handler.print_as_error
				error_handler.append("Error while loading features of cecil path file %"")
				error_handler.append(cecil_file.path)
				error_handler.append("%".")
				error_handler.print_as_fatal_error
			end
			if is_creation then
				if type_mark.class_text.creation_clause_list = Void then
					fs := any_default_create_fs.resolve_static_binding_for(smart_eiffel.type_any, type)
					if not fs.name.is_equal(feature_name) then
						error_handler.append("Type ")
						error_handler.append(type.name.to_string)
						error_handler.append(" has no creation list. You must use the default creation method %
													%(named `default_create' in class ANY, or just omit the method name).")
						error_handler.add_position(type_mark.start_position)
						error_handler.add_position(feature_name.start_position)
						error_handler.print_as_fatal_error
					end
				else
					dummy := type_mark.class_text.has_creation_check(feature_name, feature_name.start_position)
				end
				check
					dummy
				end
			end
			feature_stamp := type.lookup(feature_name)
			on_echo(cecil_file)
		ensure
			type /= Void
			feature_stamp /= Void
			may_report_an_error: error_handler.is_empty
		end

	collect is
		local
			af: ANONYMOUS_FEATURE
		do
			smart_eiffel.collect_cecil_entry(type, feature_stamp, is_creation)
			af := feature_stamp.anonymous_feature(type)
			assignment_handler.from_external(type, af.arguments, af.result_type)
			if is_creation then
				if type.canonical_type_mark.class_text_name.to_string = as_typed_internals then
					error_handler.add_position(type_mark.start_position)
					error_handler.append("The type for a creation cannot be INTERNALS.%N")
					error_handler.print_as_fatal_error
				end
				if type.is_deferred then
					error_handler.add_position(type_mark.start_position)
					error_handler.append(once "The type for a crestion procedure cannot be deferred.%N")
					error_handler.print_as_fatal_error
				end
			end
		end

	on_echo (cecil_file: CECIL_FILE) is
		do
			if is_creation then
				echo.put_string(once "%Tcreate {")
			else
				echo.put_string(once "%T{")
			end
			echo.put_string(type.name.to_string)
			echo.put_string(once "}.")
			echo.put_string(feature_name.to_string)
			echo.put_string(once " (from -cecil %"")
			echo.put_string(cecil_file.path)
			echo.put_string(once "%").%N")
		end

feature {CECIL_POOL}
	run_feature: RUN_FEATURE is
		do
			Result := feature_stamp.run_feature_for(type)
		end

feature {}
	make (cn: like c_name; tm: like type_mark; fn: like feature_name; c: like is_creation) is
		require
			cn /= Void
			tm /= Void
			fn /= Void
		do
			c_name := cn
			type_mark := tm
			feature_name := fn
			is_creation := c
		ensure
			c_name = cn
			type_mark = tm
			feature_name = fn
			is_creation = c
		end

invariant
	c_name /= Void

	type_mark /= Void

	feature_name /= Void

end -- class CECIL_ENTRY
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
