-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class CODE
	--
	-- Common direct ancestor of EXPRESSION and INSTRUCTION.
	--

inherit
	VISITABLE

insert
	GLOBALS

feature {ANY}
	start_position: POSITION is
			-- Of the left-most character of `Current'.
		deferred
		ensure
			not Result.is_unknown
		end

	specialize_in (type: TYPE): like Current is
			-- Specialize `Current' knowing that the corresponding source code was written in `type'. The `Result' 
			-- still contains all the original source code information in order to be able to `specialize_in' again 
			-- the `Result' in another `type'. (See also `specialize_thru' and `has_been_specialized' as well.)
		require
			start_position.class_text = type.class_text
		deferred
		ensure
			Result.has_been_specialized -- All the original source code information is kept in the `Result':
			Result.generating_type = generating_type
		end

	has_been_specialized: BOOLEAN is
			-- Used both for `specialize_in' and `specialize_thru' to indicate that we have all FEATURE_STAMPs 
			-- ready for use and that FEATURE_NAMEs are no longer used (except for calls on non-current target 
			-- who are available after `specialize_2' -- step). Finally, also note that the purpose of this 
			-- feature is mostly to write contracts (see also the strange ensure assertion).
		deferred
		ensure
			assertion_check_only: Result
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
			-- Assuming that `Current' is already specialized in `parent_type', now specialize `Current' for the 
			-- `new_type' knowing that this source code is inherited thru `parent_edge'. (See also 
			-- `specialize_in' and `has_been_specialized' as well.)
		require
			has_been_specialized
			start_position.class_text /= new_type.class_text
			new_type.direct_thru_step(parent_type, parent_edge)
		deferred
		ensure
			has_been_specialized
			Result.has_been_specialized -- All the original source code information is kept in the `Result':
			Result.generating_type = generating_type
		end

	specialize_2 (type: TYPE): CODE is
			-- This `specialize_2' step is applyed after `specialize_in' / `specialize_thru'. The `specialize_2' 
			-- step has been added to continue specialization which can't be made at `specialize_in' / `specialize_thru' 
			-- time (because the expression TYPE computation may involve TYPEs who don't yet exist).
			-- At `specialize_2' step, all TYPEs used by `Current' expression are ready, so TYPE may be known for all
			-- sub-members (target, arguments) using `resolve_in', if we take care to call `specialize_2' on them 
			-- before calling `resolve_in'. So we can continue the specialization process and perform some validation 
			-- checks knowing the TYPE of other components.
			--
			-- After this `specialize_2' has been called on `Current', `resolve_in' can be used to know the
			-- `Current' expression result TYPE.
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
			may_report_an_error: error_handler.is_empty
		deferred
		ensure
			has_been_specialized
			Result.has_been_specialized
			may_report_an_error: error_handler.is_empty
		end

	specialize_2_without_checks (type: TYPE): like specialize_2 is
			-- Perform specialize_2 without checking the validity (export violations, ...)
			-- Useful for generated code
			--|*** At the moment, checks are only inhibited for CREATE_INSTRUCTIONs.
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		do
			Result := specialize_2(type)
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	side_effect_free (type: TYPE): BOOLEAN is
			-- True when the evaluation of `Current' has no side effect at all (i.e. no memory allocated, no
			-- modification in the _heap_ memory, no external code with possible side effect). Actually, when
			-- `side_effect_free' is True, `Current' can be evaluated more than once or not evaluated at all.
		require
			smart_eiffel.status.collecting_done
			type /= Void
		deferred
		end

	use_current (type: TYPE): BOOLEAN is
			-- Is Current used?
		require
			smart_eiffel.status.collecting_done
			type /= Void
		deferred
		end

	safety_check (type: TYPE) is
		require
			smart_eiffel.status.is_safety_checking
			type.live_type /= Void
		deferred
		end

	compile_to_c (type: TYPE) is
			-- For an INSTRUCTION, produce the corresponding C code. For an EXPRESSION the C code to access 
			-- the value (user's expanded are no longuer pointer).
		require
			smart_eiffel.is_ready
			type.live_type /= Void
			cpp.pending_c_function
		deferred
		end

	pretty (indent_level: INTEGER) is
		require
			smart_eiffel.pretty_flag implies indent_level >= 1
		deferred
		end

feature {INSTRUCTION, WHEN_CLAUSE}
	compile_to_c_with_internal_c_local_saving (type: TYPE) is
			-- Basically, performs `compile_to_c' plus the extra unlock of INTERNAL_C_LOCAL variables.
		require
			type /= Void
		local
			internal_c_local_stack_level: INTEGER
		do
			internal_c_local_stack_level := cpp.internal_c_local_stack_level
			compile_to_c(type)
			cpp.unlock_internal_c_local(internal_c_local_stack_level)
		end

feature {ANONYMOUS_FEATURE, CODE}
	simplify (type: TYPE): CODE is
			-- May return `Current', Void or a simplified version of `Current'.
		require
			type /= Void
			not smart_eiffel.pretty_flag
			not smart_eiffel.short_or_class_check_flag
			smart_eiffel.status.collecting_done
		deferred
		ensure
			(Result /= Current or Result = Void) implies (smart_eiffel.magic_count > old smart_eiffel.magic_count);
			(Result = Current) implies (smart_eiffel.magic_count = old smart_eiffel.magic_count)
		end

end -- class CODE
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
