-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class AGENT_CREATION
	--
	-- An Eiffel agent is a delayed call. Such an Eiffel object include a feature call with closed or open
	-- arguments (or closed or open target as well). An AGENT_CREATION object is the notation to create the
	-- agent (i.e. the feature is not launched yet, see also AGENT_EXPRESSION).
	--
	-- Examples of AGENT_CREATION:
	--          agent my_array.add_last(?)         -- The target is closed and the argument is open.
	--          agent {ARRAY[INTEGER]}.add_last(1) -- The target is open and the argument is closed.
	--          agent {ARRAY[INTEGER]}.add_last(?) -- Both the target and the argument are open.
	--          agent {ARRAY[INTEGER]}.add_last    -- A short-hand for the previous notation.
	--          agent my_array.put(?, 3)           -- Only the second argument is open.
	--          agent my_array.put(3, ?)           -- Only the first argument is open.
	--          agent {ARRAY[INTEGER]}.put         -- Target and all arguments are open.
	--

inherit
	EXPRESSION
		redefine is_equal
		end
	HASHABLE

creation {EIFFEL_PARSER}
	make

feature {ANY}
	start_position: POSITION
			-- Of the first letter of the mandatory "agent" keyword (obviously provided by the 
			-- `eiffel_parser').

	delayed_call: CALL
			-- Syntactically, the call to be delayed (it can become a procedure call as well).
	
	delayed_proc_call: PROC_CALL
			-- Non Void when `delayed_call' is finally a procedure call instead.
	
feature {ANY}
	is_current, is_implicit_current: BOOLEAN is False

	extra_bracket_flag: BOOLEAN is False

	hash_code: INTEGER is
		do
			Result := delayed_call.feature_name.hash_code
		end

	is_equal (other: like Current): BOOLEAN is
		do
			if Current = other then
				Result := True
			elseif delayed_proc_call /= Void then
				Result := delayed_proc_call = other.delayed_proc_call
			else
				Result := delayed_call = other.delayed_call
			end
		end

	declaration_type: TYPE is
		do
			Result := resolve_in(start_position.class_text.declaration_type_of_like_current)
		end

	specialize_in (type: TYPE): like Current is
		do
			agent_creation_helper.under_agent_scope_notify(delayed_call.target)
			if delayed_proc_call /= Void then
				-- Because of genericity, it is possible:
				Result := current_or_twin_init(delayed_call, delayed_proc_call.specialize_in(type))
			else
				Result := current_or_twin_init(delayed_call.specialize_in(type), delayed_proc_call)
			end
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			dc: like delayed_call; dpc: like delayed_proc_call
		do
			if delayed_proc_call /= Void then
				dpc := delayed_proc_call.specialize_thru(parent_type, parent_edge, new_type)
				Result := current_or_twin_init(delayed_call, dpc)
			else
				dc := delayed_call.specialize_thru(parent_type, parent_edge, new_type)
				Result := current_or_twin_init(dc, Void)
			end
		end

	specialize_2 (type: TYPE): like Current is
		local
			dc: like delayed_call; dpc: like delayed_proc_call; dc_exp, target: EXPRESSION; dpc_inst: INSTRUCTION;
			arguments: EFFECTIVE_ARG_LIST; fs: FEATURE_STAMP; af: ANONYMOUS_FEATURE
			target_type, target_declaration_type: TYPE
		do
			if delayed_proc_call /= Void then
				target := delayed_proc_call.target.specialize_2(type)
			else
				target := delayed_call.target.specialize_2(type)
			end
			target_declaration_type := target.declaration_type
			fs := target_declaration_type.search(delayed_call.feature_name)
			if fs = Void then
				smart_eiffel.unknown_feature_fatal_error(target, target.declaration_type, delayed_call.feature_name)
			end
			target_type := target.resolve_in(type)
			fs := fs.resolve_static_binding_for(target_declaration_type, target_type)
			af := fs.anonymous_feature(target_type)
			if af.arguments = Void then
				smart_eiffel.argument_count_check(start_position, af, delayed_call.arguments)
			elseif delayed_call.arguments = Void then
				arguments := af.arguments.omitted_open_arguments(delayed_call.feature_name.start_position)
				if arguments.count = 1 then
					create {CALL_1_C} delayed_call.make(target, delayed_call.feature_name, arguments)
				   delayed_call.set_feature_stamp(fs)
				else
					create {CALL_N} delayed_call.make(target, delayed_call.feature_name, arguments)
				   delayed_call.set_feature_stamp(fs)
				end
			else
				arguments := delayed_call.arguments
				smart_eiffel.argument_count_check(start_position, af, arguments)
			end
			check
				af.arguments /= Void implies af.arguments.count = delayed_call.arguments.count
			end
			agent_creation_helper.resolve_open_operands(type, target_type, af, target, arguments)
			if arguments /= Void then
				arguments := arguments.specialize_2(type, af, target_type, delayed_call.target.is_current)
			end
			if af.result_type = Void then
				-- It is actually a procedure call:
				dc := delayed_call
				if delayed_proc_call /= Void and then
					delayed_proc_call.target = target and then
					delayed_proc_call.arguments = arguments and then
					delayed_proc_call.feature_stamp = fs
				 then
					-- Yes we can recycle:
					dpc := delayed_proc_call
				elseif arguments = Void then
					create {PROC_CALL_0} dpc.make(target, delayed_call.feature_name)
					dpc.set_feature_stamp(fs)
				elseif arguments.count = 1 then
					create {PROC_CALL_1} dpc.make(target, delayed_call.feature_name, arguments)
					dpc.set_feature_stamp(fs)
				else
					create {PROC_CALL_N} dpc.make(target, delayed_call.feature_name, arguments)
					dpc.set_feature_stamp(fs)
				end
				-- To get all exportation checks done too:
				dpc_inst := dpc.specialize_2(type)
				if dpc ?:= dpc_inst then
					dpc ::= dpc_inst
				else
					if {AGENT_INSTRUCTION} ?:= dpc_inst then
						error_handler.append("You attempted to create an agent from a call to an agent. %
													%SmartEiffel can only create agents from normal procedure or function calls.%N%N%
													%If you were just trying to call the agent, just remove the `agent' keword.")
					else
						error_handler.append("You attempted to create an agent from something that is not a procedure or function call. %
													%SmartEiffel can only create agents from normal procedure or function calls.")
					end
					error_handler.add_position(start_position)
					error_handler.add_position(dpc_inst.start_position)
					error_handler.print_as_fatal_error
				end
			else
				-- To get all exportation checks done too:
				if delayed_call.target = target and then
					delayed_call.arguments = arguments and then
					delayed_call.feature_stamp = fs
				 then
					dc := delayed_call
				else
					dc := delayed_call.twin
					dc.set_target(target)
					if arguments /= Void then
						dc.set_arguments(arguments)
					end
					dc.set_feature_stamp(fs)
				end
				dc_exp := dc.specialize_2(type)
				if dc ?:= dc_exp then
					dc ::= dc_exp
				else
					if {AGENT_EXPRESSION} ?:= dc_exp then
						error_handler.append("You attempted to create an agent from a call to an agent. %
													%SmartEiffel can only create agents from normal procedure or function calls.%N%N%
													%If you were just trying to call the agent, just remove the `agent' keword.")
					else
						error_handler.append("You attempted to create an agent from something that is not a procedure or function call. %
													%SmartEiffel can only create agents from normal procedure or function calls.")
					end
					error_handler.add_position(start_position)
					error_handler.add_position(dc_exp.start_position)
					error_handler.print_as_fatal_error
				end
			end
			Result := current_or_twin_init(dc, dpc)
		end

	has_been_specialized: BOOLEAN is
		do
			if delayed_proc_call /= Void then
				Result := delayed_proc_call.has_been_specialized
			else
				Result := delayed_call.has_been_specialized
			end
		end

	resolve_in (type: TYPE): TYPE is
		local
			target: EXPRESSION; a: EFFECTIVE_ARG_LIST; open_tuple, rt: TYPE_MARK;
			target_type: TYPE; af: ANONYMOUS_FEATURE; fs: FEATURE_STAMP
		do
			if delayed_proc_call /= Void then
				target := delayed_proc_call.target
				a := delayed_proc_call.arguments
				fs := delayed_proc_call.feature_stamp
				update_capture_open_lists(type, delayed_proc_call.target, delayed_proc_call.arguments)
			else
				target := delayed_call.target
				a := delayed_call.arguments
				fs := delayed_call.feature_stamp
				update_capture_open_lists(type, delayed_call.target, delayed_call.arguments)
			end
			compute_open_type_list(type)
			if open_type_list /= Void then
				create {NON_EMPTY_TUPLE_TYPE_MARK} open_tuple.make(start_position, open_type_list)
				--|*** Still leaky. *** Dom. July 12th 2006 ***
			else
				create {EMPTY_TUPLE_TYPE_MARK} open_tuple.make(start_position)
				--|*** Still leaky. *** Dom. July 12th 2006 ***
			end
			check
				open_tuple.is_static
			end
			target_type := target.resolve_in(type)
			af := fs.anonymous_feature(target_type)
			rt := af.result_type
			if rt = Void then
				create {AGENT_TYPE_MARK} rt.procedure(start_position, open_tuple)
			else
				rt := rt.to_static(target_type)
				if rt.is_boolean then
					create {AGENT_TYPE_MARK} rt.predicate(start_position, open_tuple)
				else
					create {AGENT_TYPE_MARK} rt.function(start_position, open_tuple, rt)
				end
			end
			check
				rt.is_static
			end
			Result := rt.type
		end

	collect (type: TYPE): TYPE is
		local
			unused: TYPE
		do
			if delayed_proc_call /= Void then
				delayed_proc_call.collect(type)
			else
				unused := delayed_call.collect(type)
			end
			Result := resolve_in(type)
			smart_eiffel.collect_create(Result)
			agent_pool.agent_creation_collect(type, Current, Result)
		end

	adapt_for (type: TYPE): like Current is
		local
			dc: like delayed_call; dpc: like delayed_proc_call
		do
			if delayed_proc_call /= Void then
				dpc := delayed_proc_call.adapt_for(type)
				Result := current_or_twin_init(delayed_call, dpc)
			else
				dc := delayed_call.adapt_for(type)
				Result := current_or_twin_init(dc, Void)
			end
		end

	non_void_no_dispatch_type (type: TYPE): TYPE is
		do
			Result := resolve_in(type)
		end

	simplify (type: TYPE): like Current is
		local
			target: EXPRESSION; arguments: EFFECTIVE_ARG_LIST; dc: like delayed_call; dpc: like delayed_proc_call
		do
			if delayed_proc_call /= Void then
				target := delayed_proc_call.target.simplify(type)
				arguments := delayed_proc_call.arguments
				if arguments /= Void then
					arguments := arguments.simplify(type)
				end
				if target = delayed_proc_call.target and then arguments = delayed_proc_call.arguments then
					Result := Current
				else
					dpc := delayed_proc_call.twin
					dpc.set_target(target)
					dpc.set_arguments(arguments)
					Result := twin
					Result.set_delayed_proc_call(dpc)
				end
			else
				target := delayed_call.target.simplify(type)
				arguments := delayed_call.arguments
				if arguments /= Void then
					arguments := arguments.simplify(type)
				end
				if target = delayed_call.target and then arguments = delayed_call.arguments then
					Result := Current
				else
					dc := delayed_call.twin
					dc.set_target(target)
					dc.set_arguments(arguments)
					Result := twin
					Result.set_delayed_call(dc)
				end
			end
		end

	accept (visitor: AGENT_CREATION_VISITOR) is
		do
			visitor.visit_agent_creation(Current)
		end

	is_void: BOOLEAN is False

	is_writable: BOOLEAN is False

	is_manifest_string: BOOLEAN is False

	is_static: BOOLEAN is False

	is_result: BOOLEAN is False

	side_effect_free (type: TYPE): BOOLEAN is
		do
			-- Memory allocation.
		end

	safety_check (type: TYPE) is
		do
			if delayed_proc_call /= Void then
				delayed_proc_call.safety_check(type)
			else
				delayed_call.safety_check(type)
			end
		end

	compile_to_c (type: TYPE) is
		local
			i: INTEGER; boost, need_comma: BOOLEAN; expression: EXPRESSION
		do
			mold_id_in(type, cpp.pending_c_function_body)
			cpp.pending_c_function_body.extend('(')
			boost := ace.boost
			if not boost then
				cpp.pending_c_function_body.append(once "&ds")
				need_comma := True
			end
			if delayed_proc_call /= Void then
				update_capture_open_lists(type, delayed_proc_call.target, delayed_proc_call.arguments)
			else
				update_capture_open_lists(type, delayed_call.target, delayed_call.arguments)
			end
			from
				i := capture_list.lower
			until
				i > capture_list.upper
			loop
				expression := capture_list.item(i)
				if need_comma then
					cpp.pending_c_function_body.extend(',')
				end
				expression.mapping_c_arg(type)
				need_comma := True
				i := i + 1
			end
			cpp.pending_c_function_body.extend(')')
		end

	mapping_c_target (type, target_formal_type: TYPE) is
		do
			compile_to_c(type)
		end

	mapping_c_arg (type: TYPE) is
		do
			compile_to_c(type)
		end

	use_current (type: TYPE): BOOLEAN is
		do
			if delayed_proc_call /= Void then
				Result := delayed_proc_call.target.use_current(type)
				if (not Result) and then delayed_proc_call.arguments /= Void then
					Result := delayed_proc_call.arguments.use_current(type)
				end
			else
				Result := delayed_call.target.use_current(type)
				if (not Result) and then delayed_call.arguments /= Void then
					 Result := delayed_call.arguments.use_current(type)
				end
			end
		end

	bracketed_pretty, pretty (indent_level: INTEGER) is
		do
			pretty_printer.keyword(once "agent")
			delayed_call.pretty(indent_level)
		end

	pretty_target (indent_level: INTEGER) is
		do
			pretty_printer.put_character('(')
			pretty(indent_level)
			pretty_printer.put_character(')')
			pretty_printer.put_character('.')
		end

	short (type: TYPE) is
		do
			short_printer.hook_or(as_agent, as_agent)
			short_printer.put_character(' ')
			delayed_call.short(type)
		end

	short_target (type: TYPE) is
		do
			bracketed_short(type)
			short_printer.put_dot
		end

	precedence: INTEGER is
		do
			Result := atomic_precedence
		end

	jvm_assign_creation, jvm_assign (type: TYPE) is
		do
			check
				False
			end
		end

	compile_target_to_jvm, compile_to_jvm (type: TYPE) is
		do
			not_yet_implemented
		end

	jvm_branch_if_false, jvm_branch_if_true (type: TYPE): INTEGER is
		do
			not_yet_implemented
		end

feature {AGENT_POOL}
	mold_id_in (type: TYPE; buffer: STRING) is
			-- Identify the corresponding AGENT_CREATION function .
		do
			buffer.append(once "agenT")
			type.id.append_in(buffer)
			buffer.extend('C')
			start_position.class_text.id.append_in(buffer)
			buffer.extend('l')
			start_position.line.append_in(buffer)
			buffer.extend('c')
			start_position.column.append_in(buffer)
		end

	same_mold_id_as (other: like Current): BOOLEAN is
		do
			Result := start_position = other.start_position
		end

	c_define_function (type: TYPE; mold_id: STRING; integer_mold_id: INTEGER) is
		local
			boost: BOOLEAN; target_type, agent_type, agent_result: TYPE; tm: TYPE_MARK;
			i: INTEGER; rf: RUN_FEATURE; target, target2, expression: EXPRESSION
			user_expanded_flag: BOOLEAN; fs: FEATURE_STAMP; eal, eal2: EFFECTIVE_ARG_LIST
			open_operand: OPEN_OPERAND
		do
			boost := ace.boost
			agent_type := c_define_type(type, boost, mold_id, integer_mold_id)
			agent_result := agent_type.agent_result

			-- The local execution function:
			cpp.prepare_c_function
			cpp.pending_c_function_signature.append(once "%N/*agent launcher*/static ")
			if agent_result /= Void then
				agent_result.canonical_type_mark.c_type_for_result_in(cpp.pending_c_function_signature)
			else
				cpp.pending_c_function_signature.append(once "void")
			end
			cpp.pending_c_function_signature.append(once " _")
			cpp.pending_c_function_signature.append(mold_id)
			cpp.pending_c_function_signature.extend('(')
			if not boost then
				cpp.pending_c_function_signature.append(once "se_dump_stack*caller,")
			end
			if ace.profile then
				cpp.pending_c_function_signature.append(once "se_local_profile_t*parent_profile,")
			end
			cpp.pending_c_function_signature.append(once "se_")
			cpp.pending_c_function_signature.append(mold_id)
			cpp.pending_c_function_signature.append(once "*u")
			if delayed_proc_call /= Void then
				update_capture_open_lists(type, delayed_proc_call.target, delayed_proc_call.arguments)
			else
				update_capture_open_lists(type, delayed_call.target, delayed_call.arguments)
			end
			from
				i := open_list.lower
			until
				i > open_list.upper
			loop
				open_operand := open_list.item(i)
				cpp.pending_c_function_signature.extend(',')
				open_operand.resolve_in(type).canonical_type_mark.c_type_for_result_in(cpp.pending_c_function_signature)
				cpp.pending_c_function_signature.extend(' ')
			   open_operand.compile_to_c_in(type, cpp.pending_c_function_signature)
				i := i + 1
			end
			cpp.pending_c_function_signature.extend(')')
			from
				i := capture_list.lower
			until
				i > capture_list.upper
			loop
				expression := capture_list.item(i)
				if not expression.is_void then
					expression.resolve_in(type).canonical_type_mark.c_type_for_result_in(cpp.pending_c_function_body)
					if expression.is_current and then type.is_user_expanded then
						cpp.pending_c_function_body.extend('*')
						user_expanded_flag := True
					else
						cpp.pending_c_function_body.extend(' ')
						user_expanded_flag := False
					end
					cpp.pending_c_function_body.append (once " c")
					i.append_in(cpp.pending_c_function_body)				
					cpp.pending_c_function_body.extend('=')
					if user_expanded_flag then
						cpp.pending_c_function_body.extend('&')
					end
					cpp.pending_c_function_body.append(once "(u->c")
					i.append_in(cpp.pending_c_function_body)				
					cpp.pending_c_function_body.append(once ");%N")
				end
				i := i + 1
			end
			if not boost then
				cpp.pending_c_function_body.append(once "static se_frame_descriptor fd={%"Agent launcher%",0,0,%"%",1};%N")
			end
			if ace.profile then
				smart_eiffel.local_profile
			end
			if not boost then
				cpp.pending_c_function_body.append(once "[
               se_dump_stack ds;
               ds.fd=&fd;
               ds.current=NULL;
               ds.p=(caller->p);
               ds.caller=caller;
               ds.exception_origin=NULL;
               ds.locals=NULL;
               
               ]")
			end
			-- Calling the actual one:
			if ace.profile then
				smart_eiffel.start_profile_agent_creation(Current)
			end
			if agent_result /= Void then
				cpp.pending_c_function_body.append(once "u->R=")
			end
			if delayed_proc_call /= Void then
				target := delayed_proc_call.target
				eal := delayed_proc_call.arguments
				target_type := target.resolve_in(type)
				fs := delayed_proc_call.feature_stamp
			else
				target := delayed_call.target
				eal := delayed_call.arguments
				target_type := target.resolve_in(type)
				fs := delayed_call.feature_stamp
			end
			rf := fs.run_feature_for(target_type)
			-- May replace `target' and `a' to use NUMBERED_CAPTURE replacement:
			target2 := numbered_capture(target)
			eal2 := numbered_capture_for(eal)
			cpp.put_cpc(type, fs, rf, target2, eal2)
			if agent_result /= Void then
				cpp.pending_c_function_body.append(once ";%N")
			end
			if ace.profile then
				smart_eiffel.stop_profile
			end
			if agent_result /= Void then
				cpp.pending_c_function_body.append(once "return u->R;%N")
			end
			cpp.dump_pending_c_function(False)

			-- The definition function:
			cpp.prepare_c_function
			cpp.pending_c_function_signature.append(once "/*agent creation*/T0*")
			cpp.pending_c_function_signature.append(mold_id)
			cpp.pending_c_function_signature.extend('(')
			if not boost then
				cpp.pending_c_function_signature.append(once "se_dump_stack*caller")
			end
			from
				i := capture_list.lower
			until
				i > capture_list.upper
			loop
				expression := capture_list.item(i)
				if cpp.pending_c_function_signature.last /= '(' then
					cpp.pending_c_function_signature.extend(',')
				end
				if expression.is_void then
					cpp.pending_c_function_signature.append(once "T0*")
				else
					expression.resolve_in(type).canonical_type_mark.c_type_for_result_in(cpp.pending_c_function_signature)
				end
				cpp.pending_c_function_signature.append(once " c")
				i.append_in(cpp.pending_c_function_signature)				
				i := i + 1
			end
			if cpp.pending_c_function_signature.last = '(' then
				cpp.pending_c_function_signature.append(once "void")
			end
			cpp.pending_c_function_signature.extend(')')
			cpp.pending_c_function_body.append(once "se_")
			cpp.pending_c_function_body.append(mold_id)
			cpp.pending_c_function_body.append(once "*u=(void*)new_agent(")
			agent_type.id.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ");%Nu->creation_mold_id=")
			integer_mold_id.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nu->afp=_")
			cpp.pending_c_function_body.append(mold_id)
			cpp.pending_c_function_body.append(once ";%Nu->eq=eq_")
			cpp.pending_c_function_body.append(agent_equal.mold_id)
			cpp.pending_c_function_body.append(once ";%N")
			if not gc_handler.is_off then
				cpp.pending_c_function_body.append(once "u->gc_mark_agent_mold=gc_mark_")
				cpp.pending_c_function_body.append(mold_id)
				cpp.pending_c_function_body.append(once ";%N")
			end
			from
				i := capture_list.lower
			until
				i > capture_list.upper
			loop
				expression := capture_list.item(i)
				cpp.pending_c_function_body.append(once "u->c")
				i.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once "=c")
				i.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once ";%N")
				i := i + 1
			end
			cpp.pending_c_function_body.append(once "return((T0*)u);%N")
			cpp.dump_pending_c_function(True)

			-- The gc_mark_MOLD_ID function:
			if not gc_handler.is_off then
				cpp.prepare_c_function
				cpp.pending_c_function_signature.append(once "void gc_mark_")
				cpp.pending_c_function_signature.append(mold_id)
				cpp.pending_c_function_signature.append(once "(se_")
				cpp.pending_c_function_signature.append(mold_id)
				cpp.pending_c_function_signature.append(once "*u)")

				cpp.pending_c_function_body.append(once "gc_agent*gcu=(gc_agent*)u;%N%
									  %if (gcu->header.flag==FSOH_UNMARKED){%N%
									  %gcu->header.flag=FSOH_MARKED;%N")
				from
					i := capture_list.lower
				until
					i > capture_list.upper
				loop
					expression := capture_list.item(i)
					if not expression.is_void then
						tm := expression.resolve_in(type).canonical_type_mark
						if tm.is_reference then
							cpp.pending_c_function_body.append(once "gc_mark(u->c")
							i.append_in(cpp.pending_c_function_body)
							cpp.pending_c_function_body.append(once ");%N")
						elseif tm.need_gc_mark_function then
							tm.gc_mark_in(cpp.pending_c_function_body)
							cpp.pending_c_function_body.append(once "(&(u->c")
							i.append_in(cpp.pending_c_function_body)
							cpp.pending_c_function_body.append(once "));%N")
						end
					end
					i := i + 1
				end
				cpp.pending_c_function_body.append(once "}%N")
				cpp.dump_pending_c_function(True)
			end

			-- The is_equal function
			agent_equal.c_define_function(type)
		end

feature {AGENT_CREATION}
	init (dc: like delayed_call; dpc: like delayed_proc_call) is
		require
			dc /= Void
		do
			delayed_call := dc
			delayed_proc_call := dpc
			agent_equal.set_call(dc, dpc)
		ensure
			delayed_call = dc
			delayed_proc_call = dpc
		end

	set_delayed_call (dc: like delayed_call) is
		require
			dc /= Void
		do
			delayed_call := dc
		ensure
			delayed_call = dc
		end
	
	set_delayed_proc_call (dpc: like delayed_proc_call) is
		require
			dpc /= Void
		do
			delayed_proc_call := dpc
		ensure
			delayed_proc_call = dpc
		end
	
feature {}
	make (sp: like start_position; dc: like delayed_call) is
		require
			not sp.is_unknown
			dc /= Void
		do
			start_position := sp
			delayed_call := dc
			agent_creation_helper.syntactical_flat_check(delayed_call)
			create agent_equal.make
		ensure
			start_position = sp
			delayed_call = dc
		end
	
	current_or_twin_init (dc: like delayed_call; dpc: like delayed_proc_call): like Current is
		require
			dc /= Void
		do
			if dc = delayed_call and then dpc = delayed_proc_call then
				Result := Current
				agent_equal.set_call(dc, dpc)
			else
				Result := twin
				Result.init(dc, dpc)
			end
		ensure
			Result.delayed_call = dc
			Result.delayed_proc_call = dpc
		end

	c_define_type (type: TYPE; boost: BOOLEAN; mold_id: STRING; integer_mold_id: INTEGER): TYPE is
			-- The Result is the `agent_type'.
		require
			boost = ace.boost
			not cpp.pending_c_function
		local
			i: INTEGER; agent_type, agent_result: TYPE; expression: EXPRESSION
		do
			if delayed_proc_call /= Void then
				update_capture_open_lists(type, delayed_proc_call.target, delayed_proc_call.arguments)
			else
				update_capture_open_lists(type, delayed_call.target, delayed_call.arguments)
			end
			agent_type := resolve_in(type)
			Result := agent_type

			cpp.out_h_buffer.copy(once "typedef struct _se_")
			cpp.out_h_buffer.append(mold_id)
			cpp.out_h_buffer.append(once " se_")
			cpp.out_h_buffer.append(mold_id)
			cpp.out_h_buffer.append(once ";%Nstruct _se_")
			cpp.out_h_buffer.append(mold_id)
			cpp.out_h_buffer.append(once "{Tid id;%Nint creation_mold_id;%N")
			agent_result := agent_type.agent_result
			if agent_result = Void then
				cpp.out_h_buffer.append(once "void")
			else
				agent_result.canonical_type_mark.c_type_for_result_in(cpp.out_h_buffer)
			end
			cpp.out_h_buffer.append(once "(*afp)(")
			if not boost then
				cpp.out_h_buffer.append(once "se_dump_stack*,")
			end
			if ace.profile then
				cpp.out_h_buffer.append(once "se_local_profile_t*,")
			end
			cpp.out_h_buffer.append(once "se_")
			cpp.out_h_buffer.append(mold_id)
			cpp.out_h_buffer.extend('*')
			from
				i := open_list.lower
			until
				i > open_list.upper
			loop
				cpp.out_h_buffer.extend(',')
				open_list.item(i).resolve_in(type).canonical_type_mark.c_type_for_result_in(cpp.out_h_buffer)
				i := i + 1
			end
			cpp.out_h_buffer.append(once ");%N")
			if not gc_handler.is_off then
				cpp.out_h_buffer.append(once "[
				     void(*gc_mark_agent_mold)(se_
				     ]")
				cpp.out_h_buffer.append(mold_id)
				cpp.out_h_buffer.append(once "*);%N")
			end
			cpp.out_h_buffer.append(once "[
			     int (*eq)(se_agent*,se_agent*);

			     ]")
			from
				i := capture_list.lower
			until
				i > capture_list.upper
			loop
				expression := capture_list.item(i)
				if expression.is_void then
					cpp.out_h_buffer.append(once "T0*")
				else
					expression.resolve_in(type).canonical_type_mark.c_type_for_result_in(cpp.out_h_buffer)
				end
				cpp.out_h_buffer.append(once " c")
				i.append_in(cpp.out_h_buffer)
				cpp.out_h_buffer.extend(';')
				i := i + 1
			end
			if agent_result /= Void then
				agent_result.canonical_type_mark.c_type_for_argument_in(cpp.out_h_buffer)
				cpp.out_h_buffer.append(once " R;")
			end
			cpp.out_h_buffer.append(once "};%N")
			cpp.write_out_h_buffer
			agent_equal.c_define_type(type)
		end

feature {}
	capture_list: FAST_ARRAY[EXPRESSION]
	agent_equal: AGENT_EQUAL
	open_list: FAST_ARRAY[OPEN_OPERAND]

	update_capture_open_lists (type: TYPE; target: EXPRESSION; arguments: EFFECTIVE_ARG_LIST) is
		require
			type /= Void
			target /= Void
		local
			i: INTEGER; expression: EXPRESSION; open_operand: OPEN_OPERAND
		do
			if capture_list = Void then
				create capture_list.with_capacity(4)
				create open_list.with_capacity(4)
			end
			capture_list.clear_count
			open_list.clear_count
			if {OPEN_OPERAND} ?:= target then
				open_operand ::= target
				open_list.add_last(open_operand)
			else
				capture_list.add_last(target)
			end
			if arguments /= Void then
				from
					i := 1
				until
					i > arguments.count
				loop
					expression := arguments.expression(i)
					if {OPEN_OPERAND} ?:= expression then
						open_operand ::= expression
						open_list.add_last(open_operand)
					else
						capture_list.add_last(expression)
					end
					i := i + 1
				end
			end
			agent_equal.set_capture_list(capture_list)
		end
	
	agent_creation_helper: AGENT_CREATION_HELPER is
		once
			create Result.make
		end

	numbered_capture_for (a: EFFECTIVE_ARG_LIST): EFFECTIVE_ARG_LIST is
		local
			i: INTEGER; first_one, e: EXPRESSION; remainder: FAST_ARRAY[EXPRESSION]
		do
			if a /= Void then
				first_one := numbered_capture(a.first_one)
				if a.remainder /= Void then
					from
						remainder := a.remainder
						i := remainder.upper
					until
						i < remainder.lower
					loop
						e := numbered_capture(remainder.item(i))
						if e /= remainder.item(i) then
							if remainder = a.remainder then
								remainder := remainder.twin
							end
							remainder.put(e, i)
						end
						i := i - 1
					end
				end
				if first_one = a.first_one and then remainder = a.remainder then
					Result := a
				else
					create Result.make_n(first_one, remainder)
				end
			end
		ensure
			(a = Void) implies (Result = Void)
			(a /= Void) implies (a.count = Result.count)
		end

	open_type_list: ARRAY[TYPE_MARK]
	
	compute_open_type_list (type: TYPE) is
			-- Assume `open_list' is uptodate.
		local
			i, j: INTEGER
		do
			if not open_list.is_empty then
				if open_type_list = Void then
					create open_type_list.make(1, open_list.count)
				else
					open_type_list.make(1, open_list.count)
				end
				from
					i := open_type_list.lower
					j := open_list.lower
				until
					i > open_type_list.upper
				loop
					open_type_list.put(open_list.item(j).resolve_in(type).canonical_type_mark, i)
					i := i + 1
					j := j + 1
				end
			end
		end

	numbered_capture (expression: EXPRESSION): EXPRESSION is
		local
			rank: INTEGER
		do
			if expression.is_static then
				Result := expression
			else
				rank := capture_list.first_index_of(expression)
				if capture_list.valid_index(rank) then
					create {NUMBERED_CAPTURE} Result.make(expression, rank)
				else
					Result := expression
				end
			end
		end
	
invariant
	not start_position.is_unknown
	delayed_call /= Void
	agent_equal /= Void

end -- class AGENT_CREATION
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
