-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class CALL
	--
	-- For all sort of feature calls with result value.
	-- So it does not include procedure calls (see PROC_CALL).
	--
	-- Classification: CALL_0 when 0 argument, CALL_1 when
	-- 1 argument and CALL_N when N arguments.
	--

inherit
	EXPRESSION
	CALL_SUPPORT

feature {ANY}
	is_writable: BOOLEAN is False

	is_current: BOOLEAN is False

	is_implicit_current: BOOLEAN is False

	is_manifest_string: BOOLEAN is False

	is_result: BOOLEAN is False

	is_void: BOOLEAN is False

	is_static: BOOLEAN is False

	frozen collect (type: TYPE): TYPE is
		do
			Result := call_support_collect(type)
		end
	
	resolve_in (type: TYPE): TYPE is
		local
			tm: TYPE_MARK; target_type: TYPE
		do
			target_type := target.resolve_in(type)
			tm := feature_stamp.anonymous_feature(target_type).result_type -- If this call fails, then check if specialize_2 has been called!
			if tm /= Void then
				Result := tm.resolve_in(target_type)
			else
				--|*** Can we remove this error message ?
				-- I am quite sure the answer is yes, but it is safer to let this here ?
				--|*** (Dom. march 30th 2004)
				error_handler.append("This feature is a procedure: it cannot be used as target.")
				error_handler.add_position(feature_stamp.anonymous_feature(target_type).start_position)
				error_handler.add_position(feature_name.start_position)
				error_handler.print_as_fatal_error
			end
		end

	declaration_type: TYPE is
		local
			target_declaration_type: TYPE; af: ANONYMOUS_FEATURE; fs: FEATURE_STAMP
		do
			if declaration_type_memory = Void then
				target_declaration_type := target.declaration_type
				fs := target_declaration_type.lookup(feature_name)
				if target_declaration_type = feature_accumulator.context_type then
					declaration_type_memory := feature_accumulator.find_type_for(fs).resolve_in(target_declaration_type)
				else
					af := fs.anonymous_feature(target_declaration_type)
					declaration_type_memory := af.result_type.resolve_in(target_declaration_type)
				end
			end
			Result := declaration_type_memory
		end

	frozen non_void_no_dispatch_type (type: TYPE): TYPE is
		local
			rt: TYPE
		do
			rt := resolve_in(type)
			if rt.is_expanded then
				Result := rt
			else
				-- Even if there is only one possibility for the dynamic
				-- type, we are not sure that this call is not Void.
			end
		end

	frozen mapping_c_target (type, target_formal_type: TYPE) is
		local
			class_invariant_flag: INTEGER; actual_type: TYPE; internal_c_local: INTERNAL_C_LOCAL
		do
			class_invariant_flag := cpp.class_invariant_call_opening(target_formal_type, True)
			actual_type := resolve_in(type)
			if actual_type.is_reference then
				cpp.pending_c_function_body.extend('(')
				target_formal_type.canonical_type_mark.c_type_for_target_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once ")(")
				compile_to_c(type)
				cpp.pending_c_function_body.extend(')')
			elseif target_formal_type.canonical_type_mark.need_c_struct then
				if extra_local_expanded(type) = Void then
					cpp.pending_c_function_body.append(once "&(")
				else
					check
						extra_local_expanded(type) = actual_type
					end
					cpp.pending_c_function_body.extend('(')
					internal_c_local := cpp.pending_c_function_lock_local(actual_type)
					internal_c_local.append_in(cpp.pending_c_function_body)
					cpp.pending_c_function_body.extend('=')
				end
				compile_to_c(type)
				if internal_c_local /= Void then
					cpp.pending_c_function_body.append(once ",&")
					internal_c_local.append_in(cpp.pending_c_function_body)
				end
				cpp.pending_c_function_body.extend(')')
			else
				compile_to_c(type)
			end
			if class_invariant_flag > 0 then
				cpp.class_invariant_call_closing(class_invariant_flag, False)
			end
		end

	frozen mapping_c_arg (type: TYPE) is
		do
			compile_to_c(type)
		end

	pretty_target (indent_level: INTEGER) is
		do
			pretty(indent_level)
			pretty_printer.put_character('.')
		end

	frozen compile_target_to_jvm (type: TYPE) is
		do
			standard_compile_target_to_jvm(type)
		end

	frozen jvm_assign_creation, jvm_assign (type: TYPE) is
		do
			check
				False
			end
		end

	has_been_specialized: BOOLEAN is
		do
			Result := target.has_been_specialized
			if Result and then arguments /= Void then
				Result := arguments.has_been_specialized
			end
			if target.is_current then
				Result := Result and then feature_stamp /= Void
			end
		end

feature {CALL_SUPPORT, IMPLICIT_CAST}
	extra_local_expanded (type: TYPE): TYPE is
			-- Assuming that `Current' is used as some target, if some extra local variable is required, the
			-- corresponding user's expanded type is returned.
		require
			smart_eiffel.status.collecting_done
			type /= Void
		local
			target_type, rt: TYPE; an_attribute: ATTRIBUTE; af: ANONYMOUS_FEATURE
		do
			rt := resolve_in(type)
			if rt.is_user_expanded and then not rt.is_empty_expanded then
				target_type := target.resolve_in(type)
				af := feature_stamp.anonymous_feature(target_type)
				if (an_attribute ?:= af) and then target.is_current then
					-- An extra local would prevent the modification of the target.
				else
					Result := rt
				end
			end
		ensure
			Result /= Void implies Result.is_user_expanded
		end

	creation_type: TYPE_MARK is
		do
		end

feature {}
	declaration_type_memory: TYPE
	
	integer_constant_overflow: STRING is "Integer constant overflow (out of INTEGER_64 range)."

	frozen function_and_argument_count_check (af: ANONYMOUS_FEATURE; actual_args: like arguments) is
			-- Check that the feature found is really a function then launch `argument_count_check'.
		require
			af /= Void
		do
			if af.result_type = Void then
				error_handler.add_position(af.start_position)
				error_handler.add_position(feature_name.start_position)
				error_handler.append("Feature found is a procedure.")
				error_handler.print_as_error
				error_handler.add_position(feature_name.start_position)
				error_handler.append("This call has no result.")
				error_handler.print_as_fatal_error
			end
			smart_eiffel.argument_count_check(feature_name.start_position, af, actual_args)
		end

	frozen true_or_false (boolean: BOOLEAN): EXPRESSION is
			-- (Often used by simplify.)
		do
			if boolean then
				create {E_TRUE} Result.make(feature_name.start_position)
			else
				create {E_FALSE} Result.make(feature_name.start_position)
			end
		end

end -- class CALL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
