-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class INTERNAL_C_LOCAL
	--
	-- An internal extra variable introduced to generate C code. As an example, we need extra C local to handle, 
	-- in some situations, user expanded types.
	--
	-- Note: unlike INTERNAL_LOCAL variables, INTERNAL_C_LOCAL variables are only useful for C code generation.
	--

insert
	GLOBALS
	
creation {INTERNAL_C_LOCAL_LIST}
	make

feature {}
	number: INTEGER
			-- of the "tmp" variable in C code.
	
feature {ANY}
	type: TYPE
			-- corresponding `type' for this INTERNAL_C_LOCAL variable.
	
	is_locked: BOOLEAN
			-- Indicates that the variable cannot be reused yet for another temporary computation. 

	append_in (buffer: STRING) is
		do
			buffer.append(once "tmp")
			number.append_in(buffer)
		end

	un_lock is
		do
			is_locked := False
		ensure
			not is_locked
		end
	
feature {INTERNAL_C_LOCAL_LIST}
	is_member_of_pending_c_function_locals: BOOLEAN
	
	lock is
		require
			not is_locked
		do
			is_locked := True
		ensure
			is_locked
		end
	
	un_lock_after_dump is
		do
			is_locked := False
			is_member_of_pending_c_function_locals := False
		ensure
			not is_locked
		end
	
	recycle (t: like type) is
		require
			t /= Void
		do
			type := t
			is_locked := True
			is_member_of_pending_c_function_locals := True
		ensure
			type = t
			is_locked
		end

	c_type_in (buffer: STRING) is
		do
			if type.is_expanded then
				type.canonical_type_mark.c_type_for_argument_in(buffer)
			else
				buffer.append(once "T0*")
			end
		end
	
	has_the_same_c_type_than (t: like type): BOOLEAN is
		do
			if type = t then
				Result := True
			elseif type.is_reference then
				Result := t.is_reference
			end
		end
	
feature {}
	make (t: like type) is
		require
			t /= Void
			not is_locked
		do
			number := counter.value
			counter.increment
			type := t
			is_locked := True
			is_member_of_pending_c_function_locals := True
		ensure
			type = t
			is_locked
		end

feature {}
	counter: COUNTER is
		once
			create Result
		end
	
end -- class INTERNAL_C_LOCAL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
