-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class EIFFELDOC_GLOBALS

insert
	GLOBALS

feature {}
	operator_filter: HASHED_DICTIONARY[STRING, CHARACTER] is
		once
			create Result.make
			Result.add("plus.", '+')
			Result.add("minus.", '-')
			Result.add("mult.", '*')
			Result.add("slash.", '/')
			Result.add("bslash.", '\')
			Result.add("and.", '&')
			Result.add("pipe.", '|')
			Result.add("inf.", '<')
			Result.add("sup.", '>')
			Result.add("eq.", '=')
			Result.add("pow.", '^')
			Result.add("space.", ' ')
			Result.add("sharp.", '#')
			Result.add("quot.", '"')
			Result.add("tilde.", '~')
			Result.add("at.", '@')
		end

	css_root: STRING is "root"
	css_all_clusters: STRING is "all_clusters"
	css_letter_cluster: STRING is "letter_cluster"
	css_cluster: STRING is "cluster"
	css_sub_cluster: STRING is "sub_cluster"
	css_cluster_comments: STRING is "cluster_comments"
	css_cluster_classes: STRING is "cluster_classes"
	css_points_of_view: STRING is "points_of_view"
	css_summary: STRING is "summary"
	css_overview: STRING is "overview"
	css_feature: STRING is "feature"
	css_invariant: STRING is "invariant"
	css_assertion: STRING is "assertion"
	css_obsolete: STRING is "obsolete"
	
	css_block_suffix: STRING is "_block"
	css_head_suffix: STRING is "_head"
	css_head_layout_suffix: STRING is "_head_layout"
	css_expand_mark_suffix: STRING is "_expand_mark"
	css_expand_mark_hover_suffix: STRING is "_expand_mark_hover"
	css_title_suffix: STRING is "_title"
	css_nav_link_suffix: STRING is "_nav_link"
	css_name_suffix: STRING is "_name"
	css_comment_suffix: STRING is "_comment"
	css_clause_comment_suffix: STRING is "_clause_comment"
	css_feature_item_suffix: STRING is "_feature_item"
	css_expanded_suffix: STRING is "_expanded"
	css_current_suffix: STRING is "_current"
	css_keyword_suffix: STRING is "_keyword"
	css_msg_suffix: STRING is "_msg"
	css_mark_suffix: STRING is "mark"
	css_deferred_suffix: STRING is "_deferred"
	css_frozen_suffix: STRING is "_frozen"
	css_attribute_suffix: STRING is "_attribute"
	css_once_suffix: STRING is "_once"
	css_blank_suffix: STRING is "_blank"

	css_nav_link: STRING is "nav_link"
	css_class_link: STRING is "class_link"
	css_feature_link: STRING is "feature_link"

	index_filename: STRING is "index.html"

	top_anchor_name: STRING is "top"
	top_anchor_address: STRING is "#top"
	top_link_name: STRING is "top"
	home_link_name: STRING is "home"
	wiki_link_name: STRING is "wiki"
	classes_and_clusters_list_link_name: STRING is "classes/clusters list"
	classes_and_clusters_list_page_title: STRING is "CLASSES and CLUSTERS list"
	class_information_link_name: STRING is "class information"
	points_of_view_menu_name: STRING is "Point of view"
	all_feature_entry_name: STRING is "All features"
	class_str: STRING is "class "
	deferred_str: STRING is "deferred"
	expanded_str: STRING is "expanded"
	frozen_str: STRING is "frozen"
	attribute_str: STRING is "attribute"
	once_str: STRING is "once"
	obsolete_class_str: STRING is "This class is obsolete: "
	obsolete_feature_str: STRING is "This feature is obsolete: "

	summary_title_str: STRING is "Summary"
	direct_parents_str: STRING is "Direct parents"
	known_children_str: STRING is "Known children"
	inherit_list_str: STRING is "inherit list"
	insert_list_str: STRING is "insert list"

	all_classes_str: STRING is "All classes"
	cluster_comments_str: STRING is "Cluster description"
	cluster_classes_str: STRING is "Cluster classes"
	overview_title_str: STRING is "Overview"
	creation_features_str: STRING is "creation features"
	exported_features_str: STRING is "exported features"
	features_str: STRING is "features"
	class_invariant_str: STRING is "Class invariant"
	require_str: STRING is "require"
	ensure_str: STRING is "ensure"
	old_str: STRING is "old"

	all_classes_id: STRING is "All classes:"
	cluster_comments_id: STRING is "Cluster description:"
	cluster_classes_id: STRING is "Cluster classes:"
	summary_id: STRING is "Summary:"
	points_of_view_id: STRING is "Points of view:"
	overview_id: STRING is "Overview:"
	class_invariant_id: STRING is "Class invariant:"

	title_bar: STRING is "Eiffeldoc : "
	real_css: STRING is "eiffeldoc.css"
	real_js : STRING is "eiffeldoc.js"

	filtered_attribute(value: STRING): STRING is
		require
			value /= Void
		local
			i: INTEGER
			c: CHARACTER
		do
			Result := once ""
			Result.clear_count
			i := value.lower
			if value.first = '#' then
				Result.extend('#')
				i := i + 1
			end
			from
			until
				i > value.upper
			loop
				c := value.item(i)
				if operator_filter.has(c) then
					Result.append(operator_filter.at(c))
				else
					Result.extend(c)
				end
				i := i + 1
			end
		end

	set_suffixed_attribute (name, base_value, suffix: STRING; html_os: EIFFELDOC_OUTPUT_STREAM) is
		require
			name /= Void
			base_value /= Void
			suffix /= Void
			html_os /= Void
		local
			s: STRING
		do
			s := once ""
			s.copy(base_value)
			s.append(suffix)
			html_os.with_attribute(name, s)
		end

	html_output_stream_for_file (filename: STRING): EIFFELDOC_OUTPUT_STREAM is
		require
			filename /= Void
		local
			path: STRING; depth: INTEGER; bd: BASIC_DIRECTORY
			tfw: TEXT_FILE_WRITE
		do
			depth := create_dirs(filename)
			path := last_dirname
			if path /= Void then
				bd.compute_file_path_with(path, basename_of(filename))
				if not  bd.last_entry.is_empty then
					path.copy(bd.last_entry)
				else
					std_error.put_string(once "*** Could not compute the file name ")
					std_error.put_string(basename_of(filename))
					std_error.put_string(once " in the directory ")
					std_error.put_string(path)
					std_error.put_line(once " -- creating file in current directory.")
					depth := 0
					path := basename_of(filename)
				end
			else
				std_error.put_string(once "*** Could not create a directory for the file ")
				std_error.put_string(filename)
				std_error.put_line(once " -- creating file in current directory.")
				depth := 0
				path := basename_of(filename)
			end
			create tfw.connect_to(path)
			create Result.connect_to(tfw)
			Result.set_depth(depth)
		end

	write_header (html_os: EIFFELDOC_OUTPUT_STREAM; title, js, css: STRING) is
		require
			html_os /= Void
			title /= Void
		do
			html_os.header
			html_os.open_title
			html_os.put_string(title_bar)
			html_os.put_string(title)
			html_os.close_title
			html_os.put_stylesheet(css)
			html_os.put_javascript(js)
			html_os.with_attribute(once "onload", once "init()")
		ensure
			html_os.in_header
		end

	open_block_head (html_os: EIFFELDOC_OUTPUT_STREAM; css_base_class, base_id: STRING) is
		require
			html_os /= Void
			css_base_class /= Void
			base_id /= Void
		local
			id: STRING
		do
			id := once ""
			id.clear_count
			id.copy(base_id)
			id.add_last('.')
			id := filtered_attribute(id)

			html_os.indent
			set_suffixed_attribute(once "class", css_base_class, css_block_suffix, html_os)
			html_os.open_div
			set_suffixed_attribute(once "class", css_base_class, css_head_suffix, html_os)
			html_os.open_div
			set_suffixed_attribute(once "class", css_base_class, css_head_layout_suffix, html_os)
			html_os.open_div
			set_suffixed_attribute(once "id", id, once "_hl", html_os)
			set_suffixed_attribute(once "class", css_base_class, css_expand_mark_suffix, html_os)
			html_os.open_div
			html_os.put_character(' ')
			html_os.close_div
			set_suffixed_attribute(once "id", id, once "", html_os)
			set_suffixed_attribute(once "class", css_base_class, css_title_suffix, html_os)
			html_os.open_div
		end

	close_block_head (html_os: EIFFELDOC_OUTPUT_STREAM; css_base_class: STRING) is
		require
			html_os /= Void
			css_base_class /= Void
		do
			html_os.close_div
			set_suffixed_attribute(once "class", css_base_class, css_nav_link_suffix, html_os)
			html_os.open_div
			html_os.with_attribute(once "class", css_nav_link)
			html_os.open_anchor_address(top_anchor_address, Void)
			html_os.put_string(top_link_name)
			html_os.close_anchor
			html_os.close_div
			html_os.close_div
			html_os.close_div
		end

	open_block (html_os: EIFFELDOC_OUTPUT_STREAM; css_base_class, name, base_id: STRING) is
		require
			html_os /= Void
			css_base_class /= Void
			name /= Void
			base_id /= Void
		do
			open_block_head(html_os, css_base_class, base_id)

			set_suffixed_attribute(once "class", css_base_class, css_name_suffix, html_os)
			html_os.open_span
			html_os.put_string(name)
			html_os.close_span

			close_block_head(html_os, css_base_class)
		end

	close_block (html_os: EIFFELDOC_OUTPUT_STREAM) is
		require
			html_os /= Void
		do
			html_os.indent
			html_os.close_div
		end
	
	open_comment_block (html_os: EIFFELDOC_OUTPUT_STREAM; css_base_class: STRING) is
		require
			html_os /= Void
			css_base_class /= Void
		do
			set_suffixed_attribute(once "class", css_base_class, css_comment_suffix, html_os)
			html_os.open_div
		end

	close_comment_block (html_os: EIFFELDOC_OUTPUT_STREAM) is
		require
			html_os /= Void
		do
			html_os.indent
			html_os.close_div
		end
	
	open_expand_block (html_os: EIFFELDOC_OUTPUT_STREAM; css_base_class: STRING; base_id: STRING; expand: BOOLEAN) is
		require
			html_os /= Void
			css_base_class /= Void
			base_id /= Void
		local
			id: STRING
		do
			id := once ""
			id.clear_count
			id.copy(base_id)
			id.add_last('.')

			set_suffixed_attribute(once "id", filtered_attribute(id), once "_expanded", html_os)
			set_suffixed_attribute(once "class", css_base_class, css_expanded_suffix, html_os)
			if expand then
				html_os.with_attribute(once "style", once "display: block")
			else
				html_os.with_attribute(once "style", once "display: none")
			end
			html_os.open_div
		end

	close_expand_block (html_os: EIFFELDOC_OUTPUT_STREAM) is
		require
			html_os /= Void
		do
			html_os.indent
			html_os.close_div
		end

	create_dirs (a_filename: STRING): INTEGER is
			-- And returns the depth of the tree
		local
			i, j: INTEGER; bd: BASIC_DIRECTORY; buffer, dirname: STRING
		do
			last_dirname := once ""
			last_dirname.copy(bd.current_working_directory)
			buffer := once ""
			buffer.copy(a_filename)
			from
				i := buffer.first_index_of('/')
			until
				not buffer.valid_index(i) or else last_dirname = Void
			loop
				if i-1 > j+1 then
					dirname := once ""
					dirname.copy(buffer)
					dirname.shrink(j+1, i-1)
					bd.compute_subdirectory_with(last_dirname, dirname)
					if bd.last_entry.is_empty then
						std_error.put_string(once "*** Could not compute the subdirectory ")
						std_error.put_string(dirname)
						std_error.put_string(once" of ")
						std_error.put_line(last_dirname)
						last_dirname := Void
					else
						last_dirname.copy(bd.last_entry)
						bd.connect_to(last_dirname)
						if bd.is_connected then
							Result := Result + 1
							bd.disconnect
						elseif bd.create_new_directory(last_dirname) then
							Result := Result + 1
						else
							std_error.put_string(once "*** Could not create ")
							std_error.put_line(last_dirname)
							last_dirname := Void
						end
					end
				end
				j := i
				i := buffer.index_of('/', j+1)
			end
		end

	last_dirname: STRING

	basename_of (a_filename: STRING): STRING is
		local
			i: INTEGER
		do
			i := a_filename.last_index_of('/')
			if a_filename.valid_index(i) then
				Result := once ""
				Result.copy(a_filename)
				Result.shrink(i+1, Result.count)
			else
				Result := a_filename
			end
		end

	filename_of (a_class: CLASS_TEXT; a_client: CLASS_NAME): STRING is
		require
			a_class /= Void
		local
			i: INTEGER; n: STRING; c, k: CHARACTER
		do
			-- We add underscores at the begining of directory names in the cluster name part to be sure not to
			-- have clashes between directory names and class names in case-insensitive file systems.
			-- We also remove double slashes

			Result := once ""
			from
				n := a_class.cluster.name
				Result.copy(once "api/")
				c := n.first
				Result.extend(c)
				i := 2
			until
				i > n.count
			loop
				k := n.item(i)
				if k = ':' then
					k := '/'
				end
				if c = '/' then
					if k /= '/' then
						Result.extend(k)
					end
				elseif k = '/' then
					Result.append(once ".d/")
				else
					Result.extend(k)
				end
				c := k
				i := i + 1
			end
			if c /= '/' then
				Result.append(once ".d/")
			end
			Result.append(a_class.name.to_string)
			if a_client /= Void then
				Result.extend('/')
				Result.append(a_client.to_string)
			end
			Result.append(once ".html")
		end

	class_name_any: CLASS_NAME is
		local
			hs: HASHED_STRING
		once
			hs := string_aliaser.hashed_string(as_any)
			create Result.unknown_position(hs)
		end

	class_name_tuple: CLASS_NAME is
		local
			hs: HASHED_STRING
		once
			hs := string_aliaser.hashed_string(as_tuple)
			create Result.unknown_position(hs)
		end

end -- class EIFFELDOC_GLOBALS
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2005: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
