-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class COMPOUND
	--
	-- An ordered sequence of INSTRUCTIONs (natural indexing from 1 to `count').
	--

inherit
	INSTRUCTION
		redefine
			simplify_routine_body_first
		end
	
creation {EIFFEL_PARSER, COMPOUND, ASSIGNMENT_ATTEMPT, INTROSPECTION_HANDLER}
	make_2

creation {EIFFEL_PARSER, MANIFEST_STRING_INSPECTOR, INTROSPECTION_HANDLER}
	make_3

creation {EIFFEL_PARSER}
	make_4, make_n

feature {ANY}
	start_position: POSITION is
		do
			Result := list.first.start_position
		end

	safety_check (type: TYPE) is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).safety_check(type)
				i := i - 1
			end
		end

	simplify (type: TYPE): INSTRUCTION is
		local
			inst1, inst2: INSTRUCTION; i, i2: INTEGER
		do
			from
				i := list.lower
			until
				i > list.upper or else inst1 /= inst2
			loop
				inst1 := list.item(i)
				inst2 := inst1.simplify(type)
				i := i + 1
			end
			if inst1 = inst2 then
				-- Nothing done by all `simplify' calls:
				Result := Current
			else
				from
					from
						i2 := list.lower
					until
						i2 = i - 1
					loop
						Result := simplify_add(Result, list.item(i2))
						i2 := i2 + 1
					end
					-- Add the one at i - 1:
					Result := simplify_add(Result, inst2)
				until
					i > list.upper
				loop
					Result := simplify_add(Result, list.item(i).simplify(type))
					i := i + 1
				end
			end
		end

	compile_to_c (type: TYPE) is
		local
			i, upper: INTEGER; instruction: INSTRUCTION
		do
			from
				i := list.lower
				upper := list.upper
			until
				i > upper
			loop
				instruction := list.item(i)
				if ace.boost then
					-- Only useful to debug:
					cpp.put_position_comment(instruction.start_position)
					cpp.pending_c_function_body.extend('%N')
				end
				instruction.compile_to_c_with_internal_c_local_saving(type)
				i := i + 1
			end
		end

	compile_to_jvm (type: TYPE) is
		local
			i, upper: INTEGER; instruction: INSTRUCTION; trace: BOOLEAN; ca: like code_attribute
		do
			from
				i := list.lower
				upper := list.upper
				ca := code_attribute
				trace := start_position.sedb_trace
			until
				i > upper
			loop
				instruction := list.item(i)
				line_number_table.add(ca.program_counter, instruction.start_position)
				if trace then
					ca.se_trace(type.canonical_type_mark, instruction.start_position)
				end
				instruction.compile_to_jvm(type)
				i := i + 1
			end
		end

	use_current (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				Result or else i < list.lower
			loop
				Result := list.item(i).use_current(type)
				i := i - 1
			end
		end

	has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			Result := True
			from
				i := list.upper
			until
				not Result or else i < list.lower
			loop
				Result := list.item(i).has_been_specialized
				i := i - 1
			end
		end

	specialize_in (type: TYPE): like Current is
		local
			l: like list; inst1, inst2: INSTRUCTION; i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower or else inst1 /= inst2
			loop
				inst1 := list.item(i)
				inst2 := inst1.specialize_in(type)
				i := i - 1
			end
			if inst1 = inst2 then
				Result := Current
			else
				Result := twin
				l := list.twin
				Result.set_list(l)
				from
					l.put(inst2, i + 1)
				until
					i < l.lower
				loop
					l.put(l.item(i).specialize_in(type), i)
					i := i - 1
				end
			end
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			l: like list; inst1, inst2: INSTRUCTION; i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower or else inst1 /= inst2
			loop
				inst1 := list.item(i)
				inst2 := inst1.specialize_thru(parent_type, parent_edge, new_type)
				i := i - 1
			end
			if inst1 = inst2 then
				Result := Current
			else
				Result := twin
				l := list.twin
				Result.set_list(l)
				from
					l.put(inst2, i + 1)
				until
					i < l.lower
				loop
					l.put(l.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
					i := i - 1
				end
			end
		end

	specialize_2 (type: TYPE): like Current is
		local
			l: like list; inst1, inst2: INSTRUCTION; i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower or else inst1 /= inst2
			loop
				inst1 := list.item(i)
				inst2 := inst1.specialize_2(type)
				i := i - 1
			end
			if inst1 = inst2 then
				Result := Current
			else
				Result := twin
				l := list.twin
				Result.set_list(l)
				from
					l.put(inst2, i + 1)
				until
					i < l.lower
				loop
					l.put(l.item(i).specialize_2(type), i)
					i := i - 1
				end
			end
		end

	adapt_for (type: TYPE): like Current is
		local
			l: like list; inst1, inst2: INSTRUCTION; i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower or else inst1 /= inst2
			loop
				inst1 := list.item(i)
				inst2 := inst1.adapt_for(type)
				i := i - 1
			end
			if inst1 = inst2 then
				Result := Current
			else
				Result := twin
				l := list.twin
				Result.set_list(l)
				from
					l.put(inst2, i + 1)
				until
					i < l.lower
				loop
					l.put(l.item(i).adapt_for(type), i)
					i := i - 1
				end
			end
		end

	end_mark_comment: BOOLEAN is False

	pretty (indent_level: INTEGER) is
		local
			i: INTEGER; instruction, previous_instruction: INSTRUCTION; lnbc: CHARACTER
			extra_semicolon_flag: BOOLEAN; comment: COMMENT; index_in_pretty_printer: INTEGER
		do
			pretty_printer.set_semi_colon_flag(True)
			i := list.lower
			instruction := list.item(i)
			-- The first one never need to be preceded by an extra ';':
			pretty_printer.set_indent_level(indent_level)
			instruction.pretty(indent_level)
			from
				i := i + 1
			until
				i > list.upper
			loop
				if pretty_printer.zen_mode and then i = list.upper then
					pretty_printer.set_semi_colon_flag(False)
				end
				previous_instruction := instruction
				instruction := list.item(i)
				-- Computing `extra_semicolon_flag' first:
				lnbc := pretty_printer.last_non_blank_character
				if lnbc = ';' then
					-- The ';' is already there.
					extra_semicolon_flag := False
				elseif lnbc = ')' then
					-- The previous got arguments, so no risk of ambiguity.
					extra_semicolon_flag := False
				elseif previous_instruction.end_mark_comment then
					-- No need of extra ';' after such a statement.
					extra_semicolon_flag := False
				elseif is_a_comment(previous_instruction) then
					-- No need of extra ';' after a comment.
					extra_semicolon_flag := False
				else
					extra_semicolon_flag := True
				end
				-- To be sure that an INSTRUCTION comment is well indented:
				pretty_printer.set_indent_level(indent_level)
				if extra_semicolon_flag then
					index_in_pretty_printer := pretty_printer.index_of_last_character
				end
				instruction.pretty(indent_level)
				if extra_semicolon_flag and then
					pretty_printer.is_opening_bracket_after(index_in_pretty_printer)
				 then
					pretty_printer.erase_everything_after(index_in_pretty_printer)
					pretty_printer.put_character(';')
					index_in_pretty_printer := pretty_printer.index_of_last_character					
					instruction.pretty(indent_level)
					pretty_printer.erase_separators_after(index_in_pretty_printer)
				end
				-- Taking care of the following comment:
				i := i + 1
				if (not ({COMMENT} ?:= instruction)) and then i <= list.upper then
					comment ?= list.item(i)
					if comment /= Void and then
						comment.start_position.line = instruction.start_position.line and then
						comment.count = 1
					 then
						-- The following comment is associated to the current `instruction'.
						pretty_printer.prepare_for_same_line_comment
						comment.pretty(indent_level)
						i := i + 1
					end
				end
			end
		end

	side_effect_free (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
				Result := True
			until
				not Result or else i <= list.lower
			loop
				Result := list.item(i).side_effect_free(type)
				i := i - 1
			end
		end

	collect (type: TYPE) is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).collect(type)
				i := i - 1
			end
		end

	accept (visitor: COMPOUND_VISITOR) is
		do
			visitor.visit_compound(Current)
		end

feature {EIFFEL_PARSER}
	add_last (instruction: INSTRUCTION) is
		require
			instruction.is_not_a_compound
		do
			list.add_last(instruction)
		ensure
			list.count = 1 + old list.count
		end

feature {COMPOUND, COMPOUND_VISITOR}
	list: FAST_ARRAY[INSTRUCTION]
			-- Contains at least 2 INSTRUCTIONs (just because this is the canonical form to be enforced).

feature {COMPOUND}
	set_list (l: like list) is
		require
			l.count >= 2
		do
			list := l
		ensure
			list = l
		end

feature {ANONYMOUS_FEATURE, INSTRUCTION}
	simplify_routine_body_first (type: TYPE; return_type: TYPE): INSTRUCTION is
		local
			i1, i2: INSTRUCTION; l: like list; compound: COMPOUND; i: INTEGER
		do
			Result := Current
			i1 := list.first
			i2 := i1.simplify_routine_body_first(type, return_type)
			if i1 /= i2 then
				if i2 = Void then
					if list.count = 2 then
						Result := list.last
					else
						compound := twin
						l := list.twin
						l.remove_first
						compound.set_list(l)
						Result := compound
					end
				else
					compound ?= i2
					if compound = Void then
						compound := twin
						l := list.twin
						l.put(i2, l.lower)
						compound.set_list(l)
						Result := compound
					else
						from
							l := compound.list.twin
							i := list.lower + 1
						until
							i > list.upper
						loop
							l.add_last(list.item(i))
							i := i + 1
						end
						compound := twin
						compound.set_list(l)
						Result := compound
					end
				end
			end
		end
	
feature {}
	make_2 (i1, i2: INSTRUCTION) is
		require
			i1.is_not_a_compound
			i2.is_not_a_compound
		do
			create list.make(2)
			list.put(i1, 0)
			list.put(i2, 1)
		ensure
			list.count = 2
			list.count = list.capacity
		end

	make_3 (i1, i2, i3: INSTRUCTION) is
		require
			i1.is_not_a_compound
			i2.is_not_a_compound
			i3.is_not_a_compound
		do
			create list.make(3)
			list.put(i1, 0)
			list.put(i2, 1)
			list.put(i3, 2)
		ensure
			list.count = 3
			list.count = list.capacity
		end

	make_4 (i1, i2, i3, i4: INSTRUCTION) is
		require
			i1.is_not_a_compound
			i2.is_not_a_compound
			i3.is_not_a_compound
			i4.is_not_a_compound
		do
			create list.make(4)
			list.put(i1, 0)
			list.put(i2, 1)
			list.put(i3, 2)
			list.put(i4, 3)
		ensure
			list.count = 4
			list.count = list.capacity
		end

	make_n (i1, i2, i3, i4, i5: INSTRUCTION) is
		require
			i1.is_not_a_compound
			i2.is_not_a_compound
			i3.is_not_a_compound
			i4.is_not_a_compound
			i5.is_not_a_compound
		do
			create list.with_capacity(8)
			list.add_last(i1)
			list.add_last(i2)
			list.add_last(i3)
			list.add_last(i4)
			list.add_last(i5)
		ensure
			list.count = 5
			list.count < list.capacity
		end

	canonical_form (l: like list): BOOLEAN is
		require
			canonical_form(l)
		local
			i: INTEGER
		do
			Result := True
			from
				i := l.upper
			until
				not Result or else i < l.lower
			loop
				Result := l.item(i).is_not_a_compound
				i := i - 1
			end
		ensure
			assertion_check_only: Result
		end

	simplify_add (head: INSTRUCTION; new_item: INSTRUCTION): INSTRUCTION is
			-- The `Result' is the next `head'. (Optimal in memory, but
			-- not in execution time, but at least, it is still readable ;-)
		local
			compound: COMPOUND; i: INTEGER; l: like list
		do
			if new_item = Void then
				Result := head
			else
				compound ?= new_item
				if compound /= Void then
					from
						Result := head
						l := compound.list
						i := l.lower
					until
						i > l.upper
					loop
						Result := simplify_add(Result, l.item(i))
						i := i + 1
					end
				elseif head = Void then
					Result := new_item
				else
					compound ?= head
					if compound /= Void then
						Result := head
						compound.list.add_last(new_item)
					else
						create {COMPOUND} Result.make_2(head, new_item)
					end
				end
			end
		ensure
			head /= Void implies Result /= Void
			new_item /= Void implies Result /= Void
		end

	is_a_comment (instruction: INSTRUCTION): BOOLEAN is
		local
			comment: COMMENT
		do
			comment ?= instruction
			Result := comment /= Void
		end

invariant
	list.count >= 2

	canonical_form(list)

end -- class COMPOUND
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
