-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class IFTHENELSE
	--
	-- The most general form of the if-then-else INSTRUCTION. Actually, if-then-else INSTRUCTIONs are handled
	-- with IFTHENELSE but also with IFTHEN. Indeed, when we have a single if-then without else part, the
	-- parser creates only an IFTHEN object. When we have an "else" part or when there are at least one
	-- "elseif", the general IFTHENELSE is used.
	--

inherit
	INSTRUCTION
		redefine
			simplify_routine_body_first
		end

insert
	IF_SUPPORT

creation {EIFFEL_PARSER, IFTHENELSE, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	ifthenelse

creation {EIFFEL_PARSER}
	ifelseif

creation {IFTHENELSE}
	ifelseifelse

feature {ANY}
	elseif_list: FAST_ARRAY[IFTHEN]
			-- Non Void when there are "else if" items.

	else_position: POSITION
			-- Of the "else" keyword if any.

	else_compound: INSTRUCTION
			-- Not Void if any.

	use_current (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			Result := expression.use_current(type)
			if not Result and then then_compound /= Void then
				Result := then_compound.use_current(type)
			end
			if not Result and then elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					Result or else i < elseif_list.lower
				loop
					Result := elseif_list.item(i).use_current(type)
					i := i - 1
				end
			end
			if not Result and then else_compound /= Void then
				Result := else_compound.use_current(type)
			end
		end

	safety_check (type: TYPE) is
		local
			i: INTEGER
		do
			expression.safety_check(type)
			if then_compound /= Void then
				then_compound.safety_check(type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower
				loop
					elseif_list.item(i).safety_check(type)
					i := i - 1
				end
			end
			if else_compound /= Void then
				else_compound.safety_check(type)
			end
		end

	compile_to_c (type: TYPE) is
		local
			i: INTEGER; ifthen: IFTHEN; c: INSTRUCTION; trace: BOOLEAN
		do
			trace := ace.no_check
			cpp.pending_c_function_body.append(once "if(")
			if trace then
				cpp.trace_boolean_expression(type, expression)
			else
				expression.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "){%N")
			if then_compound /= Void then
				then_compound.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "}%N")
			if elseif_list /= Void then
				from
					i := elseif_list.lower
				until
					i > elseif_list.upper
				loop
					ifthen := elseif_list.item(i)
					cpp.pending_c_function_body.append(once "else if(")
					if trace then
						cpp.trace_boolean_expression(type, ifthen.expression)
					else
						ifthen.expression.compile_to_c_with_internal_c_local_saving(type)
					end
					cpp.pending_c_function_body.append(once "){%N")
					c := ifthen.then_compound
					if c /= Void then
						c.compile_to_c_with_internal_c_local_saving(type)
					end
					cpp.pending_c_function_body.append(once "}%N")
					i := i + 1
				end
			end
			if else_compound /= Void then
				cpp.pending_c_function_body.append(once "else{%N")
				else_compound.compile_to_c_with_internal_c_local_saving(type)
				cpp.pending_c_function_body.append(once "}%N")
			end
		end

	compile_to_jvm (type: TYPE) is
		local
			i: INTEGER; ca: like code_attribute
		do
			ca := code_attribute
			compile_to_jvm_(type)
			if elseif_list /= Void then
				from
					i := elseif_list.lower
				until
					i > elseif_list.upper
				loop
					elseif_list.item(i).compile_to_jvm_(type)
					i := i + 1
				end
			end
			if else_compound /= Void then
				else_compound.compile_to_jvm(type)
			end
			-- Resolving all `jvm_after_point' which must go to the end:
			ca.resolve_u2_branch(jvm_after_point)
			if elseif_list /= Void then
				from
					i := elseif_list.lower
				until
					i > elseif_list.upper
				loop
					ca.resolve_u2_branch(elseif_list.item(i).jvm_after_point)
					i := i + 1
				end
			end
		end

	specialize_in (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound; eil: like elseif_list; it1, it2: IFTHEN; i: INTEGER
			ec: like else_compound
		do
			e := expression.specialize_in(type)
			if then_compound /= Void then
				tc := then_compound.specialize_in(type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower or else it1 /= it2
				loop
					it1 := elseif_list.item(i)
					it2 := it1.specialize_in(type)
					i := i - 1
				end
				if it1 = it2 then
					eil := elseif_list
				else
					from
						eil := elseif_list.twin
						eil.put(it2, i + 1)
					until
						i < eil.lower
					loop
						eil.put(elseif_list.item(i).specialize_in(type), i)
						i := i - 1
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_in(type)
			end
			Result := current_or_twin_init(e, tc, eil, ec)
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound; eil: like elseif_list; it1, it2: IFTHEN; i: INTEGER
			ec: like else_compound
		do
			e := expression.specialize_thru(parent_type, parent_edge, new_type)
			if then_compound /= Void then
				tc := then_compound.specialize_thru(parent_type, parent_edge, new_type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower or else it1 /= it2
				loop
					it1 := elseif_list.item(i)
					it2 := it1.specialize_thru(parent_type, parent_edge, new_type)
					i := i - 1
				end
				if it1 = it2 then
					eil := elseif_list
				else
					from
						eil := elseif_list.twin
						eil.put(it2, i + 1)
					until
						i < eil.lower
					loop
						eil.put(elseif_list.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
						i := i - 1
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(e, tc, eil, ec)
		end

	specialize_2 (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound; eil: like elseif_list; it1, it2: IFTHEN; i: INTEGER
			ec: like else_compound
		do
			e := expression.specialize_2(type)
			if then_compound /= Void then
				tc := then_compound.specialize_2(type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower or else it1 /= it2
				loop
					it1 := elseif_list.item(i)
					it2 := it1.specialize_2(type)
					i := i - 1
				end
				if it1 = it2 then
					eil := elseif_list
				else
					from
						eil := elseif_list.twin
						eil.put(it2, i + 1)
					until
						i < eil.lower
					loop
						eil.put(elseif_list.item(i).specialize_2(type), i)
						i := i - 1
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_2(type)
			end
			Result := current_or_twin_init(e, tc, eil, ec)
			Result.specialize_2_check(type)
		end

	has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			Result := expression.has_been_specialized
			if then_compound /= Void then
				Result := then_compound.has_been_specialized
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					not Result or else i < elseif_list.lower
				loop
					Result := elseif_list.item(i).has_been_specialized
					i := i - 1
				end
			end
			if else_compound /= Void then
				Result := else_compound.has_been_specialized
			end
		end

	simplify (type: TYPE): INSTRUCTION is
		local
			e: like expression; tc: like then_compound; eil: like elseif_list; ifthen: IFTHEN
			ec: like else_compound; bc: BOOLEAN_CONSTANT; ep: like else_position
		do
			e := expression.simplify(type)
			bc ?= e
			if bc /= Void then
				smart_eiffel.magic_count_increment
				if bc.value then
					if then_compound /= Void then
						Result := then_compound.simplify(type)
					end
				elseif elseif_list = Void then
					if else_compound /= Void then
						Result := else_compound.simplify(type)
					end
				else
					ifthen := elseif_list.first
					e := ifthen.expression.simplify(type)
					tc := ifthen.then_compound
					if elseif_list.count >= 2 then
						eil := elseif_list.twin
						eil.remove_first
						eil := simplify_elseif_list(type, True, eil)
					end
					ec := else_compound
					if eil /= Void then
						bc ?= eil.last.expression
						if bc /= Void then
							check bc.value end -- would have been removed if it was False
							ec := eil.last.then_compound
							if eil.count = 1 then
								eil := Void
							else
								if eil = elseif_list then
									eil := elseif_list.twin
								end
								eil.remove_last
							end
						end
					end
					if ec /= Void then
						ep := ec.start_position
						ec := ec.simplify(type)
					end
					if eil = Void then
						if ec = Void then
							create {IFTHEN} Result.make(ifthen.start_position, e, tc)
						else
							create {IFTHENELSE} Result.ifthenelse(ifthen.start_position, e, tc, ep, ec)
						end
					else
						create {IFTHENELSE} Result.ifelseifelse(ifthen.start_position, e, tc, eil, ep, ec)
					end
				end
			else
				if then_compound /= Void then
					tc := then_compound.simplify(type)
				end
				if elseif_list /= Void then
					eil := simplify_elseif_list(type, False, elseif_list)
				end
				ec := else_compound
				if eil /= Void then
					bc ?= eil.last.expression
					if bc /= Void then
						check bc.value end -- would have been removed if it was False
						smart_eiffel.magic_count_increment
						ec := eil.last.then_compound
						if eil.count = 1 then
							eil := Void
						else
							if eil = elseif_list then
								eil := elseif_list.twin
							end
							eil.remove_last
						end
					end
				end
				if ec /= Void then
					ec := ec.simplify(type)
				end
				Result := current_or_twin_init(e, tc, eil, ec)
			end
		end

	pretty (indent_level: INTEGER) is
		local
			i: INTEGER
		do
			pretty_(indent_level, once "if")
			if elseif_list /= Void then
				from
					i := elseif_list.lower
				until
					i > elseif_list.upper
				loop
					elseif_list.item(i).pretty_(indent_level, once "elseif")
					i := i + 1
				end
			end
			if else_compound /= Void then
				pretty_printer.set_indent_level(indent_level)
				pretty_printer.put_string(once "else")
				pretty_printer.set_indent_level(indent_level + 1)
				else_compound.pretty(indent_level + 1)
			end
			pretty_end_if(indent_level)
		end

	accept (visitor: IFTHENELSE_VISITOR) is
		do
			visitor.visit_ifthenelse(Current)
		end

	collect (type: TYPE) is
		local
			dummy: TYPE; i: INTEGER
		do
			dummy := expression.collect(type)
			if then_compound /= Void then
				then_compound.collect(type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower
				loop
					elseif_list.item(i).collect(type)
					i := i - 1
				end
			end
			if else_compound /= Void then
				else_compound.collect(type)
			end
		end

	adapt_for (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound; eil: like elseif_list; it1, it2: IFTHEN; i: INTEGER
			ec: like else_compound; bc: BOOLEAN_CONSTANT
		do
			e := expression.adapt_for(type)
			if then_compound /= Void then
				tc := then_compound.adapt_for(type)
			end
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower or else it1 /= it2
				loop
					it1 := elseif_list.item(i)
					it2 := it1.adapt_for(type)
					i := i - 1
				end
				if it1 = it2 then
					eil := elseif_list
				else
					from
						eil := elseif_list.twin
						eil.put(it2, i + 1)
					until
						i < eil.lower
					loop
						eil.put(elseif_list.item(i).adapt_for(type), i)
						i := i - 1
					end
				end
			end
			ec := else_compound
			if eil /= Void then
				bc ?= eil.last.expression
				if bc /= Void and then bc.value then
					check False end --|*** PH 06/05/05
					-- Finally the `else_compound' cannot be reached:
					ec := eil.last.then_compound
					eil.remove_last
					if eil.is_empty then
						eil := Void
					end
					-- Note: this simplification must not be done during simplify because the very first
					-- expression may be detected False in some inner `simplify' step.
					--|*** PH 05/05/05 I can't understand...
				end
			end
			if ec /= Void then
				ec := ec.adapt_for(type)
			end
			Result := current_or_twin_init(e, tc, eil, ec)
		end

feature {EIFFEL_PARSER}
	set_else (ep: like else_position; ec: like else_compound) is
		require
			not ep.is_unknown
		do
			else_position := ep
			else_compound := ec
		ensure
			else_position = ep
			else_compound = ec
		end

	add_elseif (ifthen: IFTHEN) is
		require
			ifthen /= Void
		do
			elseif_list.add_last(ifthen)
		end

feature {ANONYMOUS_FEATURE, INSTRUCTION}
	simplify_routine_body_first (type: TYPE; return_type: TYPE): INSTRUCTION is
		do
			if elseif_list = Void and then else_compound = Void then
				-- Moving to canonical IFTHEN:
				smart_eiffel.magic_count_increment
				create {IFTHEN} Result.make(start_position, expression, then_compound)
			else
				Result := Current
			end
		end
	
feature {IFTHENELSE}
	specialize_2_check (type: TYPE) is
		local
			i: INTEGER
		do
			specialize_2_check_(type)
			if elseif_list /= Void then
				from
					i := elseif_list.upper
				until
					i < elseif_list.lower
				loop
					elseif_list.item(i).specialize_2_check_(type)
					i := i - 1
				end
			end
		end

	init (e: like expression; tc: like then_compound; eil: like elseif_list; ec: like else_compound) is
		do
			expression := e
			then_compound := tc
			elseif_list := eil
			else_compound := ec
			if ec /= Void then
				else_position := ec.start_position
			end
		ensure
			expression = e
			then_compound = tc
			elseif_list = eil
			else_compound = ec
		end

feature {}
	current_or_twin_init (e: like expression; tc: like then_compound; eil: like elseif_list; ec: like else_compound): like Current is
		require
			e /= Void
		do
			if e = expression and then tc = then_compound and then eil = elseif_list and then ec = else_compound then
				Result := Current
			else
				Result := twin
				Result.init(e, tc, eil, ec)
			end
		ensure
			Result.expression = e
			Result.then_compound = tc
			Result.elseif_list = eil
			Result.else_compound = ec
		end

	ifthenelse (sp: like start_position; e: like expression; tc: like then_compound; ep: like else_position
		ec: like else_compound) is
		require
			not sp.is_unknown
			e /= Void
			not ep.is_unknown
		do
			start_position := sp
			expression := e
			then_compound := tc
			else_position := ep
			else_compound := ec
		ensure
			start_position = sp
			expression = e
			then_compound = tc
			else_position = ep
			else_compound = ec
		end

	ifelseif (sp: like start_position; e: like expression; tc: like then_compound; ifthen: IFTHEN) is
		require
			not sp.is_unknown
			e /= Void
			ifthen /= Void
		do
			start_position := sp
			expression := e
			then_compound := tc
			create elseif_list.with_capacity(2)
			elseif_list.add_last(ifthen)
		ensure
			start_position = sp
			expression = e
			then_compound = tc
		end

	ifelseifelse (sp: like start_position; e: like expression; tc: like then_compound; eil: like elseif_list
		ep: like else_position; ec: like else_compound) is
		require
			not sp.is_unknown
			e /= Void
			eil /= Void
			not ep.is_unknown
		do
			start_position := sp
			expression := e
			then_compound := tc
			elseif_list := eil
			else_position := ep
			else_compound := ec
		ensure
			start_position = sp
			expression = e
			then_compound = tc
			elseif_list = eil
			else_position = ep
			else_compound = ec
		end

	simplify_elseif_list (type: TYPE; twin_already_done: BOOLEAN; eil: like elseif_list): like elseif_list is
		require
			type /= Void
			eil.count >= 1
		local
			twin_done: BOOLEAN; e1, e2: EXPRESSION; inst1, inst2: INSTRUCTION; ifthen: IFTHEN; i: INTEGER
			bc: BOOLEAN_CONSTANT
		do
			from
				Result := eil
				twin_done := twin_already_done
				i := Result.lower
			until
				i > Result.upper
			loop
				ifthen := Result.item(i)
				e1 := ifthen.expression
				e2 := e1.simplify(type)
				inst1 := ifthen.then_compound
				if inst1 = Void then
					inst2 := Void
				else
					inst2 := inst1.simplify(type)
				end
				bc ?= e2
				if bc /= Void then
					if bc.value then
						-- All items after the current one are not reachable:
						if Result.upper > i then
							smart_eiffel.magic_count_increment
							if not twin_done then
								Result := Result.twin
								twin_done := True
							end
							Result.remove_tail(Result.upper - i)
							check
								Result.upper = i
							end
						end
						if e1 /= e2 or else inst1 /= inst2 then
							smart_eiffel.magic_count_increment
							if not twin_done then
								Result := Result.twin
								twin_done := True
							end
							Result.put(create {IFTHEN}.make(ifthen.start_position, e2, inst2), i)
						end
					else
						-- The current item can be removed:
						smart_eiffel.magic_count_increment
						if not twin_done then
							Result := Result.twin
							twin_done := True
						end
						Result.remove(i)
						i := i - 1
					end
				elseif e1 /= e2 or else inst1 /= inst2 then
					if not twin_done then
						Result := Result.twin
						twin_done := True
					end
					Result.put(create {IFTHEN}.make(ifthen.start_position, e2, inst2), i)
				end
				i := i + 1
			end
			if Result.is_empty then
				Result := Void
			end
		ensure
			(Result = elseif_list) = (smart_eiffel.magic_count = old smart_eiffel.magic_count)
		end

invariant
	elseif_list /= Void implies not elseif_list.is_empty
	
	else_compound /= Void implies not else_position.is_unknown

end -- class IFTHENELSE
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
