-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class WHEN_ITEM_2
	--
	-- To store a slice value of a when clause in an inspect instruction.
	--
	-- Exemple :
	--          inspect ...
	--              when foo .. bar, then ...
	--

inherit
	WHEN_ITEM

creation {WHEN_CLAUSE}
	make

feature {ANY}
	lower, upper: EXPRESSION
			-- The written slice.

	lower_manifest_expression, upper_manifest_expression: MANIFEST_EXPRESSION
			-- The corresponding values for `lower' and `upper' (they may be calls).

	lower_value, upper_value: INTEGER
			-- Corresponding values for this slice.

	start_position: POSITION is
		do
			Result := lower.start_position
		end

	accept (visitor: WHEN_ITEM_2_VISITOR) is
		do
			visitor.visit_when_item_2(Current)
		end

feature {WHEN_ITEM_2}
	init (l: like lower; u: like upper) is
		require
			l /= Void
			u /= Void
		do
			lower := l
			upper := u
		ensure
			lower = l
			upper = u
		end

	set_lower_upper_manifest_expression (lookup_type: TYPE) is
		do
			lower_manifest_expression := manifest_expression_check(lower, lookup_type)
			upper_manifest_expression := manifest_expression_check(upper, lookup_type)
		ensure
			lower_manifest_expression /= Void
			upper_manifest_expression /= Void
		end

	set_values_character is
		require
			lower_manifest_expression /= Void
			upper_manifest_expression /= Void
		do
			lower_value := character_check(lower, lower_manifest_expression)
			upper_value := character_check(upper, upper_manifest_expression)
		end

	set_values_integer is
		require
			lower_manifest_expression /= Void
			upper_manifest_expression /= Void
		do
			lower_value := integer_check(lower, lower_manifest_expression)
			upper_value := integer_check(upper, upper_manifest_expression)
		end

feature {WHEN_CLAUSE}
	validity_check_continued_ (occurrence_1: WHEN_ITEM) is
		do
			lower_upper_value_check
			occurrence_1.validity_check_continued_when_item_2(Current)
		end

	match_value (v: INTEGER): BOOLEAN is
		do
			Result := v.in_range(lower_value, upper_value)
		end

	collect (t: TYPE) is
		local
			dummy: TYPE
		do
			if lower /= Void then
				dummy := lower.collect(t)
			end
			if upper /= Void then
				dummy := upper.collect(t)
			end
		end

	adapt_for (t: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.adapt_for(t), upper.adapt_for(t))
		end

	compute_values (container: WHEN_CLAUSE) is
		do
			container.compute_values_with_item_2(Current)
		end

	force_internal_character_values (type: TYPE; container: WHEN_CLAUSE) is
		do
			set_lower_upper_manifest_expression(type)
			set_values_character
			lower_upper_value_check
			compute_values(container)
		end

	force_internal_integer_values (type: TYPE; container: WHEN_CLAUSE) is
		do
			set_lower_upper_manifest_expression(type)
			set_values_integer
			lower_upper_value_check
			compute_values(container)
		end

feature {}
	make (l, u: EXPRESSION) is
		require
			l /= Void
			u /= Void
		do
			lower := l
			upper := u
		ensure
			lower = l
			upper = u
		end

feature {WHEN_CLAUSE, WHEN_ITEM}
	specialize_in (new_type: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.specialize_in(new_type), upper.specialize_in(new_type))
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.specialize_thru(parent_type, parent_edge, new_type), upper.specialize_thru(parent_type, parent_edge, new_type))
		end

	specialize_2_character (type: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.specialize_2(type), upper.specialize_2(type))
			Result.set_lower_upper_manifest_expression(type)
			Result.set_values_character
			Result.lower_upper_value_check
		end

	specialize_2_integer (type: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.specialize_2(type), upper.specialize_2(type))
			Result.set_lower_upper_manifest_expression(type)
			Result.set_values_integer
			Result.lower_upper_value_check
		end

	specialize_2_string (type: TYPE): like Current is
		do
			Result := current_or_twin_init(lower.specialize_2(type), upper.specialize_2(type))
		end

	has_been_specialized: BOOLEAN is
		do
			Result := lower.has_been_specialized and then upper.has_been_specialized
		end

	pretty (indent_level: INTEGER) is
		do
			lower.pretty(indent_level)
			if not pretty_printer.zen_mode then
				pretty_printer.put_character(' ')
			end
			pretty_printer.put_string(once "..")
			if not pretty_printer.zen_mode then
				pretty_printer.put_character(' ')
			end
			upper.pretty(indent_level)
		end

feature {WHEN_ITEM_1}
	validity_check_continued_when_item_1 (occurrence_2: WHEN_ITEM_1) is
		do
			lower_upper_value_check
			if occurrence_2.expression_value.in_range(lower_value, upper_value) then
				error_handler.add_position(start_position)
				error_handler.add_position(occurrence_2.start_position)
				error_handler.append("The value is already part of previously encountered slice. (Wrong inspect statement.)")
				error_handler.print_as_fatal_error
			end
		end

feature {WHEN_ITEM_2}
	validity_check_continued_when_item_2 (occurrence_2: WHEN_ITEM_2) is
		do
			lower_upper_value_check
			if occurrence_2.lower_value.in_range(lower_value, upper_value) or else occurrence_2.upper_value.in_range(lower_value, upper_value) then
				error_handler.add_position(start_position)
				error_handler.add_position(occurrence_2.start_position)
				error_handler.append("Overlapping slices. (Wrong inspect statement.)")
				error_handler.print_as_fatal_error
			end
		end

	lower_upper_value_check is
		do
			if lower_value >= upper_value then
				error_handler.add_position(lower.start_position)
				error_handler.add_position(upper.start_position)
				error_handler.append("Not a good slice (lower bound is not smaller than upper bound).")
				error_handler.print_as_fatal_error
			end
		end

feature {}
	current_or_twin_init (l: like lower; u: like upper): like Current is
		require
			l /= Void
			u /= Void
		do
			if l = lower and then u = upper then
				Result := Current
			else
				Result := twin
				Result.init(l, u)
			end
		ensure
			Result.lower = l
			Result.upper = u
		end

invariant
	lower /= Void

	upper /= Void

end -- class WHEN_ITEM_2
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
