-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class AGENT_CREATION_HELPER

inherit
	VISITOR
	EXPRESSION_VISITOR
	EFFECTIVE_ARG_LIST_VISITOR

insert
	GLOBALS
	
creation {AGENT_CREATION}
	make

feature {EFFECTIVE_ARG_LIST}
	visit_effective_arg_list (visited: EFFECTIVE_ARG_LIST) is
		local
			i: INTEGER
		do
			inspect
				status
			when Under_agent_scope_notify_status,
				Resolve_open_operands_status, Search_open_argument_status
			 then
				-- We must do the loop from 1 to `count' in order to follow the evaluation order:
				from
					i := 1
				until
					i > visited.count
				loop
					last_rank := i
					visited.expression(i).accept(Current)
					i := i + 1
				end
			end
		end

	last_rank: INTEGER
	
feature {MANIFEST_TUPLE}
	visit_manifest_tuple (visited: MANIFEST_TUPLE) is
		local
			eal: EFFECTIVE_ARG_LIST; i: INTEGER
		do
			-- Note: a MANIFEST_TUPLE must be created together with the agent; one cannot delay the 
			-- MANIFEST_TUPLE  creation itself; well, using other words, a MANIFEST_TUPLE must be 
			-- captured  and cannot hold some open argument.
			-- The situation is the same in `visit_create_expression'.
			inspect
				status
			when Under_agent_scope_notify_status then
			when Resolve_open_operands_status then
				eal := visited.effective_arg_list
				if eal /= Void then
					from
						i := 1
					until
						i > eal.count
					loop
						search_open_argument_in(eal.expression(i))
						if last_open_argument /= Void then
							error_handler.add_position(last_open_argument.start_position)
							error_handler.append("Under %"agent%" scope, a manifest TUPLE creation (as %
                                          %well as any other create expression) must not hold an %
                                          %open argument. Actually, the expression ")
							error_handler.add_expression(visited)
							error_handler.append(" must be evaluated together with the agent. Manifest %
														%TUPLE creation cannot be delayed. Using ")
							error_handler.add_expression(last_open_argument)
							error_handler.append(" here is not allowed.")
							error_handler.print_as_fatal_error
						end
						i := i + 1
					end
				end
			when Search_open_argument_status then
				eal := visited.effective_arg_list
				if eal /= Void then
					from
						i := 1
					until
						i > eal.count
					loop
						search_open_argument_in(eal.expression(i))
						if last_open_argument /= Void then
							i := eal.count
						end
						i := i + 1
					end
				end
			end
		end

feature {FAKE_ARGUMENT}
	
	visit_fake_argument (visited: FAKE_ARGUMENT) is
		do
			check
				False
			end
		end

feature {GENERATOR_GENERATING_TYPE}
	visit_generator_generating_type (visited: GENERATOR_GENERATING_TYPE) is
		do
			check
				False
			end
		end

feature {FAKE_TUPLE}
	visit_fake_tuple (visited: FAKE_TUPLE) is
		do
			limitation_error(visited)
		end

feature {CREATE_EXPRESSION}
	visit_create_expression (visited: CREATE_EXPRESSION) is
		local
			proc_call: PROC_CALL; eal: EFFECTIVE_ARG_LIST; i: INTEGER
		do
			-- Note: a CREATE_EXPRESSION must be evaluated together with the agent; one cannot delay the 
			-- CREATE_EXPRESSION evaluation itself; well, using other words, a CREATE_EXPRESSION must be 
			-- captured and cannot hold some open argument.
			-- The situation is the same in `visit_manifest_tuple'.
			inspect
				status
			when Under_agent_scope_notify_status then
				sedb_breakpoint
			when Resolve_open_operands_status then
				if visited.call /= Void then
					search_open_argument_in(visited)
					if last_open_argument /= Void then
						error_handler.add_position(last_open_argument.start_position)
						error_handler.append("Under %"agent%" scope, a create expression must not hold an %
													%open argument. Actually, the expression ")
						error_handler.add_expression(visited)
						error_handler.append(" must be evaluated together with the agent. This create %
													%expression cannot be delayed. Using ")
						error_handler.add_expression(last_open_argument)
						error_handler.append(" here is not allowed.")
						error_handler.print_as_fatal_error
					end
				end
			when Search_open_argument_status then
				proc_call := visited.call
				if proc_call /= Void then
					eal := proc_call.arguments
					if eal /= Void then
						from
							i := 1
						until
							(i > eal.count) or else last_open_argument /= Void
						loop
							eal.expression(i).accept(Current)
							i := i + 1
						end
					end
				end
			end
		end

feature {AGENT_CREATION}
	visit_agent_creation (visited: AGENT_CREATION) is
		do
			-- Nothing to do because all will be correctly handled by the sub-expression.
		end

feature {PRECURSOR_EXPRESSION}
	visit_precursor_expression (visited: PRECURSOR_EXPRESSION) is
		do
			limitation_error(visited)
		end

feature {ADDRESS_OF}
	visit_address_of (visited: ADDRESS_OF) is
		do
		end

feature {ASSIGNMENT_TEST}
	visit_assignment_test (visited: ASSIGNMENT_TEST) is
		do
			limitation_error(visited)
		end

feature {AGENT_EXPRESSION}
	visit_agent_expression (visited: AGENT_EXPRESSION) is
		do
			limitation_error(visited)
		end

feature {FAKE_TARGET}
	visit_fake_target (visited: FAKE_TARGET) is
		do
			inspect
				status
			when Under_agent_scope_notify_status then
			when Resolve_open_operands_status then
			when Search_open_argument_status then
				last_open_argument := visited
			end
		end

feature {CREATE_WRITABLE}
	visit_create_writable (visited: CREATE_WRITABLE) is
		do
			check
				False
			end
		end

feature {OPEN_OPERAND}
	visit_open_operand (visited: OPEN_OPERAND) is
		do
			inspect
				status
			when Under_agent_scope_notify_status then
				visited.under_agent_scope_notify
			when Resolve_open_operands_status then
				if not visited.is_open_target then
					visited.notify_as_open_argument(type,
															  target_type_stack.top,
															  anonymous_feature_stack.top,
															  last_rank)
				end
			when Search_open_argument_status then
				last_open_argument := visited
			end
		end

feature {RESULT}
	visit_result (visited: RESULT) is
		do
		end

feature {IMPLICIT_CAST}
	visit_implicit_cast (visited: IMPLICIT_CAST) is
		do
			visited.expression.accept(Current)
		end

feature {EXPRESSION_WITH_COMMENT}
	visit_expression_with_comment (visited: EXPRESSION_WITH_COMMENT) is
		do
			visited.expression.accept(Current)
		end

feature {WRITABLE_ATTRIBUTE_NAME}
	visit_writable_attribute_name (visited: WRITABLE_ATTRIBUTE_NAME) is
		do
			check
				False
			end
		end

feature {STATIC_CALL_0_C}
	visit_static_call_0_c (visited: STATIC_CALL_0_C) is
		do
			visit_call(visited)
		end

feature {E_OLD}
	visit_e_old (visited: E_OLD) is
		do
			limitation_error(visited)
		end

feature {BUILT_IN_EQ_NEQ}
	visit_built_in_eq_neq (visited: BUILT_IN_EQ_NEQ) is
		do
			visited.left_side.accept(Current)
			visited.right_side.accept(Current)
		end

feature {}
	visit_fast_array_of_expression (list: FAST_ARRAY[EXPRESSION]) is
		require
			list /= Void
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).accept(Current)
				i := i - 1
			end
		end
	
feature {MANIFEST_GENERIC}
	visit_manifest_generic (visited: MANIFEST_GENERIC) is
		do
			if visited.optional_arguments /= Void then
				visit_fast_array_of_expression(visited.optional_arguments)
			end
			visit_fast_array_of_expression(visited.item_list)
		end

feature {OLD_MANIFEST_ARRAY}
	visit_old_manifest_array (visited: OLD_MANIFEST_ARRAY) is
		do
			check
				False -- (Because we have already switched to the canonical form.)
			end
		end

feature {VOID_CALL}
	visit_void_call (visited: VOID_CALL) is
		do
		end

feature {NULL_POINTER}
	visit_null_pointer (visited: NULL_POINTER) is
		do
		end

feature {NON_VOID_NO_DISPATCH}
	visit_non_void_no_dispatch (visited: NON_VOID_NO_DISPATCH) is
		do
		end

feature {}
	visit_abstract_current (visited: ABSTRACT_CURRENT) is
		do
		end

feature {IMPLICIT_CURRENT}
	visit_implicit_current (visited: IMPLICIT_CURRENT) is
		do
			visit_abstract_current(visited)
		end

feature {WRITTEN_CURRENT}
	visit_written_current (visited: WRITTEN_CURRENT) is
		do
			visit_abstract_current(visited)
		end

feature {LOCAL_NAME2}
	visit_local_name2 (visited: LOCAL_NAME2) is
		do
		end

feature {ARGUMENT_NAME2}
	visit_argument_name2 (visited: ARGUMENT_NAME2) is
		do
		end

feature {}
	visit_call (visited: CALL) is
		local
			target_type, target_declaration_type: TYPE; target: EXPRESSION
			fs: FEATURE_STAMP; arguments: EFFECTIVE_ARG_LIST			
		do
			inspect
				status
			when Under_agent_scope_notify_status,
				Search_open_argument_status
			 then
				visited.target.accept(Current)
				arguments := visited.arguments
				if arguments /= Void then
					arguments.accept(Current)
				end
			when Resolve_open_operands_status then
				-- To be sure to have the `feature_stamp' ready for use:
				target := visited.target.specialize_2(type)
				target_declaration_type := target.declaration_type
				fs := target_declaration_type.search(visited.feature_name)
				if fs = Void then
					smart_eiffel.unknown_feature_fatal_error(target, target.declaration_type, visited.feature_name)
				end
				target_type := target.resolve_in(type)
				fs := fs.resolve_static_binding_for(target.declaration_type, target_type)
				push_pop(target_type, fs.anonymous_feature(target_type), target, visited.arguments)
			end
		end
	
feature {CALL_0_C}
	visit_call_0_c (visited: CALL_0_C) is
		do
			visit_call(visited)
		end

feature {CALL_1_C}
	visit_call_1_c (visited: CALL_1_C) is
		do
			visit_call(visited)
		end

feature {CALL_N}
	visit_call_n (visited: CALL_N) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_INT_REM}
	visit_call_infix_int_rem (visited: CALL_INFIX_INT_REM) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_IMPLIES}
	visit_call_infix_implies (visited: CALL_INFIX_IMPLIES) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_INT_DIV}
	visit_call_infix_int_div (visited: CALL_INFIX_INT_DIV) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_OR_ELSE}
	visit_call_infix_or_else (visited: CALL_INFIX_OR_ELSE) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_OR}
	visit_call_infix_or (visited: CALL_INFIX_OR) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_LT}
	visit_call_infix_lt (visited: CALL_INFIX_LT) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_LE}
	visit_call_infix_le (visited: CALL_INFIX_LE) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_GT}
	visit_call_infix_gt (visited: CALL_INFIX_GT) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_GE}
	visit_call_infix_ge (visited: CALL_INFIX_GE) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_POWER}
	visit_call_infix_power (visited: CALL_INFIX_POWER) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_PLUS}
	visit_call_infix_plus (visited: CALL_INFIX_PLUS) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_XOR}
	visit_call_infix_xor (visited: CALL_INFIX_XOR) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_TIMES}
	visit_call_infix_times (visited: CALL_INFIX_TIMES) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_DIV}
	visit_call_infix_div (visited: CALL_INFIX_DIV) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_AND}
	visit_call_infix_and (visited: CALL_INFIX_AND) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_MINUS}
	visit_call_infix_minus (visited: CALL_INFIX_MINUS) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_FREEOP}
	visit_call_infix_freeop (visited: CALL_INFIX_FREEOP) is
		do
			visit_call(visited)
		end

feature {CALL_INFIX_AND_THEN}
	visit_call_infix_and_then (visited: CALL_INFIX_AND_THEN) is
		do
			visit_call(visited)
		end

feature {CALL_PREFIX_NOT}
	visit_call_prefix_not (visited: CALL_PREFIX_NOT) is
		do
			visit_call(visited)
		end

feature {CALL_PREFIX_PLUS}
	visit_call_prefix_plus (visited: CALL_PREFIX_PLUS) is
		do
			visit_call(visited)
		end

feature {CALL_PREFIX_FREEOP}
	visit_call_prefix_freeop (visited: CALL_PREFIX_FREEOP) is
		do
			visit_call(visited)
		end

feature {CALL_PREFIX_MINUS}
	visit_call_prefix_minus (visited: CALL_PREFIX_MINUS) is
		do
			visit_call(visited)
		end

feature {E_FALSE}
	visit_e_false (visited: E_FALSE) is
		do
		end

feature {E_TRUE}
	visit_e_true (visited: E_TRUE) is
		do
		end

feature {INTEGER_CONSTANT}
	visit_integer_constant (visited: INTEGER_CONSTANT) is
		do
		end

feature {REAL_CONSTANT}
	visit_real_constant (visited: REAL_CONSTANT) is
		do
		end

feature {CHARACTER_CONSTANT}
	visit_character_constant (visited: CHARACTER_CONSTANT) is
		do
		end

feature {MANIFEST_STRING}
	visit_manifest_string (visited: MANIFEST_STRING) is
		do
		end

feature {E_VOID}
	visit_e_void (visited: E_VOID) is
		do
		end

feature {NUMBERED_CAPTURE}
	visit_numbered_capture (visited: NUMBERED_CAPTURE) is
		do
		end

feature {AGENT_CREATION}
	is_open_operand (expression: EXPRESSION): BOOLEAN is
		require
			expression /= Void
		do
			Result := {OPEN_OPERAND} ?:= expression
		end
	
	syntactical_flat_check (delayed_call: CALL) is
			-- Check that the `delayed_call' is acceptable (no imbricated open arguments).
		require
			delayed_call /= Void
		local
			arguments: EFFECTIVE_ARG_LIST; i: INTEGER
		do
			syntactical_flat_check_(delayed_call, delayed_call.target)
			arguments := delayed_call.arguments
			if arguments /= Void then
				from
					i := 1
				until
					i > arguments.count
				loop
					syntactical_flat_check_(delayed_call, arguments.expression(i))
					i := i + 1
				end
			end

		end
	
	under_agent_scope_notify (expression: EXPRESSION) is
		require
			status = 0
			expression /= Void
		do
			status := Under_agent_scope_notify_status
			expression.accept(Current)
			status := 0
		ensure
			status = 0
		end
	
	resolve_open_operands (t: like type; target_type: TYPE; af: ANONYMOUS_FEATURE
								  target: EXPRESSION; arguments: EFFECTIVE_ARG_LIST) is
		require
			status = 0
			t /= Void
			target_type = target.resolve_in(t)
			af /= Void
		do
			status := Resolve_open_operands_status
			type := t
			push_pop(target_type, af, target, arguments)
			status := 0
		ensure
			status = 0
		end
	
feature {}
	status:                            INTEGER_8
	
	Under_agent_scope_notify_status:   INTEGER_8 is 1
			-- This value given to `status' indicates that we are notifying all open operands under 
			-- the scope of some "agent" creation.
	
	Resolve_open_operands_status:      INTEGER_8 is 2
			-- This value given to `status' indicates that the type of open operands is being resolved.
	
	Search_open_argument_status:       INTEGER_8 is 5
			-- This value given to `status' indicates that we are searching an open argument in some 
			-- a sub-expression (see `search_open_argument_in').
	
	type: TYPE
	
	target_stack: STACK[EXPRESSION] is
		once
			create Result.make
		end
	
	target_type_stack: STACK[TYPE] is
		once
			create Result.make
		end
	
	anonymous_feature_stack: STACK[ANONYMOUS_FEATURE] is
		once
			create Result.make
		end

	push_pop (target_type: TYPE; af: ANONYMOUS_FEATURE; target: EXPRESSION; arguments: EFFECTIVE_ARG_LIST) is
		require
			target_type /= Void
			af /= Void
			target /= Void
		do
			target_type_stack.push(target_type)
			anonymous_feature_stack.push(af)
			target_stack.push(target)
			target.accept(Current)
			if arguments /= Void then
				arguments.accept(Current)
			elseif af.arguments /= Void then
				limitation_error(target)
			end
			target_stack.pop
			anonymous_feature_stack.pop
			target_type_stack.pop
		end
	
	make is
		do
		end

	limitation_error (visited: EXPRESSION) is
		do
			error_handler.append("Cannot use expression ")
			error_handler.add_expression(visited)
			error_handler.append(" under the scope of %"agent%" keyword. If you feel this limitation to strong, %
										%feel free to contact us.")
			error_handler.print_as_fatal_error
		end
	
	singleton_memory: AGENT_CREATION_HELPER is
		once
			Result := Current
		end

	last_open_argument: EXPRESSION
	
	search_open_argument_in (expression: EXPRESSION) is
			-- If any, search for the first open argument in `expression' and make it 
			-- available in `last_open_argument'.
		require
			expression /= Void
		local
			old_status: like status
		do
			old_status := status
			last_open_argument := Void
			status := Search_open_argument_status
			expression.accept(Current)
			status := old_status
		ensure
			status = old status
		end
		
	syntactical_flat_check_ (delayed_call: CALL; expression: EXPRESSION) is
		require
			expression /= Void
		do
			if not is_open_operand(expression) then
				search_open_argument_in(expression)
				if last_open_argument /= Void then
					error_handler.add_position(last_open_argument.start_position)
					error_handler.append("Open operand ")
					error_handler.add_expression(last_open_argument)
					error_handler.append(" is out of %"agent%" scope (see also next error message for details).")
					error_handler.print_as_error
					--
					error_handler.append("Actually, the delayed routine is ")
					error_handler.add_feature_name(delayed_call.feature_name)
					error_handler.append(" and the open operand must be at the same level. It cannot be %
												%hidden in the ")
					error_handler.add_expression(expression)
					error_handler.append(" sub-expression.")
					error_handler.print_as_fatal_error
				end
			end
		end
	
invariant
	is_real_singleton: Current = singleton_memory

end -- AGENT_CREATION_HELPER
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
