-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class MANIFEST_STRING_INSPECTOR

insert
	GLOBALS

creation {ANY}
	make

feature {}
	string_pool: TUPLE_STRING_POOL

	headers: FAST_ARRAY[STRING]

	make (ei: INSPECT_STATEMENT) is
		require
			ei /= Void
		local
			when_list: FAST_ARRAY[WHEN_CLAUSE]; val: FAST_ARRAY[WHEN_ITEM]; i, j, n: INTEGER; wi1: WHEN_ITEM_1
			s: STRING; ms: MANIFEST_STRING
		do
			create headers.make(0)
			when_list := ei.when_list
			from
				n := when_list.count - 1
				i := 0
			until
				i > n
			loop
				val := when_list.item(i).list
				from
					j := val.lower
				until
					j > val.upper
				loop
					if not wi1 ?:= val.item(j) then
						error_handler.add_position(ei.start_position)
						error_handler.add_position(val.item(j).start_position)
						error_handler.append("Unexpected range for inspect strings. (This feature is not implemented).")
						error_handler.print_as_fatal_error
					else
						wi1 ::= val.item(j)
						if not ms ?:= wi1.expression then
							error_handler.add_position(ei.start_position)
							error_handler.add_position(wi1.start_position)
							error_handler.append("Only manifest strings are accepted in inspect.")
							error_handler.print_as_fatal_error
						else
							ms ::= wi1.expression
							s := ms.to_string
							if headers.has(s) then
								error_handler.add_position(ei.start_position)
								error_handler.add_position(ms.start_position)
								error_handler.append("Second occurrence of this value (%"")
								error_handler.append(s)
								error_handler.append("%") in the same inspect.")
								error_handler.print_as_fatal_error
							elseif s.is_empty then
								empty_position := ms.start_position
							end
							wi1.set_expression_value(headers.count)
							headers.add_last(s)
						end
					end
					j := j + 1
				end
				i := i + 1
			end
			if not ace.boost then
				-- in boost the string pool is not used --- see `inline_inspect'
				create string_pool.from_collection(headers)
				debug
					from
						i := headers.lower
					until
						i > headers.upper
					loop
						echo.put_string(once "checking header#")
						echo.put_integer(i)
						echo.put_string(once ": %"")
						echo.put_string(headers.item(i))
						echo.put_string(once "%"%N")
						check
							string_pool.index_of(headers.item(i)) = i
						end
						i := i + 1
					end
				end
				check
					has_empty = string_pool.has_empty
				end
			end
		end

	has_empty: BOOLEAN is
		local
			unknown_position: POSITION
		do
			Result := empty_position /= unknown_position
		end

	empty_position: POSITION

feature {} -- Generation helpers:
	var_storage: STRING is "storage"

	var_count: STRING is "count"

	var_state: STRING is "state"

	var_i: STRING is "i"

	put_var (var: STRING) is
		require
			var.count > 0
		do
			cpp.pending_c_function_body.append(var)
			cpp.pending_c_function_body.extend('_')
			cpp.put_inspect_tmp
		end

feature {INSPECT_STATEMENT}
	simplify (type: TYPE; inspect_statement: INSPECT_STATEMENT;
				 prelude: TUPLE[INSTRUCTION, INTERNAL_LOCAL, CALL_1_C, EXPRESSION]): INSTRUCTION is
		local
			exp: EXPRESSION; i, s, c: INTEGER
			exp_local, state_local: INTERNAL_LOCAL
			item_call: CALL_1_C; count_call: EXPRESSION
			compound: COMPOUND; when_clause: WHEN_CLAUSE
			eval_exp: INSTRUCTION; string_inspect: INSTRUCTION; state_inspect: INSPECT_STATEMENT
		do
			if prelude /= Void then
				eval_exp := prelude.first
				state_local := prelude.second
				item_call := prelude.third
				count_call := prelude.fourth
			else
				exp := inspect_statement.expression.simplify(type)
				exp_local := inspect_statement.exp_local(type, exp)
				state_local := inspect_statement.state_local(type)
				create {ASSIGNMENT} eval_exp.inline_make(exp_local, exp)
				item_call := inspect_statement.item_call(type, exp_local)
				count_call := inspect_statement.count_call(type, exp_local)
			end
			string_inspect := inline_inspect(type, state_local, item_call, count_call,
														once "", inspect_statement.start_position)
			create state_inspect.make(inspect_statement.start_position, state_local)
			from
				s := 1
				i := inspect_statement.when_list.lower
			until
				i > inspect_statement.when_list.upper
			loop
				create when_clause.make(state_inspect,
												inspect_statement.when_list.item(i).start_position,
												inspect_statement.when_list.item(i).header_comment)
				c := inspect_statement.when_list.item(i).list.count
				if c = 1 then
					when_clause.add_value(create {INTEGER_CONSTANT}.make(s, inspect_statement.when_list.item(i).start_position))
					s := s + 1
				else
					when_clause.add_slice(create {INTEGER_CONSTANT}.make(s, inspect_statement.when_list.item(i).start_position),
												 create {INTEGER_CONSTANT}.make(s + c - 1, inspect_statement.when_list.item(i).start_position))
					s := s + c
				end
				when_clause.set_compound(inspect_statement.when_list.item(i).compound)
				i := i + 1
			end
			if not inspect_statement.else_position.is_unknown then
				state_inspect.set_else_compound(inspect_statement.else_position, inspect_statement.else_compound)
			end
			state_inspect.force_internal_values(type)

			create compound.make_3(eval_exp, string_inspect, state_inspect)

			smart_eiffel.magic_count_increment
			Result := compound.simplify(type)
		end

	c_compile (type: TYPE; inspect_statement: INSPECT_STATEMENT) is
		require
			cpp.pending_c_function
		local
			i, cur_state, new_state, ext_state: INTEGER; cur_char: CHARACTER; octal: STRING
			no_check, all_check: BOOLEAN; transition: LINKED_LIST[TUPLE[CHARACTER, INTEGER]]
		do
			no_check := ace.no_check
			all_check := ace.all_check
			cpp.inspect_tmp_increment
			cpp.pending_c_function_body.append(once "/*[manifest INSPECT*/%N{T7* ")
			cpp.put_inspect_tmp
			cpp.pending_c_function_body.append(once "=(T7*)")
			if all_check then
				cpp.pending_c_function_body.extend('(')
				cpp.put_trace_or_sedb_expression(inspect_statement.expression.start_position)
				cpp.pending_c_function_body.extend(',')
			end
			cpp.pending_c_function_body.append(once "se_string_inspect_check(")
			inspect_statement.expression.compile_to_c(type)
			cpp.pending_c_function_body.extend(',')
			cpp.put_position(inspect_statement.expression.start_position)
			cpp.pending_c_function_body.extend(')')
			if all_check then
				cpp.pending_c_function_body.extend(')')
			end
			cpp.pending_c_function_body.append(once ";%NT3* ")
			put_var(var_storage)
			cpp.pending_c_function_body.extend('=')
			cpp.put_inspect_tmp
			cpp.pending_c_function_body.append(once "->_storage;%Nint ")
			put_var(var_count)
			cpp.pending_c_function_body.extend('=')
			cpp.put_inspect_tmp
			cpp.pending_c_function_body.append(once "->_count;%N")
			-- walk through the string to have the final state:
			cpp.pending_c_function_body.append(once "int ")
			put_var(var_state)
			cpp.pending_c_function_body.extend('=')
			string_pool.unknown_state.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nint ")
			put_var(var_i)
			cpp.pending_c_function_body.append(once "=0;%N")
			if has_empty then
				cpp.pending_c_function_body.append(once " /* has_empty */ if (")
				put_var(var_count)
				cpp.pending_c_function_body.append(once "==0) ")
				put_var(var_state)
				cpp.pending_c_function_body.append(once "=")
				string_pool.external_state(string_pool.state_empty).append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once ";%Nelse")
			end
			cpp.pending_c_function_body.append(once " while (")
			put_var(var_i)
			cpp.pending_c_function_body.append(once " < ")
			put_var(var_count)
			cpp.pending_c_function_body.append(once " && (")
			put_var(var_i)
			cpp.pending_c_function_body.append(once "==0 || ")
			put_var(var_state)
			cpp.pending_c_function_body.append(once "!=")
			string_pool.unknown_state.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ")) {%Nswitch(")
			put_var(var_state)
			cpp.pending_c_function_body.append(once ") {%N")
			from
				cur_state := string_pool.unknown_state
			until
				cur_state > string_pool.maxstate
			loop
				cpp.pending_c_function_body.append(once "case ")
				cur_state.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once ": switch(*(")
				put_var(var_storage)
				cpp.pending_c_function_body.extend('+')
				put_var(var_i)
				cpp.pending_c_function_body.append(once ")) {%N")
				transition := string_pool.transition(cur_state)
				from
					i := transition.lower
				until
					i > transition.upper
				loop
					cur_char := transition.item(i).first
					new_state := transition.item(i).second
					cpp.pending_c_function_body.append(once "case (unsigned char)'")
					inspect
						cur_char.code
					when 9 then
						cpp.pending_c_function_body.append(once "\t")
					when 10 then
						cpp.pending_c_function_body.append(once "\r")
					when 13 then
						cpp.pending_c_function_body.append(once "\n")
					when 39 then
						cpp.pending_c_function_body.append(once "\'")
					when 92 then
						cpp.pending_c_function_body.append(once "\\")
					when 0 .. 8, 11, 12, 14 .. 31 then
						octal := once ""
						octal.clear_count
						cur_char.code.to_integer_8.to_octal_in(octal)
						cpp.pending_c_function_body.extend('\')
						cpp.pending_c_function_body.append(octal)
					else
						cpp.pending_c_function_body.extend(cur_char)
					end
					cpp.pending_c_function_body.append(once "': ")
					put_var(var_state)
					cpp.pending_c_function_body.extend('=')
					if new_state < headers.count and then new_state /= string_pool.unknown_state then
						ext_state := string_pool.external_state(new_state)
						if ext_state /= new_state then
							cpp.pending_c_function_body.extend('(')
							put_var(var_i)
							cpp.pending_c_function_body.append(once "!=")
							put_var(var_count)
							cpp.pending_c_function_body.append(once "-1)?")
							new_state.append_in(cpp.pending_c_function_body)
							cpp.pending_c_function_body.extend(':')
						end
						ext_state.append_in(cpp.pending_c_function_body)
					else
						new_state.append_in(cpp.pending_c_function_body)
					end
					cpp.pending_c_function_body.append(once ";break;%N")
					i := i + 1
				end
				cpp.pending_c_function_body.append(once "default: ")
				put_var(var_state)
				cpp.pending_c_function_body.extend('=')
				string_pool.unknown_state.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once ";%Nbreak;%N}%Nbreak;%N")
				cur_state := cur_state + 1
			end
			cpp.pending_c_function_body.append(once "default: ")
			put_var(var_state)
			cpp.pending_c_function_body.extend('=')
			string_pool.unknown_state.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nbreak;%N}%N")
			put_var(var_i)
			cpp.pending_c_function_body.append(once "++;%N}%N")
			-- now compile the compounds:
			cpp.pending_c_function_body.append(once "switch(")
			put_var(var_state)
			cpp.pending_c_function_body.append(once "){%N")
			inspect_statement.compile_to_c_switch(type)
			if inspect_statement.else_compound /= Void then
				check
					not inspect_statement.else_position.is_unknown
				end
				cpp.pending_c_function_body.append(once "default:;%N")
				inspect_statement.else_compound.compile_to_c(type)
			elseif inspect_statement.else_position.is_unknown and then no_check then
				cpp.pending_c_function_body.append(once "default:;%N")
				exceptions_handler.bad_inspect_value(inspect_statement.start_position)
			end
			cpp.pending_c_function_body.append(once "}%N}/*manifest INSPECT]*/%N")
			cpp.inspect_tmp_decrement
		end

	jvm_compile (type: TYPE; inspect_statement: INSPECT_STATEMENT) is
		local
			ca: like code_attribute
			loop_point, default_state_point, default_char_point, empty_point, not_empty_point
			state_found_point: INTEGER; counter, state: INTEGER; i, cur_state, new_state, cur_char: INTEGER
			transition: LINKED_LIST[TUPLE[CHARACTER, INTEGER]]; loop_points, inspect_points: INTEGER
			pc_tableswitch, pc_lookupswitch: INTEGER
		do
			ca := code_attribute
			loop_points := ca.get_branch_array_index
			inspect_points := ca.get_branch_array_index
			inspect_statement.expression.compile_to_jvm(type)
			ca.opcode_checkcast(constant_pool.idx_eiffel_string_class)
			ca.opcode_dup
			ca.opcode_getfield(constant_pool.idx_eiffel_string_storage_fieldref, -2)
			ca.opcode_swap
			ca.opcode_getfield(constant_pool.idx_eiffel_string_count_fieldref, -2)
			counter := ca.extra_local_size1
			ca.opcode_iconst_0
			ca.opcode_istore(counter)
			state := ca.extra_local_size1
			ca.opcode_push_integer(string_pool.unknown_state)
			ca.opcode_istore(state)
			-- Walk through the string to have the final state:
			-- State of the stack at this point:
			-- 2 storage
			-- 1 count
			if has_empty then
				-- Here we test if the count is 0. In that case, we have an empty
				-- string, we must take that into account.
				ca.opcode_dup
				empty_point := ca.opcode_ifeq
				not_empty_point := ca.opcode_goto
				ca.resolve_u2_branch(empty_point)
				ca.opcode_push_integer(string_pool.state_empty)
				ca.opcode_istore(state)
				state_found_point := ca.opcode_goto
			end
			ca.resolve_u2_branch(not_empty_point)
			-- State of the stack at this point:
			-- 2 storage
			-- 1 count
			loop_point := ca.program_counter
			-- End-loop condition: if the counter is equal or greater than the
			-- count, then go to the "inspect" part.
			ca.opcode_dup
			ca.opcode_iload(counter)
			ca.add_branch(ca.opcode_if_icmpge, inspect_points)
			-- Switch on all states: since the whole complete sequence of states
			-- is generated, a table switch will be used. Maybe someday we will
			-- use a lookup if the algorithm is complexified (still more) to
			-- remove the "holes".
			ca.opcode_iload(state)
			pc_tableswitch := ca.program_counter
			default_state_point := ca.opcode_tableswitch(string_pool.unknown_state, string_pool.maxstate, jvm_state_points)
			from
				cur_state := string_pool.unknown_state
			until
				cur_state > string_pool.maxstate
			loop
				-- Resolve the current switch case position
				ca.resolve_tableswitch_branch(pc_tableswitch, jvm_state_points, cur_state - string_pool.unknown_state)
				-- We now prepare the array of available characters...
				transition := string_pool.transition(cur_state)
				jvm_char_values.clear_count
				from
					i := transition.lower
				until
					i > transition.upper
				loop
					cur_char := transition.item(i).first.code
					jvm_char_values.add_last(cur_char)
					i := i + 1
				end
				if jvm_char_values.count = 0 then
					-- ... and either it is empty, in that case there is no need to
					-- go further: the string will not be found here...
					ca.opcode_push_integer(string_pool.unknown_state)
					ca.opcode_istore(state)
					ca.add_branch(ca.opcode_goto, inspect_points)
				else
					-- ... or it's not empty; in that case, we will use a
					-- lookupswitch because obviously all the possible characters
					-- won't match. Be aware that `opcode_lookupswitch' sorts the
					-- values, so after the opcode we will have to find them again.
					jvm_char_points.clear_count
					-- Just before the lookup switch, we push the current
					-- character, from the storage (stack#2) and the counter
					ca.opcode_swap
					ca.opcode_dup_x1
					ca.opcode_iload(counter)
					ca.opcode_baload
					pc_lookupswitch := ca.program_counter
					default_char_point := ca.opcode_lookupswitch(jvm_char_values, jvm_char_points)
					from
						cur_char := jvm_char_values.lower
					until
						cur_char > jvm_char_values.upper
					loop
						-- As said above, `opcode_lookupswitch' may have changed the
						-- ordering of jvm_char_values, so we must look for the
						-- "good" `new_state'; hence the loop.
						from
							i := transition.lower
						until
							jvm_char_values.item(cur_char) = transition.item(i).first.code or else i > transition.upper
						loop
							i := i + 1
						end
						new_state := transition.item(i).second -- Resolve the current switch case position
						ca.resolve_lookupswitch_branch(pc_lookupswitch, jvm_char_points, cur_char)
						if new_state = string_pool.unknown_state then
							-- Either the new_state is unknown, in that case we exit
							-- the loop...
							ca.opcode_push_integer(new_state)
							ca.opcode_istore(state)
							ca.add_branch(ca.opcode_goto, inspect_points)
						else
							-- Either it is a valid state, in that case we store it
							-- and continue to the next character
							if new_state <= headers.count then
								ca.opcode_push_integer(string_pool.external_state(new_state))
							else
								ca.opcode_push_integer(new_state)
							end
							ca.opcode_istore(state)
							ca.add_branch(ca.opcode_goto, loop_points)
						end
						cur_char := cur_char + 1
					end
					-- Default when no character match: unknown state, exit the
					-- loop
					ca.resolve_lookupswitch_default_branch(pc_lookupswitch, default_char_point)
					ca.opcode_push_integer(string_pool.unknown_state)
					ca.opcode_istore(state)
					ca.add_branch(ca.opcode_goto, inspect_points)
				end
				cur_state := cur_state + 1
			end
			-- Default when no state match (which shouldn't happen)
			ca.resolve_tableswitch_default_branch(pc_tableswitch, default_state_point)
			ca.opcode_push_integer(string_pool.unknown_state)
			ca.opcode_istore(state)
			ca.add_branch(ca.opcode_goto, inspect_points)
			-- Here is the "continue" part of the loop. We increment the counter
			-- and go back at the beginning.
			ca.resolve_branches(loop_points)
			ca.opcode_iinc(counter, 1)
			ca.opcode_goto_backward(loop_point)
			-- Now compile the compounds:
			-- Here we have exitted the loop.
			ca.resolve_branches(inspect_points)
			-- This point is also reached from the is_empty case.
			ca.resolve_u2_branch(state_found_point)
			-- We may safely remove the two stack elements, they were useful for
			-- the loop and are not anymore.
			ca.opcode_pop2
			-- Now we push the final state we reckoned above; the rest is like a
			-- "normal" inspect.
			ca.opcode_iload(state)
			inspect_statement.when_list_compile_to_jvm(type)
			if inspect_statement.else_compound /= Void then
				inspect_statement.else_compound.compile_to_jvm(type)
			elseif inspect_statement.else_position.is_unknown then
				if ace.no_check then
					code_attribute.runtime_error_inspect(type, inspect_statement.expression)
				end
			end
			inspect_statement.when_list_compile_to_jvm_resolve_branch
			ca.opcode_pop
			ca.release_branch_array_index
			ca.release_branch_array_index
		end

feature {}
	jvm_state_points: FAST_ARRAY[INTEGER] is
		once
			create Result.with_capacity(4)
		end

	jvm_char_points: FAST_ARRAY[INTEGER] is
		once
			create Result.with_capacity(4)
		end

	jvm_char_values: FAST_ARRAY[INTEGER] is
		once
			create Result.with_capacity(4)
		end

feature {} -- For simplify:
	inline_inspect (type: TYPE; state_local: INTERNAL_LOCAL; item_call: CALL_1_C; count_call: EXPRESSION;
						 a_prefix: STRING; pos: POSITION): INSTRUCTION is
		local
			i_call: CALL_1_C
			args: EFFECTIVE_ARG_LIST
			then_compound, else_compound, when_compound: INSTRUCTION; ifthenelse: IFTHENELSE
			inspect_chars: INSPECT_STATEMENT; eq: BUILT_IN_EQ_NEQ
			when_clause: WHEN_CLAUSE
			level, i, full_index: INTEGER; c: CHARACTER; chars, pfx, full: STRING
		do
			level := a_prefix.count

			-- Which characters are available with the given prefix?
			chars := strings.new
			from
				i := headers.lower
			until
				i > headers.upper
			loop
				if headers.item(i).has_prefix(a_prefix) then
					if headers.item(i).count = level then
						full := headers.item(i)
						full_index := i
					elseif level < headers.item(i).count then
						c := headers.item(i).item(level + 1)
						if not chars.has(c) then
							chars.add_last(c)
						end
					end
				end
				i := i + 1
			end

			-- We begin to define the else_compound, which may contain nested inspect statements
			if chars.is_empty then
				-- No character is available to the chars set; so the else_compound is simply to set the state to
				-- 0 (not found).
				create {ASSIGNMENT} else_compound.inline_make(state_local, create {INTEGER_CONSTANT}.make(0, pos))
			else
				create args.make_1(create {INTEGER_CONSTANT}.make(level + 1, pos))
				i_call := item_call.twin
				i_call.set_arguments(args)
				if chars.count = 1 then
					-- Only one character is available; either it is it, or not.
					c := chars.first
					pfx := strings.new_twin(a_prefix)
					pfx.extend(c)
					create eq.make_eq(i_call, pos, create {CHARACTER_CONSTANT}.with(pos, c))
					when_compound := inline_inspect(type, state_local, i_call, count_call, pfx, pos)
					create ifthenelse.ifthenelse(pos, eq,
														  when_compound,
														  pos,
														  create {ASSIGNMENT}.inline_make(state_local, create {INTEGER_CONSTANT}.make(0, pos)))
					else_compound := ifthenelse
					strings.recycle(pfx)
				else
					-- General case: we must generate a characters inspect.
					create inspect_chars.make(pos, i_call)
					pfx := strings.best_fit(level + 1)
					pfx.copy(a_prefix)
					pfx.extend('%U')
					from
						i := chars.lower
					until
						i > chars.upper
					loop
						c := chars.item(i)
						pfx.put(c, level + 1)
						when_compound := inline_inspect(type, state_local, i_call, count_call, pfx, pos)
						create when_clause.make(inspect_chars, pos, Void)
						when_clause.add_value(create {CHARACTER_CONSTANT}.with(pos, c))
						when_clause.set_compound(when_compound)
						i := i + 1
					end
					strings.recycle(pfx)
					inspect_chars.set_else_compound(pos,
															  create {ASSIGNMENT}.inline_make(state_local, create {INTEGER_CONSTANT}.make(0, pos)))
					-- fix the inspect statement values
					inspect_chars.force_internal_values(type)
					else_compound := inspect_chars
				end
			end
			if full = Void then
				-- There is no terminal solution here; either we go on; or the state is 0 (not found).
				Result := else_compound
			else
				-- There is a terminal solution; we must test the count of the exp before concluding. Hence the
				-- if-then-else.
				create {ASSIGNMENT} then_compound.inline_make(state_local, create {INTEGER_CONSTANT}.make(full_index + 1, pos))
				create eq.make_eq(count_call, pos, create {INTEGER_CONSTANT}.make(level, pos))
				create ifthenelse.ifthenelse(pos, eq, then_compound, pos, else_compound)
				Result := ifthenelse
			end
			strings.recycle(chars)
		ensure
			a_prefix.is_equal(old (a_prefix.twin))
		end

end -- class MANIFEST_STRING_INSPECTOR
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
