-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
expanded class SWITCH
	--
	-- Set of tools (no attributes) to handle one switching site.
	--

insert
	VISITABLE
	GLOBALS

feature {SWITCH_COLLECTION}
	c_define (desc: SWITCH_SITE) is
			-- Define the switching C function for the function named `rf'. All
			-- the possible calls are the `features'.
		require
			desc /= Void
		local
			boost: BOOLEAN; arguments: FORMAL_ARG_LIST; rt: TYPE; fn: FEATURE_NAME
			current_type: TYPE; result_type, arg_type: TYPE_MARK; i: INTEGER
		do
			boost := ace.boost
			current_type := desc.current_type
			rt := desc.result_type
			fn := desc.feature_name
			arguments := desc.arguments
			if rt /= Void then
				result_type := rt.canonical_type_mark
			end
			cpp.prepare_c_function
			if result_type = Void then
				cpp.pending_c_function_signature.append(once "void")
			else
				result_type.c_type_for_result_in(cpp.pending_c_function_signature)
			end
			cpp.pending_c_function_signature.append(once " X")
			current_type.id.append_in(cpp.pending_c_function_signature)
			fn.mapping_c_in(cpp.pending_c_function_signature)
			cpp.pending_c_function_signature.extend('(')
			if ace.profile then
				cpp.pending_c_function_signature.append(once "se_local_profile_t*parent_profile,")
			end
			if boost then
				cpp.pending_c_function_signature.append(once "T0*C")
			else
				cpp.pending_c_function_signature.append(once "se_dump_stack*caller,se_position position,T0*C")
			end
			if arguments /= Void then
				from
					i := 1
				until
					i > arguments.count
				loop
					cpp.pending_c_function_signature.extend(',')
					arg_type := arguments.type_mark(i).to_static(current_type)
					arg_type.c_type_for_argument_in(cpp.pending_c_function_signature)
					cpp.pending_c_function_signature.append(once " a")
					i.append_in(cpp.pending_c_function_signature)
					i := i + 1
				end
			end
			cpp.pending_c_function_signature.extend(')')
			if not boost then
				cpp.pending_c_function_body.append(once "static se_frame_descriptor fd={%"Dynamic dispatch {")
				cpp.pending_c_function_body.append(current_type.name.to_string)
				cpp.pending_c_function_body.append(once "}.")
				fn.mapping_c_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once "%",0,0,%"%",1};%N")
			end
			if ace.profile then
				local_profile
			end
			if result_type /= Void then
				result_type.c_type_for_result_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once " R")
				if not boost then
					cpp.pending_c_function_body.extend('=')
					result_type.c_initialize_in(cpp.pending_c_function_body)
				end
				cpp.pending_c_function_body.append(once ";%N")
			end
			if not boost then
				cpp.pending_c_function_body.append(once "[
               se_dump_stack ds;
               ds.fd=&fd;
               ds.current=NULL;
               ds.p=(caller->p);
               ds.caller=caller;
               ds.exception_origin=NULL;
               ds.locals=NULL;
               
               ]")
			end
			cpp.pending_c_function_body.append(once "{Tid id=")
			if boost then
				cpp.pending_c_function_body.append(once "((T0*)C)->id;%N")
			else
				cpp.pending_c_function_body.append(once "vc(C,position)->id;%N")
			end
			if ace.profile then
				start_profile(desc)
			end
			if not boost then
				cpp.set_dump_stack_top_for(current_type, once "&ds", once "link")
			end
			if ace.all_check then
				c_switch(desc)
			else
				c_dichotomatic(desc, 1, desc.count)
			end
			cpp.pending_c_function_body.append(once "}%N")
			if ace.profile then
				stop_profile
			end
			if not boost then
				cpp.set_dump_stack_top_for(current_type, once "caller", once "unlink")
			end
			if result_type /= Void then
				cpp.pending_c_function_body.append(once "return R;%N")
			end
			cpp.dump_pending_c_function(True)	
		end

feature {C_PRETTY_PRINTER}
	put_ith_argument (i: INTEGER) is
			-- Produce C code for argument at index `i' of `fal' used inside the switching C function.
		require
			cpp.pending_c_function
			1 <= i
		do
			cpp.pending_c_function_body.extend('a')
			i.append_in(cpp.pending_c_function_body)
		end

feature {C_PRETTY_PRINTER, SWITCH, AGENT_CREATION}
	name_in (buffer: STRING; up_rf: RUN_FEATURE) is
		do
			buffer.extend('X')
			up_rf.type_of_current.live_type.id.append_in(buffer)
			up_rf.name.mapping_c_in(buffer)
		end

feature {ANY}
	jvm_descriptor (up_rf: RUN_FEATURE): STRING is
		local
			arguments: FORMAL_ARG_LIST; rt: TYPE_MARK
		do
			arguments := up_rf.arguments
			tmp_jvmd.clear_count
			tmp_jvmd.extend('(')
			tmp_jvmd.append(jvm_root_descriptor)
			if arguments /= Void then
				arguments.jvm_descriptor_in(tmp_jvmd)
			end
			rt := up_rf.result_type
			if rt = Void then
				tmp_jvmd.append(once ")V")
			else
				tmp_jvmd.extend(')')
				if rt.is_reference then
					tmp_jvmd.append(jvm_root_descriptor)
				else
					rt.jvm_descriptor_in(tmp_jvmd)
				end
			end
			Result := tmp_jvmd
		end

feature {ANY}
	idx_methodref (up_rf: RUN_FEATURE): INTEGER is
		require
			up_rf /= Void
		local
			buffer: STRING
		do
			buffer := once ".... unique ...."
			buffer.clear_count
			name_in(buffer, up_rf)
			Result := constant_pool.idx_methodref3(jvm_root_class, buffer, jvm_descriptor(up_rf))
		end

feature {JVM}
	jvm_mapping (up_rf: RUN_FEATURE; type: TYPE; target: EXPRESSION; eal: EFFECTIVE_ARG_LIST) is
		require
			target /= Void
			up_rf /= Void
		local
			idx, stack_level: INTEGER; fal: FORMAL_ARG_LIST; switch: SWITCH
		do
			target.compile_to_jvm(type)
			stack_level := 1
			if eal /= Void then
				fal := up_rf.arguments
				stack_level := stack_level + eal.compile_to_jvm(type, fal)
			end
			if up_rf.result_type /= Void then
				stack_level := stack_level - up_rf.result_type.jvm_stack_space
			end
			idx := switch.idx_methodref(up_rf)
			code_attribute.opcode_invokestatic(idx, - stack_level)
		end

feature {ANY}
	accept (visitor: SWITCH_VISITOR) is
		do
			visitor.visit_switch(Current)
		end

feature {SWITCH_SITE}
	jvm_define (up_rf: RUN_FEATURE) is
		local
			rt: TYPE_MARK; run_time_set: RUN_TIME_SET; buffer: STRING

		do
			-- Define the Java switching static method for `up_rf'.
			buffer := once ".... unique ...."
			buffer.clear_count
			name_in(buffer, up_rf)
			method_info.start(9, buffer, jvm_descriptor(up_rf))
			rt := up_rf.result_type
			run_time_set := up_rf.type_of_current.live_type.run_time_set
			jvm_switch(up_rf, run_time_set, rt)
			method_info.finish
		end

feature {}
	jvm_switch (up_rf: RUN_FEATURE; run_time_set: RUN_TIME_SET; rt: TYPE_MARK) is
			-- Produce Java sequential switch code.
		require
			rt /= Void implies rt.is_static
		local
			space, point, i: INTEGER; dyn_lt: LIVE_TYPE; dyn_rf: RUN_FEATURE; boost: BOOLEAN
		do
			from
				boost := ace.boost
				i := run_time_set.count
			until
				i = 0
			loop
				dyn_lt := run_time_set.item(i)
				dyn_rf := dyn_lt.dynamic_feature(up_rf)
				if i = 1 and then boost then
				else
					code_attribute.opcode_aload_0
					dyn_rf.type_of_current.live_type.opcode_instanceof
					point := code_attribute.opcode_ifeq
				end
				jvm.push_switch(dyn_rf)
				dyn_rf.mapping_jvm
				jvm.pop
				if rt = Void then
					code_attribute.opcode_return
				else
					rt.jvm_return_code
				end
				if i = 1 and then boost then
				else
					code_attribute.resolve_u2_branch(point)
				end
				i := i - 1
			end
			if not boost then
				code_attribute.opcode_aload_0
				code_attribute.runtime_error_bad_target(up_rf.start_position, up_rf.type_of_current,
																	 once "System-level Validity error detected inside virtual %
																			%switching code.%N%
																			%Bad target type for dynamic dispatch.")
				if rt = Void then
					code_attribute.opcode_return
				else
					space := rt.jvm_push_default
					rt.jvm_return_code
				end
			end
		end

	tmp_jvmd: STRING is
		once
			create Result.make(32)
		end

	c_dichotomatic (desc: SWITCH_SITE; bi, bs: INTEGER) is
			-- Produce dichotomic inspection code for Current id.
		require
			bi <= bs
		local
			m: INTEGER; dyn_lt: LIVE_TYPE; dyn_rf: RUN_FEATURE
		do
			if bi = bs then
				dyn_rf := desc.item(bi)
				dyn_lt := dyn_rf.type_of_current.live_type
				cpp.inside_switch_call(dyn_rf, desc.result_type)
			else
				m := (bi + bs) #// 2
				dyn_rf := desc.item(m)
				dyn_lt := dyn_rf.type_of_current.live_type
				cpp.pending_c_function_body.append(once "if(id<=")
				dyn_lt.id.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.append(once "){%N")
				c_dichotomatic(desc, bi, m)
				cpp.pending_c_function_body.append(once "}%Nelse{%N")
				c_dichotomatic(desc, m + 1, bs)
				cpp.pending_c_function_body.extend('}')
			end
		end

	c_switch (desc: SWITCH_SITE) is
			-- Produce C switch inspection code for Current id.
		local
			i: INTEGER; dyn_lt: LIVE_TYPE; dyn_rf: RUN_FEATURE
		do
			cpp.pending_c_function_body.append(once "switch(id){%N")
			from
				i := desc.count
			until
				i < 1
			loop
				dyn_rf := desc.item(i)
				dyn_lt := dyn_rf.type_of_current.live_type
				cpp.pending_c_function_body.append(once "case ")
				dyn_lt.id.append_in(cpp.pending_c_function_body)
				cpp.pending_c_function_body.extend(':')
				cpp.inside_switch_call(dyn_rf, desc.result_type)
				cpp.pending_c_function_body.append(once "%Nbreak;%N")
				i := i - 1
			end
			if ace.no_check then
				cpp.pending_c_function_body.append(once "default: error2(C,position);%N")
			end
			cpp.pending_c_function_body.append(once "}%N")
		end

feature {}
	local_profile is
		require
			ace.profile
			cpp.pending_c_function
		do
			smart_eiffel.local_profile
		end

	start_profile (desc: SWITCH_SITE) is
		require
			ace.profile
			cpp.pending_c_function
		do
			cpp.pending_c_function_body.append(once "local_profile.profile=switch_profile+")
			desc.id.append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%N")
			cpp.pending_c_function_body.append(once "start_profile(parent_profile, &local_profile);%N")
		end

	stop_profile is
		require
			ace.profile
			cpp.pending_c_function
		do
			smart_eiffel.stop_profile
		end

end -- class SWITCH
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
