-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class E_REQUIRE
	--
	-- Such an object is used to store an ordinary "require" as well as a "require else" clause.
	-- Furthermore, a E_REQUIRE object can also be used to store all the inherited require assertions of some
	-- final feature.
	--

inherit
	VISITABLE

insert
	GLOBALS
	ASSERTION_PRINT_TOOLS

creation {ANY}
	make

feature {ANY}
	start_position: POSITION is
		do
			Result := list.first.start_position
		end

	header_comment: COMMENT is
		do
			Result := list.first.header_comment
		end

	is_require_else: BOOLEAN
			-- Is it a "require else" assertion?

	pretty_print is
		do
			check
				list.count = 1
			end
			if is_require_else then
				list.first.pretty_print_(2, once "require else")
			else
				list.first.pretty_print_(2, once "require")
			end
		end

	short (type: TYPE; client: CLASS_NAME; hide_current: BOOLEAN) is
		local
			i: INTEGER
		do
			from
				i := list.lower
				short_(type, list.item(i), once "hook401", once "      require%N", client, hide_current)
				i := i + 1
			until
				i > list.upper
			loop
				short_(type, list.item(i), once "hook402", once "      require else %N", client, hide_current)
				i := i + 1
			end
			short_printer.hook(once "hook403")
		end

	collect (type: TYPE) is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).collect(type)
				i := i - 1
			end
		end

	adapt_for (type: TYPE): like Current is
		require
			type /= Void
		local
			i: INTEGER; ri1, ri2: REQUIRE_ITEM; l: like list
		do
			from
				i := list.upper
			until
				i < list.lower or else ri1 /= ri2
			loop
				ri1 := list.item(i)
				ri2 := ri1.adapt_for(type)
				i := i - 1
			end
			if ri1 = ri2 then
				Result := Current
			else
				from
					l := list.twin
					Result := twin
					Result.set_list(l)
					l.put(ri2, i + 1)
				until
					i < list.lower
				loop
					l.put(list.item(i).adapt_for(type), i)
					i := i - 1
				end
			end
		end

	accept (visitor: E_REQUIRE_VISITOR) is
		do
			visitor.visit_e_require(Current)
		end

	specialize_in (type: TYPE): like Current is
		require
			type /= Void
		local
			i: INTEGER; ri1, ri2: REQUIRE_ITEM; l: like list
		do
			from
				i := list.upper
			until
				i < list.lower or else ri1 /= ri2
			loop
				ri1 := list.item(i)
				ri2 := ri1.specialize_in(type)
				i := i - 1
			end
			if ri1 = ri2 then
				Result := Current
			else
				from
					l := list.twin
					Result := twin
					Result.set_list(l)
					l.put(ri2, i + 1)
				until
					i < list.lower
				loop
					l.put(list.item(i).specialize_in(type), i)
					i := i - 1
				end
			end
		ensure
			Result.has_been_specialized
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		require
			parent_type /= Void
			parent_edge /= Void
			new_type /= Void
			new_type /= parent_type
			has_been_specialized
		local
			i: INTEGER; ri1, ri2: REQUIRE_ITEM; l: like list
		do
			from
				i := list.upper
			until
				i < list.lower or else ri1 /= ri2
			loop
				ri1 := list.item(i)
				ri2 := ri1.specialize_thru(parent_type, parent_edge, new_type)
				i := i - 1
			end
			if ri1 = ri2 then
				Result := Current
			else
				from
					l := list.twin
					Result := twin
					Result.set_list(l)
					l.put(ri2, i + 1)
				until
					i < list.lower
				loop
					l.put(list.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
					i := i - 1
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2 (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER; ri1, ri2: REQUIRE_ITEM; l: like list
		do
			from
				i := list.upper
			until
				i < list.lower or else ri1 /= ri2
			loop
				ri1 := list.item(i)
				ri2 := ri1.specialize_2(type)
				i := i - 1
			end
			if ri1 = ri2 then
				Result := Current
			else
				from
					l := list.twin
					Result := twin
					Result.set_list(l)
					l.put(ri2, i + 1)
				until
					i < list.lower
				loop
					l.put(list.item(i).specialize_2(type), i)
					i := i - 1
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	is_always_true: BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
				Result := True
			until
				not Result or else i < list.lower
			loop
				Result := list.item(i).is_always_true
				i := i - 1
			end
		end

	side_effect_free (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
				Result := True
			until
				not Result or else i < list.lower
			loop
				Result := list.item(i).side_effect_free(type)
				i := i - 1
			end
		end
		--|*** compile_to_c (type: TYPE) is
		--|*** local
		--|*** i: INTEGER; assertion: ASSERTION
		--|*** do
		--|*** not_yet_implemented
		--|*** if not is_always_true then
		--|*** c_set_flag
		--|*** from
		--|*** i := list.lower
		--|*** until
		--|*** i > list.upper
		--|*** loop
		--|*** assertion := list.item(i)
		--|*** if not smart_eiffel.scoop or else not assertion.is_guard	then
		--|*** c_compile_assertion(assertion)
		--|*** end
		--|*** i := i + 1
		--|*** end
		--|*** c_clear_flag
		--|*** end
		--|*** end

	compile_to_c (type: TYPE) is
		local
			i: INTEGER
		do
			if not is_always_true then
				-- (Note that simplify is only called in mode boost.)
				cpp.stop_recursive_assertion_opening(True)
				if list.count = 1 then
					list.first.compile_to_c(type)
				else
					cpp.pending_c_function_body.append(once "if(")
					list.first.non_upper_most_compile_to_c(type)
					cpp.pending_c_function_body.append(once "){}%N")
					from
						i := list.lower + 1
					until
						i = list.upper
					loop
						cpp.pending_c_function_body.append(once "else if(")
						list.first.non_upper_most_compile_to_c(type)
						cpp.pending_c_function_body.append(once "){}%N")
						i := i + 1
					end
					cpp.pending_c_function_body.append(once "else{%N")
					list.item(i).compile_to_c(type)
					cpp.pending_c_function_body.append(once "}%N")
				end
				cpp.stop_recursive_assertion_closing(True)
			end
		end

	compile_to_jvm (type: TYPE) is
		local
			i: INTEGER; ca: like code_attribute
		do
			ca := code_attribute
			if list.count = 1 then
				list.first.compile_to_jvm(type, True)
				ca.opcode_pop
			else
				sucess.clear_count
				from
					i := list.lower
				until
					i > list.upper - 1
				loop
					list.item(i).compile_to_jvm(type, False)
					sucess.add_last(ca.opcode_ifne)
					i := i + 1
				end
				list.item(i).compile_to_jvm(type, True)
				ca.opcode_pop
				ca.resolve_with(sucess)
			end
		end

	use_current (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				Result or else i < list.lower
			loop
				Result := list.item(i).use_current(type)
				i := i - 1
			end
		end

	has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			from
				Result := True
				i := list.upper
			until
				i < list.lower or else not Result
			loop
				Result := list.item(i).has_been_specialized
				i := i - 1
			end
		end

feature {ANONYMOUS_FEATURE_MIXER}
	add_parent_require (parent_require: like Current): like Current is
			-- Add `parent_require' to `Current' only if it is really new code (i.e. using `start_position' to decide.).
		require
			parent_require /= Void
		local
			i: INTEGER; l: like list; ri: REQUIRE_ITEM
		do
			Result := Current
			if start_position /= parent_require.start_position then
				from
					l := parent_require.list
					i := l.lower
				until
					i > l.upper
				loop
					ri := l.item(i)
					if not has_require_item_located_at(ri.start_position) then
						if Result = Current then
							Result := twin
							Result.set_list(list.twin)
						end
						Result.list.add_last(ri)
					end
					i := i + 1
				end
			end
		ensure
			Result.start_position = start_position
		end

feature {E_REQUIRE, E_REQUIRE_VISITOR}
	list: FAST_ARRAY[REQUIRE_ITEM]
			-- Because of "require else" there is more than one list of assertion.
			-- The last one is the uppermost in the inheritance hierarchy (the last chance to pass the require).

feature {E_REQUIRE}
	set_list (l: like list) is
		require
			l /= Void
		do
			list := l
		ensure
			list = l
		end

feature {}
	has_require_item_located_at (sp: POSITION): BOOLEAN is
			-- Is there already the source code of the REQUIRE_ITEM located at `sp' ?
		require
			not sp.is_unknown
		local
			i: INTEGER
		do
			if list /= Void then
				from
					i := list.upper
				until
					Result or else i < list.lower
				loop
					if list.item(i).start_position = sp then
						Result := True
					end
					i := i - 1
				end
			end
		end

	run_feature: RUN_FEATURE
			-- Corresponding one (if any) when runnable.
			-- Note: class invariant are not inside a RUN_FEATURE.

	make (sp: like start_position; else_flag: BOOLEAN; hc: like header_comment; l: FAST_ARRAY[ASSERTION]) is
		do
			create list.with_capacity(4)
			list.add_last(create {REQUIRE_ITEM}.make(sp, hc, l))
			is_require_else := else_flag
		ensure
			is_require_else = else_flag
		end

	sucess: FAST_ARRAY[INTEGER] is
			-- To reach the sucessful code.
		once
			create Result.with_capacity(4)
		end

	short_ (type: TYPE; ri: REQUIRE_ITEM; h1, r1: STRING; client: CLASS_NAME; hide_current: BOOLEAN) is
		local
			i: INTEGER; l: FAST_ARRAY[ASSERTION]
		do
			l := ri.list
			tmp_assertion_list.clear_count
			if l /= Void then
				from
					i := 0
				until
					i > l.upper
				loop
					if client = Void or else not hidden_expression_detector.visit(l.item(i), type, client, hide_current) then
						tmp_assertion_list.add_last(l.item(i))
					end
					i := i + 1
				end
			end
			l := ri.list
			tmp_assertion_list.clear_count
			if l /= Void then
				from
					i := 0
				until
					i > l.upper
				loop
					if client = Void or else not hidden_expression_detector.visit(l.item(i), type, client, hide_current) then
						tmp_assertion_list.add_last(l.item(i))
					end
					i := i + 1
				end
			end
			if header_comment /= Void or else not tmp_assertion_list.is_empty then
				short_printer.hook_or(h1, r1)
				if header_comment = Void then
					short_printer.hook_or("hook412", "")
				else
					short_printer.hook_or("hook413", "")
					header_comment.short("hook414", "         --", "hook415", "%N")
					short_printer.hook_or("hook416", "")
				end
				if tmp_assertion_list.is_empty then
					short_printer.hook_or("hook417", "")
				else
					short_printer.hook_or("hook418", "")
					from
						i := tmp_assertion_list.lower
					until
						i = tmp_assertion_list.upper
					loop
						tmp_assertion_list.item(i).short(type, "hook419", "         ", "hook420" -- before each assertion
						, "", "hook421" -- no tag
						, "", "hook422" -- before tag
						, ": ", "hook423" -- after tag
						, "", "hook424" -- no expression
						, "", "hook425" -- before expression
						, ";", "hook426" -- after expression except last
						, "%N", "hook427" -- no comment
						, "", "hook428" -- before comment
						, " --", "hook429" -- comment begin line
						, "%N", "hook430" -- comment end of line
						, "", "hook431" -- after comment
						, "")
						-- end of each assertion
						i := i + 1
					end
					tmp_assertion_list.item(i).short(type, "hook419", "         ", "hook420" -- before each assertion
					, "", "hook421" -- no tag
					, "", "hook422" -- before tag
					, ": ", "hook423" -- after tag
					, "", "hook424" -- no expression
					, "", "hook432" -- before expression
					, "", "hook426" -- after expression except last
					, "%N", "hook427" -- no comment
					, "", "hook428" -- before comment
					, " --", "hook429" -- comment begin line
					, "%N", "hook430" -- comment end of line
					, "", "hook431" -- after comment
					, "")
					short_printer.hook_or("hook433", "")
				end
				short_printer.hook_or("hook434", "")
			end
		end

invariant
	list.count >= 1

end -- class E_REQUIRE
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
