/***************************************************************************
    smb4kipfinder.cpp  -  This class takes care of the IP address detection
    of the hosts in the browser window.
                             -------------------
    begin                : Son Feb 22 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// application specific includes
#include "smb4kipfinder.h"
#include "smb4kcore.h"

// KDE includes
#include <kapplication.h>

Smb4KIPFinder::Smb4KIPFinder( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_timer = new QTimer( this );
  m_timer->start( 50, false );
  
  m_working = false;

  m_proc = new KProcess( this, "IPFinderProcess" );
  m_proc->setUseShell( true );
  
  m_queue.setAutoDelete( true );
  
  connect( m_timer, SIGNAL( timeout() ), this, SLOT( getIPAddress() ) );
  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess* ) ), this, SLOT( slotProcessExited( KProcess * ) ) );
  connect( ((Smb4KCore *)parent)->scanner(), SIGNAL( members( const QValueList<Smb4KHostItem *> & ) ), this, SLOT( slotInput( const QValueList<Smb4KHostItem *> & ) ) );
}


Smb4KIPFinder::~Smb4KIPFinder()
{
  abort();
}


/****************************************************************************
   Aborts the network scan.
****************************************************************************/

void Smb4KIPFinder::abort()
{
  if ( m_proc->isRunning() )
    m_proc->kill();
}


/****************************************************************************
   Reads the options.
****************************************************************************/


void Smb4KIPFinder::readOptions()
{
  m_config->setGroup( "Browse Options" );
  bool wins = m_config->readBoolEntry( "WINS Support", false );

  // Get the WINS server.
  if ( wins )
    m_wins = m_config->readEntry( "WINS Server", QString::null );

  m_config->setGroup( "Samba" );
  if ( !m_config->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
    m_advNMBOpts.append( QString( " -n \"%1\"" ).arg( m_config->readEntry( "NetBIOS Name", QString::null ) ) );

  if ( !m_config->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
    m_advNMBOpts.append( QString( " -i \"%1\"" ).arg( m_config->readEntry( "NetBIOS Scope", QString::null ) ) );

  if ( !m_config->readEntry( "Socket Options", QString::null ).isEmpty() )
    m_advNMBOpts.append( QString( " -O \"%1\"" ).arg( m_config->readEntry( "Socket Options", QString::null ) ) );
    
  if ( !m_config->readEntry( "NMB Domain", QString::null ).isEmpty() )
  {
    m_config->setGroup( "System" );
    
    if ( m_config->readEntry( "Samba" ).startsWith( "3" ) )
    {
      m_config->setGroup( "Samba" );
      m_advNMBOpts.append( QString( " -W \"%1\"" ).arg( m_config->readEntry( "NMB Domain", QString::null ) ) );
    }
    
    m_config->setGroup( "Samba" );
  }
  
  if ( !m_config->readEntry( "NMB Broadcast", QString::null ).isEmpty() )
    m_advNMBOpts.append( QString( " -B \"%1\"" ).arg( m_config->readEntry( "NMB Broadcast", QString::null ) ) );
    
  if ( m_config->readBoolEntry( "NMB Port 137", false ) )
    m_advNMBOpts.append( " -r" );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KIPFinder::slotInput( const QValueList<Smb4KHostItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      m_queue.enqueue( new Smb4KHostItem( (*it)->workgroup(), (*it)->name(), (*it)->comment(), (*it)->ip() ) );
    }  
  }
}


void Smb4KIPFinder::getIPAddress()
{
  if ( !m_queue.isEmpty() && !m_working )
  {
    m_working = true;
    emit running( IPFINDER_WORKING, m_working );

    m_workgroup = (m_queue.head())->workgroup();
    m_hostList.clear();

    QString command;
    
    while ( m_queue.head() && (m_queue.head())->workgroup() == m_workgroup )
    {
      Smb4KHostItem *item = m_queue.dequeue();
      
      command.append( "nmblookup" );
      
      if ( !m_advNMBOpts.stripWhiteSpace().isEmpty() )
        command.append( m_advNMBOpts );

      // If there is a WINS server, it will be queried for the host.
      if ( !m_wins.isEmpty() )
        command.append( " -R -U "+KProcess::quote( m_wins )+" "+KProcess::quote( item->name() )+" -T -W "+KProcess::quote( item->workgroup() )+" | grep '<00>' | sed -e '/nmb_name/d'" );
      else
        command.append( " "+KProcess::quote( item->name() )+" -T -W "+KProcess::quote( item->workgroup() )+" | grep '<00>' | sed -e '/nmb_name/d'" );

      command.append( " ; " );    
    
      m_hostList.append( item->name() );
    }

    command.truncate( command.length() - 3 );

    m_proc->operator<<( command );
    m_buffer = QString::null;
    m_proc->start( KProcess::NotifyOnExit, KProcess::Stdout );  
  }
}


void Smb4KIPFinder::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KIPFinder::slotProcessExited( KProcess * )
{
  QStringList list = QStringList::split( "\n", m_buffer, false );

  for ( QStringList::ConstIterator it = m_hostList.begin(); it != m_hostList.end(); ++it )
  {
    QString entry = list.grep( *it ).first();
    QString ip;

    if ( !entry.isEmpty() )
    {
      if ( entry.contains( ",", true ) != 0 )
        ip = entry.section( ",", 1, 1 ).section( *it, 0, 0 ).stripWhiteSpace();
      else
        ip = entry.section( *it, 0, 0 ).stripWhiteSpace();
    }
    else
      ip = QString::null;

    emit ip_address( new Smb4KHostItem( m_workgroup, *it, QString::null, ip ) );
  }

  m_hostList.clear();
  m_proc->clearArguments();
  m_working = false;
  emit running( IPFINDER_STOP, false );
}

#include "smb4kipfinder.moc"
