/***************************************************************************
    smb4kmounter.cpp  -  The core class, that mounts shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qmap.h>
#include <qptrlist.h>

#ifdef __FreeBSD__
#include <qfileinfo.h>
#endif

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>

// Application specific includes
#include "smb4kmounter.h"
#include "smb4kcore.h"


Smb4KMounter::Smb4KMounter( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_proc = new KProcess( this, "MounterProcess" );
  m_proc->setUseShell( true );

  m_state = Idle;
  m_lastJob = false;

  m_config->setGroup( "System" );
  m_uid = m_config->readNumEntry( "UID" );
  m_gid = m_config->readNumEntry( "GID" );

  // The timer for the queue.
  m_timer1 = new QTimer( this );
  m_timer1->start( 50, false );

  // The timer for the checks.
  m_timer2 = new QTimer( this );
  m_timer2->start( 250, false );

  m_working = false;

  // Do some initial actions:
  // - Import all mounted SMB shares,
  // - Mount all shares that have to be mounted.
  m_queue.setAutoDelete( true );

  m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  m_queue.enqueue( new QString( QString( "%1:" ).arg( MountRecent ) ) );

  connect( m_proc,   SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc,   SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,   SIGNAL( receivedStderr( KProcess *, char *, int ) ), this, SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_timer1, SIGNAL( timeout() ),                                 this, SLOT( init() ) );
  connect( m_timer2, SIGNAL( timeout() ),                                 this, SLOT( check() ) );
}


Smb4KMounter::~Smb4KMounter()
{
  if ( !m_unmountAll && m_state != UnmountAll )
    abort();
    
  m_mounted_shares.clear();
}


/***************************************************************************
   Aborts any action of the mounter.
***************************************************************************/


void Smb4KMounter::abort()
{
  m_queue.clear();

  if ( m_useSuper && m_super )
  {
    if ( m_proc->isRunning() )
    {
      KProcess p;
      p.setUseShell( true );
      p.operator<<( QString( "super kill -15 %1" ).arg( m_proc->pid() ) );
      p.start( KProcess::DontCare, KProcess::NoCommunication );
    }
  }
  else
  {
    if ( m_proc->isRunning() )
      m_proc->kill();
  }
}


/***************************************************************************
   Mounts recently used shares.
***************************************************************************/


void Smb4KMounter::mountRecent()
{
  m_config->setGroup( "Mount Options" );
  
  if ( m_config->readBoolEntry( "Mount Recent", false ) )
  {
    if ( m_config->hasGroup( "Recently Mounted Shares" ) )
    {
      m_config->setGroup( "Recently Mounted Shares" );
      
      QString temp;
      
      for ( int index = 0; ; index++ )
      {
        temp = m_config->readEntry( QString( "%1" ).arg( index ), QString::null );
        
        if ( !temp.isEmpty() )
        {
          Smb4KShare *share = findShareByName( temp );
          
          if ( !share || share->getUID() != m_uid && share->getGID() != m_gid )
          {
#ifndef __FreeBSD__
            mountShare( QString::null, temp.section( "/", 2, 2 ), QString::null, temp.section( "/", 3, 3 ).replace( QRegExp( "_" ), " " ) );
#else
            mountShare( QString::null, temp.section( "/", 2, 2 ).section( "@", 1, 1 ), QString::null, temp.section( "/", 3, 3 ).replace( QRegExp( "_" ), " " ) );
#endif            
          }
          else
            continue;
        }
        else
          break;
      }
      
      emit running( MOUNTER_STOP, false );
    }
    else
      emit running( MOUNTER_STOP, false );
  }
  else
    emit running( MOUNTER_STOP, false );
  
  m_working = false;
}


/***************************************************************************
   Imports all shares, that are mounted externally.
***************************************************************************/

void Smb4KMounter::import()
{
#ifndef __FreeBSD__
  QValueList<Smb4KShare *> shares;
    
  // /proc/mounts is read to import the externally mounted shares.
  QFile file( "mounts" );
  QDir *dir = new QDir();

  if ( QFile::exists( "/proc/mounts" ) )
  {
    dir->setPath( "/proc" );
    QDir::setCurrent( dir->path() );
  }
  else
  {
    if ( !m_proc_error )
    {
      m_proc_error = true;
      emit error( ERROR_FILE_NOT_FOUND, QString( "%1/%2" ).arg( dir->path() ).arg( file.name() ) );
    }
  }

  if ( !dir->path().isEmpty() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );
      while( !ts.atEnd() )
      {
        QString line = ts.readLine().stripWhiteSpace();
        if ( line.contains( "smbfs", false ) != 0 )
        {
          QString mounted = line.section( "smbfs", 0, 0 ).stripWhiteSpace();

          QString name = mounted.section( " ", 0, 0 ).stripWhiteSpace();
          QString path = mounted.section( " ", 1, 1 ).stripWhiteSpace();

          // Get the uid and gid of the user, who mounted this share and append them to the list.
          QString uid = line.section( "uid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();
          QString gid = line.section( "gid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();
          
          shares.append( new Smb4KShare(name, path, uid.toInt(), gid.toInt()) );
        }
      }
      file.close();
    }
    else
      emit error( ERROR_READING_FILE, QString( "%1/%2" ).arg( dir->path() ).arg( file.name() ) );
  }

  delete dir;

  // Check, if the entries are already in the list.
  // If yes, replace the entry in the temporary by the one from the 
  // old list of shares. This way we won't get into trouble with 
  // shares that contain spaces.
  if ( m_mounted_shares.count() != 0 )
  {
    for ( QValueList<Smb4KShare *>::Iterator it = shares.begin(); it != shares.end(); ++it )
    {
      Smb4KShare *s = findShareByPath( (*it)->getPath() );
      
      if ( s )
      {
        *it = s;
      }
      else
        continue;
    }
  }
  
  if( m_mounted_shares.count() != shares.count() )
  {
    emit unmountedAllShares();
  }
  
  m_mounted_shares = shares;
  
  emit importedShares();
  
  m_working = false;


#else
  *m_proc << "mount";
  
  startProcess( Import );

#endif
}


/***************************************************************************
   This function is run on exit of the whole program. It cares about the
   last second actions, that are to be done.
***************************************************************************/

void Smb4KMounter::exit()
{
  // First tell the program, that this will be the last job
  // of the mounter for this session.
  m_lastJob = true;

  // Clean up!
  if ( m_cleanUp )
  {
    // Now set the path, where the progam has to clean up.
    QDir *dir = new QDir();
    dir->setPath( m_defaultPath );

    // Get the contents of the default directory and remove everything from
    // the list we do not need.
    QStringList dirs = dir->entryList( QDir::Dirs, QDir::DefaultSort );
    dirs.remove( "." );
    dirs.remove( ".." );

    // Now, remove all directories, that are present in the base directory
    // except those that are to remain mounted.
    for ( QStringList::ConstIterator it = dirs.begin(); it != dirs.end(); ++it )
    {
      dir->cd( *it );
      QStringList subdirs = dir->entryList( QDir::Dirs, QDir::DefaultSort );
      subdirs.remove( "." );
      subdirs.remove( ".." );

      // Make sure, that no directory of a mounted share will be touched. The
      // admins of the file servers will get after you, if something will
      // accidentally be deleted!
      for ( QStringList::ConstIterator i = subdirs.begin(); i != subdirs.end(); ++i )
      {
        QString path = dir->absPath().append( "/" ).append( *i );
        if( findShareByPath(path) == 0 )
        {
          dir->rmdir( *i );
        }
      }

      dir->cdUp();
      dir->rmdir( *it );
    }
  }

  // Now store the mounted shares in the config file, if the user wants
  // to mount recently used shares on start-up.
  if ( m_saveShares )
  {
    m_config->setGroup( "Recently Mounted Shares" );
    
    QStringList list;
    
    for ( int i = 0; ; i++ )
    {
      QString tmp = m_config->readEntry( QString( "%1" ).arg( i ), QString::null );
      
      if ( !tmp.isEmpty() )
        list.append( tmp );
      else
        break;
    }
    
    m_config->deleteGroup( "Recently Mounted Shares", true, false );
    m_config->setGroup( "Recently Mounted Shares" );

    int index = 0;

    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( (*it)->getUID() == m_uid )//&& (*it)->getGID() == m_gid )
      {
        if ( (*it)->getRealName().contains( "_" ) == 0 )
        {
          m_config->writeEntry( QString( "%1" ).arg( index++ ), (*it)->getRealName() );
        }
        else
        {
          if ( list.find( (*it)->getRealName().replace( "_", " " ) ) == list.end() )
            m_config->writeEntry( QString( "%1" ).arg( index++ ), (*it)->getRealName() );
          else
            m_config->writeEntry( QString( "%1" ).arg( index++ ), *(list.find( (*it)->getRealName().replace( "_", " " ) )) );
        }
      }
    }
  }

  // Unmount shares.
  if ( m_unmountAll )
    unmountAll();
}


/***************************************************************************
   Mounts a share. (Public part)
***************************************************************************/

void Smb4KMounter::mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  QString *input = new QString( QString( "%1:%2:%3:%4:%5" ).arg( Mount ).arg( workgroup ).arg( host ).arg( ip ).arg( share ) );
  m_queue.enqueue( input );
}



/***************************************************************************
   Mounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  m_workgroup = workgroup;
  m_host = host;
  m_ip = ip; 
  
  QString mountpoint;
  
  if ( share.stripWhiteSpace().contains( " " ) )
  {
    m_share = QString( "'%1'" ).arg( share );
    mountpoint = share;
    mountpoint = mountpoint.replace( QRegExp( " " ), "_" );
  }
  else
    m_share = share;
    
  m_path = m_defaultPath;

#ifdef __FreeBSD__
  // The mount command of FreeBSD returns the canonical path.
  // To avoid too much trouble, we adjust the path now:
  QDir *d = new QDir( m_path );
  
  if ( m_path != d->canonicalPath().append( "/" ) )
    m_path = d->canonicalPath().append( "/" );
    
  delete d;
#endif

  // Test, whether the share is already mounted. We test for the mount point
  // and the share. This has to be done, because the user might change the
  // default directory during program run.
  QString test_mp;  

  if ( !mountpoint.isEmpty() )
  {
    test_mp = QString( "%1%2/%3" ).arg( m_path ).arg( m_host ).arg( mountpoint );
  }
  else
  {
    test_mp = QString( "%1%2/%3" ).arg( m_path ).arg( m_host ).arg( m_share );
  }
  
  QString test_share = QString( "//%1/%2" ).arg( m_host ).arg( m_share );
  
  bool alreadyMounted = false;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++ it )
  {
    if ( (*it)->getPath().upper() == test_mp.upper() )
    {
      alreadyMounted = true;
      break;
    }
    else if ( ( (*it)->getName().upper() == test_share.upper() || (*it)->getName().upper() == test_share.replace( " ", "_" ).upper() ) && (*it)->getUID() == m_uid )
    {
      alreadyMounted = true;
      break;
    }
  }

  // Start the mount process.
  if ( alreadyMounted == true )
  {
    emit error( ERROR_ALREADY_MOUNTED, QString::null );
    emit running( MOUNTER_STOP, false );
    m_working = false;
  }
  else
  {
    QDir *dir = new QDir();
    dir->setPath( m_path );
    if ( !dir->exists() )
      dir->mkdir( m_path, true );

    // If the user wants the generated subdirectories to be lower case,
    // then we have to care about this right here:
    if( !m_lowercase )
    {
      m_path.append( host );
      dir->setPath( m_path );

      if ( !dir->exists() )
        dir->mkdir( m_path, true );

      if ( !mountpoint.isEmpty() )
        m_path.append( "/" ).append( mountpoint );
      else
        m_path.append( "/" ).append( share );

      dir->setPath( m_path );

      if ( !dir->exists() )
        dir->mkdir( m_path, true );
    }
    else
    {
      m_path.append( host.lower() );
      dir->setPath( m_path );

      if ( !dir->exists() )
        dir->mkdir( m_path, true );

      if ( !mountpoint.isEmpty() )
        m_path.append( "/" ).append( mountpoint.lower() );
      else
        m_path.append( "/" ).append( share.lower() );

      dir->setPath( m_path );

      if ( !dir->exists() )
        dir->mkdir( m_path, true );
    }

    // Get the authentication data:
    Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( workgroup, host, share );

    //
    // Assemble the command:
    //
    QString command;
    
#ifndef __FreeBSD__

    // The command itself:
    if ( !m_useSuper || !m_super )
      command = QString( "smbmount " );
    else if ( m_useSuper && m_super )
      command = QString( "super mount -t smbfs " );

    // The share name and the mount point:
    command.append( QString( "//%1/%2 %3 " ).arg( KProcess::quote( m_host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( m_path ) ) );

    // The options:

    // (1) The workgroup
    command.append( QString( "-o workgroup=%1," ).arg( KProcess::quote( workgroup ) ) );

    // (2) The IP address
    if ( !ip.isEmpty() )
      command.append( QString( "ip=%1," ).arg( KProcess::quote( ip ) ) );


    // (3) The advanced options
    if ( !m_advancedOpts.stripWhiteSpace().isEmpty() )
      command.append( m_advancedOpts );

    // We have to make sure, that if the user uses super,
    // the GID and the UID aren't the ones of root, except this
    // is explicitly wanted:
    if ( command.contains( "uid=", true ) == 0 && command.contains( "gid=", true ) == 0 && m_useSuper )
      command.append( QString( "uid=%1,gid=%2," ).arg( QString( "%1" ).arg( m_uid ) ).arg( QString( "%1" ).arg( m_gid ) ) );

    // (4) The authentication data:
    if ( !auth->user().isEmpty() )
    {
      command.append( QString( "username=%1," ).arg( KProcess::quote( auth->user() ) ) );

      // We want to allow empty passwords:
      if ( !auth->password().isEmpty() )
        command.append( QString( "password=%1," ).arg( auth->maskedPassword() ) );
    }
    else
      command.append( "guest" );
      
#else

    if ( !m_useSuper || !m_super )
      command = QString( "mount_smbfs " );
    else if ( m_useSuper && m_super )
      command = QString( "super mount_smbfs " );
      
    // The advanced options:
    if ( !m_advancedOpts.isEmpty() )
      command.append( m_advancedOpts );
      
    // The rest of the options.
    if ( !m_ip.isEmpty() )
      command.append( " -I "+KProcess::quote( m_ip ) );
      
    if ( !m_workgroup.isEmpty() )
      command.append( " -W "+KProcess::quote( m_workgroup ) );
      
    command.append( " -N" );
    
    // We have to make sure, that if the user uses super,
    // the GID and the UID aren't the ones of root, except this
    // is explicitly wanted:
    if ( command.contains( "-u", true ) == 0 && command.contains( "-g", true ) == 0 && m_useSuper )
      command.append( QString( " -u %1 -g %2" ).arg( QString( "%1" ).arg( m_uid ) ).arg( QString( "%1" ).arg( m_gid ) ) );
    
    if ( !auth->user().isEmpty() )
      command.append( " //"+KProcess::quote( auth->user() )+"@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( m_path ) );
    else
      command.append( " //guest@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( m_path ) );
    
#endif

    delete dir;
    
    // Start the mount process:
    *m_proc << command;

    startProcess( Mount );
  }
}


/****************************************************************************
   Unmounts a share. (Public part)
****************************************************************************/

void Smb4KMounter::unmountShare( const QString &mountpoint, const QString &uid, const QString &gid )
{
  QString *input = new QString( QString( "%1:%2:%3:%4" ).arg( Unmount ).arg( mountpoint ).arg( uid ).arg( gid ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::unmount( const QString &mountpoint, const QString &uid, const QString &gid )
{
  if ( !mountpoint.isEmpty() )
  {
    if ( uid.toInt() == m_uid && gid.toInt() == m_gid )
    {
      m_path = mountpoint;
      QString command;

      if ( !m_useSuper )
        command = QString( "smbumount " );
      else if ( m_useSuper && m_super )
        command = QString( "super umount " );

      command.append( KProcess::quote( mountpoint ) );
      m_proc->operator<<( command );
      startProcess( Unmount );
    }
    else
    {
      if ( m_allowUnmountForeign )
      {
        m_path = mountpoint;
        QString command;

        if ( !m_useSuper )
          command = QString( "smbumount " );
        else if ( m_useSuper && m_super )
          command = QString( "super umount " );

        command.append( KProcess::quote( mountpoint ) );
        m_proc->operator<<( command );
        startProcess( Unmount );
      }
      else
      {
        emit error( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        emit running( MOUNTER_STOP, false );
        m_working = false;
      }
    }
  }
}


/****************************************************************************
   Unmounts a dead share. (Public part)
****************************************************************************/

void Smb4KMounter::forceUnmountShare( const QString &mountpoint, const QString &uid, const QString &gid )
{
  QString *input = new QString( QString( "%1:%2:%3:%4" ).arg( ForceUnmount ).arg( mountpoint ).arg( uid ).arg( gid ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a dead share. (Private part)
***************************************************************************/

void Smb4KMounter::forceUnmount( const QString &mountpoint, const QString &uid, const QString &gid )
{
  if ( !mountpoint.isEmpty() )
  {
    if ( m_allowUnmountForeign )
    {
      if ( KMessageBox::questionYesNo( (QWidget *)this, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
      {
        m_path = mountpoint;
        QString command( "super umount -l " );
        command.append( KProcess::quote( mountpoint ) );

        m_proc->operator<<( command );
        startProcess( Unmount );
      }
    }
    else
    {
      if ( uid.toInt() == m_uid && gid.toInt() == m_gid )
      {
        if ( KMessageBox::questionYesNo( (QWidget *)this, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes  )
        {
          m_path = mountpoint;
          QString command( "super umount -l " );
          command.append( KProcess::quote( mountpoint ) );

          m_proc->operator<<( command );
          startProcess( Unmount );
        }
      }
      else
      {
        emit error( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        emit running( MOUNTER_STOP, false );
        m_working = false;
      }
    }
  }
}


/***************************************************************************
   Unmounts all shares at once. (Public part)
***************************************************************************/

void Smb4KMounter::unmountAllShares()
{
  QString *input = new QString( QString( "%1" ).arg( UnmountAll ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts all shares at once.
***************************************************************************/

void Smb4KMounter::unmountAll()
{
  QString command;

  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if ( m_allowUnmountForeign )
    {
      if ( !m_useSuper )
      {
        command.append( "smbumount " ).append( KProcess::quote( (*it)->getPath() ) ).append( " ; " );
      }
      else if ( m_useSuper && m_super )
      {
        command.append( "super umount " ).append( KProcess::quote( (*it)->getPath() ) ).append( " ; " );
      }
        
      QDir *d = new QDir( (*it)->getPath() );
        
      if ( m_lastJob && m_cleanUp && d->canonicalPath().startsWith( QDir( m_defaultPath ).canonicalPath() ) )
      {
        command.append( "rmdir --ignore-fail-on-non-empty" ).append( KProcess::quote( d->canonicalPath() ) ).append( " ; " );
        command.append( "rmdir --ignore-fail-on-non-empty" ).append( KProcess::quote( d->canonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
      }
        
      delete d;
    }
    else
    {
      if ( (*it)->getUID() == (int)getuid() && (*it)->getGID() == (int)getgid() )
      {
        if ( !m_useSuper )
        {
          command.append( "smbumount " ).append( KProcess::quote( (*it)->getPath() ) ).append( " ; " );
        }
        else if ( m_useSuper && m_super )
        {
          command.append( "super umount " ).append( KProcess::quote( (*it)->getPath() ) ).append( " ; " );
        }
          
        QDir *d = new QDir( (*it)->getPath() );
        
        if ( m_lastJob && m_cleanUp && d->canonicalPath().startsWith( QDir( m_defaultPath ).canonicalPath() ) )
        {
          command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( d->canonicalPath() ) ).append( " ; " );
          command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( d->canonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
        }
        
        delete d;        
      }
    }
  }
  command.truncate( command.length() - 2 );

  *m_proc << command;

  startProcess( UnmountAll );
}

/***************************************************************************
   Starts any process.
***************************************************************************/

void Smb4KMounter::startProcess( int state )
{
  m_buffer = QString::null;
  m_state = state;

  if ( m_state != Import )
    QApplication::setOverrideCursor( waitCursor );

  if ( m_lastJob )
  {
    m_proc->detach();
    m_proc->start( KProcess::DontCare, KProcess::NoCommunication );
  }
  else
    m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/***************************************************************************
   Ends any process. This functions tells the mounter what to do
   afterwards.
***************************************************************************/

void Smb4KMounter::endProcess()
{
  switch ( m_state )
  {
#ifdef __FreeBSD__
    case Import:
      processImport();
      break;
#endif
    case Mount:
      processMount();
      break;
    case Unmount:
      processUnmount();
      break;
    case ForceUnmount:
      processUnmount();
      break;
    case UnmountAll:
      processUnmountAll();
      break;
    default:
      break;
  }
  m_state = Idle;
  m_path = QString::null;
  QApplication::restoreOverrideCursor();
  m_proc->clearArguments();

  m_working = false;
  emit running( MOUNTER_STOP, m_working );
}


/***************************************************************************
   Process mounts.
***************************************************************************/

void Smb4KMounter::processMount()
{
#ifndef __FreeBSD__
    
  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "failed", true ) == 0 && 
         m_buffer.contains( "ERR", true ) == 0 && 
         m_buffer.contains( "/bin/sh:" ) == 0 &&
         m_buffer.contains( "mount:", true ) == 0 && 
         m_buffer.contains( "smbmnt" ) == 0 &&
         m_buffer.contains( m_path ) == 0 )
    {
      // If the share does contain spaces, then we have to replace them by
      // underscores and we also have to remove the single quotation marks.
      
      QString mountpoint;
      
      if ( m_share.contains( "'" ) == 2 )
      {
        m_share = m_share.replace( QRegExp( "'" ), "" );
      }

      QString name = QString( "//%1/%2" ).arg( m_host ).arg( m_share );
      Smb4KShare* smb_share = new Smb4KShare( name, m_path, m_uid, m_gid );

      m_mounted_shares.append(smb_share);

      emit mountedShare(smb_share);
    }
    else
    {
      if ( m_buffer.contains( "ERRbadpw" ) != 0 || m_buffer.contains( "ERRnoaccess" ) != 0 )
      {
        if ( m_share.contains( "'" ) == 2 )
        {
          m_share = m_share.replace( QRegExp( "'" ), "" );
        }
        
        // If the user supplied auth information, we will retry mounting.
        if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_workgroup, m_host, m_share ) )
        {
          mountShare( m_workgroup, m_host, m_ip, m_share );
        }
      }
      else if ( m_buffer.contains( "ERRnosuchshare" ) != 0 )
      {
        if ( m_share.contains( "_" ) != 0 )
        {
          m_share = m_share.replace( "_", " " );
        }
        
        // If the user supplied auth information, we will retry mounting.
        if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_workgroup, m_host, m_share ) )
        {
          mountShare( m_workgroup, m_host, m_ip, m_share );
        }        
      }
      else
        emit error( ERROR_MOUNTING_SHARE, m_buffer );
    }
  }
  
#else
    
  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "Authentication error", true ) == 0 
         && m_buffer.contains( "Connection refused", true ) == 0 
         && m_buffer.contains( "Operation not permitted", true ) == 0 )
    {
      // If the share does contain spaces, then we have to replace them by
      // underscores and we also have to remove the single quotation marks.
      if ( m_share.contains( "'" ) == 2 )
      {
        m_share = m_share.replace( QRegExp( "'" ), "" );
      }

      Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( m_workgroup, m_host, m_share );
      
      QString name = QString( "//%1@%2/%3" ).arg( auth->user().upper(), m_host.upper(), m_share.upper() );
      Smb4KShare* smb_share = new Smb4KShare(name, m_path, m_uid, m_gid );

      m_mounted_shares.append(smb_share);

      emit mountedShare(smb_share);
    }
    else
    {
      if ( m_buffer.contains( "Authentication error" ) != 0 )
      {
        // If the user supplied auth information, we will retry mounting.
        if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_workgroup, m_host, m_share ) )
        {
          mountShare( m_workgroup, m_host, m_ip, m_share );
        }
      }
      else
        emit error( ERROR_MOUNTING_SHARE, m_buffer );
    }
  }
  
#endif

  m_workgroup = QString::null;
  m_host = QString::null;
  m_share = QString::null;
  m_ip = QString::null;
}


/***************************************************************************
   Process unmounts.
***************************************************************************/

void Smb4KMounter::processUnmount()
{
  if ( m_proc->normalExit() )
  {
    // FIXME: This is a potential problem. What if the unmount
    // process returns some debugging info like the mount process
    // under SuSE? Need more info.
    if ( m_buffer.isEmpty() )
    {
      // Since the mount point is unique, the mount point is greped
      // and used to determine the share that has to be removed from
      // m_mountedShares.
      Smb4KShare *share = findShareByPath(m_path);

      emit unmountedShare( share->getName(), share->getPath() );
      m_mounted_shares.remove(share);
    }
    else
      emit error( ERROR_UNMOUNTING_SHARE, m_buffer );
  }
}


/***************************************************************************
   What to do after all shares were unmounted at once.
***************************************************************************/

void Smb4KMounter::processUnmountAll()
{
  if ( m_proc->normalExit() )
  {
    if ( m_buffer.isEmpty() )
    {
      m_mounted_shares.clear();
      emit unmountedAllShares();
    }
    else
    {
      QValueList<Smb4KShare *> tmp_list;

      for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
      {
        QString path = (*it)->getPath();
        if ( m_buffer.contains( path ) == 0 )
        {
          tmp_list.append(*it);
        }
      }

      for ( QValueListIterator<Smb4KShare *> it = tmp_list.begin(); it != tmp_list.end(); ++it )      
      {
        m_mounted_shares.remove(*it);
      }

      tmp_list.clear();

      emit unmountedAllShares();
      emit error( ERROR_UNMOUNTING_ALL, m_buffer );
    }
  }
}


#ifdef __FreeBSD__

/***************************************************************************
  Process the output of the mount command to import the SMB shares 
  mounted on the system.
***************************************************************************/

void Smb4KMounter::processImport()
{
  QStringList list = QStringList::split( '\n', m_buffer, false ).grep( "(smbfs)" );
  
  QValueList<Smb4KShare *> shares;
  
  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    QString name = (*it).section( " on ", 0, 0 ).stripWhiteSpace();
    QString mountpoint = (*it).section( " on ", 1, 1 ).section( "(smbfs)", 0, 0 ).stripWhiteSpace();
    
    // We can get the UID and GID from the rights, that were given to the 
    // mount point.
    QFileInfo *info = new QFileInfo( mountpoint+"/." );
    
    int uid = (int)info->ownerId();
    int gid = (int)info->groupId();
    
    shares.append( new Smb4KShare( name, mountpoint, uid, gid ) );
    
    delete info;
  }
  
  // Check, if the entries are already in the list.
  // If yes, replace the entry in the temporary by the one from the 
  // old list of shares. This way we won't get into trouble with 
  // shares that contain spaces.
  if ( m_mounted_shares.count() != 0 )
  {
    for ( QValueList<Smb4KShare *>::Iterator it = shares.begin(); it != shares.end(); ++it )
    {
      Smb4KShare *s = findShareByPath( (*it)->getPath() );
      
      if ( s )
      {
        *it = s;
      }
      else
        continue;
    }
  }
  
  if( m_mounted_shares.count() != shares.count() )
  {
    emit unmountedAllShares();
  }

  m_mounted_shares = shares;
  
  emit importedShares();
}

#endif


/***************************************************************************
   Read the options.
***************************************************************************/

void Smb4KMounter::readOptions()
{
  //
  // Mount options
  //
  m_config->setGroup( "Mount Options" );
  m_defaultPath = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  // Set this variable to true, so that unmountAll() knows, that it has
  // to put a 'rm -rf *' after the unmounting command when exiting.
  m_cleanUp = m_config->readBoolEntry( "Clean Up", true );
  m_unmountAll = m_config->readBoolEntry( "Unmount All", false );
  m_lowercase = m_config->readBoolEntry( "Force Lowercase", false );
  m_useSuper = m_config->readBoolEntry( "Use SU Privileges", false );
  m_saveShares = m_config->readBoolEntry( "Mount Recent", false );
  m_allowUnmountForeign = m_config->readBoolEntry( "Allow Unmount Foreign", false );
  m_super = m_config->readBoolEntry( "Super", false );

  //
  // Advanced mount options.
  //
  m_advancedOpts = QString::null;

  m_config->setGroup( "Samba" );
  
#ifndef __FreeBSD__

  if ( m_config->readBoolEntry( "Use Kerberos", false ) )
    m_advancedOpts.append( "krb," );

  if ( !m_config->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "netbiosname='%1'," ).arg( m_config->readEntry( "NetBIOS Name", QString::null ) ) );

  if ( !m_config->readEntry( "Mount UID", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "uid=%1," ).arg( m_config->readEntry( "Mount UID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount GID", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "gid=%1," ).arg( m_config->readEntry( "Mount GID", QString::null ) ) );

  if ( !m_config->readEntry( "Port", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "port=%1," ).arg( m_config->readNumEntry( "Port", 139 ) ) );

  if ( !m_config->readEntry( "Mount FMASK", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "fmask=%1," ).arg( m_config->readEntry( "Mount FMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Mount DMASK", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "dmask=%1," ).arg( m_config->readEntry( "Mount DMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Socket Options", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "sockopt=\"%1\"," ).arg( m_config->readEntry( "Socket Options", QString::null ) ) );

  if ( !m_config->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "scope=\"%1\"," ).arg( m_config->readEntry( "NetBIOS Scope", QString::null ) ) );

  if ( m_config->readBoolEntry( "Mount ReadWrite", true ) )
    m_advancedOpts.append( "rw," );
  else
    m_advancedOpts.append( "ro," );

  if ( !m_config->readEntry( "Mount Charset", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "iocharset=%1," ).arg( m_config->readEntry( "Mount Charset", QString::null ) ) );

  if ( !m_config->readEntry( "Mount Codepage", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "codepage=%1," ).arg( m_config->readEntry( "Mount Codepage", QString::null ) ) );

  if ( !m_config->readEntry( "Mount Cache", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "ttl=%1," ).arg( m_config->readEntry( "Mount Cache", QString::null ) ) );

  if ( m_config->readBoolEntry( "Mount Unicode", false ) )
    m_advancedOpts.append( "unicode," );

  if ( m_config->readBoolEntry( "Mount LFS", false ) )
    m_advancedOpts.append( "lfs," );
    
#else

  if ( !m_config->readEntry( "Mount UID", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( "-u %1" ).arg( m_config->readEntry( "Mount UID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount GID", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( " -g %1" ).arg( m_config->readEntry( "Mount GID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount FMASK", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( " -f %1" ).arg( m_config->readEntry( "Mount FMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Mount DMASK", QString::null ).isEmpty() )
    m_advancedOpts.append( QString( " -d %1" ).arg( m_config->readEntry( "Mount DMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Mount Charset", QString::null ).isEmpty() && !m_config->readEntry( "Mount Codepage", QString::null ).isEmpty() )
  {
    m_advancedOpts.append( QString( " -E %1:%2" ).arg( m_config->readEntry( "Mount Charset", QString::null ) ).arg( m_config->readEntry( "Mount Codepage", QString::null ) ) );
  }

#endif
}


/***************************************************************************
   Check if a bookmark points to a share already mounted
***************************************************************************/

bool Smb4KMounter::isMounted(const QString &bookmark)
{
  for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    QString path = (*it)->getPath();
    if( path == bookmark )
    {
      return TRUE;
    }
  }
  return FALSE;
}


/***************************************************************************
   Find a share in the list with its path
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByPath(const QString &path)
{
  QValueListIterator<Smb4KShare *> it;
  
  QDir *d = new QDir( path );
  QString p = d->canonicalPath().append( "/" );
  
  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( (*it)->getPath().upper() == path.upper() || (*it)->getPath().upper() == p.upper() )
    {
      break;
    }
  }
  
  delete d;
  
  return it == m_mounted_shares.end() ? NULL : *it;
}


/***************************************************************************
   Find a share in the list with its name
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByName(const QString &name)
{
  QString n = name;
  
  QValueListIterator<Smb4KShare *> it;
  
  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( (*it)->getName().upper() == name.upper() )
    {
      break;
    }
    else if ( (*it)->getName().upper() == n.replace( " ", "_" ).upper() )
    {
      break;
    }
  }
  
  return it == m_mounted_shares.end() ? NULL : *it;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

/****************************************************************************
   Internal slots.
****************************************************************************/


void Smb4KMounter::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KMounter::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::init()
{
  if ( !m_working && !m_queue.isEmpty() )
  {
    // Tell the mounter, that it is busy.
    m_working = true;

    QString *item = m_queue.dequeue();
    int todo = item->section( ":", 0, 0 ).toInt();

    switch ( todo )
    {
      case MountRecent:
        mountRecent();
        break;
      case Import:
        import();
        break;
      case Mount:
        emit running( MOUNTER_MOUNTING, m_working );
        mount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ) );
        break;
      case Unmount:
        emit running( MOUNTER_UNMOUNTING, m_working );
        unmount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      case ForceUnmount:
        emit running( MOUNTER_UNMOUNTING, m_working );
        forceUnmount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      case UnmountAll:
        emit running( MOUNTER_UNMOUNTING_ALL, m_working );
        unmountAll();
        break;
      default:
        break;
    }
  }
}


/***************************************************************************
   Initializes periodic checking. At the moment only the detection of 
   mounted shares is supported. (connected to second timer)
***************************************************************************/

void Smb4KMounter::check()
{
  if ( !m_working || m_queue.isEmpty() )
  {
    // Just append what has to be checked to the queue.
    m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  }
}


#include "smb4kmounter.moc"
