/***************************************************************************
    smb4kmounter.h  -  The core class, that mounts shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KMOUNTER_H
#define SMB4KMOUNTER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qptrqueue.h>

// KDE includes
#include <kprocess.h>
#include <kconfig.h>

// application specific includes
#include "smb4kshare.h"
#include "smb4kdefs.h"

/**
 * This is a core class. It mounts the shares.
 */
class Smb4KMounter : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KMounter( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KMounter();
    /**
     * Aborts a running process.
     */
    void abort();
    /**
     * Enumeration for the state of the process. The members are also
     * used in the results() signal as return values.
     */
    enum State{ MountRecent, Import, Mount, Unmount, UnmountAll, ForceUnmount, Idle };
    /**
     * Does everything, that has to be done on exit. Retruns TRUE if everything
     * has been accomplished.
     */
    void exit();
    /**
     * Reads the options.
     */
    void readOptions();
    /**
     * Unmounts a share.
     */
    void unmountShare( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Unmounts all shares.
     */
    void unmountAllShares();
    /**
     * Mounts a share.
     */
    void mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share );
    /**
     * Unmounts a dead share using super and 'umount -l' command.
     */
    void forceUnmountShare( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Returns the unsorted list of mounted shares.
     */
    const QValueList<Smb4KShare *> &getShares() { return m_mounted_shares; };
    /**
     * Check if a bookmark points to a share already mounted
     */
    bool isMounted( const QString &bookmark );
    /**
     * Find a share in the list with its path.
     */
    Smb4KShare* findShareByPath( const QString &path );
    /**
     * Find a share in the list with its name
     */
    Smb4KShare* findShareByName( const QString &name );
    /**
     * Returns the UID of the user.
     */
    int getUID(){return m_uid;}
    /**
     * Returns the GID of the user.
     */
    int getGID(){return m_gid;}

  signals:
    /**
     * This signal is emitted, when the scanner starts and ends.
     */
    void running( int mode, bool on );
    /**
     * Tells the program, that the shares list has been updated.
     */
    void importedShares();
    /**
     * Emits the mounted share data.
     */
    void mountedShare(Smb4KShare *);
    /**
     * Emits the mounted share data.
     */
    void unmountedShare(QString share, QString path);
    /**
     * Is emitted, if all shares should have been unmounted. The string list
     * is empty, if the unmount process ended successfully.
     */
    void unmountedAllShares();
    /**
     * In case of an error this signal emits the error code 
     * and an error message.
     */
    void error( int error_code, const QString &error_message );

  protected:
    /**
     * Starts a process. It takes an interger, that determines,
     * which process has to be started.
     */
    void startProcess( int state );
    /**
     * Is called, when the process ended.
     */
    void endProcess();
    /**
     * Finishes the mounting of shares.
     */
    void processMount();
    /**
     * Finishes the unmounting of a single share.
     */
    void processUnmount();
    /**
     * Finishes the unmounting of all shares.
     */
    void processUnmountAll();
#ifdef __FreeBSD__
    /**
     * This function is FreeBSD specific.It is used to detect the
     * SMB shares mounted on the system.
     */
    void processImport();     
#endif

  protected slots:
    /**
     * Is called, when the process exits.
     */
    void slotProcessExited( KProcess * );
    /**
     * Is called, if output is received at Stdout.
     */
    void slotReceivedStdout( KProcess *, char *buf, int len );
    /**
     * Is called, if output is received a Stderr.
     */
    void slotReceivedStderr( KProcess *, char *buf, int len );
    /**
     * Initiates the processing of the queue.
     */
    void init();
    /**
     * Initializes periodic checking. At the moment only the 
     * detection of mounted shares is supported.
     */
    void check();

  private:
    /**
     * Mount recently used shares.
     */
    void mountRecent();
    /**
     * Imports mounted shares.
     */
    void import();
    /**
     * Mounts a selected share.
     */
    void mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share );
    /**
     * Unmounts the selected item.
     */
    void unmount( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Unmounts dead shares.
     */
    void forceUnmount( const QString &mountpoint, const QString &uid, const QString &gid );
    /**
     * Unmounts all shares a once.
     */
    void unmountAll();
    /**
     * The KProcess object.
     */
    KProcess *m_proc;
    /**
     * The KConfig objects. With its help the options are read.
     */
    KConfig *m_config;
    /**
     * The buffer.
     */
    QString m_buffer;
    /**
     * Holds the mount point of a share.
     */
    QString m_path;
    /**
     * Holds the workgroup of the currently processed shares.
     */
    QString m_workgroup;
    /**
     * Holds the host's name of the currently processed share.
     */
    QString m_host;
    /**
     * Holds the IP address of the currently processed share.
     */
    QString m_ip;
    /**
     * Holds the name of the currently processed share.
     */
    QString m_share;
    /**
     * The default path where Smb4K mounts the shares.
     */
    QString m_defaultPath;
    /**
     * Holds the state the mounter is in.
     */
    int m_state;
    /**
     * Determines whether this is the last job of the mounter.
     */
    bool m_lastJob;
    /**
     * Determines whether the default directory should be cleaned up
     * on exit.
     */
    bool m_cleanUp;
    /**
     * Determines whether all shares should be unmounted on exit.
     */
    bool m_unmountAll;
    /**
     * The queue, where the incoming requests are stored.
     */
    QPtrQueue<QString> m_queue;
    /**
     * The timer for the queue.
     */
    QTimer *m_timer1;
    /**
     * The timer for all the checks that should be done time by time.
     */
    QTimer *m_timer2;
    /**
     * Determines, whether the mounter is running or not.
     */
    bool m_working;
    /**
     * Holds whether the generate subdirectories have to be lower case
     * or not.
     */
    bool m_lowercase;
    /**
     * Determines whether the user wants to use the super program
     * for mounting and unmounting or not.
     */
    bool m_useSuper;
    /**
     * Is true if the super program is installed, false otherwise.
     */
    bool m_super;
    /**
     * Determines whether the mounted shares should be written to the
     * config file on exit.
     */
    bool m_saveShares;
    /**
     * Holds the list of currently mounted shares as a pointer list.
     */
    QValueList<Smb4KShare *> m_mounted_shares;
    /**
     * Holds the UID of the user.
     */
    int m_uid;
    /**
     * Holds the GID of the user.
     */
    int m_gid;
    /**
     * This string holds the advanced options, that were defined by the
     * user in the configuration tab. Only options that differ from the
     * default will be entered.
     */
    QString m_advancedOpts;
    /**
     * Determines whether the user is allowed to unmount shares that are
     * owned by other users.
     */
    bool m_allowUnmountForeign;
    /**
     * Makes sure that the error message concerning the missing of
     * the file /proc/mounts is only shown once.
     */
    bool m_proc_error;
};

#endif
