/***************************************************************************
    smb4k.cpp  -  The main class of Smb4K.
                             -------------------
    begin                : Sam Mr  1 14:57:21 CET 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qtooltip.h>

// KDE includes
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kstdaction.h>
#include <ksystemtray.h>

// application specific includes
#include "smb4k.h"
#include "widgets/smb4kbrowserwidget.h"
#include "widgets/smb4knetworktab.h"
#include "widgets/smb4ksearchtab.h"
#include "widgets/smb4ksharewidget.h"
#include "widgets/smb4kbookmarkmenuwidget.h"
#include "widgets/smb4koptionsdlg.h"
#include "core/smb4kdefs.h"

Smb4KApp *Smb4KApp::sApp = 0L;
QPopupMenu *menu;

#define ID_STATUS_MSG 1


Smb4KApp::Smb4KApp( QWidget* , const char* name ) : KMainWindow( 0, name )
{
  sApp = this;

  m_config = kapp->config();

  // The caption.
  setPlainCaption( i18n( "Smb4K - The SMB Share Browser" ) );

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initStatusBar();
  initCore();
  initView();
  initActions();
  setupBookmarkMenu();
  
  readOptions();

  ///////////////////////////////////////////////////////////////////
  // disable actions at startup
  abort->setEnabled( false );
  networkPreview->setEnabled( false );
  sharesMount->setEnabled( false );
  sharesUnmount->setEnabled( false );
  sharesUnmountAll->setEnabled( false );
  sharesForceUnmount->setEnabled( false );
  sharesStartFilemanager->setEnabled( false );
  networkAskPass->setEnabled( false );
  networkPrint->setEnabled( false );
//  bookmarksAddBookmark->setEnabled( false );
}


Smb4KApp::~Smb4KApp()
{
}


void Smb4KApp::initActions()
{
  quitApp = KStdAction::quit( this, SLOT( slotQuit() ), actionCollection(), "quit_smb4k" );

  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());

  abort = new KAction( i18n( "&Abort" ) , "stop", CTRL+Key_A, this, SLOT( slotAbort() ), actionCollection(), "abort_action" );

  networkRescan = new KAction( i18n( "&Rescan" ), "reload", CTRL+Key_R, m_core->scanner(), SLOT( slotInit() ), actionCollection(), "network_rescan" );
  
  networkSearch = new KAction( i18n( "&Search" ), "find", CTRL+Key_S, m_view, SLOT( slotShowSearchTab() ), actionCollection(), "network_search" );
  
  networkPreview = new KAction( i18n( "Pre&view" ), "view_icon", CTRL+Key_V, m_view->m_NetworkView, SLOT( slotPreview() ), actionCollection(), "network_preview" );
  
  networkAskPass = new KAction( i18n( "Au&thentication" ), "personal", CTRL+Key_T, m_view->m_NetworkView, SLOT( slotAskPass() ), actionCollection(), "network_askpass" );

  networkPrint = new KAction( i18n( "&Print File" ), "printer1", CTRL+Key_P, m_view->m_NetworkView, SLOT( slotPrint() ), actionCollection(), "network_print" );
    
  sharesMount = new KAction( i18n( "&Mount" ), "hdd_mount", CTRL+Key_M, m_view->m_NetworkView, SLOT( slotMountShare() ), actionCollection(), "shares_mount" );
  
  sharesUnmount = new KAction( i18n( "&Unmount" ), "hdd_unmount", CTRL+Key_U, m_view->m_ShareView, SLOT( slotUnmountShare() ), actionCollection(), "shares_unmount_share" );
  
  sharesForceUnmount = new KAction( i18n( "&Force Unmounting" ), "hdd_unmount", CTRL+Key_F, m_view->m_ShareView, SLOT( slotForceUnmountShare() ), actionCollection(), "shares_force_unmount" );
  
  sharesUnmountAll = new KAction( i18n( "U&nmount All" ), "gear", CTRL+Key_N, m_view->m_ShareView, SLOT( slotUnmountAllShares() ), actionCollection(), "shares_unmount_all" );
  
  sharesStartFilemanager = new KAction( i18n( "&Konqueror" ), "kfm_home", CTRL+Key_K, m_view->m_ShareView, SLOT( slotOpenFilemanager() ), actionCollection(), "shares_start_filemanager" );

  bookmarksAddBookmark = new KAction( i18n( "&Add Bookmark" ), "bookmark_add", CTRL+Key_B, m_view->m_NetworkView, SLOT( slotAddBookmark() ), actionCollection(), "bookmarks_add_bookmark" );
  
  bookmarksEditBookmarks = new KAction( i18n( "&Edit Bookmarks" ), "bookmark", QString::null, this, SLOT( slotEditBookmarks() ), actionCollection(), "bookmarks_edit_bookmarks" );

  settingsSmb4KOptions = KStdAction::preferences( this, SLOT( slotSmb4KOptionsDlg() ), actionCollection(), "configure_smb4k" );

  // use the absolute path to your smb4kui.rc file for testing purpose in createGUI();
  createGUI();
}


void Smb4KApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
  // TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem( i18n( "Ready." ), ID_STATUS_MSG, 1, false );
  statusBar()->setItemAlignment( ID_STATUS_MSG, AlignLeft );

  // Labels:
  m_winsLabel = new QLabel( statusBar() );
  m_winsLabel->setAlignment( Qt::AlignCenter );
  m_listLabel = new QLabel( statusBar() );
  m_listLabel->setAlignment( Qt::AlignCenter );
  QLabel *version = new QLabel( QString( "Smb4K %1" ).arg( VERSION ), statusBar() );
  version->setAlignment( Qt::AlignCenter );
  version->setFixedWidth( version->sizeHint().width() + 5 );

  m_progress = new KProgress( statusBar(), 0, 0 );
  m_progress->setFixedWidth( 75 );
  m_progress->setFixedHeight( version->fontMetrics().height() );
  m_progress->setPercentageVisible( false );

  m_progTimer = new QTimer( this );
  connect( m_progTimer, SIGNAL( timeout() ), SLOT( slotSetProgress() ) );

  statusBar()->addWidget( m_winsLabel, false );
  statusBar()->addWidget( m_listLabel, false );
  statusBar()->addWidget( m_progress, false );
  statusBar()->addWidget( version, false );

  QToolTip::add( m_winsLabel, i18n( "WINS server" ) );
  QToolTip::add( m_listLabel, i18n( "Browse list retrieval" ) );
  QToolTip::add( version, i18n( "Version" ) );
}


void Smb4KApp::initCore()
{
  // The core:
  m_core = new Smb4KCore( this, "Core" );
  
  // Connections
  connect( m_core, SIGNAL( runStateChanged() ), this, SLOT( slotCoreRunStateChanged() ) );
}


void Smb4KApp::setupBookmarkMenu()
{
  popup_bookmarks = new Smb4KBookmarkMenuWidget(Smb4KBookmarkMenuWidget::FULL, this, "bookmark_menu" );
  menuBar()->insertItem( i18n( "&Bookmarks" ), popup_bookmarks, 3, 3 );
}


void Smb4KApp::initView()
{
  ////////////////////////////////////////////////////////////////////
  // create the main widget here that is managed by KTMainWindow's view-region and
  // connect the widget to your document to display document contents.
  m_view = new Smb4KView( this );
  setCentralWidget(m_view);

  ////////////////////////////////////////////////////////////////////
  // emable the application for the docking in the system tray
  sys_tray = new Smb4KSystemTray( this );
  sys_tray->dock();
}


void Smb4KApp::saveOptions()
{
  m_config->setGroup("General Options");
  m_config->writeEntry( "Main Window Geometry" , size());
  m_config->writeEntry("Show Toolbar", viewToolBar->isChecked());
  m_config->writeEntry("Show Statusbar",viewStatusBar->isChecked());
  m_config->writeEntry("Toolbar Position", (int)toolBar("mainToolBar")->barPos());
  m_config->writeEntry("Toolbar Text Position", toolBar("mainToolBar")->iconText());
  m_config->writeEntry("Toolbar Icon Size", toolBar("mainToolBar")->iconSize());
  // Put the sizes of the QSplitters into the config file.
  m_config->writeEntry( "Vertical Splitter Sizes", m_view->sizes() );
  m_config->writeEntry( "Horizontal Splitter Sizes", m_view->m_container->sizes() );
}


void Smb4KApp::readOptions()
{
  m_config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = m_config->readBoolEntry("Show Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = m_config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();


  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) m_config->readNumEntry("Toolbar Position", KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);

  KToolBar::IconText toolBarIconText;
  toolBarIconText=(KToolBar::IconText) m_config->readNumEntry("Toolbar Text Position", KToolBar::IconOnly);
  toolBar("mainToolBar")->setIconText(toolBarIconText);
  toolBar("mainToolBar")->setIconSize( m_config->readNumEntry( "Toolbar Icon Size", 0 ) );

  QSize size = m_config->readSizeEntry( "Main Window Geometry" );

  if(!size.isEmpty())
  {
    resize(size);
  }

  // Read the sizes of the QSplitters.
  m_view->setSizes( m_config->readIntListEntry( "Vertical Splitter Sizes" ) );
  m_view->m_container->setSizes( m_config->readIntListEntry( "Horizontal Splitter Sizes" ) );

  // Set-up the status bar.
  m_config->setGroup( "Browse Options" );
  if ( m_config->readEntry( "Browse List", "nmblookup" ) == "nmblookup" )
    m_listLabel->setText( i18n( "Network" ) );
  else
    m_listLabel->setText( m_config->readEntry( "Query Host" ) );

  if ( m_config->readBoolEntry( "WINS Support", false ) )
    m_winsLabel->setText( m_config->readEntry( "WINS Server" ) );
  else
    m_winsLabel->setText( i18n( "None" ) );

  m_listLabel->setFixedWidth( m_listLabel->sizeHint().width() + 5 );
  m_winsLabel->setFixedWidth( m_winsLabel->sizeHint().width() + 5 );

  // EXTERNAL configuration options:
  m_view->m_NetworkView->readOptions();
  m_view->m_ShareView->readOptions();
  m_core->readOptions();
}


bool Smb4KApp::event( QEvent *e )
{
  switch( (int)e->type() )
  {
    case QEvent::Hide:
    {
      this->hide();
      break;
    }
    case QEvent::Show:
    {
      this->show();
      break;
    }
    case QEvent::Close:
    {
      if ( this->isVisible() && !kapp->sessionSaving() )
      {
        QHideEvent *hev = new QHideEvent();
        return KApplication::sendEvent( this, hev );
      }
      else if ( kapp->sessionSaving() )
      {
        // This should be similar to slotQuit().
        m_core->mounter()->exit();
        saveOptions();
        
        QCloseEvent *cev = (QCloseEvent *)e;
        cev->accept();                
      }
      break;
    }
    default:
    {
      break;
    }
  }
  return KMainWindow::event( e );
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////

void Smb4KApp::slotQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  m_core->mounter()->exit();
  saveOptions();
  KApplication::exit( 0 );
}


void Smb4KApp::slotAbort()
{
  slotStatusMsg( i18n( "Aborting..." ) );
  m_core->abort();
  slotStatusMsg( i18n( "Done." ) );
}


void Smb4KApp::slotCoreRunStateChanged()
{
  int mode = m_core->currentState();
  
  if ( m_core->isRunning() )
  {
    abort->setEnabled( true );
    
    if ( m_core->scannerIsRunning() )
      networkRescan->setEnabled( false );
    else
      networkRescan->setEnabled( true );
    
    switch ( mode )
    {
      case SCANNER_INIT:
        slotStatusMsg( i18n( "Searching network..." ) );
        break;
      case SCANNER_OPENING_WORKGROUP:
        slotStatusMsg( i18n( "Opening workgroup..." ) );
        break;
      case SCANNER_OPENING_HOST:
        slotStatusMsg( i18n( "Opening host..." ) );
        break;   
      case SCANNER_RETRIEVING_INFO:
        slotStatusMsg( i18n( "Retrieving information..." ) );
        break;                
      case SCANNER_RETRIEVING_PREVIEW:
        slotStatusMsg( i18n( "Retrieving preview..." ) );
        break;      
      case SCANNER_SEARCHING:
        slotStatusMsg( i18n( "Searching for host..." ) );
        break;  
      case IPFINDER_WORKING:
        slotStatusMsg( i18n( "Looking for IP addresses..." ) );
        break;
      case MOUNTER_MOUNTING:
        slotStatusMsg( i18n( "Mounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING:
        slotStatusMsg( i18n( "Unmounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING_ALL:
        slotStatusMsg( i18n( "Unmounting all shares..." ) );
        break;
      case PRINT_SEND_FILE:
        slotStatusMsg( i18n( "Sending file to printer..." ) );
        break;
      case PRINT_CONVERT_FILE:
        slotStatusMsg( i18n( "Converting file to Postscript..." ) );
        break;
      default:
        break;
    }
    
    if ( !m_progTimer->isActive() )
    {
      // Start the busy progress bar:
      m_progress->setTotalSteps( 0 );
      m_progTimer->start( 25, false );
    }
  }
  else
  {   
    m_progTimer->stop();
    m_progress->setTotalSteps( 100 );
    m_progress->reset();
    networkRescan->setEnabled( true ); 
    abort->setEnabled( false );
    slotStatusMsg( i18n( "Done." ) );
  }
}


void Smb4KApp::slotSmb4KOptionsDlg()
{
  if ( child( "ConfigDialog", "KDialogBase", true ) == 0 )
  {
    Smb4KOptionsDlg *configDlg = new Smb4KOptionsDlg( this, "ConfigDialog", false );
    configDlg->show();
  }
  else
  {
    // FIXME: Change to the desktop where the config dialog is, make is visible
    // and give it the focus.
  }
}



void Smb4KApp::slotViewToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  if(!viewToolBar->isChecked())
  {
    toolBar("mainToolBar")->hide();
  }
  else
  {
    toolBar("mainToolBar")->show();
  }
}


void Smb4KApp::slotViewStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if(!viewStatusBar->isChecked())
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }
}


void Smb4KApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}


void Smb4KApp::slotSetProgress()
{
  m_progress->setProgress( m_progress->progress()+1 );
}


void Smb4KApp::slotReadOptions()
{
  readOptions();
}


void Smb4KApp::slotEditBookmarks()
{
  m_core->bookmarkHandler()->editBookmarks( this );
}

#include "smb4k.moc"
