/***************************************************************************
    smb4koptionsdlg.cpp  -  description
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qvaluelist.h>

#include <qscrollview.h>

// KDE includes
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kstandarddirs.h>

// application specific includes
#include "smb4koptionsdlg.h"
#include "../smb4k.h"
#include "../core/smb4kauthinfo.h"


/**
 * This class handles the configuration dialog. It initializes the pages and saves
 * and reads the options.
 */

Smb4KOptionsDlg::Smb4KOptionsDlg( QWidget *parent, const char *name, bool modal )
: KDialogBase( IconList, i18n( "Configuration" ), Ok|Apply|Cancel, Ok, parent, name, modal, true )
{
  setWFlags( Qt::WDestructiveClose );
  
  m_config = kapp->config();

  setupAppearanceOptions();
  setupNetworkOptions();
  setupShareOptions();
  setupAuthOptions();
  setupSambaOptions();
  
  readOptions();

  connect( sapp->core()->fileIO(), SIGNAL( doneSUPWriting() ), SLOT( slotDoneSUPWriting() ) );
  connect( sapp->core()->fileIO(), SIGNAL( noSuper() ), SLOT( slotNoSuper() ) );
}


Smb4KOptionsDlg::~Smb4KOptionsDlg()
{
}


/****************************************************************************
   Set up the network options page.
****************************************************************************/

void Smb4KOptionsDlg::setupNetworkOptions()
{
  QFrame *frame = addPage( i18n( "Network" ), i18n( "Network" ), DesktopIcon( "network" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_networkPage = new Smb4KNetworkOptions( container->viewport() );
  container->addChild( m_networkPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the share options page.
****************************************************************************/

void Smb4KOptionsDlg::setupShareOptions()
{
  QFrame *frame = addPage( i18n( "Shares" ), i18n( "Shares" ), DesktopIcon( "hdd_mount" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );  
  m_sharePage = new Smb4KShareOptions( container->viewport() );
  container->addChild( m_sharePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the authentication management page.
****************************************************************************/

void Smb4KOptionsDlg::setupAuthOptions()
{
  QFrame *frame = addPage( i18n( "Authentication" ), i18n( "Authentication" ), DesktopIcon( "personal" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );    
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_authPage = new Smb4KAuthOptions( container->viewport() );
  container->addChild( m_authPage, 0, 0 );
  
  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the apprearance options page.
****************************************************************************/

void Smb4KOptionsDlg::setupAppearanceOptions()
{
  QFrame *frame = addPage( i18n( "Appearance" ), i18n( "Appearance" ), DesktopIcon( "colorize" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );  
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );  
  m_appearancePage = new Smb4KAppearanceOptions( container->viewport() );
  container->addChild( m_appearancePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the advanced Samba options page.
****************************************************************************/

void Smb4KOptionsDlg::setupSambaOptions()
{
  QFrame *frame = addPage( i18n( "Samba" ), i18n( "Samba Options" ), DesktopIcon( "misc" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel ); 
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );   
  m_sambaPage = new Smb4KSambaOptions( container->viewport() );
  container->addChild( m_sambaPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Saves the options.
****************************************************************************/


bool Smb4KOptionsDlg::saveOptions()
{
  //
  // General options
  //
  m_config->setGroup( "General Options" );
  m_config->writeEntry( "Configuration Dialog Geometry", size() );

  //
  // Mount options
  //
  m_config->setGroup( "Mount Options" );
  
  // Check, if the default path ends with a slash.
  if ( m_sharePage->m_defaultPath->lineEdit()->text().endsWith( "/" ) )
    m_config->writeEntry( "Default Path", m_sharePage->m_defaultPath->lineEdit()->text() );
  else
    m_config->writeEntry( "Default Path", m_sharePage->m_defaultPath->lineEdit()->text().append( "/" ) );
  
  m_config->writeEntry( "Clean Up", m_sharePage->m_cleanUp->isChecked() );
  m_config->writeEntry( "Unmount All", m_sharePage->m_unmountAll->isChecked() );
  m_config->writeEntry( "Mount Recent", m_sharePage->m_mountRecent->isChecked() );
  m_config->writeEntry( "Allow Unmount Foreign", m_sharePage->m_allowUnmountForeign->isChecked() );
  m_config->writeEntry( "Force Lowercase", m_sharePage->m_lowercase->isChecked() );
  m_config->writeEntry( "Force Unmount", m_sharePage->m_unmountDead->isChecked() );
  m_config->writeEntry( "Use SU Privileges", m_sharePage->m_useSuper->isChecked() );

  //
  // Browse options
  //
  m_config->setGroup( "Browse Options" );

  if ( m_networkPage->m_useNMB->isChecked() )
  {
    m_config->writeEntry( "Browse List", "nmblookup" );

    if ( !m_networkPage->m_hostName->text().stripWhiteSpace().isEmpty() )
      m_config->writeEntry( "Query Host", m_networkPage->m_hostName->text().stripWhiteSpace().upper() );

    m_config->writeEntry( "Host Needs Auth", m_networkPage->m_hostNeedsAuth->isChecked() );
  }
  else if ( m_networkPage->m_useHost->isChecked() )
  {
    m_config->writeEntry( "Browse List", "host" );
    m_config->writeEntry( "Query Host", m_networkPage->m_hostName->text().stripWhiteSpace().upper() );
    m_config->writeEntry( "Host Needs Auth", m_networkPage->m_hostNeedsAuth->isChecked() );
  }

  m_config->writeEntry( "Master Browser Auth", m_networkPage->m_mastersAuth->isChecked() );

  if ( m_networkPage->m_nmbSearch->isChecked() )
    m_config->writeEntry( "Network Search", "nmblookup" );
  else
    m_config->writeEntry( "Network Search", "smbclient" );


  //
  // Appearance
  //
  m_config->setGroup( "Appearance" );
  m_config->writeEntry( "Show Hidden", m_appearancePage->m_showHidden->isChecked() );
  m_config->writeEntry( "Show IPC", m_appearancePage->m_showIPC->isChecked() );
  m_config->writeEntry( "Show ADMIN", m_appearancePage->m_showAdmin->isChecked() );
  m_config->writeEntry( "Show Printer", m_appearancePage->m_showPrinter->isChecked() );
  m_config->writeEntry( "Show IP", m_appearancePage->m_showIP->isChecked() );
  m_config->writeEntry( "Show Type", m_appearancePage->m_showType->isChecked() );
  m_config->writeEntry( "Show Comment", m_appearancePage->m_showComment->isChecked() );
  if ( m_appearancePage->m_shareview->isChecked() )
    m_config->writeEntry( "Show Shares", "list" );
  else
    m_config->writeEntry( "Show Shares", "icons" );
  m_config->writeEntry( "Show Mount Point", m_appearancePage->m_description->isChecked() );
  m_config->writeEntry( "Show All Shares", m_appearancePage->m_external->isChecked() );

  //
  // Authentication
  //
  m_config->setGroup( "Authentication" );
  m_config->writeEntry( "Default Authentication", m_authPage->m_defaultAuth->isChecked() );
  
  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null, m_authPage->m_defaultLogin->text().stripWhiteSpace(), m_authPage->m_defaultPasswd->text().stripWhiteSpace() );
    sapp->core()->passwdReader()->writeDefaultAuthData( auth );
  }

  QValueList<Smb4KAuthInfo *> authList;
  
  for ( QListViewItem *item = m_authPage->m_loginData->firstChild(); item; item = item->nextSibling() )
  {
    authList.append( new Smb4KAuthInfo( item->text( 0 ), item->text( 1 ), item->text( 2 ), item->text( 3 ), item->text( 4 ) ) );
  }

  sapp->core()->passwdReader()->updateAuthList( authList );

  //
  // Samba options
  //
  m_config->setGroup( "Samba" );

  // General
  if ( !m_sambaPage->m_netbios->text().stripWhiteSpace().isEmpty() && *(sapp->core()->fileIO()->getSMBOptions().find( "netbios name")) != m_sambaPage->m_netbios->text().stripWhiteSpace().upper() )
    m_config->writeEntry( "NetBIOS Name", m_sambaPage->m_netbios->text().stripWhiteSpace().upper() );
  else
    m_config->deleteEntry( "NetBIOS Name" );

  if ( !m_sambaPage->m_sockopt->text().stripWhiteSpace().isEmpty() && *(sapp->core()->fileIO()->getSMBOptions().find( "socket options" )) != m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() )
    m_config->writeEntry( "Socket Options", m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() );
  else
    m_config->deleteEntry( "Socket Options" );

  if ( !m_sambaPage->m_scope->text().stripWhiteSpace().isEmpty() && *(sapp->core()->fileIO()->getSMBOptions().find( "netbios scope")) != m_sambaPage->m_scope->text().stripWhiteSpace().upper() )
    m_config->writeEntry( "Scope", m_sambaPage->m_scope->text().stripWhiteSpace().upper() );
  else
    m_config->deleteEntry( "NetBIOS Scope" );

  if ( m_sambaPage->m_port->value() == 139 )
    m_config->deleteEntry( "Port" );
  else
    m_config->writeEntry( "Port", m_sambaPage->m_port->value() );

  if ( m_sambaPage->m_krb->isEnabled() )
    m_config->writeEntry( "Use Kerberos", m_sambaPage->m_krb->isChecked() );
  else
    m_config->deleteEntry( "Use Kerberos" );

  // smbmount
  if ( m_sambaPage->m_uid->currentText() != i18n( "default" ) )
    m_config->writeEntry( "Mount UID", m_sambaPage->m_uid->currentText() );
  else
    m_config->deleteEntry( "Mount UID" );

  if ( m_sambaPage->m_gid->currentText() != i18n( "default" ) )
    m_config->writeEntry( "Mount GID", m_sambaPage->m_gid->currentText() );
  else
    m_config->deleteEntry( "Mount GID" );

  if ( m_sambaPage->m_charset->currentText() != i18n( "default" ) && m_sambaPage->m_charset->isEnabled() )
    m_config->writeEntry( "Mount Charset", m_sambaPage->m_charset->currentText() );
  else
    m_config->deleteEntry( "Mount Charset" );

  if ( m_sambaPage->m_codepage->currentText() != i18n( "default" ) && m_sambaPage->m_codepage->isEnabled() )
    m_config->writeEntry( "Mount Codepage", m_sambaPage->m_codepage->currentText() );
  else
    m_config->deleteEntry( "Mount Codepage" );

  m_config->writeEntry( "Mount FMASK", m_sambaPage->m_fmask->text().stripWhiteSpace() );
  m_config->writeEntry( "Mount DMASK", m_sambaPage->m_dmask->text().stripWhiteSpace() );

  m_config->writeEntry( "Mount ReadWrite", m_sambaPage->m_rw->isChecked() );

  if ( m_sambaPage->m_ttl->value() != 1000 && m_sambaPage->m_ttl->isEnabled() )
    m_config->writeEntry( "Mount Cache", m_sambaPage->m_ttl->value() );
  else
    m_config->deleteEntry( "Mount Cache" );

  if ( m_sambaPage->m_unicode->isEnabled() )
    m_config->writeEntry( "Mount Unicode", m_sambaPage->m_unicode->isChecked() );
  else
    m_config->deleteEntry( "Mount Unicode" );

  if ( m_sambaPage->m_lfs->isEnabled() )
    m_config->writeEntry( "Mount LFS", m_sambaPage->m_lfs->isChecked() );
  else
    m_config->deleteEntry( "Mount LFS" );

  // smbclient
  if ( !m_sambaPage->m_resolve->text().stripWhiteSpace().isEmpty() && *(sapp->core()->fileIO()->getSMBOptions().find( "name resolve order" )) != m_sambaPage->m_resolve->text().stripWhiteSpace().upper() )
    m_config->writeEntry( "Client Resolve Order", m_sambaPage->m_resolve->text().stripWhiteSpace().upper() );
  else
    m_config->deleteEntry( "Client Resolve Order" );

  if ( m_sambaPage->m_buffer->value() == 65520 )
    m_config->deleteEntry( "Client Buffer Size" );
  else
    m_config->writeEntry( "Client Buffer Size", m_sambaPage->m_buffer->value() );
    
  if ( m_sambaPage->m_domain->isEnabled() )
  {
    if ( !m_sambaPage->m_domain->text().stripWhiteSpace().isEmpty() && m_sambaPage->m_domain->text().stripWhiteSpace().upper() != *(sapp->core()->fileIO()->getSMBOptions().find( "workgroup" )) )
      m_config->writeEntry( "NMB Domain", m_sambaPage->m_domain->text().stripWhiteSpace().upper() );
    else
      m_config->deleteEntry( "NMB Domain" );
  }
  
  if ( !m_sambaPage->m_broadcast->text().stripWhiteSpace().isEmpty() && *(sapp->core()->fileIO()->getSMBOptions().find( "interfaces" )) != m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() )
    m_config->writeEntry( "NMB Broadcast", m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() );
  else
    m_config->deleteEntry( "NMB Broadcast" );
    
  m_config->writeEntry( "NMB Port 137", m_sambaPage->m_udp->isChecked() );
  
  
  m_config->sync();
  return true;
}


/****************************************************************************
   Reads the options.
****************************************************************************/


void Smb4KOptionsDlg::readOptions()
{
  //
  // System options
  //
  m_config->setGroup( "System" );
  QString kernel = m_config->readEntry( "Version", QString::null );
  int k_major = kernel.section( ".", 0, 0 ).toInt();
  int k_minor = kernel.section( ".", 1, 1 ).toInt();
  int k_rev = kernel.section( ".", 2, 2 ).toInt();
  QString samba = m_config->readEntry( "Samba", QString::null );
  int s_major = samba.section( ".", 0, 0 ).toInt();
  int mask = m_config->readNumEntry( "UMASK" );
  QString system = m_config->readEntry( "Name", QString::null );

  //
  // Mount options
  //
  m_config->setGroup( "Mount Options" );
  m_sharePage->m_defaultPath->lineEdit()->setText( m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) ) );
  m_sharePage->m_cleanUp->setChecked( m_config->readBoolEntry( "Clean Up", true ) );
  m_sharePage->m_unmountAll->setChecked( m_config->readBoolEntry( "Unmount All", false ) );
  m_sharePage->m_mountRecent->setChecked( m_config->readBoolEntry( "Mount Recent", false ) );
  m_sharePage->m_allowUnmountForeign->setChecked( m_config->readBoolEntry( "Allow Unmount Foreign", false ) );
  m_sharePage->m_lowercase->setChecked( m_config->readBoolEntry( "Force Lowercase", false ) );
  m_sharePage->m_useSuper->setChecked( m_config->readBoolEntry( "Use SU Privileges", false ) );
  bool useSuper = m_config->readBoolEntry( "Super", false );

  if ( !useSuper )
  {
    m_sharePage->m_unmountDead->setEnabled( false );
    m_sharePage->m_useSuper->setEnabled( false );
    m_sharePage->m_removeSUEntries->setEnabled( false );
  }

  if ( system == "Linux" && k_major >= 2 && ( ( k_minor == 4 && k_rev >= 11 ) || k_minor > 4 ) )
    m_sharePage->m_unmountDead->setChecked( m_config->readBoolEntry( "Force Unmount", false ) );
  else
  {
    m_sharePage->m_unmountDead->setChecked( false );
    m_sharePage->m_unmountDead->setEnabled( false );
  }

  //
  // Browse options
  //
  m_config->setGroup( "Browse Options" );
  // Toggle the browse list buttons.
  bool winsSupport = m_config->readBoolEntry( "WINS Support", false );
  QString browselist = m_config->readEntry( "Browse List", "nmblookup" );
  QString wins =  m_config->readEntry( "WINS Server", QString::null );
  QString host = m_config->readEntry( "Query Host", wins );

  if ( browselist.stripWhiteSpace() == "nmblookup" )
  {
    m_networkPage->m_useNMB->setChecked( true );
    m_networkPage->m_hostLabel->setEnabled( false );
    m_networkPage->m_hostName->setEnabled( false );
    m_networkPage->m_hostNeedsAuth->setEnabled( false );
  }
  else if ( browselist.stripWhiteSpace() == "host" )
  {
    m_networkPage->m_useHost->setChecked( true );
  }

  if ( !host.isEmpty() )
    m_networkPage->m_hostName->setText( host );

  if ( winsSupport )
    m_networkPage->m_winsYesNo->setText( wins );

  m_networkPage->m_hostNeedsAuth->setChecked( m_config->readBoolEntry( "Host Needs Auth", false ) );
  m_networkPage->m_mastersAuth->setChecked( m_config->readBoolEntry( "Master Browser Auth", false ) );

  if ( m_config->readEntry( "Network Search", "nmblookup" ) == "nmblookup" )
    m_networkPage->m_nmbSearch->setChecked( true );
  else
    m_networkPage->m_clientSearch->setChecked( true );


  //
  // Appearance
  //
  m_config->setGroup( "Appearance" );
  m_appearancePage->m_showHidden->setChecked( m_config->readBoolEntry( "Show Hidden", true ) );

  // The program has to be notified that the toggle state changed.
  m_appearancePage->init();

  m_appearancePage->m_showIPC->setChecked( m_config->readBoolEntry( "Show IPC", false ) );
  m_appearancePage->m_showAdmin->setChecked( m_config->readBoolEntry( "Show ADMIN", false ) );
  m_appearancePage->m_showPrinter->setChecked( m_config->readBoolEntry( "Show Printer", true ) );
  m_appearancePage->m_showIP->setChecked( m_config->readBoolEntry( "Show IP", true ) );
  m_appearancePage->m_showType->setChecked( m_config->readBoolEntry( "Show Type", true ) );
  m_appearancePage->m_showComment->setChecked( m_config->readBoolEntry( "Show Comment", true ) );
           
  if ( m_config->readEntry( "Show Shares", "icons" ) == "icons" )
    m_appearancePage->m_shareview->setChecked( false );
  else if ( m_config->readEntry( "Show Shares", "icons" ) == "list" )
    m_appearancePage->m_shareview->setChecked( true );
  
  if ( !m_config->readBoolEntry( "Show Mount Point", false ) )
    m_appearancePage->m_description->setChecked( false );
  else
    m_appearancePage->m_description->setChecked( true );
    
  m_appearancePage->m_external->setChecked( m_config->readBoolEntry( "Show All Shares", false ) );
  
  
  //
  // General options
  //
  m_config->setGroup( "General Options" );
  setInitialSize( m_config->readSizeEntry( "Configuration Dialog Geometry", 0L ), false );


  //
  // Authentication
  //
  m_config->setGroup( "Authentication" );
  if ( m_config->readBoolEntry( "Default Authentication", false ) )
  {
    m_authPage->m_defaultAuth->setChecked( true );
    m_authPage->m_authWidget->setEnabled( true );
  }
  else
  {
    m_authPage->m_defaultAuth->setChecked( false );
    m_authPage->m_authWidget->setEnabled( false );
  }
  
  // Now read the passwords:
  QValueList<Smb4KAuthInfo *> authList = sapp->core()->passwdReader()->getAuthList();
  
  for ( QValueList<Smb4KAuthInfo *>::ConstIterator it = authList.begin(); it != authList.end(); ++it  )
  {
    QListViewItem *item = new QListViewItem( m_authPage->m_loginData );
    item->setText( 0, (*it)->workgroup() );
    item->setText( 1, (*it)->host() );
    item->setText( 2, (*it)->share() );
    item->setText( 3, (*it)->user() );
    item->setText( 4, (*it)->password() );
  }
  
  // And the default authentication:
  Smb4KAuthInfo *defaultAuth = sapp->core()->passwdReader()->readDefaultAuthData();
  m_authPage->m_defaultLogin->setText( defaultAuth->user() );
  m_authPage->m_defaultPasswd->setText( defaultAuth->password() );

  
  //
  // Samba options
  //
  m_config->setGroup( "Samba" );
  m_sambaPage->m_netbios->setText( m_config->readEntry( "NetBIOS Name", *(sapp->core()->fileIO()->getSMBOptions().find( "netbios name" )) ) );
  m_sambaPage->m_sockopt->setText( m_config->readEntry( "Socket Options", *(sapp->core()->fileIO()->getSMBOptions().find( "socket options" )) ) );
  m_sambaPage->m_scope->setText( m_config->readEntry( "NetBIOS Scope", *(sapp->core()->fileIO()->getSMBOptions().find( "netbios scope" )) ) );
  m_sambaPage->m_port->setValue( m_config->readNumEntry( "Port", 139 ) );

  if ( s_major >= 3 )
  {
    m_sambaPage->m_krb->setChecked( m_config->readBoolEntry( "Use Kerberos", false ) );
    m_sambaPage->m_unicode->setChecked( m_config->readBoolEntry( "Mount Unicode", false ) );
    m_sambaPage->m_lfs->setChecked( m_config->readBoolEntry( "Mount LFS", false ) );
  }
  else
  {
    m_sambaPage->m_krb->setEnabled( false );
    m_sambaPage->m_lfs->setEnabled( false );
    m_sambaPage->m_unicode->setEnabled( false );
  }
  
  QValueList<Smb4KUser *> list = sapp->core()->fileIO()->getUsers();
  QStringList uid, gid;
    
  for ( QValueList<Smb4KUser *>::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    // Put the UIDs in the respective combo box.
    if ( uid.find( QString( "%1" ).arg( (*it)->uid() ) ) == uid.end() )
      uid.append( QString( "%1" ).arg( (*it)->uid() ) );
    
    // Put the GIDs in the respective combo box.
    if ( gid.find( QString( "%1" ).arg( (*it)->gid() ) ) == gid.end() )
      gid.append( QString( "%1" ).arg( (*it)->gid() ) );    
  }
    
  uid.sort();
  gid.sort();

  m_sambaPage->m_uid->insertStringList( uid, -1 );
  m_sambaPage->m_uid->setCurrentText( m_config->readEntry( "Mount UID", i18n( "default" ) ) );

  m_sambaPage->m_gid->insertStringList( gid, -1 );
  m_sambaPage->m_gid->setCurrentText( m_config->readEntry( "Mount GID", i18n( "default" ) ) );
  
  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 0 )
  {
    m_sambaPage->m_codepage->setCurrentText( m_config->readEntry( "Mount Codepage", i18n( "default" ) ) );
    m_sambaPage->m_charset->setCurrentText( m_config->readEntry( "Mount Charset", i18n( "default" ) ) );
  }
  else
  {
    m_sambaPage->m_codepage->setEnabled( false );
    m_sambaPage->m_charset->setEnabled( false );
  }

  m_sambaPage->m_fmask->setText( m_config->readEntry( "Mount FMASK", QString( "%1" ).arg( 0777 - mask, 0, 8 ) ) );
  m_sambaPage->m_dmask->setText( m_config->readEntry( "Mount DMASK", QString( "%1" ).arg( 0777 - mask, 0, 8 ) ) );

  if ( m_config->readBoolEntry( "Mount ReadWrite", true ) )
    m_sambaPage->m_rw->setChecked( true );
  else
    m_sambaPage->m_ro->setChecked( true );

  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 2 )
    m_sambaPage->m_ttl->setValue( m_config->readNumEntry( "Mount Cache", 1000 ) );
  else
    m_sambaPage->m_ttl->setEnabled( false );

  m_sambaPage->m_buffer->setValue( m_config->readNumEntry( "Client Buffer Size", 65520 ) );
  m_sambaPage->m_resolve->setText( m_config->readEntry( "Client Resolve Order", *(sapp->core()->fileIO()->getSMBOptions().find( "name resolve order" )) ) );
  
  if ( s_major >= 3 )
    m_sambaPage->m_domain->setEnabled( true );
  else
    m_sambaPage->m_domain->setEnabled( false );

  m_sambaPage->m_domain->setText( m_config->readEntry( "NMB Domain", *(sapp->core()->fileIO()->getSMBOptions().find( "workgroup" )) ) );
  m_sambaPage->m_broadcast->setText( m_config->readEntry( "NMB Broadcast", *(sapp->core()->fileIO()->getSMBOptions().find( "interfaces" )) ) );
  m_sambaPage->m_udp->setChecked( m_config->readBoolEntry( "NMB Port 137", false ) );
  
}


/****************************************************************************
   Checks, whether everything has been done
****************************************************************************/


bool Smb4KOptionsDlg::check()
{
  bool ok = true;

  //
  // Network page
  //
  if ( m_networkPage->m_useHost->isChecked() && m_networkPage->m_hostName->text().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the host you want to query to retrieve the browse list.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Network );
      m_networkPage->m_hostName->setFocus();
    }
  }

  //
  // Shares page
  //
  if ( m_sharePage->m_defaultPath->url().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the path where Smb4K will mount the shares to.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Shares );
      m_sharePage->m_defaultPath->setFocus();
    }
  }

  //
  // Authentication page
  //
  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login and password.\nDo you want to enter them now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && !m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( !m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default password.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultPasswd->setFocus();
      }
    }
  }

  if ( !m_authPage->m_workgroupEdit->text().isEmpty() || !m_authPage->m_hostEdit->text().isEmpty() || ( m_authPage->m_oneShare->isChecked() && !m_authPage->m_shareEdit->text().isEmpty() ) || !m_authPage->m_loginEdit->text().isEmpty() || !m_authPage->m_passwdEdit->text().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to commit the authentication data you entered last.\nDo you want to do it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Authentication );
      m_authPage->showPage( m_authPage->page( 1 ) );
    }
    else
      m_authPage->clear();
  }

  return ok;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KOptionsDlg::slotOk()
{
  if ( check() )
  {
    if ( m_sharePage->m_unmountDead->isChecked() || m_sharePage->m_useSuper->isChecked() )
    {
      m_button = "OK";
      // Modify the /etc/super.tab file to gain super user privileges for
      // mounting/unmounting shares.
      sapp->core()->fileIO()->writeSuperUserEntries( m_sharePage->m_unmountDead->isChecked(), m_sharePage->m_useSuper->isChecked(), false );
    }
    else
    {
      if ( saveOptions() )
      {
        sapp->readOptions();
        accept();      
      }
    }
  }
}


void Smb4KOptionsDlg::slotApply()
{
  if ( check() )
  {
    if ( m_sharePage->m_unmountDead->isChecked() || m_sharePage->m_useSuper->isChecked() )
    {
      m_button = "Apply";
      // Modify the /etc/super.tab file to gain super user privileges for
      // mounting/unmounting shares.
      sapp->core()->fileIO()->writeSuperUserEntries( m_sharePage->m_unmountDead->isChecked(), m_sharePage->m_useSuper->isChecked(), false );
    }
    else
    {
      if ( saveOptions() )
      {
        sapp->readOptions();
      }
    }
  }
}


void Smb4KOptionsDlg::slotDoneSUPWriting()
{
  if ( m_button == "Apply" )
  {
    if ( saveOptions() )
      sapp->readOptions();
  }
  else if ( m_button == "OK" )
  {
    if ( saveOptions() )
    {
      sapp->readOptions();
      accept();    
    }
  }
}


void Smb4KOptionsDlg::slotNoSuper()
{
  m_sharePage->m_unmountDead->setChecked( false );
  m_sharePage->m_useSuper->setChecked( false );
}


#include "smb4koptionsdlg.moc"

