/***************************************************************************
    smb4ksharewidget.cpp  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam Mr 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qregexp.h>

// KDE includes
#include <klocale.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <kactionclasses.h>
#include <kpopupmenu.h>
#include <kaction.h>

// system includes
#include <unistd.h>
#include <sys/types.h>

// application specific includes
#include "smb4ksharewidget.h"
#include "../smb4k.h"
#include "smb4kmountdialog.h"


Smb4KShareWidget::Smb4KShareWidget( QWidget *parent, const char *name ) : KIconView( parent, name )
{
  m_config = kapp->config();

  setFrameShape( KIconView::PopupPanel );
  setFrameShadow( KIconView::Sunken );
  setSelectionMode( KIconView::Single );
  setResizeMode( KIconView::Adjust );
  setShowToolTips( true );
  setAutoArrange( true );
  setSorting( true, true );
  
  // Internal connections
  connect( this, SIGNAL( mouseButtonPressed( int, QIconViewItem *, const QPoint & ) ), this, SLOT( slotButtonPressed( int, QIconViewItem *, const QPoint & ) ) );
  connect( this, SIGNAL( executed( QIconViewItem * ) ),                                this, SLOT( slotOpenFilemanager() ) );
  connect( this, SIGNAL( selectionChanged( QIconViewItem * ) ),                        this, SLOT( slotSelectionChanged( QIconViewItem * ) ) );
  connect( sapp->core()->mounter(), SIGNAL( importedShares() ),                        this, SLOT( slotMountedShares() ) );
}


Smb4KShareWidget::~Smb4KShareWidget()
{
}

/****************************************************************************
   Reads the options for the share view widget.
****************************************************************************/

void Smb4KShareWidget::readOptions()
{  
  // Which appearance does the user want?
  m_config->setGroup( "Appearance" );
  m_display = m_config->readEntry( "Show Shares", "icons" );
  
  if ( m_display == "icons" )
  {
    setItemTextPos( KIconView::Bottom );
    setMaxItemWidth( 150 );
    setArrangement( KIconView::LeftToRight );
    setWordWrapIconText( true );
  }
  else if ( m_display == "list" )
  {
    setItemTextPos( KIconView::Right );
    setMaxItemWidth( 300 );
    setArrangement( KIconView::TopToBottom );
    setWordWrapIconText( false );
  }

  m_mountpoint = m_config->readBoolEntry( "Show Mount Point", false );

  // Should all share be shown, that were mounted on the system?
  m_showExternal = m_config->readBoolEntry( "Show All Shares", false );
  
  if ( count() != 0 )
  {
    for ( QIconViewItem *item = this->firstItem(); item; item = item->nextItem() )
    {
      if ( m_display == "icons" )
      {
        item->setPixmap( DesktopIcon( "hdd_mount" ) );
      }
      else if ( m_display == "list" )
      {
        item->setPixmap( SmallIcon( "hdd_mount" ) );
      }
      
      if ( m_mountpoint )
      {
        item->setText( ((Smb4KShareWidgetItem *)item)->mountpoint() );
      }
      else
      {
        item->setText( ((Smb4KShareWidgetItem *)item)->name() );
      }
    }
  }
  
  slotMountedShares();
  
  adjustItems();

  // Mount options.
  m_config->setGroup( "Programs" );
  QString super = m_config->readPathEntry( "super" );
  QString sudo = m_config->readPathEntry( "sudo" );
  
  m_config->setGroup( "Super User Privileges" );
  if ( !super.isEmpty() || !sudo.isEmpty() )
  {
    m_forcedUnmount = m_config->readBoolEntry( "Force Unmount", false );
  }
  else
  {
    m_forcedUnmount = false;
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KShareWidget::slotButtonPressed( int button, QIconViewItem *item, const QPoint &pos )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    // Enable the actions:
    if ( m_forcedUnmount )
    {
      sapp->sharesForceUnmount->setEnabled( true );
    }
    
    if ( !i->isBroken() )
    {
      sapp->sharesStartFilemanager->setEnabled( true );
      sapp->core()->setShareForUsageInfo( i->shareObject() );
    }
    else
    {
      sapp->sharesStartFilemanager->setEnabled( false );
      sapp->core()->setShareForUsageInfo( NULL );
    }
      
    sapp->sharesUnmount->setEnabled( true );
  }
  else
  {
    sapp->sharesUnmount->setEnabled( false );
    sapp->sharesStartFilemanager->setEnabled( false );
    sapp->sharesForceUnmount->setEnabled( false );
    
    sapp->core()->setShareForUsageInfo( NULL );
    clearSelection();    
  }
  
  if ( button == KIconView::RightButton )
  {
    KActionMenu *menu = new KActionMenu( this, "SharesViewPopupMenu" );
    menu->popupMenu()->insertTitle( SmallIcon( "hdd_mount" ), i18n( "Shares" ), 0, 0 );
    menu->insert( sapp->actionCollection()->action( "shares_unmount_share" ) );
    menu->insert( sapp->actionCollection()->action( "shares_force_unmount" ) );
    menu->insert( sapp->actionCollection()->action( "shares_unmount_all" ) );
    menu->popupMenu()->insertSeparator( -1 );
    menu->insert( sapp->actionCollection()->action( "shares_manual_mount" ) );
    menu->popupMenu()->insertSeparator( -1 );
    menu->insert( sapp->actionCollection()->action( "shares_start_filemanager" ) );
    
    menu->popupMenu()->exec( pos, 0 );
  }
}


void Smb4KShareWidget::slotSelectionChanged( QIconViewItem *item )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    if ( !i->isBroken() )
    {
      sapp->sharesStartFilemanager->setEnabled( true );
      sapp->core()->setShareForUsageInfo( i->shareObject() );
    }
    else
    {
      sapp->sharesStartFilemanager->setEnabled( false );
      sapp->core()->setShareForUsageInfo( NULL );
    }
    
    if ( m_forcedUnmount )
    {
      sapp->sharesForceUnmount->setEnabled( true );
    }
      
    sapp->sharesUnmount->setEnabled( true );
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KShareWidget::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = sapp->core()->mounter()->getShares();
  Smb4KShareWidgetItem *current = (Smb4KShareWidgetItem *)currentItem();
  
  if ( !shares.isEmpty() )
  {
    // First remove those, that are not mounted anymore:
    for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
    {
      if( !sapp->core()->mounter()->findShareByPath( item->mountpoint() )
          || ( item->shareObject()->isForeign() && !m_showExternal ) )
      {
        if ( current == item )
        {
          sapp->core()->setShareForUsageInfo( NULL );
        }
        
        delete sapp->actionCollection()->action( ((Smb4KShareWidgetItem *)item)->name() );
        delete item;
        this->sort( this->sortDirection() );
      }
    }

    // Now process the entries in the list:
    for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    {
      // Check, whether the share is already in the list. Look for the
      // mount point.
      bool alreadyInView = false;
      
      for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
      {
        if ( item->mountpoint() == (*it)->getPath() || item->mountpoint() == (*it)->getCanonicalPath() )
        {
          if ( (*it)->isBroken() && !item->isBroken() )
          {
            item->setBroken( true );
            
            KAction *a = sapp->actionCollection()->action( (*it)->getName() );
            a->setEnabled( false );
            a->setIcon( "file_broken" );
          }
          else if ( !(*it)->isBroken() && item->isBroken() )
          {
            item->setBroken( false );
            
            KAction *a = sapp->actionCollection()->action( (*it)->getName() );
            a->setEnabled( true );
            a->setIcon( "hdd_mount" );
          }
          
          alreadyInView = true;
          break;
        }
        else
        {
          continue;
        }
      }

      if ( !alreadyInView )
      {
        if ( m_display == "icons" && !m_showExternal && (*it)->isForeign() )
        {
          continue;
        }
        else if ( m_display == "icons" )
        {
          new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::Icons, m_mountpoint );

          KAction *a = new KAction( (*it)->getName(), 0, 0, 0, 0, sapp->actionCollection(), (*it)->getName() );
          a->setGroup( "MountedShares" );
          a->setIcon( (*it)->isBroken() ? "file_broken" : "hdd_mount" );
          a->setEnabled( !(*it)->isBroken() );
        }

        if ( m_display == "list" && !m_showExternal && (*it)->isForeign() )
        {
          continue;
        }
        else if ( m_display == "list" )
        {
          new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::List, m_mountpoint );
            
          KAction *a = new KAction( (*it)->getName(), 0, 0, 0, 0, sapp->actionCollection(), (*it)->getName() );
          a->setGroup( "MountedShares" );
          a->setIcon( (*it)->isBroken() ? "file_broken" : "hdd_mount" );
          a->setEnabled( !(*it)->isBroken() );
        }

        // Do the sorting:
        this->sort( this->sortDirection() );
      }
    }
    
    if ( current != (Smb4KShareWidgetItem *)currentItem() )
    {
      sapp->sharesUnmount->setEnabled( false );
      sapp->sharesForceUnmount->setEnabled( false );
      sapp->sharesStartFilemanager->setEnabled( false );
      
      clearSelection();
      sapp->shareTab()->clear();
    }
    
    sapp->sharesUnmountAll->setEnabled( true );
  }
  else
  {
    sapp->core()->setShareForUsageInfo( NULL );
    delete current;
    clear();
    
    KActionPtrList list = sapp->actionCollection()->actions( "MountedShares" );
    
    for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); ++it )
    {
      delete *it;
    }
    
    sapp->sharesUnmount->setEnabled( false );
    sapp->sharesForceUnmount->setEnabled( false );
    sapp->sharesUnmountAll->setEnabled( false );
    sapp->sharesStartFilemanager->setEnabled( false );    
    
    sapp->shareTab()->clear();
  }
}


/***************************************************************************
   Slots for the menu.
***************************************************************************/


void Smb4KShareWidget::slotOpenFilemanager()
{
  if ( currentItem() && !((Smb4KShareWidgetItem *)currentItem())->isBroken() )
  {
    sapp->core()->open( ((Smb4KShareWidgetItem *)currentItem())->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    sapp->core()->mounter()->unmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


void Smb4KShareWidget::slotUnmountAllShares()
{
  sapp->core()->mounter()->unmountAllShares();
}


void Smb4KShareWidget::slotForceUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    sapp->core()->mounter()->forceUnmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


void Smb4KShareWidget::slotManualMount()
{
  Smb4KMountDialog *dlg = new Smb4KMountDialog( 0, "ManualMountDialog" );
  dlg->show();
}


/***************************************************************************
 *                                                                         *
 *   The Smb4KShareWidgetItem class.                                       *
 *                                                                         *
 ***************************************************************************/


Smb4KShareWidgetItem::Smb4KShareWidgetItem( Smb4KShare *share, Smb4KShareWidget *parent, int alignment, bool mp ) 
: KIconViewItem( parent, QString::null ), m_share( *share ), m_alignment( alignment )
{
  setBroken( m_share.isBroken() );
     
  if ( mp )
  {
    setText( m_share.getPath() );
  }
  else
  {
    setText( m_share.getName() );
  }
}


Smb4KShareWidgetItem::~Smb4KShareWidgetItem()
{
}

/****************************************************************************
   Paints an item.
****************************************************************************/


void Smb4KShareWidgetItem::paintItem( QPainter *p, const QColorGroup &cg )
{
  QColorGroup colorgrp( cg );

  if ( m_share.isForeign() || m_share.isBroken() )
  {
    colorgrp.setColor( QColorGroup::Text, Qt::gray );
  }

  QIconViewItem::paintItem( p, colorgrp );
}


void Smb4KShareWidgetItem::setBroken( bool broken )
{
  m_share.setBroken( broken );
  
  if ( broken )
  {  
    if ( !m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( DesktopIcon( "file_broken" ) );
    }
    else if ( !m_share.isForeign() && m_alignment == List )
    {
      setPixmap( SmallIcon( "file_broken" ) );
    }
    else if ( m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "file_broken", KIcon::Desktop, 0, KIcon::DisabledState, 0L, false ) );
    }
    else if ( m_share.isForeign() && m_alignment == List )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "file_broken", KIcon::Small, 0, KIcon::DisabledState, 0L, false ) );
    }
  }
  else
  {
    if ( !m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( DesktopIcon( "hdd_mount" ) );
    }
    else if ( !m_share.isForeign() && m_alignment == List )
    {
      setPixmap( SmallIcon( "hdd_mount" ) );
    }
    else if ( m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Desktop, 0, KIcon::DisabledState, 0L, false ) );
    }
    else if ( m_share.isForeign() && m_alignment == List )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Small, 0, KIcon::DisabledState, 0L, false ) );
    }
  }
}


#include "smb4ksharewidget.moc"

