/***************************************************************************
                          smb4kglobal  -  description
                             -------------------
    begin                : Sa Apr 2 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

#include <kconfig.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kapplication.h>
#include <kdebug.h>

#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>

// application specific includes
#include "smb4kglobal.h"

#define TIME 50

class Smb4KGlobalPrivate
{
  public:
    Smb4KGlobalPrivate()
    {
      m_timer = new QTimer();
      m_timer->start( TIME, false );

      m_config = NULL;
    }
    ~Smb4KGlobalPrivate()
    {
      delete m_timer;
      delete m_config;
    }
    QTimer *timer()
    {
      return m_timer;
    }
    KConfig *config()
    {
      return m_config ? m_config : (m_config = new KConfig( "smb4krc", false, false, "config" ));
    }
  private:
    QTimer *m_timer;
    KConfig *m_config;
};


Smb4KGlobalPrivate p;


/****************************************************************************
   Returns the global options for smbclient
****************************************************************************/

const QString Smb4K_Global::getSmbclientOptions()
{
  QString options;

  p.config()->setGroup( "Samba" );

  if ( !p.config()->readEntry( "Client Resolve Order", QString::null ).isEmpty() )
  {
    options.append( QString( " -R %1" ).arg( KProcess::quote( p.config()->readEntry( "Client Resolve Order", QString::null ) ) ) );
  }

  options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );

  if ( !p.config()->readEntry( "Client Buffer Size", QString::null ).isEmpty() )
  {
    options.append( QString( " -b %1" ).arg( p.config()->readNumEntry( "Client Buffer Size", 65520 ) ) );
  }

  if ( p.config()->readBoolEntry( "Use Kerberos", false ) )
  {
    if ( p.config()->readEntry( "Version", QString::null ).startsWith( "3" ) )
    {
      options.append( " -k" );
    }
  }

  if ( !p.config()->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
  {
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
  {
    options.append( QString( " -i %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Scope", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Socket Options", QString::null ).isEmpty() )
  {
    options.append( QString( " -O %1" ).arg( KProcess::quote( p.config()->readEntry( "Socket Options", QString::null ) ) ) );
  }

  return options;
}


/****************************************************************************
   Returns the global options for nmblookup
****************************************************************************/

const QString Smb4K_Global::getNmblookupOptions()
{
  QString options;

  p.config()->setGroup( "Samba" );

  if ( !p.config()->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
  {
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
  {
    options.append( QString( " -i %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Scope", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Socket Options", QString::null ).isEmpty() )
  {
    options.append( QString( " -O %1" ).arg( KProcess::quote( p.config()->readEntry( "Socket Options", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Domain", QString::null ).isEmpty() )
  {
    if ( p.config()->readEntry( "Version", QString::null ).startsWith( "3" ) )
    {
      options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", QString::null ) ) ) );
    }
  }

  if ( !p.config()->readEntry( "NMB Broadcast", QString::null ).isEmpty() )
  {
    options.append( QString( " -B %1" ).arg( KProcess::quote( p.config()->readEntry( "NMB Broadcast", QString::null ) ) ) );
  }

  if ( p.config()->readBoolEntry( "NMB Port 137", false ) )
  {
    options.append( " -r" );
  }

  return options;
}


/****************************************************************************
   Reads the [global section of the smb.conf file
****************************************************************************/

const QMap<QString,QString> Smb4K_Global::readGlobalSMBOptions()
{
  QStringList paths;
  paths << "/etc";
  paths << "/etc/samba";
  paths << "/usr/local/etc";
  paths << "/usr/local/etc/samba";

  QFile f( "smb.conf" );

  QStringList contents;

  // Locate the file and read its contents:
  for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it )
  {
    QDir::setCurrent( *it );

    if ( f.exists() )
    {
      if ( f.open( IO_ReadOnly ) )
      {
        QTextStream ts( &f );

        contents = QStringList::split( '\n', ts.read(), false );
      }

      f.close();

      break;
    }
    else
    {
      continue;
    }
  }

  // Process the file contents.
  for ( QStringList::Iterator it = contents.erase( contents.begin(), ++(contents.find( "[global]" )) ); it != contents.end(); ++it )
  {
    if ( (*it).stripWhiteSpace().startsWith( "#" ) )
    {
      *it = QString::null;
    }
    else if ( (*it).stripWhiteSpace().startsWith( "include" ) )
    {
      // Put the contents of the included at this position.
      QString file = (*it).section( "=", 1, 1 ).stripWhiteSpace();
      *it = QString::null;
      f.setName( file );

      QStringList include;

      if ( f.exists() )
      {
        if ( f.open( IO_ReadOnly ) )
        {
          QTextStream ts( &f );

          include = QStringList::split( '\n', ts.read(), false );
        }

        f.close();
      }

      for ( QStringList::Iterator i = include.begin(); i != include.end(); ++i )
      {
        if ( !(*i).stripWhiteSpace().isEmpty() )
        {
          contents.insert( it, *i );

          continue;
        }
        else
        {
          continue;
        }
      }

      continue;
    }
    else if ( (*it).startsWith( "[" ) )
    {
      contents.erase( it, contents.end() );

      break;
    }
    else
    {
      continue;
    }
  }

  contents.remove( QString::null );

  QMap<QString,QString> map;

  // Write all options into the map:
  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    QString key = (*it).section( "=", 0, 0 ).stripWhiteSpace().lower();
    map[key] = QString( (*it).section( "=", 1, 1 ).stripWhiteSpace().upper() );
  }

  // Post-processing. Some values should be entered with their defaults, if they are
  // not already present.
  if ( !map.contains( "netbios name" ) )
  {
    size_t hostnamelen = 255;
    char *hostname = new char[hostnamelen];

    if ( gethostname( hostname, hostnamelen ) == 0 )
    {
      map["netbios name"] = ( QString( "%1" ).arg( hostname ) ).upper();
    }

    delete [] hostname;
  }

  return map;
}



/****************************************************************************
   Returns the WINS server the system is using
****************************************************************************/

const QString Smb4K_Global::getWINSServer()
{
  QString wins = QString::null;
  QMap<QString, QString> options = readGlobalSMBOptions();

  if ( options.find( "wins server" ) != options.end() )
  {
    wins = options["wins server"];
  }
  else if ( options.find( "wins support" ) != options.end() &&
            (QString::compare( options["wins support"].lower(), "yes" ) == 0 ||
             QString::compare( options["wins support"].lower(), "true" ) == 0) )
  {
    wins = "127.0.0.1";
  }

  return wins;
}


/****************************************************************************
   Returns the kernel version
****************************************************************************/

const QString Smb4K_Global::getKernelVersion()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.release ).section( "-", 0, 0 );
}


/****************************************************************************
   Returns the system name
****************************************************************************/

const QString Smb4K_Global::getSystemName()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.sysname );
}


/****************************************************************************
   Returns the UMASK
****************************************************************************/

int Smb4K_Global::getUMASK()
{
  return umask( 0 );
}


/****************************************************************************
   Returns the global options of the net command
****************************************************************************/

const QString Smb4K_Global::getNetOptions( const QString &command )
{
  QMap<QString,QString> smb_map = readGlobalSMBOptions();

  QString options;

  if ( QString::compare( command, "share" ) == 0 )
  {
    p.config()->setGroup( "Samba" );

    if ( QString::compare( p.config()->readEntry( "Net Protocol", "auto" ), "auto" ) != 0 )
    {
      // The 'share' command only works with the RAP or RPC protocol.
      // Because the user can also choose the ADS protocol, we have to
      // permit only RAP and RPC protocol here:
      QString protocol = QString::compare( p.config()->readEntry( "Net Protocol", "auto" ).lower(), "rap" ) == 0 ? "rap" : "rpc";
      options.append( " "+protocol );
    }

    options.append( " share" );

    if ( p.config()->readBoolEntry( "Net Machine Account", false ) )
    {
      options.append( " -P" );
    }

    options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );
    options.append( " -l" );
    options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", smb_map["workgroup"] ) ) ) );
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", smb_map["netbios name"] ) ) ) );
  }
  else if ( QString::compare( command, "share_no_protocol" ) == 0 )
  {
    p.config()->setGroup( "Samba" );

    options.append( " share" );

    if ( p.config()->readBoolEntry( "Net Machine Account", false ) )
    {
      options.append( " -P" );
    }

    options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );
    options.append( " -l" );
    options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", smb_map["workgroup"] ) ) ) );
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", smb_map["netbios name"] ) ) ) );
  }
  else if ( QString::compare( command, "server domain" ) == 0 )
  {
    p.config()->setGroup( "Samba" );

    options.append( " rap "+command );

    if ( p.config()->readBoolEntry( "Net Machine Account", false ) )
    {
      options.append( " -P" );
    }

    options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );
    options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", smb_map["workgroup"] ) ) ) );
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", smb_map["netbios name"] ) ) ) );
  }
  else if ( command.startsWith( "lookup host" ) )
  {
    options.append( " "+command );

    if ( p.config()->readBoolEntry( "Net Machine Account", false ) )
    {
      options.append( " -P" );
    }

    options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );
    options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", smb_map["workgroup"] ) ) ) );
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", smb_map["netbios name"] ) ) ) );
  }
  else
  {
    KMessageBox::error( 0, i18n( "The net command '%1' is not supported by Smb4K." ).arg( command ) );
  }

  return options;
}


/***************************************************************************
   Returns a pointer to the global KConfig object.
***************************************************************************/

KConfig *Smb4K_Global::config()
{
  return p.config();
}


/***************************************************************************
   Returns a pointer to the global timer object
***************************************************************************/

QTimer *Smb4K_Global::timer()
{
  return p.timer();
}


/***************************************************************************
   Returns the timer interval
***************************************************************************/

const int Smb4K_Global::timerInterval()
{
  return TIME;
}
