/***************************************************************************
    smb4ksharewidget.h  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam Mr 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KSHAREWIDGET_H
#define SMB4KSHAREWIDGET_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstringlist.h>
#include <qpainter.h>
#include <qpalette.h>

// KDE includes
#include <kiconview.h>
#include <kconfig.h>
#include <kprocess.h>
#include <kactioncollection.h>
#include <kaction.h>

// Application includes
#include "../core/smb4kshare.h"

/**
 * This class provides the ability display the mounted shares
 * and manipulate them.
 */
class Smb4KShareWidgetItem;
class Smb4KShareWidget : public KIconView
{
  Q_OBJECT

  friend class Smb4KApp;
  friend class Smb4KShareWidgetItem;

  public:
    /**
     * The constructor.
     */
    Smb4KShareWidget( QWidget *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KShareWidget();
    /**
     * Reads the options.
     */
    void readOptions();
    /**
     * This function returns a pointer to the action collection of this class.
     * You should use it to include the action into the action collection of the
     * main action collection (e.g. KMainWindow::actionCollection()).
     *
     * The entries in this action collection are members of the group "ShareWidget".
     *
     * @returns       a pointer to the action collection of this class
     */
    KActionCollection *action_collection() { return m_collection; }

  signals:
    /**
     * This signal is emitted when the widget has been cleared, i. e. when alls
     * shares have been unmounted.
     */
    void cleared();
    /**
     * This signal is emitted when the selection has been cleared.
     */
    void clearedSelection();

  protected slots:
    /**
     * Is called by the menu. Does the unmounting.
     */
    void slotUnmountShare();
    /**
     * Is called by the menu. Unmounts all shares at once.
     */
    void slotUnmountAllShares();
    /**
     * Is called by the menu. Forces the unmount of a share.
     */
    void slotForceUnmountShare();
    /**
     * Is called to import mounted shares into the share view.
     */
    void slotMountedShares();
    /**
     * Is called, when a share is to be opened with the filemanager.
     * This slot behaves exactly the same as slotItemExecuted(), except
     * this is an external one.
     */
    void slotOpenFilemanager();
    /**
     * Is called when any mouse button was pressed.
     */
    void slotButtonPressed( int button, QIconViewItem *item, const QPoint &pos );
    /**
     * Is called when the current item changed.
     */
    void slotSelectionChanged( QIconViewItem *item );
    /**
     * Is called when the user want to mount a share manually.
     */
    void slotManualMount();

  private:
    /**
     * Holds the display style.
     */
    QString m_display;
    /**
     * Holds whether the mount point should be shown as description.
     */
    bool m_mountpoint;
    /**
     * Determines whether all shares that were mounted should be shown.
     */
    bool m_showExternal;
    /**
     * Determines whether the user wants to force the unmounting of
     * dead shares or not.
     */
    bool m_forcedUnmount;
    /**
     * This is the action collection of this class.
     */
    KActionCollection *m_collection;
    /**
     * This function initializes the actions that are provided by this class.
     */
    void initActions();
    /**
     * The unmount action.
     */
    KAction *m_unmount_action;
    /**
     * Action to force the unmount.
     */
    KAction *m_force_action;
    /**
     * The unmount all action
     */
    KAction *m_unmount_all_action;
    /**
     * The manual mount action
     */
    KAction *m_manual_action;
    /**
     * The filemanager action
     */
    KAction *m_filemanager_action;
};


/**
 * This class handles the icon view items that are to be displayed.
 */
class Smb4KShareWidget;
class Smb4KShareWidgetItem : public KIconViewItem
{
  public:
    /**
     * The constructor.
     *
     * @param share     The Smb4KShare object for which the item is for.
     *
     * @param parent    The parent Smb4KShareWidget.
     *
     * @param alignment The alignment of the icon text. It takes one of the values present in the Smb4KShareWidget::DisplayStyle enumeration.
     *
     * @param mp        Whether the mountpoint instead of the share name should be shown. Default is FALSE.
     */
    Smb4KShareWidgetItem( Smb4KShare *share, Smb4KShareWidget *parent = 0, int alignment = Icons, bool mp = false );
    /**
     * The destructor.
     */
    virtual ~Smb4KShareWidgetItem();
    /**
     * Returns the UID of the item.
     */
    int uid() const { return m_share.getUID(); }
    /**
     * Returns the GID of the item.
     */
    int gid() const { return m_share.getGID(); }
    /**
     * Returns the name of the share.
     */
    QString name() const { return m_share.getName(); }
    /**
     * Returns the mountpoint of this share.
     */
    QString mountpoint() const { return m_share.getPath(); }
    /**
     * Returns the Smb4KShare object.
     */
    Smb4KShare *shareObject() { return &m_share; }
    /**
     * The display style enumeration.
     */
    enum DisplayStyle{ List, Icons };
    /**
     * Marks the Smb4KShare object to be broken.
     */
    void setBroken( bool broken );
    /**
     * Returns TRUE if the share is broken and FALSE otherwise.
     */
    bool isBroken() const { return m_share.isBroken(); }

  protected:
    /**
     * Reimplemented from QIconViewItem.
     */
    void paintItem( QPainter *p, const QColorGroup &cg );

  private:
    /**
     * The Smb4KShare object.
     */
    Smb4KShare m_share;
    /**
     * The alignment object.
     */
    int m_alignment;
};
#endif
