/***************************************************************************
    smb4k_umount  -  This is the unmount utility of Smb4K.
                             -------------------
    begin                : Sa Sep 25 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <cstring>
#include <cerrno>
#include <unistd.h>
#ifndef __FreeBSD__
#include <sys/statfs.h>
#else
#include <sys/param.h>
#include <sys/mount.h>
#endif
using namespace std;

//
// Global variable that carries the name of the file system
//

char fs[5];

//
// Returns the information about the program
//

void info()
{
  cout << "This is smb4k_umount, the unmount utility for Smb4K." << endl;
  cout << "(c) 2004-2005, Alexander Reinholdt" << endl << endl;
  cout << "Usage:" << endl;
  cout << " smb4k_umount [--no-suid|--suid] [--cifs|--smbfs] <mountpoint>" << endl;
  cout << " smb4k_umount --help" << endl << endl;
  cout << "Arguments:" << endl;
  cout << "  --no-suid\tsmb4k_umount invokes smbumount." << endl;
  cout << "  --suid\tsmb4k_umount invokes umount." << endl;
  cout << "  --cifs\tUnmount a CIFS share." << endl;
  cout << "  --smbfs\tUnmount an SMBFS share." << endl;
  cout << "  --help\tDisplays this help screen and exits." << endl;
  cout << "  <mountpoint>\tThe mountpoint of the share." << endl;
  cout << endl;
}


//
// Finds the umount program
//

const char *findprog( const char *name )
{
  const char *paths[] = { "/bin/", "/sbin/", "/usr/bin/", "/usr/sbin/", "/usr/local/bin/", "/usr/local/sbin/" };
  string file = "";

  for ( uint i = 0; i < sizeof( paths ) / sizeof( char * ); i++ )
  {
    string path( paths[i] );
    path.append( name );

    if ( access( path.c_str(), X_OK ) == 0 )
    {
      file.assign( path );
      break;
    }
  }

  if ( !strcmp( file.c_str(), "" ) )
  {
    cerr << "smb4k_umount: Could not find " << name << "." << endl;
    exit( EXIT_FAILURE );
  }

  return file.c_str();
}


//
// Replaces special characters with their ASCII codes. Some
// characters have been omitted, because they either are non-
// critical or are necessary for mounting (like $).
//

const char *replace_special_characters( const char *str )
{
  string s( str );

  for ( uint i = 0; i < s.length(); i++ )
  {
    switch ( s[i] )
    {
      case '\040':
        s.replace( i, 1, "\040" );
        break;
      case '\041':
        s.replace( i, 1, "\041" );
        break;
      case '\042':
        s.replace( i, 1, "\042" );
        break;
      case '\043':
        s.replace( i, 1, "\043" );
        break;
      case '\045':
        s.replace( i, 1, "\045" );
        break;
      case '\046':
        s.replace( i, 1, "\046" );
        break;
      case '\047':
        s.replace( i, 1, "\047" );
        break;
      case '\050':
        s.replace( i, 1, "\050" );
        break;
      case '\051':
        s.replace( i, 1, "\051" );
        break;
      case '\052':
        s.replace( i, 1, "\052" );
        break;
      case '\054':
        s.replace( i, 1, "\054" );
        break;
      case '\074':
        s.replace( i, 1, "\074" );
        break;
      case '\076':
        s.replace( i, 1, "\076" );
        break;
      case '\130':
        s.replace( i, 1, "\130" );
        break;
      case '\131':
        s.replace( i, 1, "\131" );
        break;
      case '\132':
        s.replace( i, 1, "\132" );
        break;
      case '\133':
        s.replace( i, 1, "\133" );
        break;
      case '\134':
        s.replace( i, 1, "\134" );
        break;
      case '\135':
        s.replace( i, 1, "\135" );
        break;
      case '\136':
        s.replace( i, 1, "\136" );
        break;
      case '\137':
        s.replace( i, 1, "\137" );
        break;
      case '\140':
        s.replace( i, 1, "\140" );
        break;
      case '\141':
        s.replace( i, 1, "\141" );
        break;
      case '\142':
        s.replace( i, 1, "\142" );
        break;
      case '\143':
        s.replace( i, 1, "\143" );
        break;
      case '\144':
        s.replace( i, 1, "\144" );
        break;
      case '\145':
        s.replace( i, 1, "\145" );
        break;
      case '\146':
        s.replace( i, 1, "\146" );
        break;
      case '\147':
        s.replace( i, 1, "\147" );
        break;
      case '\150':
        s.replace( i, 1, "\150" );
        break;
      case '\151':
        s.replace( i, 1, "\151" );
        break;
      case '\153':
        s.replace( i, 1, "\153" );
        break;
      case '\154':
        s.replace( i, 1, "\154" );
        break;
      case '\156':
        s.replace( i, 1, "\156" );
        break;
      case '\157':
        s.replace( i, 1, "\157" );
        break;
      case '\160':
        s.replace( i, 1, "\160" );
        break;
      case '\161':
        s.replace( i, 1, "\161" );
        break;
      case '\162':
        s.replace( i, 1, "\162" );
        break;
      case '\163':
        s.replace( i, 1, "\163" );
        break;
      case '\164':
        s.replace( i, 1, "\164" );
        break;
      case '\165':
        s.replace( i, 1, "\165" );
        break;
      case '\173':
        s.replace( i, 1, "\173" );
        break;
      case '\174':
        s.replace( i, 1, "\174" );
        break;
      case '\175':
        s.replace( i, 1, "\175" );
        break;
      case '\176':
        s.replace( i, 1, "\176" );
        break;
      default:
        break;
    }
  }

  return s.c_str();
}


//
// Checks wether the user is allowed to unmount the
// specified filesystem
//
bool check( const char *path )
{
  struct statfs filesystem;

  if ( statfs( path, &filesystem ) == -1 )
  {
    int err_code = errno;

    if ( err_code != EIO && err_code != EACCES )
    {
      cerr << "smb4k_umount: The filesystem check failed: " << strerror( err_code ) << endl;
      exit( EXIT_FAILURE );
    }
    else
    {
      return true; // Bypass the check below, because it would yield ok = FALSE
                   // and we want to be able to unmount broken shares as well.
    }
  }

  bool ok = false;

#ifndef __FreeBSD__
  // First entry is for CIFS, the second for SMBFS.
  if ( (uint)filesystem.f_type == 0xFF534D42 || (uint)filesystem.f_type == 0x517B )
  {
//     if ( ((uint)filesystem.f_type == 0xFF534D42 && strcmp( fs, "cifs" )) ||
//          ((uint)filesystem.f_type == 0x517B && strcmp( fs, "smbfs" )) )
//     {
//       cerr << "smb4k_umount: Wrong filesystem argument! Unmounting might fail." << endl;
//     }
#else
  if ( !strcmp( filesystem.f_fstypename, "smbfs" ) )
  {
#endif
    ok = true;
  }

  return ok;
}



//
// The main function
//

int main( int argc, char *argv[], char *envp[] )
{
  if ( argc < 2 )
  {
    info();
    exit( EXIT_FAILURE );
  }

  char *args[25];
  int k = 0;
  char *path = (char *)malloc( 250*sizeof(char) );

  if ( !path )
  {
    cerr << "smb4k_umount: Out of memory" << endl;
    exit( EXIT_FAILURE );
  }

  bool suid_arg_exists = false;
  bool fs_arg_exists = false;

  int pos;

  for ( int i = 1; i < argc; i++ )
  {
    if ( !strcmp( argv[i], "--help" ) )
    {
      info();
      exit( EXIT_SUCCESS );
    }
    else if ( !strcmp( argv[i], "--no-suid" ) )
    {
      char *p = (char *)malloc( 100*sizeof(char) );

      if ( !p )
      {
        cerr << "smb4k_umount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( p, findprog( "smbumount" ) );
      pos = k;
      k++;
      suid_arg_exists = true;
      continue;
    }
    else if ( !strcmp( argv[i], "--suid" ) )
    {
      char *p = (char *)malloc( 100*sizeof(char) );

      if ( !p )
      {
        cerr << "smb4k_umount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( p, findprog( "umount" ) );
      pos = k;
      k++;
      suid_arg_exists = true;
      continue;
    }
    else if ( !strcmp( argv[i], "--smbfs" ) )
    {
      fs_arg_exists = true;
      strcpy( fs, "smbfs" );
      continue;
    }
    else if ( !strcmp( argv[i], "--cifs" ) )
    {
      fs_arg_exists = true;
      strcpy( fs, "cifs" );

      if ( strcmp( args[pos], findprog( "umount" ) ) )
      {
        free( args[pos] );

        char *p = (char *)malloc( 100*sizeof(char) );

        if ( !p )
        {
          cerr << "smb4k_umount: Out of memory" << endl;
          exit( EXIT_FAILURE );
        }

        args[k] = strcpy( p, findprog( "umount" ) );
      }

      continue;
    }
    else
    {
      char *t = (char *)malloc( 250*sizeof(char) );

      if ( !t )
      {
        cerr << "smb4k_umount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      if ( strlen( argv[i] ) > 250 )
      {
        cerr << "smb4k_umount: One or more arguments exceed 250 characters." << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( t, replace_special_characters( argv[i] ) );

      if ( args[k][0] == '/' )
      {
        path = strcpy( path, args[k] );
      }

      k++;

      continue;
    }
  }

  args[k] = NULL;

  if ( !suid_arg_exists )
  {
    cerr << "smb4k_mount: You have neither supplied the '--suid' nor '--no-suid' argument." << endl;
    exit( EXIT_FAILURE );
  }

  if ( !fs_arg_exists )
  {
    cerr << "smb4k_umount: The filesystem argument is missing." << endl;
    exit( EXIT_FAILURE );
  }

  if ( !check( path ) )
  {
    cerr << "smb4k_umount: You are not allowed to unmount this filesystem type." << endl;
    exit( EXIT_FAILURE );
  }

  free( path );

  if ( execve( args[0], args, envp ) == -1 )
  {
    int err_code = errno;

    cerr << "smb4k_umount: " << strerror( err_code ) << endl;
    exit( EXIT_FAILURE );
  }

  return EXIT_SUCCESS;
}
