/***************************************************************************
    smb4kscanner.h  -  The network scan core class of Smb4K.
                             -------------------
    begin                : Sam Mai 31 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KSCANNER_H
#define SMB4KSCANNER_H

#ifndef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstringlist.h>
#include <qptrqueue.h>

// KDE includes
#include <kprocess.h>

// application specific includes
#include "smb4kdefs.h"
#include "smb4knetworkitems.h"
#include "smb4kpasswordhandler.h"
#include "smb4kdataitem.h"


/**
 * This is the core class, that communicates with the network. All look-up
 * stuff is done here.
 */

class Smb4KScanner : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     *
     * @param parent          The parent of this class.
     *
     * @param name            The name of this class.
     */
    Smb4KScanner( QObject *parent = 0, const char *name = 0 );

    /**
     * The destructor.
     */
    ~Smb4KScanner();

    /**
     * Initiates a network scan.
     */
    void rescan();

    /**
     * Aborts the network scan.
     */
    void abort();

    /**
     * Reads the options.
     */
    void readOptions();

    /**
     * Scan for the shares on a selected host.
     *
     * @param workgroup       The workgroup of the host you want to scan.
     *
     * @param host            The host you want to scan.
     *
     * @param ip              The IP address of the host.
     *
     * @param protocol        With this argument you can force a special protocol
     *                        the net command has to use. Normally, you do not have
     *                        to set here anything.
     */
    void getShares( const QString &workgroup, const QString &host, const QString &ip, const QString &protocol = QString::null );

    /**
     * Scans for workgroup members.
     *
     * @param workgroup       The workgroup of the master browser that should be scanned.
     *
     * @param master          The name of the master browser.
     *
     * @param ip              The IP address of the master browser.
     */
    void getWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip );

    /**
     * Get more info about a share (i.e. server and OS string, etc.).
     *
     * @param workgroup       The workgroup of the host
     *
     * @param host            The host's name
     *
     * @param ip              The host's IP address
     */
    void getInfo( const QString &workgroup, const QString &host, const QString &ip );

    /**
     * Searches for a given host.
     */
    void makeSearch( const QString &host );

    /**
     * Get the preview of a share.
     *
     * @param workgroup       The workgroup the share is in
     *
     * @param host            The host where the share is located
     *
     * @param ip              The IP address of the host
     *
     * @param share           The share name
     *
     * @param path            The path within the share that should be displayed
     *
     * @returns               The name of the share. This is useful, if you want to
     *                        preview a 'homes' share with a certain user name. If the
     *                        the user canceled the input, QString::null will be returned.
     */
    const QString getPreview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path );

    /**
     * This function returns an Smb4KWorkgroupItem, if the the workgroup
     * exists in the list, or NULL, if it does not.
     *
     * @param workgroup       The name of the workgroup
     */
    Smb4KWorkgroupItem *getWorkgroup( const QString &workgroup );

    /**
     * This function takes an Smb4KWorkgroupItem and appends it to the list, if the represented
     * workgroup isn't already in it.
     */
    void appendWorkgroup( Smb4KWorkgroupItem *item );

    /**
     * This function reports if the scanner is running or not.
     *
     * @returns             TRUE if the scanner is running and FALSE otherwise.
     */
    bool isRunning() { return m_working; }

    /**
     * This function returns the specified host item, or NULL, if this host was not found.
     * The name of the host is mandatory. The workgroup may be empty, but should be given,
     * because this will speed up the search process.
     *
     * @param name          The name of the host
     *
     * @param workgroup     The workgroup of the host item
     *
     * @returns             A host item
     */
    Smb4KHostItem *getHost( const QString &name, const QString &workgroup = QString::null );

  signals:
    /**
     * This signal emits the run state.
     *
     * @param state         The so-called run state. There are several defined
     *                      in the smb4kdefs.h header file.
     */
    void state( int state );

    /**
     * This signal is emitted, when the workgroup list has been updated.
     *
     * @param list          The list of workgroups in the network neighborhood.
     */
    void workgroups( const QValueList<Smb4KWorkgroupItem *> &list );

    /**
     * Emits the list of workgroup members.
     *
     * @param list          The list of workgroup members.
     */
    void members( const QValueList<Smb4KHostItem *> &list );

    /**
     * Emits the list of shares.
     */
    void shares( const QValueList<Smb4KShareItem *> &list );

    /**
     * This signal provides info about a certain host. It passes the server
     * and the OS string.
     */
    void info( Smb4KHostItem *host );

    /**
     * This signal emits the directory list of the share that is to be
     * previewed or 0 if an error occurred.
     */
    void previewResult( const QValueList<Smb4KPreviewItem *> &list );

    /**
     * Is emitted, when the results of a network search are to be passed.
     */
    void searchResult( Smb4KHostItem *host );

    /**
     * This signal emits the host item for which an IP address has
     * been found.
     *
     * @param item          The host item with the new IP address
     */
    void ipAddress( Smb4KHostItem *item );

    /**
     * This signal is emitted when the list of hosts is changed.
     */
    void hostListChanged();

  protected:
    /**
     * Starts the main process.
     *
     * @param state         The state
     */
    void startMainProcess( int state );

    /**
     * Starts the background process.
     *
     * @param state         The state
     */
    void startBackgroundProcess( int state );

    /**
     * Is called, when the main process ends.
     */
    void endMainProcess();

    /**
     * Is invoked, when the background process exited.
     */
    void endBackgroundProcess();

    /**
     * Processes the output of the network group scan.
     */
    void processWorkgroups();

    /**
     * Processes the output of the master host scan.
     */
    void processHosts();

    /**
     * Processes the output of the host scan.
     */
    void processShares();

    /**
     * Processes the output of the scan for the OS and Server
     * string.
     */
    void processInfo();

    /**
     * Processes the output of the preview request.
     */
    void processPreview();

    /**
     * Processes the output of a search request.
     */
    void processSearch();

    /**
     * Process the data received from the IP address look-ups.
     */
    void processIPAddresses();

  protected slots:
    /**
     * Is called, if something is received on stdout from the main process.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedMainProcessStdout( KProcess *, char *buf, int len );

    /**
     * Is called, when the KProcess exited.
     */
    void slotMainProcessExited( KProcess * );

    /**
     * Is called, if something is received on stderr from the main process.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedMainProcessStderr( KProcess *, char  *buf, int len );

    /**
     * Starts processing the queue.
     */
    void start();

    /**
     * This slot is connected to the receivedStdout() signal of the background
     * process.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedBackgroundProcessStdout( KProcess *proc, char *buf, int len );

    /**
     * This slot is connected to the receivedStderr() signal of the background
     * process.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedBackgroundProcessStderr( KProcess *proc, char *buf, int len );

    /**
     * This slot is connected to the processExited() signal of the background
     * process. It clears the arguments of the process.
     *
     * @param proc          The process
     */
    void slotBackgroundProcessExited( KProcess *proc );

  private:
    /**
     * Inits the network (re-)scan.
     */
    void init();

    /**
     * Scans the group master for the group members.
     */
    void scanForWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip );

    /**
     * Scans the chosen host for its shares. This is the private part
     * of @see Smb4KScanner::getShares().
     */
    void scanForShares( const QString &workgroup, const QString &host, const QString &ip, const QString &protocol );

    /**
     * Scans a given host for its OS and Server string.
     */
    void scanForInfo( const QString &workgroup, const QString &host, const QString &ip );

    /**
     * Searches for a given host.
     */
    void searchForHost( const QString &host );

    /**
     * Gets a preview of a share.
     *
     * @param workgroup     The workgroup of the host
     *
     * @param host          The name of the host
     *
     * @param ip            The IP address of the host
     *
     * @param share         The name of the remote share that should be previewed
     *
     * @param path          The path within the share that should be previewed
     */
    void preview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path );

    /**
     * This function initiates the scanning for IP addresses.
     */
    void getIPAddresses();

    /**
     * Internal enumeration.
     */
    enum TODO{ Workgroups, QueryHost, Hosts, Shares, Info, Preview, Search, IPAddresses, Init, Idle };

    /**
     * The main KProcess object.
     */
    KProcess *m_main_proc;

    /**
     * The background KProcess object for the IP address searches.
     */
    KProcess *m_bg_proc;

    /**
     * The buffer for the main process.
     */
    QString m_main_buffer;

    /**
     * The buffer for the background process.
     */
    QString m_bg_buffer;

    /**
     * This queue stores the incoming requests until they are processed.
     */
    QPtrQueue<QString> m_queue;

    /**
     * Is set to true, if the scanner is running.
     */
    bool m_working;

    /**
     * The list of workgroups including the master browser and their
     * IP addresses.
     */
    QValueList<Smb4KWorkgroupItem *> m_workgroup_list;

    /**
     * This list contains all hosts, that are found by the scanner and
     * that are currently active.
     */
    QValueList<Smb4KHostItem *> m_hosts_list;

    /**
     * This is the password handler for this class.
     */
    Smb4KPasswordHandler *m_password_handler;

    /**
     * The internal state of the main process. Do not mix this up with
     * the state that's emitted to notify the application about changes.
     */
    int m_main_state;

    /**
     * The internal state of the background process. Do not mix this up with
     * the state that's emitted to notify the application about changes.
     */
    int m_bg_state;

    /**
     * This determines if the background process is running.
     */
    bool m_bg_proc_working;

    /**
     * Connect or disconnect the timer.
     *
     * @param con           TRUE if the timer should be connected and false otherwise.
     */
    void connect_timer( bool con );
};

#endif
