/***************************************************************************
    smb4k_mount  -  This is the mount utility for Smb4K.
                             -------------------
    begin                : Di Sep 21 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <cstring>
#include <cerrno>
#include <unistd.h>
using namespace std;


//
// Returns the information about the program
//

void info()
{
  cout << "This is smb4k_mount, the mount utility for Smb4K." << endl;
  cout << "(c) 2004-2006, Alexander Reinholdt" << endl << endl;
  cout << "Usage:" << endl;
  cout << " smb4k_mount [--no-suid|--suid] <options>" << endl;
  cout << " smb4k_mount --help" << endl << endl;
  cout << "Arguments:" << endl;
  cout << "  --no-suid\tsmb4k_mount invokes smbmount." << endl;
  cout << "  --suid\tsmb4k_mount invokes mount." << endl;
  cout << "  --help\tDisplays this help screen and exits." << endl;
  cout << "  <options>\tThese are the options that you have to pass to the mount" << endl;
  cout << "\t\tbinary. Note, that under Linux you must specify the \'-t\'" << endl;
  cout << "\t\toption under any circumstances. Under FreeBSD, however, it" << endl;
  cout << "\t\tis not recognized and might cause problems." << endl;
  cout << endl;
}


//
// Finds the mount program
//

const char *findprog( const char *name )
{
  const char *paths[] = { "/bin/", "/sbin/", "/usr/bin/", "/usr/sbin/", "/usr/local/bin/", "/usr/local/sbin/" };
  string file = "";

  for ( uint i = 0; i < sizeof( paths ) / sizeof( char * ); i++ )
  {
    string path( paths[i] );
    path.append( name );

    if ( access( path.c_str(), X_OK ) == 0 )
    {
      file.assign( path );
      break;
    }
  }

  if ( !strcmp( file.c_str(), "" ) )
  {
    cerr << "smb4k_mount: Could not find " << name << "." << endl;
    exit( EXIT_FAILURE );
  }

  return file.c_str();
}


//
// Replaces special characters with their ASCII codes. Some
// characters have been omitted, because they either are non-
// critical or are necessary for mounting (like $).
//

const char *replace_special_characters( const char *str )
{
  string s( str );

  for ( uint i = 0; i < s.length(); i++ )
  {
    switch ( s[i] )
    {
      case '\040':
        s.replace( i, 1, "\040" );
        break;
      case '\041':
        s.replace( i, 1, "\041" );
        break;
      case '\042':
        s.replace( i, 1, "\042" );
        break;
      case '\043':
        s.replace( i, 1, "\043" );
        break;
      case '\045':
        s.replace( i, 1, "\045" );
        break;
      case '\046':
        s.replace( i, 1, "\046" );
        break;
      case '\047':
        s.replace( i, 1, "\047" );
        break;
      case '\050':
        s.replace( i, 1, "\050" );
        break;
      case '\051':
        s.replace( i, 1, "\051" );
        break;
      case '\052':
        s.replace( i, 1, "\052" );
        break;
      case '\054':
        s.replace( i, 1, "\054" );
        break;
      case '\074':
        s.replace( i, 1, "\074" );
        break;
      case '\076':
        s.replace( i, 1, "\076" );
        break;
      case '\130':
        s.replace( i, 1, "\130" );
        break;
      case '\131':
        s.replace( i, 1, "\131" );
        break;
      case '\132':
        s.replace( i, 1, "\132" );
        break;
      case '\133':
        s.replace( i, 1, "\133" );
        break;
      case '\134':
        s.replace( i, 1, "\134" );
        break;
      case '\135':
        s.replace( i, 1, "\135" );
        break;
      case '\136':
        s.replace( i, 1, "\136" );
        break;
      case '\137':
        s.replace( i, 1, "\137" );
        break;
      case '\140':
        s.replace( i, 1, "\140" );
        break;
      case '\141':
        s.replace( i, 1, "\141" );
        break;
      case '\142':
        s.replace( i, 1, "\142" );
        break;
      case '\143':
        s.replace( i, 1, "\143" );
        break;
      case '\144':
        s.replace( i, 1, "\144" );
        break;
      case '\145':
        s.replace( i, 1, "\145" );
        break;
      case '\146':
        s.replace( i, 1, "\146" );
        break;
      case '\147':
        s.replace( i, 1, "\147" );
        break;
      case '\150':
        s.replace( i, 1, "\150" );
        break;
      case '\151':
        s.replace( i, 1, "\151" );
        break;
      case '\153':
        s.replace( i, 1, "\153" );
        break;
      case '\154':
        s.replace( i, 1, "\154" );
        break;
      case '\156':
        s.replace( i, 1, "\156" );
        break;
      case '\157':
        s.replace( i, 1, "\157" );
        break;
      case '\160':
        s.replace( i, 1, "\160" );
        break;
      case '\161':
        s.replace( i, 1, "\161" );
        break;
      case '\162':
        s.replace( i, 1, "\162" );
        break;
      case '\163':
        s.replace( i, 1, "\163" );
        break;
      case '\164':
        s.replace( i, 1, "\164" );
        break;
      case '\165':
        s.replace( i, 1, "\165" );
        break;
      case '\173':
        s.replace( i, 1, "\173" );
        break;
      case '\174':
        s.replace( i, 1, "\174" );
        break;
      case '\175':
        s.replace( i, 1, "\175" );
        break;
      case '\176':
        s.replace( i, 1, "\176" );
        break;
      default:
        break;
    }
  }

  return s.c_str();
}


//
// The main function
//

int main( int argc, char *argv[], char *envp[] )
{
  if ( argc < 2 )
  {
    info();
    exit( EXIT_FAILURE );
  }

  bool fs_exists = false;
  bool suid_arg_exists = false;
  bool use_suid = false;
  char *args[25];
  int k = 0;

  // Loop through the argument
  for ( int i = 1; i < argc; i++ )
  {
    if ( !strcmp( argv[i], "--help" ) )
    {
      info();
      exit( EXIT_SUCCESS );
    }
    else if ( !strcmp( argv[i], "--no-suid" ) )
    {
      // Under FreeBSD, we can put the binary in the array already
      // at this point, because there is only one. Under Linux, we
      // have to determine the filesystem type before.

#ifdef __FreeBSD__
      char *p = new char[100];

      if ( !p )
      {
        cerr << "smb4k_mount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( p, findprog( "mount_smbfs" ) );
#endif

      k++;
      suid_arg_exists = true;
      use_suid = false;
      continue;
    }
    else if ( !strcmp( argv[i], "--suid" ) )
    {
      // Under FreeBSD, we can put the binary in the array already
      // at this point, because there is only one. Under Linux, we
      // have to determine the filesystem type before.

#ifdef __FreeBSD__
      char *p = new char[100];

      if ( !p )
      {
        cerr << "smb4k_mount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( p, findprog( "mount_smbfs" ) );
#endif

      k++;
      suid_arg_exists = true;
      use_suid = true;
      continue;
    }
    else
    {
#ifndef __FreeBSD__
      if ( !strcmp( argv[i], "-t" ) )
      {
        fs_exists = true;

        if ( strcmp( argv[i+1], "smbfs" ) != 0 && strcmp( argv[i+1], "cifs" ) != 0 )
        {
          cerr << "smb4k_mount: You specified an invalid filesystem." << endl;
          exit( EXIT_FAILURE );
        }
        else
        {
          // Put the binary into the first position.

          if ( suid_arg_exists )
          {
            if ( use_suid )
            {
              char *p = new char[100];

              if ( !p )
              {
                cerr << "smb4k_mount: Out of memory" << endl;
                exit( EXIT_FAILURE );
              }

              args[0] = strcpy( p, findprog( "mount" ) );
            }
            else
            {
              char *p = new char[100];

              if ( !p )
              {
                cerr << "smb4k_mount: Out of memory" << endl;
                exit( EXIT_FAILURE );
              }

              if ( !strcmp( argv[i+1], "smbfs" ) )
              {
                args[0] = strcpy( p, findprog( "smbmount" ) );
              }
              else if ( !strcmp( argv[i+1], "cifs" ) )
              {
                args[0] = strcpy( p, findprog( "mount.cifs" ) );
              }

              // For smbmount and mount.cifs, we do not need the -t option.
              // Jump to the next argument.
              i++;

              continue;
            }
          }
          else
          {
            cerr << "smb4k_mount: You have to supply either the '--suid' or '--no-suid' argument." << endl;
            exit( EXIT_FAILURE );
          }
        }
      }
#endif

      char *t = new char[strlen( argv[i] )+1];

      if ( !t )
      {
        cerr << "smb4k_mount: Out of memory" << endl;
        exit( EXIT_FAILURE );
      }

      args[k] = strcpy( t, replace_special_characters( argv[i] ) );
      k++;

      continue;
    }
  }

  args[k] = NULL;

  // FIXME: Do we need the following under FreeBSD?

  if ( !suid_arg_exists )
  {
    cerr << "smb4k_mount: You have to supply either the '--suid' or '--no-suid' argument." << endl;
    exit( EXIT_FAILURE );
  }

#ifndef __FreeBSD__
  if ( !fs_exists )
  {
    cerr << "smb4k_mount: You haven't specified a filesystem (smbfs or cifs)." << endl;
    exit( EXIT_FAILURE );
  }
#endif

  if ( execve( args[0], args, envp ) == -1 )
  {
    int err_code = errno;

    cerr << "smb4k_mount: " << strerror( err_code ) << endl;
    exit( EXIT_FAILURE );
  }

  return EXIT_SUCCESS;
}
