/***************************************************************************
    smb4kpasswordhandler  -  This class handles the passwords for Smb4K.
                             -------------------
    begin                : So Jan 16 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/


// Qt includes
#include <qframe.h>
#include <qlayout.h>
#include <qstring.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qdir.h>
#include <qfile.h>
#include <qstringlist.h>

// KDE includes
#include <klocale.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kdebug.h>

#ifdef __FreeBSD__
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#endif

// application specific includes
#include "smb4kpasswordhandler.h"
#include "smb4kdefs.h"
#include "smb4kerror.h"


#ifndef __FreeBSD__
Smb4KPasswordHandler::Smb4KPasswordHandler( KConfig *config, Smb4KHomesSharesHandler *handler, QObject *parent, const char *name )
: QObject( parent, name ), m_config( config ), m_handler( handler )
#else
Smb4KPasswordHandler::Smb4KPasswordHandler( KConfig *config, Smb4KHomesSharesHandler *s_handler, Smb4KSambaOptionsHandler *o_handler, QObject *parent, const char *name )
: QObject( parent, name ), m_config( config ), m_handler( s_handler ), m_options_handler( o_handler )
#endif
{
  if ( !m_config )
  {
    kdFatal() << "Smb4KPasswordHandler: No KConfig object" << endl;
  }

  if ( !m_handler )
  {
    kdFatal() << "Smb4KPasswordHandler: No Smb4KHomesSharesHandler object" << endl;
  }

#ifdef __FreeBSD__
  if ( !m_options_handler )
  {
    kdFatal() << "Smb4KPasswordHandler: No Smb4KSambaOptionsHandler object" << endl;
  }
#endif

  m_auth = NULL;
  m_dlg = NULL;
  m_wallet = NULL;
  m_temp_auth = NULL;

  m_config->setGroup( "Authentication" );

  if ( !m_config->hasKey( "Use Wallet" ) )
  {
    m_config->writeEntry( "Use Wallet", true );
  }

  m_config->sync();
}


Smb4KPasswordHandler::~Smb4KPasswordHandler()
{
  for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_auth_list.begin(); it != m_auth_list.end(); ++it )
  {
    delete *it;
  }

  m_auth_list.clear();

  delete m_wallet;
}


void Smb4KPasswordHandler::open_close_wallet()
{
  m_config->setGroup( "Authentication" );

  if ( m_config->readBoolEntry( "Use Wallet", true ) )
  {
    if ( !m_wallet || (m_wallet && !m_wallet->isOpen()) )
    {
      m_wallet = KWallet::Wallet::openWallet( KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous );

      if ( m_wallet )
      {
        if ( !m_wallet->hasFolder( "Smb4K" ) )
        {
          m_wallet->createFolder( "Smb4K" );
        }

        m_wallet->setFolder( "Smb4K" );

        import();
      }
      else
      {
        Smb4KError::error( ERROR_OPENING_WALLET_FAILED, KWallet::Wallet::NetworkWallet(), QString::null );

        delete m_wallet;
        m_wallet = NULL;

        // Do not remove the following line or the entry won't be written
        // properly. This seems to have to do with the transition to the global
        // config object.
        m_config->setGroup( "Authentication" );
        m_config->writeEntry( "Use Wallet", false );
        m_config->sync();

        import();
      }
    }
  }
  else
  {
    if ( m_wallet )
    {
      delete m_wallet;
      m_wallet = NULL;
    }
  }
}


bool Smb4KPasswordHandler::askpass( const QString &workgroup, const QString &host, const QString &share, int desc, QWidget *parent, const char *name )
{
  m_auth = readAuth( workgroup, host, share );

  // Set up the askpass dialog:

  m_dlg = new KDialogBase( KDialogBase::Plain, i18n( "Authentication" ), KDialogBase::Ok|KDialogBase::Cancel,
                           KDialogBase::Ok, parent, name, true, true );

  QFrame *frame = m_dlg->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 10 );
  layout->setMargin( 0 );

  QLabel *pixmap_label = new QLabel( frame );
  pixmap_label->setPixmap( DesktopIcon( "identity" ) );
  pixmap_label->adjustSize();

  layout->addWidget( pixmap_label, 0, 0, Qt::AlignCenter );

  QString message;

  switch ( desc )
  {
    case NewData:
      break;
    case AccessDenied:
      message = i18n( "The access was denied. " );
      break;
    case BadPassword:
      message = i18n( "The password is not correct. " );
      break;
    case PermDenied:
      message = i18n( "The permission was denied. " );
      break;
    case AuthError:
      message = i18n( "An authentication error occurred. " );
      break;
    case LogonFailure:
      message = i18n( "The logon failed. " );
      break;
    default:
      break;
  }

  message.append( i18n( "Please enter authentication data for %1." ).arg(
                  m_auth->share().stripWhiteSpace().isEmpty() ?
                  m_auth->host() :
                  "//"+m_auth->host()+"/"+m_auth->share() ) );

  QLabel *message_label = new QLabel( frame );
  message_label->setText( message.stripWhiteSpace() );
  message_label->setTextFormat( Qt::RichText );

  layout->addWidget( message_label, 0, 1, 0 );

  QLabel *user_label = new QLabel( i18n( "User:" ), frame );
  layout->addWidget( user_label, 1, 0, 0 );

  KLineEdit *user_edit = NULL;
  KComboBox *user_combo = NULL;

  if ( QString::compare( m_auth->share(), "homes" ) != 0 )
  {
    user_edit = new KLineEdit( frame, "AskPassUserEdit" );
    user_edit->setMinimumWidth( 200 );
    layout->addWidget( user_edit, 1, 1, 0 );
  }
  else
  {
    user_combo = new KComboBox( frame, "AskPassUserCombo" );
    user_combo->setEditable( true );
    user_combo->setMinimumWidth( 200 );
    layout->addWidget( user_combo, 1, 1, 0 );
  }

  QLabel *password_label = new QLabel( i18n( "Password:" ), frame );
  layout->addWidget( password_label, 2, 0, 0 );

  KLineEdit *pass_edit = new KLineEdit( frame, "AskPassPasswordEdit" );
  pass_edit->setEchoMode( KLineEdit::Password );
  layout->addWidget( pass_edit, 2, 1, 0 );

  m_dlg->setMainWidget( frame );
  m_dlg->setFixedSize( 350, m_dlg->sizeHint().height() );
  m_dlg->enableButtonOK( false );

  // Since we have to allow empty passwords, we will only connect
  // the edit line for the user to enable/disable the OK button.
  if ( user_edit )
  {
    connect( user_edit, SIGNAL( textChanged( const QString & ) ),
             this,      SLOT( slotEnableOKButton( const QString& ) ) );
  }
  else
  {
    connect( user_combo, SIGNAL( textChanged( const QString & ) ),
             this,       SLOT( slotEnableOKButton( const QString& ) ) );
  }

  // Process the authentication data:

  if ( QString::compare( share, "homes" ) != 0 )
  {
    user_edit->setText( m_auth->user() );
    pass_edit->setText( m_auth->password() );

    if ( m_auth->user().isEmpty() )
    {
      user_edit->setFocus();
    }
    else
    {
      pass_edit->setFocus();
    }
  }
  else
  {
    QStringList list = m_handler->homesUsers( host );

    user_combo->insertStringList( list );
    user_combo->setCurrentText( QString::null );

    connect( user_combo, SIGNAL( activated( const QString & ) ),
             this,       SLOT( slotGetPassword( const QString & ) ) );

    user_combo->setFocus();
  }

  bool ok = false;

  if ( m_dlg->exec() == KDialogBase::Accepted )
  {
    if ( QString::compare( share, "homes" ) != 0 )
    {
      QString user = user_edit->text();
      QString pass = pass_edit->text();

      m_auth->setUser( user );
      m_auth->setPassword( pass );

      writeAuth( m_auth );
    }
    else
    {
      QString user = user_combo->currentText();
      QString pass = pass_edit->text();

      m_auth->setUser( user );
      m_auth->setPassword( pass );

      writeAuth( m_auth );
    }

    ok = true;
  }

  // Clean up:

  delete m_dlg;
  m_dlg = NULL;

  delete m_auth;
  m_auth = NULL;

  return ok;
}


Smb4KAuthInfo *Smb4KPasswordHandler::readAuth( const QString &workgroup, const QString &host, const QString &share )
{
  open_close_wallet();

  m_config->setGroup( "Authentication" );
  bool use_default = m_config->readBoolEntry( "Default Authentication" );

  Smb4KAuthInfo *auth = new Smb4KAuthInfo( workgroup, host, share );

  if ( m_wallet && m_wallet->isOpen() )
  {
    QStringList full_list = m_wallet->entryList();
    QStringList host_list = full_list.grep( host, false );

    if ( !host_list.isEmpty() )
    {
      for ( QStringList::ConstIterator it = host_list.begin(); it != host_list.end(); ++it )
      {
         if ( !share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":"+share.upper() ) != 0 )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          break;
        }
        else if ( !share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":*" ) != 0 )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          continue;
        }
        else if ( share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":" ) )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          break;
        }
        else if ( share.isEmpty() && (*it).upper().startsWith( "DEFAULT:" ) && use_default )
        {
          QString user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          continue;
        }
        else
        {
          continue;
        }
      }
    }
    else
    {
      m_config->setGroup( "Authentication" );

      if ( m_config->readBoolEntry( "Default Authentication", false ) )
      {
        QStringList default_list = full_list.grep( "DEFAULT", false );
        QString user, pass;

        for ( QStringList::ConstIterator it = default_list.begin(); it != default_list.end(); ++it )
        {
          if ( (*it).startsWith( "DEFAULT:" ) )
          {
            user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
            m_wallet->readPassword( *it, pass );

            auth->setUser( user );
            auth->setPassword( pass );

            break;
          }
          else
          {
            continue;
          }
        }
      }
    }
  }
  else
  {
    m_config->setGroup( "Authentication" );
    bool remember = m_config->readBoolEntry( "Remember Passwords", true );

    if ( remember && !m_auth_list.isEmpty() )
    {
      for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_auth_list.begin(); it != m_auth_list.end(); ++it )
      {
        if ( (QString::compare( (*it)->workgroup().upper(), workgroup.upper() ) == 0 &&
              QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
              QString::compare( (*it)->share().upper(), share.upper() ) == 0) ||
             (QString::compare( (*it)->workgroup().upper(), "*" ) == 0 &&
              QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
              QString::compare( (*it)->share().upper(), share.upper() ) == 0) )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );

          break;
        }
        else if ( (QString::compare( (*it)->workgroup().upper(), workgroup.upper() ) == 0 &&
                   QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
                   QString::compare( (*it)->share().upper(), "*" ) == 0) ||
                  (QString::compare( (*it)->workgroup().upper(), "*" ) == 0 &&
                   QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
                   QString::compare( (*it)->share().upper(), "*" ) == 0) )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );

          continue;
        }
        else
        {
          continue;
        }
      }
    }
    else if ( !remember )
    {
      if ( m_temp_auth )
      {
        auth->setUser( m_temp_auth->user() );
        auth->setPassword( m_temp_auth->password() );

        delete m_temp_auth;
        m_temp_auth = NULL;
      }
    }
  }

#ifdef __FreeBSD__

  writeToSMBConfFile( auth );

#endif

  return auth;
}


void Smb4KPasswordHandler::writeAuth( Smb4KAuthInfo *auth )
{
  open_close_wallet();

  if ( m_wallet && m_wallet->isOpen() )
  {
    // Remove old entry:
    QString key;
    key.append( auth->workgroup().isEmpty() ? "*" : auth->workgroup().upper() );
    key.append( ":" );
    key.append( auth->host().upper() );
    key.append( ":" );
    key.append( auth->share().isEmpty() ? "*" : auth->share().upper() );

    QStringList entries = m_wallet->entryList();

    for ( QStringList::ConstIterator it = entries.begin(); it != entries.end(); ++it )
    {
      if ( (*it).stripWhiteSpace().startsWith( key ) )
      {
        m_wallet->removeEntry( *it );

        break;
      }
      else
      {
        continue;
      }
    }

    key.append( ":" );
    key.append( auth->user() );

    m_wallet->writePassword( key, auth->password() );
    m_wallet->sync();
  }
  else
  {
    m_config->setGroup( "Authentication" );

    if ( m_config->readBoolEntry( "Remember Passwords", true ) )
    {
      bool is_included = false;

      for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_auth_list.begin(); it != m_auth_list.end(); ++it )
      {
        if ( (QString::compare( (*it)->workgroup().upper(), auth->workgroup().upper() ) == 0 ||
              QString::compare( (*it)->workgroup().upper(), "*" ) == 0) &&
             QString::compare( (*it)->host().upper(), auth->host().upper() ) == 0 &&
             QString::compare( (*it)->share().upper(), auth->share().upper() ) == 0 )
        {
          delete *it;
          *it = new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() );

          is_included = true;

          break;
        }
      }

      if ( !is_included )
      {
        m_auth_list.append( new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() ) );
      }
    }
    else
    {
      if ( !m_temp_auth )
      {
        m_temp_auth = new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() );
      }
    }
  }

#ifdef __FreeBSD__

  writeToSMBConfFile( auth );

#endif
}


Smb4KAuthInfo *Smb4KPasswordHandler::readDefaultAuth()
{
  open_close_wallet();

  Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null );

  if ( m_wallet && m_wallet->isOpen() )
  {
    QStringList full_list = m_wallet->entryList();
    QStringList default_list = full_list.grep( "DEFAULT" );

    QString user, pass;

    for ( QStringList::ConstIterator it = default_list.begin(); it != default_list.end(); ++it )
    {
      if ( (*it).startsWith( "DEFAULT:" ) )
      {
        user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
        m_wallet->readPassword( *it, pass );

        auth->setUser( user );
        auth->setPassword( pass );

        break;
      }
      else
      {
        continue;
      }
    }
  }
  else
  {
    delete auth;
    auth = NULL;
  }

  return auth;
}


void Smb4KPasswordHandler::writeDefaultAuth( Smb4KAuthInfo *auth )
{
  open_close_wallet();

  if ( m_wallet && m_wallet->isOpen() )
  {
    QString key = "DEFAULT:"+auth->user();

    m_wallet->writePassword( key, auth->password() );
  }
}


void Smb4KPasswordHandler::import()
{
  m_config->setGroup( "Authentication" );

  int key = 0;

  if ( m_config->hasKey( "Random Number" ) )
  {
    key = m_config->readNumEntry( "Random Number" );
    m_config->deleteEntry( "Random Number" );
  }

  QDir::setCurrent( locateLocal( "appdata", QString::null, KGlobal::instance() ) );

  QFile file( "passwords" );

  if ( file.exists() && key != 0 )
  {
    KConfig *pw = new KConfig( "passwords", false, true, "appdata" );

    if ( pw->hasGroup( "Default" ) )
    {
      pw->setGroup( "Default" );

      QString default_user = pw->readEntry( "User" );
      QString default_pass = pw->readEntry( "Password" );

      QString dp;
      uint index( 0 );

      while ( index < default_pass.length() )
      {
        dp[index] = (char)( (int)default_pass[index].unicode()-key%62 );
        index++;
      }

      if ( m_wallet && m_wallet->isOpen() )
      {
        m_wallet->writePassword( "DEFAULT:"+default_user, dp );
      }
    }

    if ( pw->hasGroup( "Logins" ) )
    {
      pw->setGroup( "Logins" );

      for ( int i = 0; ; i++ )
      {
        QStringList list = pw->readListEntry( QString( "%1" ).arg( i ).stripWhiteSpace(), ',' );

        if ( list.isEmpty() )
        {
          break;
        }
        else
        {
          QString p;
          uint index( 0 );

          while( index < list[4].length() )
          {
            p[index] = (char)( (int)list[4][index].unicode()-key%62 );
            index++;
          }

          QString entry = list[0].upper()+":"+list[1].upper()+":"+list[2].upper()+":"+list[3];

          if ( m_wallet && m_wallet->isOpen() )
          {
            m_wallet->writePassword( entry, p );
          }
        }
      }
    }

    if ( m_wallet && m_wallet->isOpen() )
    {
      m_wallet->sync();
    }

    delete pw;

    file.remove();
  }

  m_config->sync();
}


#ifdef __FreeBSD__

void Smb4KPasswordHandler::writeToSMBConfFile( Smb4KAuthInfo *auth )
{
  m_nsmbrc_auth = *auth;

  KProcess *p = new KProcess();
  p->setUseShell( true );

  connect( p,    SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this, SLOT( slotReceivePassword( KProcess *, char *, int ) ) );
  connect( p,    SIGNAL( processExited( KProcess * ) ),
           this, SLOT( slotWritePassword( KProcess * ) ) );

  *p << QString( "smbutil crypt %1" ).arg( m_nsmbrc_auth.password() );

  p->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}

#endif


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KPasswordHandler::slotGetPassword( const QString &username )
{
  if ( m_dlg && m_auth )
  {
    Smb4KAuthInfo *auth = readAuth( m_auth->workgroup().upper(), m_auth->host().upper(), username );
    ((KLineEdit *)m_dlg->child( "AskPassPasswordEdit", "KLineEdit", true ))->setText( auth->password() );
    delete auth;

    m_auth->setShare( username );
  }
}


void Smb4KPasswordHandler::slotEnableOKButton( const QString &text )
{
  if ( m_dlg )
  {
    m_dlg->enableButtonOK( !text.isEmpty() );
  }
}


void Smb4KPasswordHandler::slotReceivePassword( KProcess *, char *buffer, int buflen )
{
#ifdef __FreeBSD__
  m_buffer.append( QString::fromLocal8Bit( buffer, buflen ) );
#endif
}


void Smb4KPasswordHandler::slotWritePassword( KProcess *proc )
{

  delete proc;

#ifdef __FreeBSD__

  QString pass = m_buffer.remove( "\n" ).stripWhiteSpace();
  m_buffer = QString::null;

  QDir::setCurrent( QDir::homeDirPath() );

  QFile file( ".nsmbrc" );

  QStringList contents;

  if ( file.exists() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );
      ts.setEncoding( QTextStream::Locale );

      while ( !ts.atEnd() )
      {
        contents.append( ts.readLine().stripWhiteSpace() );
      }

      file.close();
    }
  }

  // Check if the [default] section is present.
  if ( contents.find( "[default]" ) == contents.end() &&
       contents.find( "[DEFAULT]" ) == contents.end() )
  {
    QMap<QString,QString> map = m_options_handler->globalSambaOptions();
    QString workgroup = map["workgroup"];
    QString wins = m_options_handler->winsServer();

    QStringList::Iterator it = contents.prepend( "[default]" );
    it++;

    if ( !workgroup.isEmpty() )
    {
      it = contents.insert( it, "workgroup="+workgroup );
    }

    if ( !wins.isEmpty() )
    {
      it = contents.insert( it, "nbns="+wins );
    }

    // FIXME: Should we write the charsets here, too??
  }

  QString section;

  if ( m_nsmbrc_auth.share().isEmpty() )
  {
    section.append( QString( "[%1:%2]" ).arg( m_nsmbrc_auth.host().upper(), m_nsmbrc_auth.user().isEmpty() ? "GUEST" : m_nsmbrc_auth.user().upper() ) );
  }
  else
  {
    section.append( QString( "[%1:%2:%3]" ).arg( m_nsmbrc_auth.host().upper(), m_nsmbrc_auth.user().isEmpty() ? "GUEST" : m_nsmbrc_auth.user().upper(), m_nsmbrc_auth.share().upper() ) );
  }

  QStringList::Iterator it = contents.find( section );

  if ( it == contents.end() )
  {
    if ( contents.last() != QString::null )
    {
      contents.append( QString::null );
    }

    contents.append( section );
    contents.append( "password="+pass );

    if ( !m_nsmbrc_auth.workgroup().isEmpty() )
    {
      contents.append( "workgroup="+m_nsmbrc_auth.workgroup() );
    }
  }
  else
  {
    for ( QStringList::Iterator i = ++it; i != contents.end(); ++i )
    {
      if ( (*i).contains( "password=" ) )
      {
        QString old_pass = (*i).section( "password=", 1, 1 ).stripWhiteSpace();

        if ( QString::compare( pass, old_pass ) == 0 )
        {
          // The passwords are identical, so stop here.
          return;
        }
        else
        {
          *i = "password="+pass;
          break;
        }
      }
      else if ( (*i).startsWith( "[" ) )
      {
        break;
      }
      else
      {
        continue;
      }
    }
  }

  QDir::setCurrent( QDir::homeDirPath() );

  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
    {
      ts << *it << endl;
    }

    file.close();
  }
  else
  {
    Smb4KError::error( ERROR_WRITING_FILE, "~/.nsmbrc", QString::null );

    return;
  }

  // Get minimal security: Fix permissions.
  QString path = QDir::homeDirPath()+"/"+file.name();

  int fd;

  if ( (fd = open( path.ascii(), O_WRONLY )) == -1 )
  {
    int err_no = errno;
    Smb4KError::error( ERROR_OPENING_FILE, path, strerror( err_no ) );

    return;
  }
  else
  {
    if ( fchmod( fd, 00600 ) == -1 )
    {
      // FIXME: Do we need to emit an error here at all?
    }
  }
#endif
}

#include "smb4kpasswordhandler.moc"
