/***************************************************************************
                          smb4ksynchronizer  -  description
                             -------------------
    begin                : Mo Jul 4 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qdir.h>
#include <qlabel.h>
#include <qregexp.h>

// KDE includes
#include <kmessagebox.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <klineedit.h>
#include <klocale.h>
#include <kprogress.h>
#include <kurlrequester.h>
#include <kguiitem.h>
#include <kapplication.h>

// application specific includes
#include "smb4ksynchronizer.h"
#include "smb4kdefs.h"
#include "smb4kerror.h"
#include "smb4kglobal.h"

using namespace Smb4KGlobal;

bool cancel = false;


Smb4KSynchronizer::Smb4KSynchronizer( QObject *parent, const char *name )
: QObject( parent, name )
{
  m_proc = new KProcess( this, "SynchronizerProcess" );

  m_proc->setUseShell( true );

  m_working = false;

  m_dialog = NULL;

  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,   SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess* ) ),
           this,   SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,   SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( kapp,   SIGNAL( shutDown() ),
           this,   SLOT( slotShutdown() ) );
}


Smb4KSynchronizer::~Smb4KSynchronizer()
{
  if ( m_dialog )
  {
    delete m_dialog;
  }
}


/****************************************************************************
   Synchronizes a share with a local copy or vice versa.
****************************************************************************/

void Smb4KSynchronizer::synchronize( Smb4KShare *share, QWidget *parent )
{
  config()->setGroup( "Programs" );

  if ( config()->readPathEntry( "rsync" ).isEmpty() )
  {
    return;
  }

  config()->setGroup( "Rsync" );
  QString default_prefix = config()->readEntry( "Prefix", QDir::homeDirPath()+"/smb4k_sync/" );

  m_chooser = new KDialogBase( KDialogBase::Plain, i18n( "Synchronization" ),
                               KDialogBase::User1|KDialogBase::Ok|KDialogBase::Cancel,
                               KDialogBase::Ok,
                               parent,
                               "SynchronizationDialog",
                               false,
                               true );

  m_chooser->setButtonGuiItem( KDialogBase::User1, KGuiItem( i18n( "Swap Paths" ), QString::null, i18n( "Swap source and destination" ) ) );
  connect( m_chooser, SIGNAL( user1Clicked() ),
           this,      SLOT( slotSwapURLs() ) );

  QFrame *f = m_chooser->plainPage();
  QGridLayout *l = new QGridLayout( f );
  l->setSpacing( 10 );
  l->setMargin( 0 );

  QLabel *srcLabel = new QLabel( i18n( "Source:" ), f );
  QLabel *destLabel = new QLabel( i18n( "Destination:" ), f );

  KURLRequester *src = new KURLRequester( share->path()+"/", f, "SourceURL" );
  src->setShowLocalProtocol( false );
  src->setMode( KFile::Directory | KFile::LocalOnly );

  KURLRequester *dest = new KURLRequester( default_prefix, f, "DestinationURL" );
  dest->setShowLocalProtocol( false );
  dest->setMode( KFile::Directory | KFile::LocalOnly );

  l->addWidget( srcLabel, 0, 0, 0 );
  l->addWidget( destLabel, 1, 0, 0 );
  l->addWidget( src, 0, 1, 0 );
  l->addWidget( dest, 1, 1, 0 );

  f->setFixedWidth( 350 );

  QString source, destination;

  if ( m_chooser->exec() == KDialogBase::Accepted )
  {
    source = src->url();
    destination = dest->url();
    delete m_chooser;
  }
  else
  {
    delete m_chooser;
    return;
  }

  m_working = true;
  emit state( SYNCER_START );

  // We will only define the stuff we urgently need
  // here. The options that actually influence rsync's
  // behavior will be retrieved by readRsyncOptions().
  QString command = "rsync --progress ";

  command.append( readRsyncOptions() );
  command.append( " " );
  command.append( KProcess::quote( source ) );
  command.append( " " );
  command.append( KProcess::quote( destination ) );

  *m_proc << command;

  // Set up the progress dialog.
  m_dialog = new KDialogBase( KDialogBase::Plain,
                              i18n( "Synchronization" ),
                              KDialogBase::Cancel,
                              KDialogBase::Cancel,
                              parent,
                              "SynchronizationProgressDialog",
                              false,
                              true );

  QFrame *frame = m_dialog->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 10 );
  layout->setMargin( 0 );

  KLineEdit *el = new KLineEdit( i18n( "Preparing Synchronization ..." ), frame, "SyncProgDlgEdit" );
  el->setEnableSqueezedText( true );
  el->setReadOnly( true );

  KProgress *prog_part = new KProgress( frame, "SyncProgDlgProgPart", 0 );
  KProgress *prog_total = new KProgress( frame, "SyncProgDlgProgTotal", 0 );

  QLabel *trans_label = new QLabel( i18n( "Files transferred:" ), frame );
  QLabel *files_label = new QLabel( i18n( "0 of 0" ), frame, "FilesLabel" );

  QLabel *rate_label = new QLabel( i18n( "Transfer rate:" ), frame );
  QLabel *trans_rate = new QLabel( "Unknown", frame, "TransferRate" );

  layout->addMultiCellWidget( el, 0, 0, 0, 1, 0 );
  layout->addMultiCellWidget( prog_part, 1, 1, 0, 1, 0 );
  layout->addMultiCellWidget( prog_total, 2, 2, 0, 1, 0 );
  layout->addWidget( rate_label, 3, 0, 0 );
  layout->addWidget( trans_rate, 3, 1, Qt::AlignRight );
  layout->addWidget( trans_label, 4, 0, 0 );
  layout->addWidget( files_label, 4, 1, Qt::AlignRight );

  connect( m_dialog, SIGNAL( cancelClicked() ), this, SLOT( abort() ) );

  frame->setFixedWidth( 350 );

  m_dialog->setFixedSize( m_dialog->sizeHint() );

  // Never change this, because otherwise we will start the
  // process AFTER the dialog has been closed:
  m_dialog->show();

  // Use KProcess::OwnGroup instead of KProcess::NotifyOnExit here, because
  // this garantees that the process is indeed killed when using abort().
  // See KProcess docs for further information.
  m_proc->start( KProcess::OwnGroup, KProcess::AllOutput );
}


/****************************************************************************
   Reads the options that should be used with rsync
****************************************************************************/

const QString Smb4KSynchronizer::readRsyncOptions()
{
  QString options;

  config()->setGroup( "Rsync" );

  if ( config()->readBoolEntry( "archive", true ) )
  {
    options.append( " --archive" );
  }
  else
  {
    options.append( config()->readBoolEntry( "recursive", true ) ? " --recursive" : "" );
    options.append( config()->readBoolEntry( "links", true ) ? " --links" : "" );
    options.append( config()->readBoolEntry( "perms", true ) ? " --perms" : "" );
    options.append( config()->readBoolEntry( "times", true ) ? " --times" : "" );
    options.append( config()->readBoolEntry( "group", true ) ? " --group" : "" );
    options.append( config()->readBoolEntry( "owner", true ) ? " --owner" : "" );
    options.append( config()->readBoolEntry( "devices", true ) ? " --devices" : "" );
  }

  options.append( config()->readBoolEntry( "relative", false ) ? " --relative" : " --no-relative" );
  options.append( config()->readBoolEntry( "omit dir times", false ) ? " --omit-dir-times" : "" );
  options.append( config()->readBoolEntry( "no implied dirs", false ) ? " --no-implied-dirs" : "" );
  options.append( config()->readBoolEntry( "update", true ) ? " --update" : "" );
  options.append( config()->readBoolEntry( "inplace", false ) ? " --inplace" : "" );
  options.append( config()->readBoolEntry( "dirs", false ) ? " --dirs" : "" );
  options.append( config()->readBoolEntry( "copy links", false ) ? " --copy-links" : "" );
  options.append( config()->readBoolEntry( "copy unsafe links", false ) ? " --copy-unsafe-links" : "" );
  options.append( config()->readBoolEntry( "safe links", false ) ? " --safe-links" : "" );
  options.append( config()->readBoolEntry( "hard links", false ) ? " --hard-links" : "" );
  options.append( config()->readBoolEntry( "keep dirlinks", false ) ? " --keep-dirlinks" : "" );
  options.append( config()->readBoolEntry( "delete", false ) ? " --delete" : "" );
  options.append( config()->readBoolEntry( "remove sent files", false ) ? " --remove-sent-files" : "" );
  options.append( config()->readBoolEntry( "delete before", false ) ? " --delete-before" : "" );
  options.append( config()->readBoolEntry( "delete during", false ) ? " --delete-during" : "" );
  options.append( config()->readBoolEntry( "delete after", false ) ? " --delete-after" : "" );
  options.append( config()->readBoolEntry( "delete excluded", false ) ? " --delete-excluded" : "" );
  options.append( config()->readBoolEntry( "ignore errors", false ) ? " --ignore-errors" : "" );
  options.append( config()->readBoolEntry( "force", false ) ? " --force" : "" );
  options.append( config()->readBoolEntry( "whole file", false ) ? " --whole-file" : " --no-whole-file" );
  options.append( config()->readBoolEntry( "sparse", false ) ? " --sparse" : "" );
  options.append( config()->readBoolEntry( "one file system", false ) ? " --one-file-system" : "" );
  options.append( config()->readBoolEntry( "existing", false ) ? " --existing" : "" );
  options.append( config()->readBoolEntry( "ignore existing", false ) ? " --ignore-existing" : "" );
  options.append( config()->readBoolEntry( "delay updates", false ) ? " --delay-updates" : "" );
  options.append( config()->readBoolEntry( "compress", false ) ? " --compress" : "" );

  if ( config()->readBoolEntry( "backup", false ) )
  {
    options.append( " --backup" );
    options.append( config()->readBoolEntry( "Use backup dir", false ) ? " --backup-dir="+config()->readPathEntry( "backup dir", QDir::homeDirPath() ) : "" );
    options.append( config()->readBoolEntry( "Use suffix", false ) ? " --suffix="+config()->readEntry( "suffix", "~" ) : "" );
  }

  options.append( config()->readBoolEntry( "Use max delete", false ) ? " --max-delete="+QString( "%1" ).arg( config()->readNumEntry( "max delete", 0 ) ) : "" );
  options.append( config()->readBoolEntry( "checksum", false ) ? " --checksum" : "" );
  options.append( config()->readBoolEntry( "Use block size", false ) ? " --block-size="+QString( "%1" ).arg( config()->readNumEntry( "block size", 0 ) ) : "" );
  options.append( config()->readBoolEntry( "Use checksum seed", false ) ? " --checksum-seed="+QString( "%1" ).arg( config()->readNumEntry( "checksum seed", 0 ) ) : "" );
  options.append( " "+config()->readEntry( "filter", QString::null ) );
  options.append( config()->readBoolEntry( "Use min size", false ) ? " --min-size="+QString( "%1" ).arg( config()->readNumEntry( "min size", 0 ) )+"K" : "" );
  options.append( config()->readBoolEntry( "Use max size", false ) ? " --max-size="+QString( "%1" ).arg( config()->readNumEntry( "max size", 0 ) )+"K" : "" );

  if ( config()->readBoolEntry( "partial", false ) )
  {
    options.append( " --partial" );
    options.append( config()->readBoolEntry( "Use partial dir", false ) ? " --partial-dir="+config()->readPathEntry( "partial dir", QDir::homeDirPath() ) : "" );
  }

  options.append( config()->readBoolEntry( "cvs exclude", false ) ? " --cvs-exclude" : "" );
  options.append( config()->readBoolEntry( "F-Filter", false ) ? " -F" : "" );
  options.append( config()->readBoolEntry( "FF-Filter", false ) ? " -F -F" : "" );
  options.append( config()->readBoolEntry( "Use exclude", false ) ? " --exclude="+config()->readEntry( "exclude", QString::null ) : "" );
  options.append( config()->readBoolEntry( "Use exclude from", false ) ? " --exclude-from="+config()->readPathEntry( "exclude from", QDir::homeDirPath()+"/exclude.txt" ) : "" );
  options.append( config()->readBoolEntry( "Use include", false ) ? " --include="+config()->readEntry( "include", QString::null ) : "" );
  options.append( config()->readBoolEntry( "Use include from", false ) ? " --include-from="+config()->readPathEntry( "include from", QDir::homeDirPath()+"/include.txt" ) : "" );

  return options;
}


/////////////////////////////////////////////////////////////////////////////
//   SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KSynchronizer::abort()
{
  cancel = true;

  if ( m_proc->isRunning() )
  {
    m_proc->kill();
  }
}


void Smb4KSynchronizer::slotProcessExited( KProcess * )
{
  if ( !cancel )
  {
    ((KProgress *)m_dialog->child( "SyncProgDlgProgTotal", "KProgress", true ))->setValue( 100 );
  }

  m_proc->clearArguments();

  m_dialog->hide();

  delete m_dialog;
  m_dialog = NULL;

  m_working = false;
  emit state( SYNCER_STOP );
}


void Smb4KSynchronizer::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer = QString::fromLocal8Bit( buf, len );
  QString partial, total, files, rate;

  if ( m_buffer[0].isSpace() && m_buffer.contains( "/s ", true ) > 0 )
  {
    partial = m_buffer.section( "%", 0, 0 ).section( " ", -1, -1 ).stripWhiteSpace();

    if ( !partial.isEmpty() )
    {
      KProgress *prog = (KProgress *)m_dialog->child( "SyncProgDlgProgPart", "KProgress", true );

      if ( prog )
      {
        prog->setValue( partial.toInt() );
      }
    }

    if ( m_buffer.contains( "to-check=" ) > 0 )
    {
      // Newer versions of rsync:
      QString tmp = m_buffer.section( "to-check=", 1, 1 ).section( ")", 0, 0 ).stripWhiteSpace();

      if ( !tmp.isEmpty() )
      {
        double tmp_total = tmp.section( "/", 1, 1 ).stripWhiteSpace().toInt();
        double tmp_done = tmp.section( "/", 0, 0 ).stripWhiteSpace().toInt();
        double tmp_percent = ((tmp_total-tmp_done)/tmp_total)*100;

        total = QString( "%1" ).arg( tmp_percent ).section( ".", 0, 0 ).stripWhiteSpace();
      }
    }
    else
    {
      // Older versions of rsync:
      total = m_buffer.section( " (", 1, 1 ).section( ",", 1, 1 ).section( "%", 0, 0 ).section( ".", 0, 0 ).stripWhiteSpace();
    }

    if ( !total.isEmpty() )
    {
      KProgress *prog = (KProgress *)m_dialog->child( "SyncProgDlgProgTotal", "KProgress", true );

      if ( prog )
      {
        prog->setValue( total.toInt() );
      }
    }

    if ( m_buffer.contains( "xfer#" ) > 0 )
    {
      // Newer versions of rsync:
      files = m_buffer.section( "xfer#", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();
    }
    else
    {
      // Older versions of rsync:
      files = m_buffer.section( " (", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();
    }

    if ( !files.isEmpty() )
    {
      QLabel *f = (QLabel *)m_dialog->child( "FilesLabel", "QLabel", true );

      if ( f )
      {
        f->setText( i18n( "%1 of %2" ).arg( files, m_total_files ) );
      }
    }

    rate = m_buffer.section( "/s ", 0, 0 ).section( " ", -1, -1 ).stripWhiteSpace();

    if ( !rate.isEmpty() )
    {
      rate.append( "/s" );
      rate.insert( rate.length() - 4, " " );

      QLabel *r = (QLabel *)m_dialog->child( "TransferRate", "QLabel", true );

      if ( r )
      {
        r->setText( rate );
      }
    }

    m_buffer = QString::null;
  }
  else if ( !m_buffer[0].isSpace() && m_buffer.endsWith( "\n" ) && m_buffer.contains( "/s ", true ) == 0 )
  {
    KLineEdit *el = (KLineEdit *)m_dialog->child( "SyncProgDlgEdit", "KLineEdit", true );

    if ( el )
    {
      el->setSqueezedText( m_buffer );
    }

    if ( m_buffer.contains( "files to consider" ) != 0 )
    {
      m_total_files = m_buffer.section( " files to consider", 0, 0 ).section( " ", -1, -1 ).stripWhiteSpace();

      QLabel *f = (QLabel *)m_dialog->child( "FilesLabel", "QLabel", true );

      if ( f )
      {
        f->setText( QString( "0 of %1" ).arg( m_total_files ) );
      }
    }

    m_buffer = QString::null;
  }
}



void Smb4KSynchronizer::slotReceivedStderr( KProcess *, char *buf, int len )
{
  QString error_message = QString::fromLocal8Bit( buf, len );

  // At least under Debian unstable (20051216), rsync emits an error
  // when you kill the process (using SIGTERM). Pressing the cancel
  // button will exactly do this. Thus, we need to exclude this occasion
  // from here.
  if ( !cancel && error_message.contains( "rsync error:", true ) != 0 )
  {
    // Cancel the whole synchronization in case an error occurred.
    // If we don't do it, the user might be flooded with error messages
    // especially if you try to synchronize a local copy with a remote
    // share that is mounted read-only.
    abort();
    Smb4KError::error( ERROR_SYNCHRONIZING, QString::null, error_message );
  }
  else
  {
    cancel = false;
  }
}


void Smb4KSynchronizer::slotSwapURLs()
{
  if ( m_chooser )
  {
    KURLRequester *src = (KURLRequester *)(m_chooser->child( "SourceURL", "KURLRequester", true ));
    KURLRequester *dest = (KURLRequester *)(m_chooser->child( "DestinationURL", "KURLRequester", true ));

    if ( src && dest )
    {
      QString srcURL = src->url();
      QString destURL = dest->url();

      src->setURL( destURL );
      dest->setURL( srcURL );
    }
  }
}


void Smb4KSynchronizer::slotShutdown()
{
  abort();
}

#include "smb4ksynchronizer.moc"
