/***************************************************************************
    smb4kbookmarkmenuwidget.cpp  - A bookmark popup menu.
                             -------------------
    begin                : Feb 04 2004
    copyright            : (C) 2004 by Franck Babin
                           (C) 2004 by Alexander Reinholdt
    email                : babinfranck@yahoo.ca
                           dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qvaluelist.h>
#include <qstringlist.h>
#include <qdir.h>

// KDE includes
#include <kpopupmenu.h>
#include <klocale.h>
#include <kdebug.h>
#include <kapplication.h>

// application specific includes
#include "smb4kbookmarkmenuwidget.h"
#include "smb4kbookmarkeditor.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"

using namespace Smb4KGlobal;

/**
 * The bookmark menu widget.
 */

Smb4KBookmarkMenuWidget::Smb4KBookmarkMenuWidget( QObject *parent, const char *name )
: KActionMenu( parent, name )
{
  m_collection = new KActionCollection( popupMenu(), "BookmarkMenu_ActionCollection", KGlobal::instance() );
  m_collection->setHighlightingEnabled( true );

  m_action = NULL;

  initActions();

  insertBookmarks();

  connect( smb4k_core->mounter(),         SIGNAL( updated() ),
           this,                          SLOT( slotUpdatedShareList() ) );
  connect( smb4k_core->bookmarkHandler(), SIGNAL( bookmarksUpdated() ),
           this,                          SLOT( slotBookmarksUpdated() ) );
  connect( m_collection,                  SIGNAL( actionHighlighted( KAction * ) ),
           this,                          SLOT( slotActionHighlighted( KAction * ) ) );
}


Smb4KBookmarkMenuWidget::~Smb4KBookmarkMenuWidget()
{
  if ( m_action )
  {
    delete m_action;
    m_action = NULL;
  }

  uint index = 0;

  while ( index < m_collection->count() )
  {
    delete m_collection->action( index++ );
  }

  m_collection->clear();
}


void Smb4KBookmarkMenuWidget::initActions()
{
  KAction *edit_bookmarks = new KAction( i18n( "&Edit Bookmarks" ), "bookmark", QString::null, this, SLOT( slotEditBookmarks() ), m_collection, "bookmarks_edit_bookmarks" );
  edit_bookmarks->setGroup( "BookmarkAction" );

  this->insert( edit_bookmarks );

  popupMenu()->insertSeparator();
}


void Smb4KBookmarkMenuWidget::insertBookmarks()
{
  QValueList<Smb4KBookmark *> bookmarks = smb4k_core->bookmarkHandler()->getBookmarks();
  QStringList shareNames;

  // Due to sorting problems, we have to do this:
  for ( QValueListIterator<Smb4KBookmark *> it = bookmarks.begin(); it != bookmarks.end(); ++it )
  {
    shareNames.append( (*it)->bookmark() );
  }

  shareNames.sort();

  KActionPtrList list = m_collection->actions( "Bookmarks" );

  if ( list.count() == 0 )
  {
    for ( QStringList::ConstIterator it = shareNames.begin(); it != shareNames.end(); ++it )
    {
      KAction *a = new KAction( *it, "folder", KShortcut::null(), 0, 0, m_collection, *it );
      a->setGroup( "Bookmarks" );
      connect( a, SIGNAL( activated() ), this, SLOT( slotActivated() ) );
      insert( a );
    }
  }
  else
  {
    // Remove entries, that do not exist anymore.

    for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); ++it )
    {
      if ( shareNames.find( (*it)->plainText() ) == shareNames.end() )
      {
        (*it)->unplug( popupMenu() );
        m_collection->remove( *it );
      }
    }

    // It turned out, that it is safest to reread the list of actions
    // at this point to avoid crashes below.
    list.clear();
    list = m_collection->actions( "Bookmarks" );

    // Insert items into the menu

    for ( QStringList::ConstIterator it = shareNames.begin(); it != shareNames.end(); ++it )
    {
      bool insert = true;

      for ( KActionPtrList::ConstIterator i = list.begin(); i != list.end(); ++i )
      {
        if ( QString::compare( *it, (*i)->plainText() ) == 0 )
        {
          insert = false;
        }
      }

      if ( insert )
      {
        KAction *a = new KAction( *it, "folder", KShortcut::null(), 0, 0, m_collection, *it );
        a->setGroup( "Bookmarks" );
        connect( a, SIGNAL( activated() ), this, SLOT( slotActivated() ) );
      }
    }

    // Reread list of actions here. The action collection has been updated.
    list.clear();
    list = m_collection->actions( "Bookmarks" );

    uint index = 0;

    while ( index < list.count() )
    {
      remove( list[index++] );
    }

    for ( QStringList::ConstIterator it = shareNames.begin(); it != shareNames.end(); ++it )
    {
      for ( KActionPtrList::ConstIterator i = list.begin(); i != list.end(); ++i )
      {
        if ( QString::compare( (*i)->plainText(), *it ) == 0 )
        {
          insert( *i );
          break;
        }
      }
    }
  }
}


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KBookmarkMenuWidget::slotUpdatedShareList()
{
  config()->setGroup( "Mount Options" );
  QString mount_prefix = config()->readPathEntry( "Default Directory", QString::null );

  KActionPtrList list = m_collection->actions( "Bookmarks" );
  QValueList<Smb4KShare *> shares = smb4k_core->mounter()->getShares();
  bool enable = true;

  if ( !list.isEmpty() )
  {
    for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); ++it )
    {
      enable = true;

      if ( !shares.isEmpty() )
      {
        for ( QValueListIterator<Smb4KShare *> i = shares.begin(); i != shares.end(); ++i )
        {
          if ( QString::compare( (*it)->plainText().upper(), (*i)->name().upper() ) == 0 &&
               (((int)getuid() == (*i)->uid() && (int)getgid() == (*i)->gid()) ||
               qstrncmp( (*i)->canonicalPath(), QDir::home().canonicalPath(), QDir::home().canonicalPath().length() ) == 0 ||
               qstrncmp( (*i)->canonicalPath(), QDir( mount_prefix ).canonicalPath(), QDir( mount_prefix ).canonicalPath().length() ) == 0) )
          {
            enable = false;
            break;
          }
        }
      }

      (*it)->setEnabled( enable );
    }
  }
}


void Smb4KBookmarkMenuWidget::slotActivated()
{
  if ( m_action && m_action->plainText().stripWhiteSpace().startsWith( "//" ) )
  {
    Smb4KBookmark *bookmark = smb4k_core->bookmarkHandler()->findBookmark( m_action->plainText() );

    if ( bookmark )
    {
      smb4k_core->mounter()->mountShare( bookmark->workgroup(), bookmark->host(), bookmark->ip(), bookmark->share() );
    }
  }
}


void Smb4KBookmarkMenuWidget::slotBookmarksUpdated()
{
  insertBookmarks();
}


void Smb4KBookmarkMenuWidget::slotEditBookmarks()
{
  Smb4KBookmarkEditor *editor = new Smb4KBookmarkEditor( kapp->mainWidget() && kapp->mainWidget()->isShown() ?
                                                         kapp->mainWidget() : 0,
                                                         "BookmarkEditor" );
  editor->exec();
}


void Smb4KBookmarkMenuWidget::slotActionHighlighted( KAction *action )
{
  m_action = action;
}

#include "smb4kbookmarkmenuwidget.moc"
