/***************************************************************************
    smb4ksambaoptions.cpp  -  This is the configuration page for the
    Samba settings of Smb4K
                             -------------------
    begin                : Mo Jan 26 2004
    copyright            : (C) 2004-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qmap.h>

// KDE includes
#include <klocale.h>
#include <klineedit.h>
#include <knuminput.h>
#include <kcombobox.h>
#include <kuser.h>
#include <klistview.h>
#include <kpushbutton.h>

// System includes
#include <unistd.h>
#include <sys/types.h>

// application specific includes
#include "smb4ksambaoptions.h"
#include "../core/smb4kglobal.h"
#include "../core/smb4ksettings.h"

using namespace Smb4KGlobal;


Smb4KSambaOptions::Smb4KSambaOptions( QWidget *parent, const char *name )
: QTabWidget( parent, name )
{
  setMargin( 10 );

  //
  // General options
  //
  QWidget *general_tab =        new QWidget( this, "GeneralSambaOptions" );
  QGridLayout *general_layout = new QGridLayout( general_tab );
  general_layout->setSpacing( 10 );

  QGroupBox *general_options =  new QGroupBox( 2, Qt::Horizontal,
                                i18n( "General Options" ), general_tab );
  general_options->setInsideSpacing( 5 );

  (void) new QLabel( i18n( "NetBIOS name:"), general_options );
  (void) new KLineEdit( general_options, "kcfg_NetBIOSName" );

  (void) new QLabel( i18n( "Domain:" ), general_options );
  (void) new KLineEdit( general_options, "kcfg_DomainName" );

  (void) new QLabel( i18n( "Socket options:" ), general_options );
  (void) new KLineEdit( general_options, "kcfg_SocketOptions" );

  (void) new QLabel( i18n( "NetBIOS scope:" ), general_options );
  (void) new KLineEdit( general_options, "kcfg_NetBIOSScope" );

  (void) new QLabel( i18n( "Remote SMB port:" ), general_options );
  (void) new KIntNumInput( general_options, "kcfg_RemotePort" );

  QGroupBox *authentication =   new QGroupBox( 1, Qt::Horizontal,
                                i18n( "Authentication" ), general_tab );

  (void) new QCheckBox( i18n( "Try to authenticate with Kerberos" ), authentication, "kcfg_UseKerberos" );
  (void) new QCheckBox( i18n( "Authenticate as machine account" ), authentication, "kcfg_MachineAccount" );

  QSpacerItem *spacer1 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Expanding );

  general_layout->addWidget( general_options, 0, 0, 0 );
  general_layout->addWidget( authentication, 1, 0, 0 );
  general_layout->addItem( spacer1, 2, 0 );

  addTab( general_tab, i18n( "General Settings" ) );

  //
  // Options for the mount commands
  //
  QWidget *mount_tab =          new QWidget( this, "MountOptions" );
  QGridLayout *mount_layout =   new QGridLayout( mount_tab );
  mount_layout->setSpacing( 10 );

  QGroupBox *filesystem_box =   new QGroupBox( 1, Qt::Horizontal,
                                i18n( "File System" ), mount_tab );

#ifndef __FreeBSD__
  KComboBox *filesystem =       new KComboBox( filesystem_box, "kcfg_Filesystem" );
  filesystem->insertItem( "CIFS", Smb4KSettings::EnumFilesystem::CIFS );
  filesystem->insertItem( "SMBFS", Smb4KSettings::EnumFilesystem::SMBFS );
#else
  (void) new QLabel( "SMBFS", filesystem_box );
#endif

  QLabel *note =                new QLabel( i18n( "NOTE: You might need to enable support for either \"super\" or \"sudo\" in the Super User page." ), filesystem_box );
  note->setTextFormat( Qt::RichText );

  QGroupBox *user_group_box =   new QGroupBox( 2, Qt::Horizontal,
                                i18n( "User and Group" ), mount_tab );
  user_group_box->setInsideSpacing( 5 );

  (void) new QLabel( i18n( "User ID:" ), user_group_box );
  KLineEdit *user_id =          new KLineEdit( user_group_box, "kcfg_UserID" );
  user_id->setAlignment( Qt::AlignRight );

  (void) new QLabel( i18n( "Group ID:" ), user_group_box );
  KLineEdit *group_id =         new KLineEdit( user_group_box, "kcfg_GroupID" );
  group_id->setAlignment( Qt::AlignRight );

  QGroupBox *charset_box =      new QGroupBox( 2, Qt::Horizontal,
                                i18n( "Charset and Codepage" ), mount_tab );
  charset_box->setInsideSpacing( 5 );

  (void) new QLabel( i18n( "Client charset:" ), charset_box );
  KComboBox *charset =          new KComboBox( charset_box, "kcfg_ClientCharset" );
  charset->insertItem( i18n( "default" ), Smb4KSettings::EnumClientCharset::default_charset );
  charset->insertItem( "iso8859-1", Smb4KSettings::EnumClientCharset::iso8859_1 );
  charset->insertItem( "iso8859-2", Smb4KSettings::EnumClientCharset::iso8859_2 );
  charset->insertItem( "iso8859-3", Smb4KSettings::EnumClientCharset::iso8859_3 );
  charset->insertItem( "iso8859-4", Smb4KSettings::EnumClientCharset::iso8859_4 );
  charset->insertItem( "iso8859-5", Smb4KSettings::EnumClientCharset::iso8859_5 );
  charset->insertItem( "iso8859-6", Smb4KSettings::EnumClientCharset::iso8859_6 );
  charset->insertItem( "iso8859-7", Smb4KSettings::EnumClientCharset::iso8859_7 );
  charset->insertItem( "iso8859-8", Smb4KSettings::EnumClientCharset::iso8859_8 );
  charset->insertItem( "iso8859-9", Smb4KSettings::EnumClientCharset::iso8859_9 );
  charset->insertItem( "iso8859-13", Smb4KSettings::EnumClientCharset::iso8859_13 );
  charset->insertItem( "iso8859-14", Smb4KSettings::EnumClientCharset::iso8859_14 );
  charset->insertItem( "iso8859-15", Smb4KSettings::EnumClientCharset::iso8859_15 );
  charset->insertItem( "utf8", Smb4KSettings::EnumClientCharset::utf8 );
  charset->insertItem( "koi8-r", Smb4KSettings::EnumClientCharset::koi8_r );
  charset->insertItem( "koi8-u", Smb4KSettings::EnumClientCharset::koi8_u );
  charset->insertItem( "koi8-ru", Smb4KSettings::EnumClientCharset::koi8_ru );
  charset->insertItem( "cp1251", Smb4KSettings::EnumClientCharset::cp1251 );
  charset->insertItem( "gb2312", Smb4KSettings::EnumClientCharset::gb2312 );
  charset->insertItem( "big5", Smb4KSettings::EnumClientCharset::big5 );
  charset->insertItem( "euc-jp", Smb4KSettings::EnumClientCharset::euc_jp );
  charset->insertItem( "euc-kr", Smb4KSettings::EnumClientCharset::euc_kr );
  charset->insertItem( "tis-620", Smb4KSettings::EnumClientCharset::tis_620 );

  (void) new QLabel( i18n( "Server codepage:" ), charset_box, "CodepageLabel" );
  KComboBox *codepage =         new KComboBox( charset_box, "kcfg_ServerCodepage" );
  codepage->insertItem( i18n( "default" ), Smb4KSettings::EnumServerCodepage::default_codepage );
  codepage->insertItem( "cp437", Smb4KSettings::EnumServerCodepage::cp437 );
  codepage->insertItem( "cp720", Smb4KSettings::EnumServerCodepage::cp720 );
  codepage->insertItem( "cp737", Smb4KSettings::EnumServerCodepage::cp737 );
  codepage->insertItem( "cp775", Smb4KSettings::EnumServerCodepage::cp775 );
  codepage->insertItem( "cp850", Smb4KSettings::EnumServerCodepage::cp850 );
  codepage->insertItem( "cp852", Smb4KSettings::EnumServerCodepage::cp852 );
  codepage->insertItem( "cp855", Smb4KSettings::EnumServerCodepage::cp855 );
  codepage->insertItem( "cp857", Smb4KSettings::EnumServerCodepage::cp857 );
  codepage->insertItem( "cp858", Smb4KSettings::EnumServerCodepage::cp858 );
  codepage->insertItem( "cp860", Smb4KSettings::EnumServerCodepage::cp860 );
  codepage->insertItem( "cp861", Smb4KSettings::EnumServerCodepage::cp861 );
  codepage->insertItem( "cp862", Smb4KSettings::EnumServerCodepage::cp862 );
  codepage->insertItem( "cp863", Smb4KSettings::EnumServerCodepage::cp863 );
  codepage->insertItem( "cp864", Smb4KSettings::EnumServerCodepage::cp864 );
  codepage->insertItem( "cp865", Smb4KSettings::EnumServerCodepage::cp865 );
  codepage->insertItem( "cp866", Smb4KSettings::EnumServerCodepage::cp866 );
  codepage->insertItem( "cp869", Smb4KSettings::EnumServerCodepage::cp869 );
  codepage->insertItem( "cp874", Smb4KSettings::EnumServerCodepage::cp874 );
  codepage->insertItem( "cp932", Smb4KSettings::EnumServerCodepage::cp932 );
  codepage->insertItem( "cp936", Smb4KSettings::EnumServerCodepage::cp936 );
  codepage->insertItem( "cp949", Smb4KSettings::EnumServerCodepage::cp949 );
  codepage->insertItem( "cp950", Smb4KSettings::EnumServerCodepage::cp950 );
  codepage->insertItem( "cp1250", Smb4KSettings::EnumServerCodepage::cp1250 );
  codepage->insertItem( "cp1251", Smb4KSettings::EnumServerCodepage::cp1251 );
  codepage->insertItem( "cp1252", Smb4KSettings::EnumServerCodepage::cp1252 );
  codepage->insertItem( "cp1253", Smb4KSettings::EnumServerCodepage::cp1253 );
  codepage->insertItem( "cp1254", Smb4KSettings::EnumServerCodepage::cp1254 );
  codepage->insertItem( "cp1255", Smb4KSettings::EnumServerCodepage::cp1255 );
  codepage->insertItem( "cp1256", Smb4KSettings::EnumServerCodepage::cp1256 );
  codepage->insertItem( "cp1257", Smb4KSettings::EnumServerCodepage::cp1257 );
  codepage->insertItem( "cp1258", Smb4KSettings::EnumServerCodepage::cp1258 );
  codepage->insertItem( "unicode", Smb4KSettings::EnumServerCodepage::unicode );

  QGroupBox *perms_box =        new QGroupBox( 2, Qt::Horizontal,
                                i18n( "Permissions" ), mount_tab );
  perms_box->setInsideSpacing( 5 );

  (void) new QLabel( i18n( "File mask:" ), perms_box );
  KLineEdit *fmask =            new KLineEdit( perms_box, "kcfg_FileMask" );
  fmask->setAlignment( Qt::AlignRight );

  (void) new QLabel( i18n( "Directory mask:" ), perms_box );
  KLineEdit *dmask =            new KLineEdit( perms_box, "kcfg_DirectoryMask" );
  dmask->setAlignment( Qt::AlignRight );

  (void) new QLabel( i18n( "Write access:" ), perms_box );
  KComboBox *read_mode =        new KComboBox( perms_box, "kcfg_WriteAccess" );
  read_mode->insertItem( i18n( "read-write" ), Smb4KSettings::EnumWriteAccess::ReadWrite );
  read_mode->insertItem( i18n( "read-only" ), Smb4KSettings::EnumWriteAccess::ReadOnly );

#ifndef __FreeBSD__
  QGroupBox *advanced_cifs =    new QGroupBox( 1, Qt::Horizontal,
                                i18n( "Advanced CIFS Options" ), mount_tab, "AdvancedCIFSOptions" );
  advanced_cifs->setInsideSpacing( 5 );

  QWidget *c_advanced_widget =  new QWidget( advanced_cifs );
  QGridLayout *c_advanced_layout = new QGridLayout( c_advanced_widget );
  c_advanced_layout->setSpacing( 5 );
  c_advanced_layout->setMargin( 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Do permission checks" ),
                                c_advanced_widget, "kcfg_PermissionChecks" ), 0, 0, 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Attempt to set UID and GID" ),
                                c_advanced_widget, "kcfg_ClientControlsIDs" ), 0, 1, 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Use server inode numbers" ),
                                c_advanced_widget, "kcfg_ServerInodeNumbers" ), 1, 0, 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Do not cache inode data" ),
                                c_advanced_widget, "kcfg_InodeDataCaching" ), 1, 1, 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Translate reserved characters" ),
                                c_advanced_widget, "kcfg_TranslateReservedChars" ), 2, 0, 0 );
  c_advanced_layout->addWidget( new QCheckBox( i18n( "Do not use locking" ),
                                c_advanced_widget, "kcfg_NoLocking" ), 2, 1, 0 );

  QWidget *c_extra_widget =     new QWidget( advanced_cifs );
  QGridLayout *c_extra_layout = new QGridLayout( c_extra_widget );
  c_extra_layout->setSpacing( 5 );
  c_extra_layout->setMargin( 0 );
  c_extra_layout->addWidget( new QLabel( i18n( "Additional options:" ), c_extra_widget ), 0, 0, 0 );
  c_extra_layout->addWidget( new KLineEdit( c_extra_widget, "kcfg_CustomCIFSOptions" ), 0, 1, 0 );

  QGroupBox *advanced_smbfs =   new QGroupBox( 1, Qt::Horizontal,
                                i18n( "Advanced SMBFS Options" ), mount_tab, "AdvancedSMBFSOptions" );
  advanced_smbfs->setInsideSpacing( 5 );

  (void) new QCheckBox( i18n( "Use Unicode when communicating with the server" ), advanced_smbfs, "kcfg_UnicodeSupport" );
  (void) new QCheckBox( i18n( "Use large file system support" ), advanced_smbfs, "kcfg_LargeFileSystemSupport" );

  QWidget *s_advanced_widget =  new QWidget( advanced_smbfs );
  QGridLayout *s_advanced_layout = new QGridLayout( s_advanced_widget );
  s_advanced_layout->setSpacing( 5 );
  s_advanced_layout->setMargin( 0 );
  s_advanced_layout->addWidget( new QLabel( i18n( "Caching time of directory listings:" ), s_advanced_widget ), 0, 0, 0 );
  KIntNumInput *ttl =           new KIntNumInput( s_advanced_widget, "kcfg_CachingTime" );
  ttl->setSuffix( " ms" );
  s_advanced_layout->addWidget( ttl, 0, 1, 0 );

#endif

  QSpacerItem *spacer2 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Expanding );

  mount_layout->addMultiCellWidget( filesystem_box, 0, 0, 0, 1, 0 );
  mount_layout->addMultiCellWidget( perms_box, 0, 0, 2, 3, 0 );
  mount_layout->addMultiCellWidget( charset_box, 1, 1, 0, 1, 0 );
  mount_layout->addMultiCellWidget( user_group_box, 1, 1, 2, 3, 0 );
#ifndef __FreeBSD__
  mount_layout->addMultiCellWidget( advanced_cifs, 2, 2, 0, 3, 0 );
  mount_layout->addMultiCellWidget( advanced_smbfs, 3, 3, 0, 3, 0 );
  mount_layout->addMultiCell( spacer2, 4, 4, 0, 3, 0 );
#else
  mount_layout->addMultiCell( spacer2, 2, 2, 0, 3, 0 );
#endif

  addTab( mount_tab, "mount" );

  //
  // Options for the 'net' command
  //
  QWidget *net_tab =            new QWidget( this, "NetOptions" );
  QGridLayout *net_layout =     new QGridLayout( net_tab );
  net_layout->setSpacing( 10 );

  QButtonGroup *protocol_hint = new QButtonGroup( 1, Qt::Horizontal,
                                i18n( "Protocol Hint" ), net_tab, "kcfg_ProtocolHint" );

  protocol_hint->insert( new QRadioButton( i18n( "Automatic detection" ), protocol_hint ),
                         Smb4KSettings::EnumProtocolHint::Automatic );
  protocol_hint->insert( new QRadioButton( i18n( "RPC: Modern operating systems" ), protocol_hint ),
                         Smb4KSettings::EnumProtocolHint::RPC );
  protocol_hint->insert( new QRadioButton( i18n( "RAP: Older operating systems" ), protocol_hint ),
                         Smb4KSettings::EnumProtocolHint::RAP );
  protocol_hint->insert( new QRadioButton( i18n( "ADS: Active Directory environment (LDAP/Kerberos)" ), protocol_hint ),
                         Smb4KSettings::EnumProtocolHint::ADS );

  QSpacerItem *spacer3 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Expanding );

  net_layout->addWidget( protocol_hint, 0, 0, 0 );
  net_layout->addItem( spacer3, 1, 0 );

  addTab( net_tab, "net" );

  //
  // Options for the 'smbclient' command
  //
  QWidget *smbclient_tab =      new QWidget( this, "SmbclientOptions" );
  QGridLayout *smbclient_layout = new QGridLayout( smbclient_tab );
  smbclient_layout->setSpacing( 10 );

  QGroupBox *smbclient_misc =   new QGroupBox( 2, Qt::Horizontal, i18n( "Miscellaneous" ),
                                smbclient_tab, "SmbclientMiscellaneous" );
  smbclient_misc->setInsideSpacing( 5 );

  (void) new QLabel( i18n( "Name resolve order:" ), smbclient_misc );
  (void) new KLineEdit( smbclient_misc, "kcfg_NameResolveOrder" );

  (void) new QLabel( i18n( "Buffer size:" ), smbclient_misc );
  KIntNumInput *buffer_size =   new KIntNumInput( smbclient_misc, "kcfg_BufferSize" );
  buffer_size->setSuffix( i18n( " Bytes" ) );

  (void) new QLabel( i18n( "Signing state:" ), smbclient_misc );
  KComboBox *signing_state =    new KComboBox( smbclient_misc, "kcfg_SigningState" );
  signing_state->insertItem( i18n( "none" ), Smb4KSettings::EnumSigningState::None );
  signing_state->insertItem( i18n( "on" ), Smb4KSettings::EnumSigningState::On );
  signing_state->insertItem( i18n( "off" ), Smb4KSettings::EnumSigningState::Off );
  signing_state->insertItem( i18n( "required" ), Smb4KSettings::EnumSigningState::Required );

  QSpacerItem *spacer4 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Expanding );

  smbclient_layout->addWidget( smbclient_misc, 0, 0, 0 );
  smbclient_layout->addItem( spacer4, 1, 0 );

  addTab( smbclient_tab, "smbclient" );

  //
  // Options for the 'nmblookup' command
  //
  QWidget *nmblookup_tab =      new QWidget( this, "NmblookupOptions" );
  QGridLayout *nmblookup_layout = new QGridLayout( nmblookup_tab );
  nmblookup_layout->setSpacing( 10 );

  QGroupBox *nmblookup_misc =   new QGroupBox( 1, Qt::Horizontal,
                                i18n( "Miscellaneous" ), nmblookup_tab );
  nmblookup_misc->setInsideSpacing( 5 );

  QWidget *broadcast_widget =   new QWidget( nmblookup_misc );
  QGridLayout *broadcast_layout = new QGridLayout( broadcast_widget );
  broadcast_layout->setSpacing( 5 );
  broadcast_layout->setMargin( 0 );
  broadcast_layout->addWidget( new QLabel( i18n( "Broadcast address:" ), broadcast_widget ), 0, 0, 0 );
  broadcast_layout->addWidget( new KLineEdit( broadcast_widget, "kcfg_BroadcastAddress" ), 0, 1, 0 );

  (void) new QCheckBox( i18n( "Try and bind to UDP port 137 to send and receive UDP datagrams" ), nmblookup_misc, "kcfg_UsePort137" );

  QSpacerItem *spacer5 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Expanding );

  nmblookup_layout->addWidget( nmblookup_misc, 0, 0, 0 );
  nmblookup_layout->addItem( spacer5, 1, 0 );

  addTab( nmblookup_tab, "nmblookup" );


  //
  // Custom options
  //
  QWidget *custom_tab =         new QWidget( this, "CustomOptions" );
  QGridLayout *custom_layout =  new QGridLayout( custom_tab );
  custom_layout->setSpacing( 10 );


  // FIXME: Set "What's this' texts!!!

  KListView *custom_options =   new KListView( custom_tab, "CustomOptionsList" );
//   custom_options->setAllColumnsShowFocus( true );
  custom_options->setSelectionMode( QListView::Single );
  custom_options->setVScrollBarMode( KListView::Auto );
  custom_options->addColumn( i18n( "Item" ), -1 );
  custom_options->addColumn( i18n( "Protocol" ), -1 );
#ifndef __FreeBSD__
  custom_options->addColumn( i18n( "File System" ), -1 );
  custom_options->addColumn( i18n( "Write Access" ), -1 );
#endif
  custom_options->addColumn( i18n( "Kerberos" ), -1 );
  custom_options->addColumn( i18n( "UID" ), -1 );
  custom_options->addColumn( i18n( "GID" ), -1 );
  custom_options->addColumn( i18n( "Port" ), -1 );

  QWidget *custom_input =       new QWidget( custom_tab, "CustomInputWidget" );
  QGridLayout *custom_input_layout = new QGridLayout( custom_input );
  custom_input_layout->setSpacing( 5 );

  QLabel *custom_proto_label =  new QLabel( i18n( "Protocol:" ), custom_input, "CustomProtocolLabel" );

  KComboBox *custom_protocol =  new KComboBox( custom_input, "CustomProtocol" );
  custom_protocol->insertItem( "-", -1 );
  custom_protocol->insertItem( i18n( "auto" ), -1 );
  custom_protocol->insertItem( "RPC", -1 );
  custom_protocol->insertItem( "RAP", -1 );
  custom_protocol->insertItem( "ADS", -1 );

#ifndef __FreeBSD__

  QLabel *custom_fs_label =     new QLabel( i18n( "File system:" ), custom_input, "CustomFileSystemLabel" );

  KComboBox *custom_fs =        new KComboBox( custom_input, "CustomFileSystem" );
  custom_fs->insertItem( "-", -1 );
  custom_fs->insertItem( "CIFS", -1 );
  custom_fs->insertItem( "SMBFS", -1 );

  QLabel *custom_rw_label =     new QLabel( i18n( "Write access:" ), custom_input, "CustomWriteAccessLabel" );

  KComboBox *custom_rw =        new KComboBox( custom_input, "CustomWriteAccess" );
  custom_rw->insertItem( "-", -1 );
  custom_rw->insertItem( i18n( "read-write" ), -1 );
  custom_rw->insertItem( i18n( "read-only" ), -1 );

#endif

  QLabel *custom_krb_label =    new QLabel( i18n( "Kerberos:" ), custom_input, "CustomKerberosLabel" );

  KComboBox *custom_krb =       new KComboBox( custom_input, "CustomKerberos" );
  custom_krb->insertItem( "-", -1 );
  custom_krb->insertItem( i18n( "no" ), -1 );
  custom_krb->insertItem( i18n( "yes" ), -1 );

  QLabel *custom_uid_label =    new QLabel( i18n( "User ID:" ), custom_input, "CustomUIDLabel" );

  KLineEdit *custom_uid =       new KLineEdit( custom_input, "CustomUID" );
  custom_uid->setText( "-" );

  QLabel *custom_gid_label =    new QLabel( i18n( "Group ID:" ), custom_input, "CustomGIDLabel" );

  KLineEdit *custom_gid =       new KLineEdit( custom_input, "CustomGID" );
  custom_gid->setText( "-" );

  QLabel *custom_port_label =   new QLabel( i18n( "Port:" ), custom_input, "CustomPortLabel" );

  KIntNumInput *custom_port =   new KIntNumInput( Smb4KSettings::remotePort(), custom_input, 10, "CustomPort" );
  custom_port->setMinValue( 0 );
  custom_port->setMaxValue( 65535 );

  QWidget *custom_rm =          new QWidget( custom_input );
  QGridLayout *custom_rm_layout = new QGridLayout( custom_rm );
  custom_rm_layout->setSpacing( 5 );

  KPushButton *remove_custom =  new KPushButton( KGuiItem( i18n( "Remove" ), "remove" ), custom_rm, "CustomRemoveItem" );
  KPushButton *rm_all_custom =  new KPushButton( KGuiItem( i18n( "Remove All" ), "editdelete" ), custom_rm, "CustomRemoveAllItems" );

  QSpacerItem *spacer6 =        new QSpacerItem( 0, 0, QSizePolicy::Expanding, QSizePolicy::Preferred );

  custom_rm_layout->addItem( spacer6, 0, 0 );
  custom_rm_layout->addWidget( remove_custom, 0, 1, 0 );
  custom_rm_layout->addWidget( rm_all_custom, 0, 2, 0 );

  custom_input_layout->addWidget( custom_proto_label, 0, 0, 0 );
  custom_input_layout->addWidget( custom_protocol, 0, 1, 0 );
#ifndef __FreeBSD__
  custom_input_layout->addWidget( custom_fs_label, 0, 2, 0 );
  custom_input_layout->addWidget( custom_fs, 0, 3, 0 );
  custom_input_layout->addWidget( custom_rw_label, 1, 0, 0 );
  custom_input_layout->addWidget( custom_rw, 1, 1, 0 );
  custom_input_layout->addWidget( custom_krb_label, 1, 2, 0 );
  custom_input_layout->addWidget( custom_krb, 1, 3, 0 );
  custom_input_layout->addWidget( custom_uid_label, 2, 0, 0 );
  custom_input_layout->addWidget( custom_uid, 2, 1, 0 );
  custom_input_layout->addWidget( custom_gid_label, 2, 2, 0 );
  custom_input_layout->addWidget( custom_gid, 2, 3, 0 );
  custom_input_layout->addWidget( custom_port_label, 3, 0, 0 );
  custom_input_layout->addWidget( custom_port, 3, 1, 0 );
  custom_input_layout->addMultiCellWidget( custom_rm, 3, 3, 2, 3, 0 );
#else
  custom_input_layout->addWidget( custom_krb_label, 0, 2, 0 );
  custom_input_layout->addWidget( custom_krb, 0, 3, 0 );
  custom_input_layout->addWidget( custom_uid_label, 1, 0, 0 );
  custom_input_layout->addWidget( custom_uid, 1, 1, 0 );
  custom_input_layout->addWidget( custom_gid_label, 1, 2, 0 );
  custom_input_layout->addWidget( custom_gid, 1, 3, 0 );
  custom_input_layout->addWidget( custom_port_label, 2, 0, 0 );
  custom_input_layout->addWidget( custom_port, 2, 1, 0 );
  custom_input_layout->addMultiCellWidget( custom_rm, 2, 2, 2, 3, 0 );
#endif

  QSpacerItem *spacer7 =        new QSpacerItem( 0, 0, QSizePolicy::Preferred, QSizePolicy::Minimum );

  custom_layout->addWidget( custom_options, 0, 0, 0 );
  custom_layout->addWidget( custom_input, 1, 0, 0 );
  custom_layout->addItem( spacer7, 2, 0 );

  addTab( custom_tab, i18n( "Custom" ) );

  //
  // Connections
  //
#ifndef __FreeBSD__
  connect( filesystem,      SIGNAL( activated( int ) ),
           this,            SLOT( slotSambaFileSystem( int ) ) );
#endif

  connect( remove_custom,   SIGNAL( clicked() ),
           this,            SLOT( slotRemoveCustomOption() ) );

  connect( rm_all_custom,   SIGNAL( clicked() ),
           this,            SLOT( slotRemoveAllCustomOptions() ) );

  connect( custom_options,  SIGNAL( clicked( QListViewItem * ) ),
           this,            SLOT( slotCustomItemClicked( QListViewItem * ) ) );

  connect( custom_protocol, SIGNAL( activated( int ) ),
           this,            SLOT( slotCustomProtocolChanged( int ) ) );

#ifndef __FreeBSD__
  connect( custom_fs,       SIGNAL( activated( int ) ),
           this,            SLOT( slotCustomFileSystemChanged( int ) ) );

  connect( custom_rw,       SIGNAL( activated( int ) ),
           this,            SLOT( slotCustomWriteAccessChanged( int ) ) );
#endif

  connect( custom_krb,      SIGNAL( activated( int ) ),
           this,            SLOT( slotCustomKerberosChanged( int ) ) );

  connect( custom_uid,      SIGNAL( textChanged( const QString & ) ),
           this,            SLOT( slotCustomUIDChanged( const QString & ) ) );

  connect( custom_gid,      SIGNAL( textChanged( const QString & ) ),
           this,            SLOT( slotCustomGIDChanged( const QString & ) ) );

  connect( custom_port,     SIGNAL( valueChanged( int ) ),
           this,            SLOT( slotCustomPortChanged( int ) ) );

  //
  // Do last things before we are ready to go
  //
#ifndef __FreeBSD__  
  slotSambaFileSystem( filesystem->currentItem() );
#endif
  slotCustomItemClicked( NULL );
}


Smb4KSambaOptions::~Smb4KSambaOptions()
{
}


void Smb4KSambaOptions::resetCustomTab()
{
  slotCustomItemClicked( NULL );
}


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

#ifndef __FreeBSD__
void Smb4KSambaOptions::slotSambaFileSystem( int item_index )
{
  switch ( item_index )
  {
    case Smb4KSettings::EnumFilesystem::CIFS:
    {
      QGroupBox *adv_cifs = static_cast<QGroupBox *>( child( "AdvancedCIFSOptions", "QGroupBox", true ) );

      if ( adv_cifs )
      {
        adv_cifs->setEnabled( true );
      }

      QGroupBox *adv_smbfs = static_cast<QGroupBox *>( child( "AdvancedSMBFSOptions", "QGroupBox", true ) );

      if ( adv_smbfs )
      {
        adv_smbfs->setEnabled( false );
      }

      QLabel *codepage = static_cast<QLabel *>( child( "CodepageLabel", "QLabel", true ) );

      if ( codepage )
      {
        codepage->setEnabled( false );
      }

      KComboBox *srv_codepage = static_cast<KComboBox *>( child( "kcfg_ServerCodepage", "KComboBox", true ) );

      if ( srv_codepage )
      {
        srv_codepage->setEnabled( false );
      }

      break;
    }
    case Smb4KSettings::EnumFilesystem::SMBFS:
    {
      QGroupBox *adv_cifs = static_cast<QGroupBox *>( child( "AdvancedCIFSOptions", "QGroupBox", true ) );

      if ( adv_cifs )
      {
        adv_cifs->setEnabled( false );
      }

      QGroupBox *adv_smbfs = static_cast<QGroupBox *>( child( "AdvancedSMBFSOptions", "QGroupBox", true ) );

      if ( adv_smbfs )
      {
        adv_smbfs->setEnabled( true );
      }

      QLabel *codepage = static_cast<QLabel *>( child( "CodepageLabel", "QLabel", true ) );

      if ( codepage )
      {
        codepage->setEnabled( true );
      }

      KComboBox *srv_codepage = static_cast<KComboBox *>( child( "kcfg_ServerCodepage", "KComboBox", true ) );

      if ( srv_codepage )
      {
        srv_codepage->setEnabled( true );
      }

      break;
    }
    default:
    {
      break;
    }
  };
#else
void Smb4KSambaOptions::slotSambaFileSystem( int )
{
#endif
}


void Smb4KSambaOptions::slotCustomItemClicked( QListViewItem *item )
{
  if ( item )
  {
    // Enable the input widget:
    QWidget *input = static_cast<QGroupBox *>( child( "CustomInputWidget", "QGroupBox", true ) );

    if ( input )
    {
      input->setEnabled( true );
    }

    // Enable/disable the labels, combo boxes, buttons, etc:
    QLabel *protocol_label = static_cast<QLabel *>( child( "CustomProtocolLabel", "QLabel", true ) );

    if ( protocol_label )
    {
      if ( QString::compare( item->text( Protocol ).stripWhiteSpace(), "-" ) == 0 )
      {
        protocol_label->setEnabled( false );
      }
      else
      {
        protocol_label->setEnabled( true );
      }
    }

    KComboBox *protocol = static_cast<KComboBox *>( child( "CustomProtocol", "KComboBox", true ) );

    if ( protocol )
    {
      if ( QString::compare( item->text( Protocol ).stripWhiteSpace(), "-" ) == 0 )
      {
        protocol->setEnabled( false );
      }
      else
      {
        protocol->setEnabled( true );
      }

      protocol->setCurrentText( item->text( Protocol ) );
    }

#ifndef __FreeBSD__

    QLabel *fs_label = static_cast<QLabel *>( child( "CustomFileSystemLabel", "QLabel", true ) );

    if ( fs_label )
    {
      if ( QString::compare( item->text( FileSystem ).stripWhiteSpace(), "-" ) == 0 )
      {
        fs_label->setEnabled( false );
      }
      else
      {
        fs_label->setEnabled( true );
      }
    }

    KComboBox *filesystem = static_cast<KComboBox *>( child( "CustomFileSystem", "KComboBox", true ) );

    if ( filesystem )
    {
      if ( QString::compare( item->text( FileSystem ).stripWhiteSpace(), "-" ) == 0 )
      {
        filesystem->setEnabled( false );
      }
      else
      {
        filesystem->setEnabled( true );
      }

      filesystem->setCurrentText( item->text( FileSystem ) );
    }

    QLabel *write_access_label = static_cast<QLabel *>( child( "CustomWriteAccessLabel", "QLabel", true ) );

    if ( write_access_label )
    {
      if ( QString::compare( item->text( WriteAccess ).stripWhiteSpace(), "-" ) == 0 )
      {
        write_access_label->setEnabled( false );
      }
      else
      {
        write_access_label->setEnabled( true );
      }
    }

    KComboBox *write_access = static_cast<KComboBox *>( child( "CustomWriteAccess", "KComboBox", true ) );

    if ( write_access )
    {
      if ( QString::compare( item->text( WriteAccess ).stripWhiteSpace(), "-" ) == 0 )
      {
        write_access->setEnabled( false );
      }
      else
      {
        write_access->setEnabled( true );
      }

      write_access->setCurrentText( item->text( WriteAccess ) );
    }

#endif

    QLabel *krb_label = static_cast<QLabel *>( child( "CustomKerberosLabel", "QLabel", true ) );

    if ( krb_label )
    {
      if ( QString::compare( item->text( Kerberos ).stripWhiteSpace(), "-" ) == 0 )
      {
        krb_label->setEnabled( false );
      }
      else
      {
        krb_label->setEnabled( true );
      }
    }

    KComboBox *kerberos = static_cast<KComboBox *>( child( "CustomKerberos", "KComboBox", true ) );

    if ( kerberos )
    {
      if ( QString::compare( item->text( Kerberos ).stripWhiteSpace(), "-" ) == 0 )
      {
        kerberos->setEnabled( false );
      }
      else
      {
        kerberos->setEnabled( true );
      }

      kerberos->setCurrentText( item->text( Kerberos ) );
    }

    QLabel *uid_label = static_cast<QLabel *>( child( "CustomUIDLabel", "QLabel", true ) );

    if ( uid_label )
    {
      if ( QString::compare( item->text( UID ).stripWhiteSpace(), "-" ) == 0 )
      {
        uid_label->setEnabled( false );
      }
      else
      {
        uid_label->setEnabled( true );
      }
    }

    KLineEdit *uid = static_cast<KLineEdit *>( child( "CustomUID", "KLineEdit", true ) );

    if ( uid )
    {
      if ( QString::compare( item->text( UID ).stripWhiteSpace(), "-" ) == 0 )
      {
        uid->setEnabled( false );
      }
      else
      {
        uid->setEnabled( true );
      }

      uid->setText( item->text( UID ) );
    }

    QLabel *gid_label = static_cast<QLabel *>( child( "CustomGIDLabel", "QLabel", true ) );

    if ( gid_label )
    {
      if ( QString::compare( item->text( GID ).stripWhiteSpace(), "-" ) == 0 )
      {
        gid_label->setEnabled( false );
      }
      else
      {
        gid_label->setEnabled( true );
      }
    }

    KLineEdit *gid = static_cast<KLineEdit *>( child( "CustomGID", "KLineEdit", true ) );

    if ( gid )
    {
      if ( QString::compare( item->text( GID ).stripWhiteSpace(), "-" ) == 0 )
      {
        gid->setEnabled( false );
      }
      else
      {
        gid->setEnabled( true );
      }

      gid->setText( item->text( GID ) );
    }

    QLabel *port_label = static_cast<QLabel *>( child( "CustomPortLabel", "QLabel", true ) );

    if ( port_label )
    {
      if ( QString::compare( item->text( Port ).stripWhiteSpace(), "-" ) == 0 )
      {
        port_label->setEnabled( false );
      }
      else
      {
        port_label->setEnabled( true );
      }
    }

    KIntNumInput *port = static_cast<KIntNumInput *>( child( "CustomPort", "KIntNumInput", true ) );

    if ( port )
    {
      if ( QString::compare( item->text( Port ).stripWhiteSpace(), "-" ) == 0 )
      {
        port->setEnabled( false );
      }
      else
      {
        port->setEnabled( true );
      }

      port->setValue( item->text( Port ).toInt() );
    }

    KPushButton *remove = static_cast<KPushButton *>( child( "CustomRemoveItem", "KPushButton", true ) );

    if ( remove )
    {
      remove->setEnabled( true );
    }

    KPushButton *remove_all = static_cast<KPushButton *>( child( "CustomRemoveAllItems", "KPushButton", true ) );

    if ( remove_all )
    {
      remove_all->setEnabled( true );
    }
  }
  else
  {
    // Get the list view:
    KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

    if ( view )
    {
      QLabel *protocol_label = static_cast<QLabel *>( child( "CustomProtocolLabel", "QLabel", true ) );

      if ( protocol_label )
      {
        protocol_label->setEnabled( false );
      }

      KComboBox *protocol = static_cast<KComboBox *>( child( "CustomProtocol", "KComboBox", true ) );

      if ( protocol )
      {
        protocol->setCurrentText( "-" );
        protocol->setEnabled( false );
      }

#ifndef __FreeBSD__

      QLabel *fs_label = static_cast<QLabel *>( child( "CustomFileSystemLabel", "QLabel", true ) );

      if ( fs_label )
      {
        fs_label->setEnabled( false );
      }

      KComboBox *filesystem = static_cast<KComboBox *>( child( "CustomFileSystem", "KComboBox", true ) );

      if ( filesystem )
      {
        filesystem->setCurrentText( "-" );
        filesystem->setEnabled( false );
      }

      QLabel *write_access_label = static_cast<QLabel *>( child( "CustomWriteAccessLabel", "QLabel", true ) );

      if ( write_access_label )
      {
        write_access_label->setEnabled( false );
      }

      KComboBox *write_access = static_cast<KComboBox *>( child( "CustomWriteAccess", "KComboBox", true ) );

      if ( write_access )
      {
        write_access->setCurrentText( "-" );
        write_access->setEnabled( false );
      }

#endif

      QLabel *krb_label = static_cast<QLabel *>( child( "CustomKerberosLabel", "QLabel", true ) );

      if ( krb_label )
      {
        krb_label->setEnabled( false );
      }

      KComboBox *kerberos = static_cast<KComboBox *>( child( "CustomKerberos", "KComboBox", true ) );

      if ( kerberos )
      {
        kerberos->setCurrentText( "-" );
        kerberos->setEnabled( false );
      }

      QLabel *uid_label = static_cast<QLabel *>( child( "CustomUIDLabel", "QLabel", true ) );

      if ( uid_label )
      {
        uid_label->setEnabled( false );
      }

      KLineEdit *uid = static_cast<KLineEdit *>( child( "CustomUID", "KLineEdit", true ) );

      if ( uid )
      {
        uid->setText( "-" );
        uid->setEnabled( false );
      }

      QLabel *gid_label = static_cast<QLabel *>( child( "CustomGIDLabel", "QLabel", true ) );

      if ( gid_label )
      {
        gid_label->setEnabled( false );
      }

      KLineEdit *gid = static_cast<KLineEdit *>( child( "CustomGID", "KLineEdit", true ) );

      if ( gid )
      {
        gid->setText( "-" );
        gid->setEnabled( false );
      }

      QLabel *port_label = static_cast<QLabel *>( child( "CustomPortLabel", "QLabel", true ) );

      if ( port_label )
      {
        port_label->setEnabled( false );
      }

      KIntNumInput *port = static_cast<KIntNumInput *>( child( "CustomPort", "KIntNumInput", true ) );

      if ( port )
      {
        port->setValue( Smb4KSettings::remotePort() );
        port->setEnabled( false );
      }

      KPushButton *remove = static_cast<KPushButton *>( child( "CustomRemoveItem", "KPushButton", true ) );

      if ( remove )
      {
        remove->setEnabled( false );
      }

      KPushButton *remove_all = static_cast<KPushButton *>( child( "CustomRemoveAllItems", "KPushButton", true ) );

      if ( remove_all )
      {
        remove_all->setEnabled( (view->childCount() != 0) );
      }
    }
  }
}


void Smb4KSambaOptions::slotCustomProtocolChanged( int index )
{
  KListView *view =            static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );
  KComboBox *custom_protocol = static_cast<KComboBox *>( child( "CustomProtocol", "KComboBox", true ) );

  if ( view && view->selectedItem() && custom_protocol )
  {
    if ( QString::compare( "-", custom_protocol->text( index ) ) != 0 )
    {
      // Set the new value:
      view->selectedItem()->setText( Protocol, custom_protocol->text( index ) );

      emit customSettingsChanged();
    }
    else
    {
      // Reset the combo box:
      custom_protocol->setCurrentText( view->selectedItem()->text( Protocol ) );
    }
  }
  else
  {
    // Nothing to do.
  }
}


#ifndef __FreeBSD__
void Smb4KSambaOptions::slotCustomFileSystemChanged( int index )
{
  KListView *view =              static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );
  KComboBox *custom_filesystem = static_cast<KComboBox *>( child( "CustomFileSystem", "KComboBox", true ) );

  if ( view && view->selectedItem() && custom_filesystem )
  {
    if ( QString::compare( "-", custom_filesystem->text( index ) ) != 0 )
    {
      // Set the new value:
      view->selectedItem()->setText( FileSystem, custom_filesystem->text( index ) );

      emit customSettingsChanged();
    }
    else
    {
      // Reset the combo box:
      custom_filesystem->setCurrentText( view->selectedItem()->text( FileSystem ) );
    }
  }
  else
  {
    // Nothing to do.
  }
#else
void Smb4KSambaOptions::slotCustomFileSystemChanged( int )
{
#endif
}


#ifndef __FreeBSD__
void Smb4KSambaOptions::slotCustomWriteAccessChanged( int index )
{
  KListView *view =      static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );
  KComboBox *custom_rw = static_cast<KComboBox *>( child( "CustomWriteAccess", "KComboBox", true ) );

  if ( view && view->selectedItem() && custom_rw )
  {
    if ( QString::compare( "-", custom_rw->text( index ) ) != 0 )
    {
      // Set the new value:
      view->selectedItem()->setText( WriteAccess, custom_rw->text( index ) );

      emit customSettingsChanged();
    }
    else
    {
      // Reset the combo box:
      custom_rw->setCurrentText( view->selectedItem()->text( WriteAccess ) );
    }
  }
  else
  {
    // Nothing to do.
  }
#else
void Smb4KSambaOptions::slotCustomWriteAccessChanged( int )
{
#endif
}


void Smb4KSambaOptions::slotCustomKerberosChanged( int index )
{
  // FIXME: Do we need to adjust something here with respect to
  // FreeBSD? It does not know Kerberos for mounting!

  KListView *view =            static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );
  KComboBox *custom_kerberos = static_cast<KComboBox *>( child( "CustomWriteAccess", "KComboBox", true ) );

  if ( view && view->selectedItem() && custom_kerberos )
  {
    if ( QString::compare( "-", custom_kerberos->text( index ) ) != 0 )
    {
      // Set the new value:
      view->selectedItem()->setText( Kerberos, custom_kerberos->text( index ) );

      emit customSettingsChanged();
    }
    else
    {
      // Reset the combo box:
      custom_kerberos->setCurrentText( view->selectedItem()->text( Kerberos ) );
    }
  }
  else
  {
    // Nothing to do.
  }
}


void Smb4KSambaOptions::slotCustomUIDChanged( const QString &uid )
{
  KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

  if ( view && view->selectedItem() )
  {
    // Set the new value:
    view->selectedItem()->setText( UID, uid );

    emit customSettingsChanged();
  }
  else
  {
    // Nothing to do.
  }
}


void Smb4KSambaOptions::slotCustomGIDChanged( const QString &gid )
{
  KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

  if ( view && view->selectedItem() )
  {
    // Set the new value:
    view->selectedItem()->setText( GID, gid );

    emit customSettingsChanged();
  }
  else
  {
    // Nothing to do.
  }
}


void Smb4KSambaOptions::slotCustomPortChanged( int port )
{
  KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

  if ( view && view->selectedItem() )
  {
    view->selectedItem()->setText( Port, QString( "%1" ).arg( port ) );

    emit customSettingsChanged();
  }
  else
  {
    // Nothing to do.
  }
}


void Smb4KSambaOptions::slotRemoveCustomOption()
{
  KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

  if ( view && view->selectedItem() )
  {
    delete view->currentItem();

    slotCustomItemClicked( NULL );

    emit customSettingsChanged();
  }
  else
  {
    // Nothing to do.
  }
}


void Smb4KSambaOptions::slotRemoveAllCustomOptions()
{
  KListView *view = static_cast<KListView *>( child( "CustomOptionsList", "KListView", true ) );

  if ( view )
  {
    while ( view->firstChild() )
    {
      delete view->firstChild();
    }

    slotCustomItemClicked( NULL );

    emit customSettingsChanged();
  }
  else
  {
    // Nothing to do.
  }
}

#include "smb4ksambaoptions.moc"
