/**************************************************************************
 * Configuration file handling functions.
 *
 * Copyright (C) 2000, Matthew Palmer.  Released under the GPL version 2.
 * You should have received a copy of this licence with this software, if not
 * see http://www.fsf.org/copyleft/gpl.html for a full copy of the licence.
 */

#include <pthread.h>

#ifndef SNUI_FILECONF_H
#define SNUI_FILECONF_H

/************\
* Data types *
\************/
struct conf_item {
	char *tag;
	char *data;
};

typedef struct conf_item_ll_s {
	struct conf_item *item;
	struct conf_item_ll_s *nextitem;
} conf_item_ll_t;

typedef struct conf_section_ll_s {
	char *sectname;
	conf_item_ll_t *headitem;
	conf_item_ll_t *curitem;
	struct conf_section_ll_s *nextsect;
} conf_section_ll_t;

typedef struct {
	conf_section_ll_t *headsect;
	conf_section_ll_t *cursect;
	pthread_mutex_t *mutex;
} conf_file_t;

/***********\
* Functions *
\***********/

/* Read filename for config options.  Returns NULL on error (with errno) or
 * a pointer to the config info on success.
 */
conf_file_t *conf_read_file(char *filename);

/* Delete all the data associated with the config info given.
 */
void conf_free_file(conf_file_t *conf);

/* Returns the name of the next section in the config file (which we are now
 * referring to for all our config needs) or NULL if there are no more config
 * sections.
 */
char *conf_next_section(conf_file_t *conffile);

/* Find a config section called sectname, and change the config record to
 * use that section.  Returns 1 if we found it, or 0 if not.
 */
int conf_find_section(conf_file_t *conffile, char *sectname);

/* Move to the nth section in the config file.  Returns 1 if success, or 0
 * if there aren't that many sections in the config file.
 */
int conf_seek_section(conf_file_t *conffile, int n);

/* Returns a pointer to a struct containing the tag and value of the next
 * config item in the currect section, or NULL if there are no more items in
 * the current section.
 */
struct conf_item *conf_next_item(conf_file_t *conffile);

/* Returns the value of the given tag in the current section, or NULL if no
 * such tag exists in this section.
 */
char *conf_find_item(conf_file_t *conffile, char *tag);

/* Moves the internal pointer to the nth config item in the current section.
 * Returns 1 on success or 0 if there are less than n items in the section.
 */
int conf_seek_item(conf_file_t *conffile, int n);


#endif  /* MARKET_FILECONF_H */
