/**************************************************************************
 * Network related functions for SNUI.
 *
 * Copyright (C) 2002, Matthew Palmer.  Released under the GPL version 2.
 * You should have received a copy of this licence with this software, if not
 * see http://www.fsf.org/copyleft/gpl.html for a full copy of the licence.
 */

static char rcsid[] = "$Id$";
 
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <netdb.h>
#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "fileconf.h"
#include "net.h"

int net_connect(char *, unsigned short);

/* Initialise a network connection using the data in the given section of
 * the default config file.  Places the FD of the socket of interest in
 * server_socket.  Returns errno.
 */
int net_client_init(char *section, FILE **server_socket)
{
	conf_file_t *cfile;
	char *server_addr, *server_port;
	int sock;
	
	cfile = conf_read_file(CONFIG_FILENAME);
	if (!cfile) {
		syslog(LOG_ERR, "Failed to open config file %s: %m", CONFIG_FILENAME);
		return errno;
	}
	
	if (!conf_find_section(cfile, section)) {
		syslog(LOG_ERR, "No section named %s in %s - aborting", section, CONFIG_FILENAME);
		conf_free_file(cfile);
		return errno;
	}

	server_addr = conf_find_item(cfile, "server_host");
	syslog(LOG_DEBUG, "Server is at [%s]", server_addr);
	server_port = conf_find_item(cfile, "server_port");
	
	if (!server_addr) {
		syslog(LOG_ERR, "Failed to find config item server_host - aborting");
		conf_free_file(cfile);
		return errno;
	}
	if (!server_port) {
		syslog(LOG_ERR, "Failed to find config item server_port - aborting");
		conf_free_file(cfile);
		return errno;
	}

	
	sock = net_connect(server_addr, atoi(server_port));
	conf_free_file(cfile);
	if (sock < 0) {
		return errno;
	}
	*server_socket = fdopen(sock, "a+");
	if (!(*server_socket)) {
		return errno;
	}

	return 0;
}

/* Sets up the networking system for a network server, placing the fd of a
 * socket suitable for accept() or select() to work on in srvsock.  Returns
 * errno on error or 0 on OK.
 */
int net_server_init(int port, int *srvsock)
{
	char localname[257];
	int sock;
	struct sockaddr_in saddr;
	struct hostent *host;
	
	if (gethostname(localname, 256) < 0) {
		syslog(LOG_ERR, "gethostname() failed");
		return errno;
	}

	if ((host = gethostbyname(localname)) == NULL) {
		syslog(LOG_ERR, "gethostbyname(%s) failed: %m", localname);
		return errno;
	}
	
	bzero(&saddr, sizeof(struct sockaddr_in));
	saddr.sin_family = host->h_addrtype;
	saddr.sin_port = htons(port);
	
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		syslog(LOG_ERR, "socket() failed");
		return errno;
	}
	
	if (bind(sock, (struct sockaddr *)&saddr, sizeof(saddr)) < 0) {
		syslog(LOG_ERR, "bind() failed");
		return errno;
	}
	
	if (listen(sock, 200) < 0) {
		syslog(LOG_ERR, "listen() failed");
		return errno;
	}
	
	*srvsock = sock;
	return 0;
}

/* Places a connected socket in newsock, and returns errno on error or 0 on OK.
 */
int net_get_connection(int sock, int port, int *newsock)
{
	char localname[257];
	int s, addrlen;
	struct sockaddr_in saddr;
	struct hostent *host;
	
	gethostname(localname, 256);
	if ((host = gethostbyname(localname)) == NULL) {
		syslog(LOG_ERR, "gethostbyname() failed");
		return errno;
	}
	
	bzero(&saddr, sizeof(struct sockaddr_in));
	saddr.sin_family = host->h_addrtype;
	saddr.sin_port = htons(port);

	addrlen = sizeof(struct sockaddr_in);
	if ((s = accept(sock, (struct sockaddr *)&saddr, &addrlen)) < 0) {
		syslog(LOG_ERR, "accept failed");
		return errno;
	}
	
	*newsock = s;
	return 0;
}
/* Read a line from sock, parsing it into 3 parts separated by whitespace.
 */
int net_read_response(FILE *sock, int *numresp, char *data, int datalen)
{
	char s[1024];
	int rv;
	
	if (!fgets(s, 1024, sock)) {
		return errno;
	}
	if (s[strlen(s)-1] == '\n') {
		s[strlen(s)-1] = '\0';
	}
	syslog(LOG_DEBUG, "Got line [%s]", s);
		
	s[3] = '\0';
	*numresp = atoi(s);
	strncpy(data, &s[4], datalen);
	/* Terminate if needed */
	if (strlen(&s[4]) > datalen) {
		data[datalen-1] = '\0';
	}

	return 0;
}

int net_connect(char *host, unsigned short port)
{
	struct hostent *he;
	struct sockaddr_in sa;
	char s[256];
	int rv, sock;
		
	he = NULL;
	while(!he) {
		he = gethostbyname(host);
		if (!he) {
			if (h_errno != TRY_AGAIN) {
				return -1;
			}
		}
	}

	memset(&sa, 0, sizeof(sa));
	sa.sin_family = AF_INET;
	sa.sin_port = htons(port);
	sa.sin_addr.s_addr = *((unsigned long *) he->h_addr);
	sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock < 0) {
		return -1;
	}

	if (connect(sock, (struct sockaddr *) &sa, sizeof(sa)) < 0) {
		close(sock);
		return -1;
	}

	return sock;
}
