#include <grp.h>
#include <errno.h>
#include <nss.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <stdio.h>

#include <pthread.h>

#include "../lib/lib.h"

static int copy_group(struct group *, char *, int, char *);
static int member_count(char *);

/* Global state */
static unsigned int grouparray_idx;
static unsigned int grouparray_count;

static char **grouparray_list;

static pthread_mutex_t grouparray_mutex;

enum nss_status _nss_snui_setgrent_r (void)
{
	FILE *sock;
	int rv, numresp, i;
	char response[1024];
	char **new_list;
	
	pthread_mutex_lock(&grouparray_mutex);
	
	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		pthread_mutex_unlock(&grouparray_mutex);
		return NSS_STATUS_UNAVAIL;
	}
	
	fprintf(sock, "group list\n");
	
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		pthread_mutex_unlock(&grouparray_mutex);
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (numresp != 300) {
		syslog(LOG_ERR, "Error while asking for list: %s", response);
		pthread_mutex_unlock(&grouparray_mutex);
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	/* OK, now we can read lines until the cows come home */
	grouparray_count = grouparray_idx = 0;	
	fgets(response, 1024, sock);
	while (strlen(response) != 0) {
		grouparray_count++;
		new_list = realloc(grouparray_list, grouparray_count);
		if (!new_list) {
			/* Memory limit reached */
			for (i = 0; i < grouparray_count - 1; i++) {
				free(grouparray_list[i]);
			}
			free(grouparray_list);
			grouparray_list = NULL;
			syslog(LOG_CRIT, "Out of memory error");
			errno = ENOMEM;
			pthread_mutex_unlock(&grouparray_mutex);
			fclose(sock);
			return NSS_STATUS_TRYAGAIN;
		}
		grouparray_list = new_list;
		grouparray_list[grouparray_count-1] = malloc(strlen(response) + 1);
		if (!grouparray_list[grouparray_count-1]) {
			/* Memory limit again */
			for (i = 0; i < grouparray_count - 1; i++) {
				free(grouparray_list[i]);
			}
			free(grouparray_list);
			grouparray_list = NULL;
			syslog(LOG_CRIT, "Out of memory error");
			errno = ENOMEM;
			pthread_mutex_unlock(&grouparray_mutex);
			fclose(sock);
			return NSS_STATUS_TRYAGAIN;
		}
		
		strcpy(grouparray_list[grouparray_count-1], response);
	}

	pthread_mutex_unlock(&grouparray_mutex);
	fclose(sock);
	
	return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_snui_endgrent_r(void)
{
	int i;
	
	pthread_mutex_lock(&grouparray_mutex);
	
	for (i = 0; i < grouparray_count; i++) {
		free(grouparray_list[i]);
	}
	free(grouparray_list);
	grouparray_list = 0;
	grouparray_count = 0;
	grouparray_idx = 0;

	pthread_mutex_unlock(&grouparray_mutex);

	return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_snui_getgrent_r(struct group *result_buf, char *buffer, int buflen, int *errnop)
{
	int rv;
	enum nss_status retval;
	
	if (grouparray_list == NULL) {
		return NSS_STATUS_UNAVAIL;
	}
	
	pthread_mutex_lock(&grouparray_mutex);
	
	if (grouparray_idx >= grouparray_count) {
		retval = NSS_STATUS_NOTFOUND;
	} else {
		rv = copy_group(result_buf, buffer, buflen, grouparray_list[grouparray_idx]);
		if (!rv) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else if (rv == -1) {
			retval = NSS_STATUS_UNAVAIL;
		} else {
			grouparray_idx++;
			retval = NSS_STATUS_SUCCESS;
		}
	}
	
	pthread_mutex_unlock(&grouparray_mutex);
	
	return retval;
}

enum nss_status _nss_snui_getgruid_r (
			gid_t gid,
			struct group *result_buf,
			char *data_buf,
			size_t buflen,
			int *errnop)
{
	FILE *sock;
	int rv, numresp;
	char response[1024];
	enum nss_status retval;
	
	syslog(LOG_DEBUG, "Looking for GID %i", gid);

	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		return NSS_STATUS_UNAVAIL;
	}
	
	fprintf(sock, "group id %i\n", gid);
	
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (numresp == 500) {
		syslog(LOG_ERR, "Error while searching for GID %i: %s", gid, response);
		retval = NSS_STATUS_UNAVAIL;
	} else if (numresp == 400) {
		retval = NSS_STATUS_NOTFOUND;
	} else if (numresp == 200) {
		rv = copy_group(result_buf, data_buf, buflen, response);
		if (rv < 0) {
			retval = NSS_STATUS_UNAVAIL;
		} else if (rv == 0) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else {
			retval = NSS_STATUS_SUCCESS;
		}
	} else {
		syslog(LOG_WARNING, "Unknown response from server: %i %s", numresp, response);
		retval = NSS_STATUS_UNAVAIL;
	}
	
	fclose(sock);
	
	return retval;
}

enum nss_status _nss_snui_getgrnam_r(
			char *name,
			struct group *result_buf,
			char *data_buf,
			size_t buflen,
			int *errnop)
{
	FILE *sock;
	int rv, numresp;
	char response[1024];
	enum nss_status retval;
	
	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		return NSS_STATUS_UNAVAIL;
	}
	
	if (strlen(name) > 1000) {
		syslog(LOG_ERR, "Name given to search for is *way* too long (%i chars)", strlen(name));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	fprintf(sock, "group name %s\n", name);
	
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (numresp == 500) {
		syslog(LOG_ERR, "Error while searching for name %s: %s", name, response);
		retval = NSS_STATUS_UNAVAIL;
	} else if (numresp == 400) {
		retval = NSS_STATUS_NOTFOUND;
	} else if (numresp == 200) {
		rv = copy_group(result_buf, data_buf, buflen, response);
		if (rv < 0) {
			retval = NSS_STATUS_UNAVAIL;
		} else if (rv == 0) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else {
			retval = NSS_STATUS_SUCCESS;
		}
	} else {
		syslog(LOG_WARNING, "Unknown response from server: %i %s", numresp, response);
		retval = NSS_STATUS_UNAVAIL;
	}
	
	fclose(sock);
	
	return retval;


}

/* Parse a group line and put it into the group struct.  Return 0 on
 * insufficient space, -1 on parse error, and 1 on successful completion.
 */
static int copy_group(struct group *dest, char *destbuf, int destbuflen, char *src)
{
	char *p;
	int count, i;
	
	syslog(LOG_DEBUG, "Copying group [%s]", src);
	count = member_count(src);
	if (count < 0) {
		/* Parse error */
		return -1;
	}

	if (strlen(src) + (count+1)*sizeof(char *) + 1 > destbuflen) {
		return 0;
	}
	strcpy(destbuf, src);
	dest->gr_name = destbuf;
			
	p = index(destbuf, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	
	p++;

	dest->gr_passwd = p;
	p = index(dest->gr_passwd, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	p++;

	dest->gr_gid = atoi(p);
	p = index(p, ':');
	p++;
	
	/* Members time.  This is a perverse way to do things - we keep the
	 * array of pointers in the databuf immediately beyond the string
	 * containing the member list; these point to strings in the databuf
	 * which have been parsed out previously.
	 * Kids, don't try this at home.
	 */
	dest->gr_mem = (char **) (destbuf + strlen(src) + 1);
	for (i = 0; i < count; i++) {
		dest->gr_mem[i] = p;
		p = index(p, ',');
		if (p) {
			*p = '\0';
			p++;
		}
	}
	dest->gr_mem[i] = NULL;
	
	return 1;
}

static int member_count(char *group_data)
{
	char *p;
	int count = 1;
	
	p = group_data;
	p = index(p, ':');
	if (!p) {
		return 0;
	}
	p++;
	p = index(p, ':');
	if (!p) {
		return 0;
	}
	p++;
	p = index(p, ':');
	if (!p) {
		return 0;
	}
	p++;
	if (!*p) {
		/* No members data */
		return 0;
	}
	while (*p) {
		if (*p == ',') {
			count++;
		}
		p++;
	}
	
	return count;
}
